/*
 * Copyright (c) 2015, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2018 KYOCERA Corporation
 * (C) 2019 KYOCERA Corporation
 */

#include <linux/kernel.h>
#include <asm/mach/map.h>
#include <asm/mach/arch.h>
#include "board-dt.h"
#include <soc/qcom/smem.h>

static const char *mdm9607_dt_match[] __initconst = {
	"qcom,mdm9607",
	NULL
};

#define KCC_SMEM_OK                         0
#define KCC_SMEM_NG                        -1
#define KCC_SMEM_BIT_16_MAX_BYTE            0x1FFF
#define KCC_SMEM_CRC_SIZE                   (uint32_t)1      /* CRC Size = 4Byte */

/* The CRC table size is based on how many bits at a time we are going
   to process through the table.  Given that we are processing the data
   8 bits at a time, this gives us 2^8 (256) entries. */
#define KCC_SMEM_CRC_TAB_SIZE               256    /* 2^CRC_TAB_BITS */

/* Most significant bitmask for a CRC. */
#define KCC_SMEM_CRC_32_MSB_MASK            0x80000000UL

/* Mask for a 32-bit CRC polynomial:
  x^32+x^26+x^23+x^22+x^16+x^12+x^11+x^10+x^8+x^7+x^5+x^4+x^2+x^1+1

  note1hang The x^32 tap is left off; it is implicit. */
#define KCC_SMEM_CRC_32_POLYNOMIAL          0x04c11db7UL

/* DATA((KCC_SMEM_xxx_SIZE + 3) / sizeof(unsigned int)) + CRC */
typedef struct {
	/* SMEM area addition */
	unsigned int vgm_flash_address_map[ (((KCC_SMEM_VGM_FLASH_ADDRESS_MAP_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int sum_sleep_ram_info[ (((KCC_SMEM_SUM_SLEEP_RAM_INFO_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int sum_sleep_rom_info[ (((KCC_SMEM_SUM_SLEEP_ROM_INFO_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int memory_reserved[ (((KCC_SMEM_MEMORY_RESERVED_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int status_table[ (((KCC_SMEM_STATUS_TABLE_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int modem_write_status[ (((KCC_SMEM_MODEM_WRITE_STATUS_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int linux_write_status[ (((KCC_SMEM_LINUX_WRITE_STATUS_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int factory_options[ (((KCC_SMEM_FACTORY_OPTIONS_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int kc_diag_mode[ (((KCC_SMEM_KC_DIAG_MODE_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int kc_diag_ethr_key[ (((KCC_SMEM_KC_DIAG_ETHR_KEY_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int hw_id[ (((KCC_SMEM_HW_ID_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int vendor_id[ (((KCC_SMEM_VENDOR_ID_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int crash_log[ (((KCC_SMEM_CRASH_LOG_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int ethernet_macaddress[ (((KCC_SMEM_ETHERNET_MACADDRESS_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int fota_boot_type[ (((KCC_SMEM_FOTA_BOOT_TYPE_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int usb_developers_func_disable[ (((KCC_SMEM_USB_DEVELOPERS_FUNC_DISABLE_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int usb_configuration[ (((KCC_SMEM_USB_CONFIGURATION_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int usb_fullspeed_fixed[ (((KCC_SMEM_USB_FULLSPEED_FIXED_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int at_kcdiag_esim_chk[ (((KCC_SMEM_AT_KCDIAG_ESIM_CHK_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int at_kcdiag_rf_init_chk[ (((KCC_SMEM_AT_KCDIAG_RF_INIT_CHK_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int at_kcdiag_audio_codec_ic_chk[ (((KCC_SMEM_AT_KCDIAG_AUDIO_CODEC_IC_CHK_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int at_kcdiag_sleep_sum_check_chk[ (((KCC_SMEM_AT_KCDIAG_SLEEP_SUM_CHECK_CHK_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_reservation_1[ (((KCC_SMEM_AT_RESERVATION_1_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_reservation_2[ (((KCC_SMEM_AT_RESERVATION_2_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_reservation_3[ (((KCC_SMEM_AT_RESERVATION_3_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_kcdiag_reservation_chk_1[ (((KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_1_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_kcdiag_reservation_chk_2[ (((KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_2_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_kcdiag_reservation_chk_3[ (((KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_3_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_kcdiag_reservation_chk_4[ (((KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_4_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_kcdiag_reservation_chk_5[ (((KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_5_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int kc_diag_imei_sec_key[ (((KCC_SMEM_KC_DIAG_IMEI_SEC_KEY_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int ds_userapp1_disconnect_cause[ (((KCC_SMEM_DS_USERAPP1_DISCONNECT_CAUSE_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int ds_userapp2_disconnect_cause[ (((KCC_SMEM_DS_USERAPP2_DISCONNECT_CAUSE_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int ecall_callback_data1[ (((KCC_SMEM_ECALL_CALLBACK_DATA1_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int ecall_callback_data2[ (((KCC_SMEM_ECALL_CALLBACK_DATA2_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
    unsigned int exclusive_lock_status[ (((KCC_SMEM_EXCLUSIVE_LOCK_STATUS_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int atcmdif_debug_log[ (((KCC_SMEM_ATCMDIF_DEBUG_LOG_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int mcfg_carrier_info[ (((KCC_SMEM_MCFG_CARRIER_INFO_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int kc_power_on_status_info[ (((KCC_SMEM_KC_POWER_ON_STATUS_INFO_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int systime_status[ (((KCC_SMEM_SYSTIME_STATUS_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int nand_write_status_modem[ (((KCC_SMEM_NAND_WRITE_STATUS_MODEM_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int nand_write_status_linux[ (((KCC_SMEM_NAND_WRITE_STATUS_LINUX_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int usb_usb_dtroff[ (((KCC_SMEM_USB_USB_DTROFF_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int rcc_reset_chk[ (((KCC_SMEM_RCC_RESET_CHK_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int fota_update_log[ (((KCC_SMEM_FOTA_UPDATE_LOG_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int lsm_developers_func_disable[ (((KCC_SMEM_LSM_DEVELOPERS_FUNC_DISABLE_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int detect_resin_int[ (((KCC_SMEM_DETECT_RESIN_INT_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int area_id[ (((KCC_SMEM_AREA_ID_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
	unsigned int at_kcecallmsdmods_set_msd[ (((KCC_SMEM_AT_KCECALLMSDMODS_SET_MSD_SIZE + (uint32_t)3) / sizeof(unsigned int)) + KCC_SMEM_CRC_SIZE) ];
}type_kcc_smem;

static type_kcc_smem  *kc_smem_addr  = NULL;

/* CRC table for 32 bit CRC with generator polynomial 0x04c11db7,
   calculated 8 bits at a time, MSB first. */
const uint32_t smem_crc32_table[ KCC_SMEM_CRC_TAB_SIZE ] = {  
/* PRQA S:m3cm_A4_500_0006 1257 A4_500_0006 */
	0x00000000UL,  0x04c11db7UL,  0x09823b6eUL,  0x0d4326d9UL,
	0x130476dcUL,  0x17c56b6bUL,  0x1a864db2UL,  0x1e475005UL,
	0x2608edb8UL,  0x22c9f00fUL,  0x2f8ad6d6UL,  0x2b4bcb61UL,
	0x350c9b64UL,  0x31cd86d3UL,  0x3c8ea00aUL,  0x384fbdbdUL,
	0x4c11db70UL,  0x48d0c6c7UL,  0x4593e01eUL,  0x4152fda9UL,
	0x5f15adacUL,  0x5bd4b01bUL,  0x569796c2UL,  0x52568b75UL,
	0x6a1936c8UL,  0x6ed82b7fUL,  0x639b0da6UL,  0x675a1011UL,
	0x791d4014UL,  0x7ddc5da3UL,  0x709f7b7aUL,  0x745e66cdUL,
	0x9823b6e0UL,  0x9ce2ab57UL,  0x91a18d8eUL,  0x95609039UL,
	0x8b27c03cUL,  0x8fe6dd8bUL,  0x82a5fb52UL,  0x8664e6e5UL,
	0xbe2b5b58UL,  0xbaea46efUL,  0xb7a96036UL,  0xb3687d81UL,
	0xad2f2d84UL,  0xa9ee3033UL,  0xa4ad16eaUL,  0xa06c0b5dUL,
	0xd4326d90UL,  0xd0f37027UL,  0xddb056feUL,  0xd9714b49UL,
	0xc7361b4cUL,  0xc3f706fbUL,  0xceb42022UL,  0xca753d95UL,
	0xf23a8028UL,  0xf6fb9d9fUL,  0xfbb8bb46UL,  0xff79a6f1UL,
	0xe13ef6f4UL,  0xe5ffeb43UL,  0xe8bccd9aUL,  0xec7dd02dUL,
	0x34867077UL,  0x30476dc0UL,  0x3d044b19UL,  0x39c556aeUL,
	0x278206abUL,  0x23431b1cUL,  0x2e003dc5UL,  0x2ac12072UL,
	0x128e9dcfUL,  0x164f8078UL,  0x1b0ca6a1UL,  0x1fcdbb16UL,
	0x018aeb13UL,  0x054bf6a4UL,  0x0808d07dUL,  0x0cc9cdcaUL,
	0x7897ab07UL,  0x7c56b6b0UL,  0x71159069UL,  0x75d48ddeUL,
	0x6b93dddbUL,  0x6f52c06cUL,  0x6211e6b5UL,  0x66d0fb02UL,
	0x5e9f46bfUL,  0x5a5e5b08UL,  0x571d7dd1UL,  0x53dc6066UL,
	0x4d9b3063UL,  0x495a2dd4UL,  0x44190b0dUL,  0x40d816baUL,
	0xaca5c697UL,  0xa864db20UL,  0xa527fdf9UL,  0xa1e6e04eUL,
	0xbfa1b04bUL,  0xbb60adfcUL,  0xb6238b25UL,  0xb2e29692UL,
	0x8aad2b2fUL,  0x8e6c3698UL,  0x832f1041UL,  0x87ee0df6UL,
	0x99a95df3UL,  0x9d684044UL,  0x902b669dUL,  0x94ea7b2aUL,
	0xe0b41de7UL,  0xe4750050UL,  0xe9362689UL,  0xedf73b3eUL,
	0xf3b06b3bUL,  0xf771768cUL,  0xfa325055UL,  0xfef34de2UL,
	0xc6bcf05fUL,  0xc27dede8UL,  0xcf3ecb31UL,  0xcbffd686UL,
	0xd5b88683UL,  0xd1799b34UL,  0xdc3abdedUL,  0xd8fba05aUL,
	0x690ce0eeUL,  0x6dcdfd59UL,  0x608edb80UL,  0x644fc637UL,
	0x7a089632UL,  0x7ec98b85UL,  0x738aad5cUL,  0x774bb0ebUL,
	0x4f040d56UL,  0x4bc510e1UL,  0x46863638UL,  0x42472b8fUL,
	0x5c007b8aUL,  0x58c1663dUL,  0x558240e4UL,  0x51435d53UL,
	0x251d3b9eUL,  0x21dc2629UL,  0x2c9f00f0UL,  0x285e1d47UL,
	0x36194d42UL,  0x32d850f5UL,  0x3f9b762cUL,  0x3b5a6b9bUL,
	0x0315d626UL,  0x07d4cb91UL,  0x0a97ed48UL,  0x0e56f0ffUL,
	0x1011a0faUL,  0x14d0bd4dUL,  0x19939b94UL,  0x1d528623UL,
	0xf12f560eUL,  0xf5ee4bb9UL,  0xf8ad6d60UL,  0xfc6c70d7UL,
	0xe22b20d2UL,  0xe6ea3d65UL,  0xeba91bbcUL,  0xef68060bUL,
	0xd727bbb6UL,  0xd3e6a601UL,  0xdea580d8UL,  0xda649d6fUL,
	0xc423cd6aUL,  0xc0e2d0ddUL,  0xcda1f604UL,  0xc960ebb3UL,
	0xbd3e8d7eUL,  0xb9ff90c9UL,  0xb4bcb610UL,  0xb07daba7UL,
	0xae3afba2UL,  0xaafbe615UL,  0xa7b8c0ccUL,  0xa379dd7bUL,
	0x9b3660c6UL,  0x9ff77d71UL,  0x92b45ba8UL,  0x9675461fUL,
	0x8832161aUL,  0x8cf30badUL,  0x81b02d74UL,  0x857130c3UL,
	0x5d8a9099UL,  0x594b8d2eUL,  0x5408abf7UL,  0x50c9b640UL,
	0x4e8ee645UL,  0x4a4ffbf2UL,  0x470cdd2bUL,  0x43cdc09cUL,
	0x7b827d21UL,  0x7f436096UL,  0x7200464fUL,  0x76c15bf8UL,
	0x68860bfdUL,  0x6c47164aUL,  0x61043093UL,  0x65c52d24UL,
	0x119b4be9UL,  0x155a565eUL,  0x18197087UL,  0x1cd86d30UL,
	0x029f3d35UL,  0x065e2082UL,  0x0b1d065bUL,  0x0fdc1becUL,
	0x3793a651UL,  0x3352bbe6UL,  0x3e119d3fUL,  0x3ad08088UL,
	0x2497d08dUL,  0x2056cd3aUL,  0x2d15ebe3UL,  0x29d4f654UL,
	0xc5a92679UL,  0xc1683bceUL,  0xcc2b1d17UL,  0xc8ea00a0UL,
	0xd6ad50a5UL,  0xd26c4d12UL,  0xdf2f6bcbUL,  0xdbee767cUL,
	0xe3a1cbc1UL,  0xe760d676UL,  0xea23f0afUL,  0xeee2ed18UL,
	0xf0a5bd1dUL,  0xf464a0aaUL,  0xf9278673UL,  0xfde69bc4UL,
	0x89b8fd09UL,  0x8d79e0beUL,  0x803ac667UL,  0x84fbdbd0UL,
	0x9abc8bd5UL,  0x9e7d9662UL,  0x933eb0bbUL,  0x97ffad0cUL,
	0xafb010b1UL,  0xab710d06UL,  0xa6322bdfUL,  0xa2f33668UL,
	0xbcb4666dUL,  0xb8757bdaUL,  0xb5365d03UL,  0xb1f740b4UL
};
/*PRQA L:A4_500_0006*/

static void __init mdm9607_init(void)
{
	board_dt_populate(NULL);
}

static uint32_t kc_smem_crc_32_calc
(
	/* Pointer to data over which to compute CRC */
	uint8_t  *buf_ptr,

	/* Number of bits over which to compute CRC */
	uint16_t  len,

	/* Seed for CRC computation */
	uint32_t  seed
)
{
	/* Input data buffer */
	uint32_t  data;

	/* Working CRC buffer */
	uint32_t  crc;
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

	/*-------------------------------------------------------------------------
	 Generate the CRC by looking up the transformation in a table and
	 XOR-ing it into the CRC, one byte at a time.
	-------------------------------------------------------------------------*/
	for (crc = seed; len >= 8; len -= 8, buf_ptr++)
	{
		crc = smem_crc32_table[ ( ( crc >> 24 ) ^ ( *buf_ptr ) ) ] ^ ( crc << 8 );
	}

	/*-------------------------------------------------------------------------
	 Finish calculating the CRC over the trailing data bits. This is done by
	 aligning the remaining data bits with the CRC MSB, and then computing the
	 CRC one bit at a time.
	-------------------------------------------------------------------------*/
	if (len != 0)
	{
		/*-----------------------------------------------------------------------
		 Align data MSB with CRC MSB.
		-----------------------------------------------------------------------*/
		data = ( (uint32_t)( *buf_ptr ) ) << 24;

		/*-----------------------------------------------------------------------
		 While there are bits left, compute CRC one bit at a time.
		-----------------------------------------------------------------------*/
		while (len-- != 0)
		{
			/*---------------------------------------------------------------------
			 If the MSB of the XOR combination of the data and CRC is one, then
			 advance the CRC register by one bit XOR with the CRC polynomial.
			 Otherwise, just advance the CRC by one bit.
			---------------------------------------------------------------------*/
			if (( ( crc ^ data ) & KCC_SMEM_CRC_32_MSB_MASK ) != 0)
			{
				crc <<= 1;
				crc ^= KCC_SMEM_CRC_32_POLYNOMIAL;
			}
			else
			{
				crc <<= 1;
			}

			/*---------------------------------------------------------------------
			 Advance to the next input data bit and continue.
			---------------------------------------------------------------------*/
			data <<= 1;
		}
	}

	/*-------------------------------------------------------------------------
	 Return the resulting CRC value.
	-------------------------------------------------------------------------*/
	return( crc );

}

static void kc_smem_init( void )
{
	uint32_t smem_size = sizeof(type_kcc_smem);

	kc_smem_addr = smem_get_entry(SMEM_ID_VENDOR2, &smem_size, 0, SMEM_ANY_HOST_FLAG);
	if (kc_smem_addr == NULL)
	{
		printk(KERN_ERR
			"%s: kc_smem_init ret error=NULL\n",__func__);
	}

	return;
}

static void *kc_smem_alloc_internal(unsigned smem_type, unsigned buf_size, unsigned *data_max_size)
{
	if (kc_smem_addr == NULL)
	{
		kc_smem_init();
		if (kc_smem_addr == NULL)
		{
			return( NULL );
		}
	}

	switch ((short)smem_type)
	{
	case SMEM_VGM_FLASH_ADDRESS_MAP:
		if (buf_size <= KCC_SMEM_VGM_FLASH_ADDRESS_MAP_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->vgm_flash_address_map) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->vgm_flash_address_map[0] );
		}
		break;

	case SMEM_SUM_SLEEP_RAM_INFO:
		if (buf_size <= KCC_SMEM_SUM_SLEEP_RAM_INFO_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->sum_sleep_ram_info) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->sum_sleep_ram_info[0] );
		}
		break;

	case SMEM_SUM_SLEEP_ROM_INFO:
		if (buf_size <= KCC_SMEM_SUM_SLEEP_ROM_INFO_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->sum_sleep_rom_info) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->sum_sleep_rom_info[0] );
		}
		break;

	case SMEM_MEMORY_RESERVED:
		if (buf_size <= KCC_SMEM_MEMORY_RESERVED_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->memory_reserved) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->memory_reserved[0] );
		}
		break;

	case SMEM_STATUS_TABLE:
		if (buf_size <= KCC_SMEM_STATUS_TABLE_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->status_table) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->status_table[0] );
		}
		break;

	case SMEM_MODEM_WRITE_STATUS:
		if (buf_size <= KCC_SMEM_MODEM_WRITE_STATUS_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->modem_write_status) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->modem_write_status[0] );
		}
		break;

	case SMEM_LINUX_WRITE_STATUS:
		if (buf_size <= KCC_SMEM_LINUX_WRITE_STATUS_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->linux_write_status) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->linux_write_status[0] );
		}
		break;

	case SMEM_FACTORY_OPTIONS:
		if (buf_size <= KCC_SMEM_FACTORY_OPTIONS_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->factory_options) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->factory_options[0] );
		}
		break;

	case SMEM_KC_DIAG_MODE:
		if (buf_size <= KCC_SMEM_KC_DIAG_MODE_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->kc_diag_mode) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->kc_diag_mode[0] );
		}
		break;

	case SMEM_KC_DIAG_ETHR_KEY:
		if (buf_size <= KCC_SMEM_KC_DIAG_ETHR_KEY_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->kc_diag_ethr_key) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->kc_diag_ethr_key[0] );
		}
		break;

	case SMEM_HW_ID:
		if (buf_size <= KCC_SMEM_HW_ID_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->hw_id) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->hw_id[0] );
		}
		break;

	case SMEM_VENDOR_ID:
		if (buf_size <= KCC_SMEM_VENDOR_ID_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->vendor_id) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->vendor_id[0] );
		}
		break;

	case SMEM_CRASH_LOG:
		if (buf_size <= KCC_SMEM_CRASH_LOG_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->crash_log) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->crash_log[0] );
		}
		break;

	case SMEM_ETHERNET_MACADDRESS:
		if (buf_size <= KCC_SMEM_ETHERNET_MACADDRESS_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->ethernet_macaddress) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->ethernet_macaddress[0] );
		}
		break;

	case SMEM_FOTA_BOOT_TYPE:
		if (buf_size <= KCC_SMEM_FOTA_BOOT_TYPE_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->fota_boot_type) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->fota_boot_type[0] );
		}
		break;

	case SMEM_USB_DEVELOPERS_FUNC_DISABLE:
		if (buf_size <= KCC_SMEM_USB_DEVELOPERS_FUNC_DISABLE_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->usb_developers_func_disable) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->usb_developers_func_disable[0] );
		}
		break;

	case SMEM_USB_CONFIGURATION:
		if (buf_size <= KCC_SMEM_USB_CONFIGURATION_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->usb_configuration) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->usb_configuration[0] );
		}
		break;

	case SMEM_USB_FULLSPEED_FIXED:
		if (buf_size <= KCC_SMEM_USB_FULLSPEED_FIXED_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->usb_fullspeed_fixed) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->usb_fullspeed_fixed[0] );
		}
		break;

    case SMEM_AT_KCDIAG_ESIM_CHK:
        if (buf_size <= KCC_SMEM_AT_KCDIAG_ESIM_CHK_SIZE)
        {
            *data_max_size = (sizeof(kc_smem_addr->at_kcdiag_esim_chk) - sizeof(unsigned int));
            return( (void *)&kc_smem_addr->at_kcdiag_esim_chk[0] );
        }
        break;

    case SMEM_AT_KCDIAG_RF_INIT_CHK:
        if (buf_size <= KCC_SMEM_AT_KCDIAG_RF_INIT_CHK_SIZE)
        {
            *data_max_size = (sizeof(kc_smem_addr->at_kcdiag_rf_init_chk) - sizeof(unsigned int));
            return( (void *)&kc_smem_addr->at_kcdiag_rf_init_chk[0] );
        }
        break;

    case SMEM_AT_KCDIAG_AUDIO_CODEC_IC_CHK:
        if (buf_size <= KCC_SMEM_AT_KCDIAG_AUDIO_CODEC_IC_CHK_SIZE)
        {
            *data_max_size = (sizeof(kc_smem_addr->at_kcdiag_audio_codec_ic_chk) - sizeof(unsigned int));
            return( (void *)&kc_smem_addr->at_kcdiag_audio_codec_ic_chk[0] );
        }
        break;

    case SMEM_AT_KCDIAG_SLEEP_SUM_CHECK_CHK:
        if (buf_size <= KCC_SMEM_AT_KCDIAG_SLEEP_SUM_CHECK_CHK_SIZE)
        {
            *data_max_size = (sizeof(kc_smem_addr->at_kcdiag_sleep_sum_check_chk) - sizeof(unsigned int));
            return( (void *)&kc_smem_addr->at_kcdiag_sleep_sum_check_chk[0] );
        }
        break;

	case SMEM_AT_RESERVATION_1:
		if (buf_size <= KCC_SMEM_AT_RESERVATION_1_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_reservation_1) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_reservation_1[0] );
		}
		break;

	case SMEM_AT_RESERVATION_2:
		if (buf_size <= KCC_SMEM_AT_RESERVATION_2_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_reservation_2) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_reservation_2[0] );
		}
		break;

	case SMEM_AT_RESERVATION_3:
		if (buf_size <= KCC_SMEM_AT_RESERVATION_3_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_reservation_3) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_reservation_3[0] );
		}
		break;

	case SMEM_AT_KCDIAG_RESERVATION_CHK_1:
		if (buf_size <= KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_1_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_kcdiag_reservation_chk_1) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_kcdiag_reservation_chk_1[0] );
		}
		break;

	case SMEM_AT_KCDIAG_RESERVATION_CHK_2:
		if (buf_size <= KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_2_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_kcdiag_reservation_chk_2) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_kcdiag_reservation_chk_2[0] );
		}
		break;

	case SMEM_AT_KCDIAG_RESERVATION_CHK_3:
		if (buf_size <= KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_3_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_kcdiag_reservation_chk_3) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_kcdiag_reservation_chk_3[0] );
		}
		break;

	case SMEM_AT_KCDIAG_RESERVATION_CHK_4:
		if (buf_size <= KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_4_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_kcdiag_reservation_chk_4) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_kcdiag_reservation_chk_4[0] );
		}
		break;

	case SMEM_AT_KCDIAG_RESERVATION_CHK_5:
		if (buf_size <= KCC_SMEM_AT_KCDIAG_RESERVATION_CHK_5_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_kcdiag_reservation_chk_5) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_kcdiag_reservation_chk_5[0] );
		}
		break;

	case SMEM_KC_DIAG_IMEI_SEC_KEY:
		if (buf_size <= KCC_SMEM_KC_DIAG_IMEI_SEC_KEY_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->kc_diag_imei_sec_key) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->kc_diag_imei_sec_key[0] );
		}
		break;

	case SMEM_DS_USERAPP1_DISCONNECT_CAUSE:
		if (buf_size <= KCC_SMEM_DS_USERAPP1_DISCONNECT_CAUSE_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->ds_userapp1_disconnect_cause) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->ds_userapp1_disconnect_cause[0] );
		}
		break;

	case SMEM_DS_USERAPP2_DISCONNECT_CAUSE:
		if (buf_size <= KCC_SMEM_DS_USERAPP2_DISCONNECT_CAUSE_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->ds_userapp2_disconnect_cause) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->ds_userapp2_disconnect_cause[0] );
		}
		break;

	case SMEM_ECALL_CALLBACK_DATA1:
		if (buf_size <= KCC_SMEM_ECALL_CALLBACK_DATA1_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->ecall_callback_data1) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->ecall_callback_data1[0] );
		}
		break;

	case SMEM_ECALL_CALLBACK_DATA2:
		if (buf_size <= KCC_SMEM_ECALL_CALLBACK_DATA2_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->ecall_callback_data2) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->ecall_callback_data2[0] );
		}
		break;

	case SMEM_EXCLUSIVE_LOCK_STATUS:
		if (buf_size <= KCC_SMEM_EXCLUSIVE_LOCK_STATUS_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->exclusive_lock_status) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->exclusive_lock_status[0] );
		}
		break;

	case SMEM_ATCMDIF_DEBUG_LOG:
		if (buf_size <= KCC_SMEM_ATCMDIF_DEBUG_LOG_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->atcmdif_debug_log) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->atcmdif_debug_log[0] );
		}
		break;

	case SMEM_MCFG_CARRIER_INFO:
		if (buf_size <= KCC_SMEM_MCFG_CARRIER_INFO_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->mcfg_carrier_info) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->mcfg_carrier_info[0] );
		}
		break;

	case SMEM_KC_POWER_ON_STATUS_INFO:
		if (buf_size <= KCC_SMEM_KC_POWER_ON_STATUS_INFO_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->kc_power_on_status_info) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->kc_power_on_status_info[0] );
		}
		break;

	case SMEM_SYSTIME_STATUS:
		if (buf_size <= KCC_SMEM_SYSTIME_STATUS_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->systime_status) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->systime_status[0] );
		}
		break;

	case SMEM_NAND_WRITE_STATUS_MODEM:
		if (buf_size <= KCC_SMEM_NAND_WRITE_STATUS_MODEM_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->nand_write_status_modem) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->nand_write_status_modem[0] );
		}
		break;

	case SMEM_NAND_WRITE_STATUS_LINUX:
		if (buf_size <= KCC_SMEM_NAND_WRITE_STATUS_LINUX_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->nand_write_status_linux) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->nand_write_status_linux[0] );
		}
		break;

	case SMEM_USB_USB_DTROFF:
		if (buf_size <= KCC_SMEM_USB_USB_DTROFF_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->usb_usb_dtroff) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->usb_usb_dtroff[0] );
		}
		break;

	case SMEM_RCC_RESET_CHK:
		if (buf_size <= KCC_SMEM_RCC_RESET_CHK_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->rcc_reset_chk) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->rcc_reset_chk[0] );
		}
		break;

	case SMEM_FOTA_UPDATE_LOG:
		if (buf_size <= KCC_SMEM_FOTA_UPDATE_LOG_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->fota_update_log) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->fota_update_log[0] );
		}
		break;

	case SMEM_LSM_DEVELOPERS_FUNC_DISABLE:
		if (buf_size <= KCC_SMEM_LSM_DEVELOPERS_FUNC_DISABLE_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->lsm_developers_func_disable) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->lsm_developers_func_disable[0] );
		}
		break;

	case SMEM_DETECT_RESIN_INT:
		if (buf_size <= KCC_SMEM_DETECT_RESIN_INT_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->detect_resin_int) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->detect_resin_int[0] );
		}
		break;

	case SMEM_AREA_ID:
		if (buf_size <= KCC_SMEM_AREA_ID_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->area_id) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->area_id[0] );
		}
		break;

	case SMEM_AT_KCECALLMSDMODS_SET_MSD:
		if (buf_size <= KCC_SMEM_AT_KCECALLMSDMODS_SET_MSD_SIZE)
		{
			*data_max_size = (sizeof(kc_smem_addr->at_kcecallmsdmods_set_msd) - sizeof(unsigned int));
			return( (void *)&kc_smem_addr->at_kcecallmsdmods_set_msd[0] );
		}
		break;

	default:
		break;
	}

	return( NULL );
}

void *kc_smem_alloc(unsigned smem_type, unsigned buf_size)
{
  uint8_t  *addr         = NULL;
  uint32_t data_max_size = 0;

  addr = (uint8_t*)kc_smem_alloc_internal(smem_type, buf_size, &data_max_size);
  return( (void *)addr );
}

void *kc_smem_alloc_get(unsigned smem_type, unsigned buf_size)
{
	uint8_t  *addr          = NULL;
	uint8_t  *calc_addr     = NULL;
	uint32_t crc32          = 0;
	uint32_t smem_crc       = 0;
	uint32_t data_calc_size = 0;

	/* SMEM Address get */
	addr = (uint8_t*)kc_smem_alloc_internal(smem_type, buf_size, &data_calc_size);
	if (addr == NULL)
	{
		return( NULL );
	}

	/* CRC get */
	(void) memcpy(&smem_crc, &addr[data_calc_size], sizeof(smem_crc));

	if (smem_crc != (uint32_t)0)
	{
		calc_addr = addr;

		/* CRC calculate */
		while (data_calc_size != (uint32_t)0)
		{
			if (data_calc_size > (uint32_t)KCC_SMEM_BIT_16_MAX_BYTE)
			{
				crc32 = kc_smem_crc_32_calc(calc_addr, ((uint16_t)KCC_SMEM_BIT_16_MAX_BYTE * (uint16_t)8), crc32);
				data_calc_size -= (uint32_t)KCC_SMEM_BIT_16_MAX_BYTE;
				calc_addr += KCC_SMEM_BIT_16_MAX_BYTE;
			}
			else
			{
				crc32 = kc_smem_crc_32_calc(calc_addr, (uint16_t)(data_calc_size * (uint16_t)8), crc32);
				data_calc_size = 0;
			}
		}

		/* CRC judge */
		if (smem_crc != crc32)
		{
			return( NULL );
		}
	}
	return( (void *)addr );
}

int32_t kc_smem_alloc_set(unsigned smem_type)
{
	uint8_t  *addr          = NULL;
	uint8_t  *calc_addr     = NULL;
	uint32_t crc32          = 0;
	uint32_t data_calc_size = 0;
    uint32_t data_max_size  = 0;

	/* SMEM Address get */
	addr = (uint8_t*)kc_smem_alloc_internal(smem_type, 0, &data_max_size);

	if (addr == NULL)
	{
		return KCC_SMEM_NG;
	}

	data_calc_size = data_max_size;
	calc_addr = addr;

	/* CRC calculate */
	while (data_calc_size != (uint32_t)0)
	{
		if (data_calc_size > (uint32_t)KCC_SMEM_BIT_16_MAX_BYTE)
		{
			crc32 = kc_smem_crc_32_calc(calc_addr, ((uint16_t)KCC_SMEM_BIT_16_MAX_BYTE * (uint16_t)8), crc32);
			data_calc_size -= (uint32_t)KCC_SMEM_BIT_16_MAX_BYTE;
			calc_addr += KCC_SMEM_BIT_16_MAX_BYTE;
		}
		else
		{
			crc32 = kc_smem_crc_32_calc(calc_addr, (uint16_t)(data_calc_size * (uint16_t)8), crc32);
			data_calc_size = 0;
		}
	}

	/* CRC set */
	(void) memcpy(&addr[data_max_size], &crc32, sizeof(crc32));

	return KCC_SMEM_OK;
}

DT_MACHINE_START(MDM9607_DT,
	"Qualcomm Technologies, Inc. MDM 9607 (Flattened Device Tree)")
	.init_machine	= mdm9607_init,
	.dt_compat	= mdm9607_dt_match,
MACHINE_END
