/*
 * pb_command.c
 *
 * this program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 */
/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2014 KYOCERA Corporation
 * (C) 2015 KYOCERA Corporation
 * (C) 2018 KYOCERA Corporation
 * (C) 2019 KYOCERA Corporation
 */
#ifdef __KERNEL__

#include <linux/init.h>
#include <linux/module.h>

#else /* !__KERNEL__ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#endif /* __KERNEL__ */

#include "pb_command.h"

#define PB_PREFIX	'#'
#define PB_POSTFIX	'$'
#define PB_PADCHAR	'z'
#define PB_COMMAND_POS	1
#define PB_COMMAND_LEN	4
#define PB_PARAM_POS	5
#define PB_PARAM_LEN	10

#define PB_USB_ACTIVITY_CMD			"USBA"

#define PB_ACTIVITY_ON_STRING	"ON_"
#define PB_ACTIVITY_OFF_STRING	"OFF"

typedef struct {
	pb_command_id_type_e id;
	const char *name;
} command_id_elem;

static const command_id_elem command_id_table[] = {
	{PB_USB_ACTIVITY_ID, PB_USB_ACTIVITY_CMD},
};

static const int COUNT_OF_PB_COMMAND_ID_TABLE =
		sizeof(command_id_table) / sizeof(command_id_elem);

typedef struct {
	pb_parm_id_type_e id;
	const char *name;
} parm_id_elem;

static const parm_id_elem parm_id_table[] = {
	{PB_ACTIVITY_ON			,	PB_ACTIVITY_ON_STRING		},
	{PB_ACTIVITY_OFF		,	PB_ACTIVITY_OFF_STRING		},
};

static const int COUNT_OF_PB_LINE_ID_TABLE =
			sizeof(parm_id_table) / sizeof(parm_id_elem);

static const char *
get_parameter_string(const pb_command_universal *parm, char *buf);
static const char *
get_parameter_string_usba(const pb_usb_activity_cmd *parm, char *buf);

static const char *command_id_to_name(const pb_command_id_type_e id);
static const char *parm_id_to_name(const pb_parm_id_type_e id);

pb_result_type_e pb_encode_command(const pb_command_universal *parm, char *cmd)
{
	int index = 0;
	const char *wrk;
	char buf[PB_PARAM_LEN + 1];

	if (parm == (const pb_command_universal *)NULL) {
		return PB_RESULT_NG;
	}

	wrk = command_id_to_name(parm->id);
	if (wrk == (const char *)NULL) {
		return PB_RESULT_NG;
	}

	*(cmd + index) = PB_PREFIX;
	index++;

	while (*wrk != '\0') {
		*(cmd + index) = *wrk;
		index++;
		wrk++;
	}

	wrk = get_parameter_string(parm, buf);
	if (wrk == (const char *)NULL) {
		return PB_RESULT_NG;
	}

	while (*wrk != '\0') {
		*(cmd + index) = *wrk;
		index++;
		wrk++;
	}

	for (; index < (PB_PARAM_POS + PB_PARAM_LEN); index++) {
		*(cmd + index) = PB_PADCHAR;
	}

	*(cmd + PB_COMMAND_ALL_LEN - 1) = PB_POSTFIX;
	*(cmd + PB_COMMAND_ALL_LEN) = '\0';

	return PB_RESULT_OK;
}

static const char *
get_parameter_string(const pb_command_universal *parm, char *buf)
{
	switch (parm->id) {
	case PB_USB_ACTIVITY_ID:
		return get_parameter_string_usba(&parm->cmd.usba, buf);
	case PB_UNKNOWN_COMMAND_ID:
	default:
		return (const char *)NULL;
	}
}

static const char *get_parameter_string_usba(
					   const pb_usb_activity_cmd *parm, char *buf)
{
	const char *wrk;
	int index = 0;

	wrk = parm_id_to_name(parm->activity);
	if (wrk == (const char *)NULL) {
		return wrk;
	}

	while (*wrk != '\0') {
		*(buf + index) = *wrk;
		index++;
		wrk++;
	}

	*(buf + index) = '\0';

	return buf;
}

static const char *command_id_to_name(const pb_command_id_type_e id)
{
	int i;
	const command_id_elem *tab;

	for (i = 0; i < COUNT_OF_PB_COMMAND_ID_TABLE; i++) {
		tab = &command_id_table[i];
		if (tab->id == id) {
			return tab->name;
		}
	}
	return (const char *)NULL;
}

static const char *parm_id_to_name(const pb_parm_id_type_e id)
{
	int i;
	const parm_id_elem *tab;

	for (i = 0; i < COUNT_OF_PB_LINE_ID_TABLE; i++) {
		tab = &parm_id_table[i];
		if (tab->id == id) {
			return tab->name;
		}
	}
	return (const char *)NULL;
}

