/*
 *  conflict.c
 *
 *  This file is conflict control driver's source file.
 *
 *  This software is contributed or developed by KYOCERA Corporation.
 *  (C) 2014 KYOCERA Corporation
 *  (C) 2015 KYOCERA Corporation
 *  (C) 2018 KYOCERA Corporation
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License.  See the file COPYING in the main directory of this archive for
 *  more details.
 */

/*******************************************************************************
 * INCLUDE FILES
 ******************************************************************************/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/types.h>
#include <linux/device.h>
#include <linux/wakelock.h>
#include <asm/uaccess.h>
#include <linux/sysfs.h>
#include "conflict.h"

#define FEATURE_SMEM_API

#ifdef FEATURE_SMEM_API
#include <soc/qcom/smem.h>

#define KCC_SMEM_OK                         0
#define KCC_SMEM_NG                        -1

#else /* FEATURE_SMEM_API */
#include <linux/slab.h>
#include <linux/mm.h>
#endif /* FEATURE_SMEM_API */

/*******************************************************************************
 * CONSTANT / DEFINE DECLARATIONS
 ******************************************************************************/

/*----------------------------------------------------------------------------*/
/* DEFINE                                                                     */
/*----------------------------------------------------------------------------*/

#define EVENT_CONFLICT_NONE         0U
#define EVENT_CONFLICT_MAX          65U

/*
 * Shared State Library
 * SState use the status bit of the conflict library.
 * In order to avoid duplicate use, SState use the status bit
 * from the largest value in decreasing order.
 *
 * SState use the flag bit (1 bit) and the state bit(s).
 * The flag bit means the state bit has been set explicitly
 * by the set API liboem_sstate_set().
 */

#define STATUS_SSTATE_ENGINE_FLAG   127U
#define STATUS_SSTATE_ENGINE_BIT    128U

#define EVENT_SSTATE_ENGINE_BIT     64U

#define CONF_TABLE_PART_SIZE 32U


#define CONFLICT_PROC_DEVICE "conflict"
#define CONFLICT_CLS_NAME    "conflict"
#define CONFLICT_DEV_NAME    "conflict"
#define CONFLICT_DEV_COUNT   1

#define CONFLICT_IDLE         0x00U
#define CONFLICT_WAITING_FOR  0xAAU
#define CONFLICT_WRITING      0xFFU


#ifndef FEATURE_SMEM_API
#define SMEM_STATUS_TABLE            0
#define SMEM_MODEM_WRITE_STATUS      0
#define SMEM_LINUX_WRITE_STATUS      0
#define KCC_SMEM_STATUS_TABLE_SIZE       16U
#define KCC_SMEM_MODEM_WRITE_STATUS_SIZE 1U
#define KCC_SMEM_LINUX_WRITE_STATUS_SIZE 1U
#endif /* FEATURE_SMEM_API */

#define VOIDCAST (void)

#define CONFLICT_NR_START 0U
#define CONFLICT_NR_END   1U
#define CONFLICT_NR_GET   2U

#define CNFL_ATTR_NAME_ENGINE "s_eng"

/*
 kernel/include/uapi/asm-generic/ioctl.h
 */

static uint32_t CONFLICT_LEFTSHIFT(uint32_t val, uint32_t shift);
static uint32_t CONFLICT_RIGHTSHIFT(uint32_t val, uint32_t shift);
static uint32_t CONFLICT_IOC_NRMASK(void);
static uint32_t CONFLICT_IOC_NR(uint32_t nr);


static uint32_t CONFLICT_LEFTSHIFT(uint32_t val, uint32_t shift)
{
    return ((uint32_t)(val<<shift));
}

static uint32_t CONFLICT_RIGHTSHIFT(uint32_t val, uint32_t shift)
{
    return ((uint32_t)(val>>shift));
}

static uint32_t CONFLICT_IOC_NRMASK(void)
{
  return ((uint32_t)(CONFLICT_LEFTSHIFT(1U , _IOC_NRBITS)-1U));
}

static uint32_t CONFLICT_IOC_NR(uint32_t nr)
{
     return ((uint32_t)(CONFLICT_RIGHTSHIFT(nr , _IOC_NRSHIFT) & CONFLICT_IOC_NRMASK() ));
}

/*----------------------------------------------------------------------------*/
/* STRUCTURE                                                                  */
/*----------------------------------------------------------------------------*/
typedef struct
{
    struct wake_lock wlock;
} cnfl_env_type;

/*----------------------------------------------------------------------------*/
/* ENUM                                                                       */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* STATIC                                                                     */
/*----------------------------------------------------------------------------*/

static volatile uint8_t * p_linux_state = NULL;
static volatile uint8_t * p_modem_state = NULL;
static volatile t_CONF_TABLE * p_state_table = NULL;
static dev_t region_dev;
static struct cdev char_dev;
static struct class *pdevcls;
static struct device *pdevice;
static cnfl_env_type cnfl_env;
static cnfl_env_type *pcnfl_env = &cnfl_env;

/*******************************************************************************
 * FUNCTIONS
 ******************************************************************************/
static uint32_t conflict_rd_u32(volatile uint32_t *p)
{
    return *p;
}
static uint8_t conflict_rd_u8(volatile uint8_t *p)
{
    return *p;
}
static void conflict_wr_u8(volatile uint8_t *p, uint8_t val)
{
    *p = val;
    return;
}
static void conflict_wror_u32(volatile uint32_t *p, uint32_t val)
{
    *p |= val;
    return;
}
static void conflict_wrxor_u32(volatile uint32_t *p, uint32_t val)
{
    *p ^= val;
    return;
}
static void conflict_wrand_u32(volatile uint32_t *p, uint32_t val)
{
    *p &= val;
    return;
}

static int32_t conflict_status_bit_check ( uint32_t status_type )
{
    uint32_t tbl_mng_num = 0U;
    uint32_t tbl_bit_shift = 0U;
    uint32_t tbl_part_tmp  = 0U;

    if( ( status_type - 1U ) < CONF_TABLE_PART_SIZE )
    {
        tbl_bit_shift = status_type - 1U;
    }
    else
    {
        tbl_mng_num   = ( status_type - 1U ) / CONF_TABLE_PART_SIZE;
        tbl_bit_shift = ( status_type - 1U ) % CONF_TABLE_PART_SIZE;
    }
    tbl_part_tmp = (uint32_t)( 1UL << tbl_bit_shift );

    if( ( p_state_table->table[tbl_mng_num] & tbl_part_tmp ) != 0U )
    {
        return 1;
    }

    return 0;
}

static ssize_t show_s_eng( struct device *dev, struct device_attribute *attr, char *buf )
{
    if ( conflict_status_bit_check( STATUS_SSTATE_ENGINE_FLAG ) == 0 )
    {
        /* the ENGINE_BIT is invalid */
        return scnprintf( buf, PAGE_SIZE, "%d\n", -1 );
    }

    if ( conflict_status_bit_check( STATUS_SSTATE_ENGINE_BIT ) == 0 )
    {
        return scnprintf( buf, PAGE_SIZE, "%d\n", 0 );
    }

    return scnprintf( buf, PAGE_SIZE, "%d\n", 1 );
}

static DEVICE_ATTR( s_eng, 0400, show_s_eng, NULL );


#ifdef CNFL_SMEM_ENABLE_CRC
static void conflict_set_CRC(unsigned id);
static void conflict_check_CRC(unsigned id, unsigned buf_size);

static void conflict_set_CRC(unsigned id)
{
    int32_t res;
    res = kc_smem_alloc_set(id);
    if (res != KCC_SMEM_OK)
    {
        BUG();
        /* fatal error. reset! */
    }
}

static void conflict_check_CRC(unsigned id, unsigned buf_size)
{
    void * p;
    p = kc_smem_alloc_get(id, buf_size);
    if (p == NULL)
    {
        BUG();
        /* CRC mismatch. fatal error. reset! */
    }
}
#endif

static void conflict_sub_wait(void)
{
    uint8_t mst = CONFLICT_IDLE;

    while(1)
    {
        /* Exclusion check */
        conflict_wr_u8(p_linux_state, CONFLICT_WAITING_FOR);

        mst = conflict_rd_u8(p_modem_state);

        if( CONFLICT_IDLE == mst )
        {
            conflict_wr_u8(p_linux_state, CONFLICT_WRITING);

            break;
        }
        else
        {
            if( (CONFLICT_WAITING_FOR != mst) && (CONFLICT_WRITING != mst) )
            {
                BUG();
            }
        }

        conflict_wr_u8(p_linux_state, CONFLICT_IDLE);
    }

    return;
}


static int conflict_check (unsigned int ev_id)
{
    int cnt = 0;
    int rc  = 0;

    for(cnt=0; cnt<CONF_TABLE_MNG_SIZE; cnt++)
    {
        if(0U != (conflict_rd_u32( &(p_state_table->table[cnt]))
                   & competitiveManagementTable[ev_id][cnt]) )
        {
            rc = 1;
            break;
        }
    }

    return rc;
}


static void conflict_op_start (conf_devctrl_s_type * param_t)
{
    uint32_t tbl_mng_num = 0U;
    uint32_t tbl_bit_shift = 0U;
    uint32_t tbl_part_tmp  = 0U;
    uint32_t status_type = 0U;

    VOIDCAST printk( KERN_DEBUG "conflict_op_start start\n" );

    if(!p_state_table)
    {
        VOIDCAST printk( KERN_ERR "conflict_op_start EFAULT! state_table is NULL\n");
        param_t->rc = (uint32_t)EFAULT;

        BUG();

        return;
    }

    if ((param_t->event_id <= EVENT_CONFLICT_NONE)
        || (EVENT_CONFLICT_MAX <= param_t->event_id))
    {
        VOIDCAST printk( KERN_ERR "conflict_op_start EFAULT! Invalid event_id\n");
        param_t->rc = (uint32_t)EFAULT;
        return;
    }

    param_t->rc = 0U;

    local_irq_disable();
    wake_lock ( &(pcnfl_env->wlock) );

    switch (param_t->event_id)
    {
    case EVENT_SSTATE_ENGINE_BIT:
        /* don't check conflict */

        status_type = STATUS_SSTATE_ENGINE_FLAG;
        tbl_mng_num   = (status_type - 1U) / CONF_TABLE_PART_SIZE;
        tbl_bit_shift = (status_type - 1U) % CONF_TABLE_PART_SIZE;
        tbl_part_tmp = (uint32_t)(1UL << tbl_bit_shift);

        conflict_sub_wait();

#ifdef CNFL_SMEM_ENABLE_CRC
        conflict_check_CRC(SMEM_STATUS_TABLE, KCC_SMEM_STATUS_TABLE_SIZE);
#endif

        conflict_wror_u32( &(p_state_table->table[tbl_mng_num]), tbl_part_tmp);

#ifdef CNFL_SMEM_ENABLE_CRC
        conflict_set_CRC(SMEM_STATUS_TABLE);
#endif

        conflict_wr_u8( p_linux_state, CONFLICT_IDLE );

        status_type = STATUS_SSTATE_ENGINE_BIT;
        tbl_mng_num   = (status_type - 1U) / CONF_TABLE_PART_SIZE;
        tbl_bit_shift = (status_type - 1U) % CONF_TABLE_PART_SIZE;
        tbl_part_tmp = (uint32_t)(1UL << tbl_bit_shift);

        conflict_sub_wait();

#ifdef CNFL_SMEM_ENABLE_CRC
        conflict_check_CRC(SMEM_STATUS_TABLE, KCC_SMEM_STATUS_TABLE_SIZE);
#endif

        conflict_wror_u32( &(p_state_table->table[tbl_mng_num]), tbl_part_tmp);

#ifdef CNFL_SMEM_ENABLE_CRC
        conflict_set_CRC(SMEM_STATUS_TABLE);
#endif

        conflict_wr_u8( p_linux_state, CONFLICT_IDLE );

        sysfs_notify( &(pdevice->kobj), NULL, CNFL_ATTR_NAME_ENGINE );
        break;
    default:
        /* conflict check */
        if(0 == conflict_check(param_t->event_id))
        {
            tbl_mng_num = afterStateTable[param_t->event_id][AFT_STATE_TBL_MNG];
            if ( (uint32_t)CONF_TABLE_MNG_SIZE <= tbl_mng_num)
            {
                VOIDCAST printk( KERN_ERR "conflict_op_start EFAULT! Invalid tbl_mng_num\n");
                param_t->rc = (uint32_t)EFAULT;

                BUG();

            }
            else
            {
                conflict_sub_wait();

#ifdef CNFL_SMEM_ENABLE_CRC
                conflict_check_CRC(SMEM_STATUS_TABLE, KCC_SMEM_STATUS_TABLE_SIZE);
#endif

                if(0 == conflict_check(param_t->event_id))
                {
                    conflict_wror_u32( &(p_state_table->table[tbl_mng_num]),
                       afterStateTable[param_t->event_id][AFT_STATE_TBL_STS]);

#ifdef CNFL_SMEM_ENABLE_CRC
                    conflict_set_CRC(SMEM_STATUS_TABLE);
#endif

                }
                else
                {
                    param_t->rc = (uint32_t)EBUSY;
                }
                conflict_wr_u8( p_linux_state, CONFLICT_IDLE );
            }
        }
        else
        {
            param_t->rc = (uint32_t)EBUSY;
        }
        break;
    }

    wake_unlock ( &(pcnfl_env->wlock) );
    local_irq_enable();

    VOIDCAST printk( KERN_DEBUG "conflict_op_start end\n" );

    return;
}


static void conflict_op_end (conf_devctrl_s_type * param_t)
{
    uint32_t tbl_mng_num = 0U;
    uint32_t tbl_bit_shift = 0U;
    uint32_t tbl_part_tmp  = 0U;
    uint32_t status_type = 0U;

    VOIDCAST printk( KERN_DEBUG "conflict_op_end start\n" );

    if(!p_state_table)
    {
        VOIDCAST printk( KERN_ERR "conflict_op_end EFAULT! state_table is NULL\n");
        param_t->rc = (uint32_t)EFAULT;

        BUG();

        return;
    }

    if ((param_t->event_id <= EVENT_CONFLICT_NONE)
        || (EVENT_CONFLICT_MAX <= param_t->event_id))
    {
        VOIDCAST printk( KERN_ERR "conflict_op_end EFAULT! Invalid evet_id\n");
        param_t->rc = (uint32_t)EFAULT;
        return;
    }

    param_t->rc = 0U;

    local_irq_disable();
    wake_lock ( &(pcnfl_env->wlock) );

    switch (param_t->event_id)
    {
    case EVENT_SSTATE_ENGINE_BIT:
        /* don't check ALREADY */

        status_type = STATUS_SSTATE_ENGINE_FLAG;
        tbl_mng_num   = (status_type - 1U) / CONF_TABLE_PART_SIZE;
        tbl_bit_shift = (status_type - 1U) % CONF_TABLE_PART_SIZE;
        tbl_part_tmp = (uint32_t)(1UL << tbl_bit_shift);

        conflict_sub_wait();

#ifdef CNFL_SMEM_ENABLE_CRC
        conflict_check_CRC(SMEM_STATUS_TABLE, KCC_SMEM_STATUS_TABLE_SIZE);
#endif

        conflict_wror_u32( &(p_state_table->table[tbl_mng_num]), tbl_part_tmp);

#ifdef CNFL_SMEM_ENABLE_CRC
        conflict_set_CRC(SMEM_STATUS_TABLE);
#endif

        conflict_wr_u8( p_linux_state, CONFLICT_IDLE );

        status_type = STATUS_SSTATE_ENGINE_BIT;
        tbl_mng_num   = (status_type - 1U) / CONF_TABLE_PART_SIZE;
        tbl_bit_shift = (status_type - 1U) % CONF_TABLE_PART_SIZE;
        tbl_part_tmp = (uint32_t)(1UL << tbl_bit_shift);

        conflict_sub_wait();

#ifdef CNFL_SMEM_ENABLE_CRC
        conflict_check_CRC(SMEM_STATUS_TABLE, KCC_SMEM_STATUS_TABLE_SIZE);
#endif

        conflict_wrand_u32( &(p_state_table->table[tbl_mng_num]), (~tbl_part_tmp));

#ifdef CNFL_SMEM_ENABLE_CRC
        conflict_set_CRC(SMEM_STATUS_TABLE);
#endif

        conflict_wr_u8( p_linux_state, CONFLICT_IDLE );

        sysfs_notify( &(pdevice->kobj), NULL, CNFL_ATTR_NAME_ENGINE );
        break;

    default:
        tbl_mng_num = afterStateTable[param_t->event_id][AFT_STATE_TBL_MNG];

        if ( (uint32_t)CONF_TABLE_MNG_SIZE <= tbl_mng_num)
        {
            VOIDCAST printk( KERN_ERR "conflict_op_end EFAULT! Invalid tbl_mng_num\n");
            param_t->rc = (uint32_t)EFAULT;

            BUG();

        }
        else
        {
            /* ALREADY check */
            if(0U != (conflict_rd_u32( &(p_state_table->table[tbl_mng_num]))
                     & afterStateTable[param_t->event_id][AFT_STATE_TBL_STS]))
            {
                conflict_sub_wait();

#ifdef CNFL_SMEM_ENABLE_CRC
                conflict_check_CRC(SMEM_STATUS_TABLE, KCC_SMEM_STATUS_TABLE_SIZE);
#endif

                conflict_wrxor_u32( &(p_state_table->table[tbl_mng_num]),
                   afterStateTable[param_t->event_id][AFT_STATE_TBL_STS]);

#ifdef CNFL_SMEM_ENABLE_CRC
                conflict_set_CRC(SMEM_STATUS_TABLE);
#endif

                conflict_wr_u8( p_linux_state, CONFLICT_IDLE );
            }
            else
            {
                VOIDCAST printk( KERN_NOTICE "conflict_op_end EALREADY! evt:%d \n", param_t->event_id );
                param_t->rc = (uint32_t)EALREADY;
            }
        }
        break;
    }

    wake_unlock ( &(pcnfl_env->wlock) );
    local_irq_enable();

    VOIDCAST printk( KERN_DEBUG "conflict_op_end end\n" );

    return;
}


static void conflict_op_get(conf_devctrl_s_type * param_t)
{
    VOIDCAST printk( KERN_DEBUG "conflict_op_get start\n" );

    if(p_state_table!=NULL)
    {
        param_t->rc = 0U;
        param_t->table = *p_state_table;
    }
    else
    {
        VOIDCAST printk( KERN_ERR "conflict_op_get EFAULT! state_table is NULL\n");
        param_t->rc = (uint32_t)EFAULT;

        BUG();

    }
    VOIDCAST printk( KERN_DEBUG "conflict_op_get end\n" );

    return;
}


static long conflict_ioctl (struct file *filp, unsigned int cmd, unsigned long arg)
{
    conf_devctrl_s_type param_t;
    void __user *argp;
    uint32_t nrval;

    VOIDCAST printk( KERN_DEBUG "conflict_ioctl start\n" );

    argp = (void __user *)arg;

    if (!argp)
    {
        VOIDCAST printk( KERN_WARNING "Invalid argument, NULL!" );
        return -EFAULT;
    }

    VOIDCAST memset(&param_t, 0, (size_t)sizeof(param_t));

    if( copy_from_user(&param_t, argp, (unsigned long) sizeof(param_t)) != 0U)
    {
        VOIDCAST printk( KERN_WARNING "copy_from_user error!" );
        return -EFAULT;
    }

    nrval = (uint32_t) CONFLICT_IOC_NR((uint32_t)cmd);

    switch(nrval)
    {
        case CONFLICT_NR_START:
            conflict_op_start(&param_t);
            break;
        case CONFLICT_NR_END:
            conflict_op_end(&param_t);
            break;
        case CONFLICT_NR_GET:
            conflict_op_get(&param_t);
            break;
        default:
            VOIDCAST printk( KERN_WARNING "conflict_ioctl cmd not found! nr:%d cmd:0x%08x\n", nrval, cmd );
            param_t.rc = (uint32_t)EFAULT;
            break;
    }

    if( copy_to_user(argp, &param_t, (unsigned long) sizeof(param_t)) != 0U)
    {
        VOIDCAST printk( KERN_WARNING "copy_to_user error!" );
        return -EFAULT;
    }

    VOIDCAST printk( KERN_DEBUG "conflict_ioctl end\n" );
    return 0;
}


static struct file_operations conflict_fops ={
    .owner   = THIS_MODULE,
    .open    = NULL,
    .release = NULL,
    .read    = NULL,
    .write   = NULL,
    .unlocked_ioctl = conflict_ioctl,
};


static void *conflict_sub_salloc(unsigned id, unsigned size)
{
    void *pv = NULL;

#ifdef FEATURE_SMEM_API
    pv = kc_smem_alloc(id, size);
#else /* FEATURE_SMEM_API */
    pv = kmalloc(size, GFP_KERNEL);
#endif /* FEATURE_SMEM_API */
    if(!pv)
    {
        VOIDCAST printk( KERN_WARNING "kc_smem_alloc error! param:id=%d,size=%d\n", id, size );

        BUG();

    }
    return pv;
}


static int __init conflict_init(void)
{
    int rc = 0;

    VOIDCAST printk( KERN_DEBUG "conflict_init start\n" );

    /* register a range of char device numbers */
    rc = alloc_chrdev_region( &region_dev, 0, CONFLICT_DEV_COUNT, CONFLICT_DEV_NAME );
    if( 0 > rc )
    {
        VOIDCAST printk( KERN_WARNING "alloc_chrdev_region error! return:%d\n", rc );

        BUG();

        return -EFAULT;
    }
    pdevcls = class_create(THIS_MODULE, CONFLICT_CLS_NAME);
    /*
    * if (IS_ERR_OR_NULL(pdevcls))
    */
    if (!pdevcls)
    {
        VOIDCAST printk( KERN_WARNING "class_create failed\n" );

        BUG();

        rc = -ENOMEM;
        return rc;
    }

    /* Only 1 device is created. It is region_dev. */
    /* /sys/devices/virtual/CLS_NAME/DEV_NAME is created. */
    /* /dev/DEV_NAME is created according to mdev.conf */
    pdevice = device_create(pdevcls, NULL, region_dev, NULL, CONFLICT_DEV_NAME);
    /*
    * if (IS_ERR_OR_NULL(pdevice))
    */
    if (!pdevice)
    {
        VOIDCAST printk( KERN_WARNING "device_create failed\n" );

        BUG();

        rc = -ENOMEM;
        return rc;
    }

    /* initialize a cdev structure */
    cdev_init( &char_dev, &conflict_fops );
    char_dev.owner = THIS_MODULE;

    /* add a char device to the system */
    rc = cdev_add( &char_dev, region_dev, CONFLICT_DEV_COUNT );
    if( 0 > rc )
    {
        VOIDCAST printk( KERN_WARNING "cdev_add error! return:%d\n", rc );

        BUG();

        return -EFAULT;
    }

    /* create attribute as sysfs */
    if ( device_create_file( pdevice, &dev_attr_s_eng ) != 0 )
    {
        VOIDCAST printk(KERN_ALERT "Sysfs Attribute Creation failed for s_eng\n");

        /* remove the cdev from the system */
        cdev_del( &char_dev );
        /* destroy the device instance. */
        device_destroy( pdevcls, region_dev );
        pdevice=NULL;
        /* destroy the class instance. */
        class_destroy( pdevcls );
        pdevcls=NULL;
        /* return a range of device numbers */
        unregister_chrdev_region( region_dev, CONFLICT_DEV_COUNT );

        BUG();

        return -EFAULT;
    }

    /* get SMEM param */
    p_state_table = (volatile t_CONF_TABLE *)conflict_sub_salloc(SMEM_STATUS_TABLE, KCC_SMEM_STATUS_TABLE_SIZE);
    p_modem_state = (volatile uint8_t *)conflict_sub_salloc(SMEM_MODEM_WRITE_STATUS, KCC_SMEM_MODEM_WRITE_STATUS_SIZE);
    p_linux_state = (volatile uint8_t *)conflict_sub_salloc(SMEM_LINUX_WRITE_STATUS, KCC_SMEM_LINUX_WRITE_STATUS_SIZE);
    if(!p_state_table || !p_modem_state || !p_linux_state )
    {
        VOIDCAST printk( KERN_WARNING "conflict_sub_salloc failed.\n" );
        return -EFAULT;
    }

#ifdef FEATURE_SMEM_API
    /* state:IDLE */
    conflict_wr_u8( p_linux_state, CONFLICT_IDLE );
#else /* FEATURE_SMEM_API */
    VOIDCAST memset((void*)p_state_table, 0, (size_t)sizeof(*p_state_table));
    VOIDCAST memset((void*)p_modem_state, 0, (size_t)sizeof(*p_modem_state));
    VOIDCAST memset((void*)p_linux_state, 0, (size_t)sizeof(*p_linux_state));
#endif /* FEATURE_SMEM_API */

    wake_lock_init ( &(pcnfl_env->wlock), WAKE_LOCK_SUSPEND, "conflict_driver");

    VOIDCAST printk( KERN_DEBUG "conflict_init end\n" );
    
    return 0;
}


static void __exit conflict_exit(void)
{
    VOIDCAST printk( KERN_DEBUG "conflict_exit start\n" );

    /* remove attribute (sysfs) */
    if( pdevice != NULL )
    {
        device_remove_file( pdevice, &dev_attr_s_eng );
    }

    wake_lock_destroy ( &(pcnfl_env->wlock) );

#ifndef FEATURE_SMEM_API
    kfree((void*)p_state_table);
    kfree((void*)p_modem_state);
    kfree((void*)p_linux_state);
#endif /* FEATURE_SMEM_API */

    /* remove a cdev from the system */
    cdev_del( &char_dev );

    /* destroy the device instance. */
    if( pdevice != NULL )
    {
        device_destroy( pdevcls, region_dev );
        pdevice=NULL;
    }

    /* destroy the class instance. */
    if( pdevcls != NULL )
    {
        class_destroy( pdevcls );
        pdevcls=NULL;
    }

    /* return a range of device numbers */
    unregister_chrdev_region( region_dev, CONFLICT_DEV_COUNT );

    VOIDCAST printk( KERN_DEBUG "conflict_exit end\n" );

    return;
}


MODULE_LICENSE("GPL");

module_init(conflict_init);
module_exit(conflict_exit);
