 /***************************************************************************
 *
 * Copyright (C) 2007-2008 SMSC
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *****************************************************************************/

/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2019 KYOCERA Corporation
 */

#include <linux/module.h>
#include <linux/kmod.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#include <linux/usb.h>
#include <linux/bitrev.h>
#include <linux/crc16.h>
#include <linux/crc32.h>
#include <linux/usb/usbnet.h>
#include <linux/slab.h>
#include "smsc95xx.h"
#include "smsc95xx_cfile_parser.h"

#define SMSC_CHIPNAME			"smsc95xx"
#define SMSC_DRIVER_VERSION		"1.0.4"
#define HS_USB_PKT_SIZE			(512)
#define FS_USB_PKT_SIZE			(64)
#define DEFAULT_HS_BURST_CAP_SIZE	(16 * 1024 + 5 * HS_USB_PKT_SIZE)
#define DEFAULT_FS_BURST_CAP_SIZE	(6 * 1024 + 33 * FS_USB_PKT_SIZE)
#define DEFAULT_BULK_IN_DELAY		(0x00002000)
#define MAX_SINGLE_PACKET_SIZE		(2048)
#define LAN95XX_EEPROM_MAGIC		(0x9500)
#define EEPROM_MAC_OFFSET		(0x01)
#define DEFAULT_TX_CSUM_ENABLE		(true)
#define DEFAULT_RX_CSUM_ENABLE		(true)
#define SMSC95XX_INTERNAL_PHY_ID	(1)
#define SMSC95XX_TX_OVERHEAD		(8)
#define SMSC95XX_TX_OVERHEAD_CSUM	(12)
#define SUPPORTED_WAKE			(WAKE_PHY | WAKE_UCAST | WAKE_BCAST | \
					 WAKE_MCAST | WAKE_ARP | WAKE_MAGIC)

#define FEATURE_8_WAKEUP_FILTERS	(0x01)
#define FEATURE_PHY_NLP_CROSSOVER	(0x02)
#define FEATURE_REMOTE_WAKEUP		(0x04)

#define SUSPEND_SUSPEND0		(0x01)
#define SUSPEND_SUSPEND1		(0x02)
#define SUSPEND_SUSPEND2		(0x04)
#define SUSPEND_SUSPEND3		(0x08)
#define SUSPEND_ALLMODES		(SUSPEND_SUSPEND0 | SUSPEND_SUSPEND1 | \
					 SUSPEND_SUSPEND2 | SUSPEND_SUSPEND3)

struct smsc95xx_priv {
	u32 mac_cr;
	u32 hash_hi;
	u32 hash_lo;
	u32 wolopts;
	spinlock_t mac_cr_lock;
	u8 features;
	u8 suspend_flags;
	struct mutex access_mutex;
	u32 mii_data;
	u32 mii_access;
};

static bool turbo_mode = false;
module_param(turbo_mode, bool, 0644);
MODULE_PARM_DESC(turbo_mode, "Enable multiple frames per Rx transaction");

struct usbnet *tmp_dev;

u32 debug_device = 0;
module_param(debug_device, uint, 0600);
MODULE_PARM_DESC(debug_device, "device type");

static unsigned long debug_reg_id = 0;
static int smsc95xx_set_reg_id(const char *val, struct kernel_param *kp);
module_param_call(debug_reg_id, smsc95xx_set_reg_id, param_get_ulong, &debug_reg_id, 0600);
MODULE_PARM_DESC(debug_reg_id, "register id write and read");

static unsigned long debug_reg_data = 0;
static int smsc95xx_set_reg_data(const char *val, struct kernel_param *kp);
static int smsc95xx_get_reg_data(char *buffer, struct kernel_param *kp);
module_param_call(debug_reg_data, smsc95xx_set_reg_data, smsc95xx_get_reg_data, &debug_reg_data, 0600);
MODULE_PARM_DESC(debug_reg_data, "register data write and read");

static struct mutex disconnect_mutex;

static int __must_check __smsc95xx_read_reg(struct usbnet *dev, u32 index,
					    u32 *data, int in_pm)
{
	u32 buf;
	int ret;
	int (*fn)(struct usbnet *, u8, u8, u16, u16, void *, u16);

	BUG_ON(!dev);

	if (!in_pm)
		fn = usbnet_read_cmd;
	else
		fn = usbnet_read_cmd_nopm;

	ret = fn(dev, USB_VENDOR_REQUEST_READ_REGISTER, USB_DIR_IN
		 | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		 0, index, &buf, 4);
	pr_info("%s READ__REGISTER,Address[h]:%03x,DATA[h]:%08x,ret:%d\n",SMSC_CHIPNAME,index,buf,ret);//REGISTER_CHECK
	if (unlikely(ret < 0))
		netdev_warn(dev->net, "Failed to read reg index 0x%08x: %d\n",
			    index, ret);

	le32_to_cpus(&buf);
	*data = buf;

	return ret;
}

static int __must_check __smsc95xx_write_reg(struct usbnet *dev, u32 index,
					     u32 data, int in_pm)
{
	u32 buf;
	int ret;
	int (*fn)(struct usbnet *, u8, u8, u16, u16, const void *, u16);

	BUG_ON(!dev);

	if (!in_pm)
		fn = usbnet_write_cmd;
	else
		fn = usbnet_write_cmd_nopm;

	buf = data;
	cpu_to_le32s(&buf);

	ret = fn(dev, USB_VENDOR_REQUEST_WRITE_REGISTER, USB_DIR_OUT
		 | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		 0, index, &buf, 4);
	pr_info("%s WRITE_REGISTER,Address[h]:%03x,DATA[h]:%08x,ret:%d\n",SMSC_CHIPNAME,index,buf,ret);//REGISTER_CHECK
	if (unlikely(ret < 0))
		netdev_warn(dev->net, "Failed to write reg index 0x%08x: %d\n",
			    index, ret);

	return ret;
}

static int __must_check smsc95xx_read_reg(struct usbnet *dev, u32 index,
					  u32 *data)
{
	return __smsc95xx_read_reg(dev, index, data, 0);
}

static int __must_check smsc95xx_write_reg(struct usbnet *dev, u32 index,
					   u32 data)
{
	return __smsc95xx_write_reg(dev, index, data, 0);
}

/* Loop until the read is completed with timeout
 * called with phy_mutex held */
static int __must_check __smsc95xx_phy_wait_not_busy(struct usbnet *dev,
						     int in_pm)
{
	unsigned long start_time = jiffies;
	struct smsc95xx_priv *pdata = (struct smsc95xx_priv *)(dev->data[0]);
	u32 val;
	int ret;

	do {
		ret = __smsc95xx_read_reg(dev, pdata->mii_access, &val, in_pm);
		if (ret < 0) {
			netdev_warn(dev->net, "Error reading MII_ACCESS\n");
			return ret;
		}

		if (!(val & MII_BUSY_))
			return 0;
	} while (!time_after(jiffies, start_time + HZ));

	return -EIO;
}

static int __smsc95xx_mdio_read(struct net_device *netdev, int phy_id, int idx,
				int in_pm)
{
	struct usbnet *dev = netdev_priv(netdev);
	struct smsc95xx_priv *pdata = (struct smsc95xx_priv *)(dev->data[0]);
	u32 val, addr;
	int ret;

	mutex_lock(&dev->phy_mutex);

	/* confirm MII not busy */
	ret = __smsc95xx_phy_wait_not_busy(dev, in_pm);
	if (ret < 0) {
		netdev_warn(dev->net, "MII is busy in smsc95xx_mdio_read\n");
		goto done;
	}

	/* set the address, index & direction (read from PHY) */
	phy_id &= dev->mii.phy_id_mask;
	idx &= dev->mii.reg_num_mask;
	addr = (phy_id << 11) | (idx << 6) | MII_READ_ | MII_BUSY_;
	ret = __smsc95xx_write_reg(dev, pdata->mii_access, addr, in_pm);
	if (ret < 0) {
		netdev_warn(dev->net, "Error writing MII_ADDR\n");
		goto done;
	}

	ret = __smsc95xx_phy_wait_not_busy(dev, in_pm);
	if (ret < 0) {
		netdev_warn(dev->net, "Timed out reading MII reg %02X\n", idx);
		goto done;
	}

	ret = __smsc95xx_read_reg(dev, pdata->mii_data, &val, in_pm);
	if (ret < 0) {
		netdev_warn(dev->net, "Error reading MII_DATA\n");
		goto done;
	}

	ret = (u16)(val & 0xFFFF);

done:
	mutex_unlock(&dev->phy_mutex);
	return ret;
}

static int __smsc95xx_mdio_write(struct net_device *netdev, int phy_id,
				  int idx, int regval, int in_pm)
{
	struct usbnet *dev = netdev_priv(netdev);
	struct smsc95xx_priv *pdata = (struct smsc95xx_priv *)(dev->data[0]);
	u32 val, addr;
	int ret;

	mutex_lock(&dev->phy_mutex);

	/* confirm MII not busy */
	ret = __smsc95xx_phy_wait_not_busy(dev, in_pm);
	if (ret < 0) {
		netdev_warn(dev->net, "MII is busy in smsc95xx_mdio_write\n");
		goto done;
	}

	val = regval;
	ret = __smsc95xx_write_reg(dev, pdata->mii_data, val, in_pm);
	if (ret < 0) {
		netdev_warn(dev->net, "Error writing MII_DATA\n");
		goto done;
	}

	/* set the address, index & direction (write to PHY) */
	phy_id &= dev->mii.phy_id_mask;
	idx &= dev->mii.reg_num_mask;
	addr = (phy_id << 11) | (idx << 6) | MII_WRITE_ | MII_BUSY_;
	ret = __smsc95xx_write_reg(dev, pdata->mii_access, addr, in_pm);
	if (ret < 0) {
		netdev_warn(dev->net, "Error writing MII_ADDR\n");
		goto done;
	}

	ret = __smsc95xx_phy_wait_not_busy(dev, in_pm);
	if (ret < 0) {
		netdev_warn(dev->net, "Timed out writing MII reg %02X\n", idx);
		goto done;
	}

done:
	mutex_unlock(&dev->phy_mutex);
	return ret;
}

static int smsc95xx_mdio_read(struct net_device *netdev, int phy_id, int idx)
{
	return __smsc95xx_mdio_read(netdev, phy_id, idx, 0);
}

static int smsc95xx_mdio_write(struct net_device *netdev, int phy_id, int idx,
				int regval)
{
	return __smsc95xx_mdio_write(netdev, phy_id, idx, regval, 0);
}

static int smsc95xx_link_reset(struct usbnet *dev)
{
	struct mii_if_info *mii = &dev->mii;
	struct ethtool_cmd ecmd = { .cmd = ETHTOOL_GSET };
	int ret;

	ret = smsc95xx_cfile_register_control(dev, "B", NULL);
	if (ret < 0) {
		netdev_warn(dev->net, "failed to register control.\n");
		return ret;
	}

	ret = mii_check_media(mii, 1, 1);
	if (ret) {
		ret = smsc95xx_cfile_register_control(dev, "E", NULL);
		if (ret < 0) {
			netdev_warn(dev->net, "failed to read/write registr.\n");
		}
	} else {
		ret = smsc95xx_cfile_register_control(dev, "D", NULL);
		if (ret < 0) {
			netdev_warn(dev->net, "failed to read/write registr.\n");
		}
	}

	mii_ethtool_gset(&dev->mii, &ecmd);

	return ret;
}

static void smsc95xx_status(struct usbnet *dev, struct urb *urb)
{
	u32 intdata;

	if (urb->actual_length != 4) {
		netdev_warn(dev->net, "unexpected urb length %d\n",
			    urb->actual_length);
		return;
	}

	memcpy(&intdata, urb->transfer_buffer, 4);
	le32_to_cpus(&intdata);

	netif_dbg(dev, link, dev->net, "intdata: 0x%08X\n", intdata);

	if (intdata & INT_ENP_PHY_INT_)
		usbnet_defer_kevent(dev, EVENT_LINK_RESET);
	else
		netdev_warn(dev->net, "unexpected interrupt, intdata=0x%08X\n",
			    intdata);
}

static int smsc95xx_ethtool_getregslen(struct net_device *netdev)
{
	/* all smsc95xx registers */
	return COE_CR - ID_REV + sizeof(u32);
}

static void
smsc95xx_ethtool_getregs(struct net_device *netdev, struct ethtool_regs *regs,
			 void *buf)
{
	struct usbnet *dev = netdev_priv(netdev);
	unsigned int i, j;
	int retval;
	u32 *data = buf;

	retval = smsc95xx_read_reg(dev, ID_REV, &regs->version);
	if (retval < 0) {
		netdev_warn(netdev, "REGS: cannot read ID_REV\n");
		return;
	}

	for (i = ID_REV, j = 0; i <= COE_CR; i += (sizeof(u32)), j++) {
		retval = smsc95xx_read_reg(dev, i, &data[j]);
		if (retval < 0) {
			netdev_warn(netdev, "REGS: cannot read reg[%x]\n", i);
			return;
		}
	}
}

static void smsc95xx_ethtool_get_wol(struct net_device *net,
				     struct ethtool_wolinfo *wolinfo)
{
	struct usbnet *dev = netdev_priv(net);
	struct smsc95xx_priv *pdata = (struct smsc95xx_priv *)(dev->data[0]);

	wolinfo->supported = SUPPORTED_WAKE;
	wolinfo->wolopts = pdata->wolopts;
}

static int smsc95xx_ethtool_set_wol(struct net_device *net,
				    struct ethtool_wolinfo *wolinfo)
{
	struct usbnet *dev = netdev_priv(net);
	struct smsc95xx_priv *pdata = (struct smsc95xx_priv *)(dev->data[0]);
	int ret;

	pdata->wolopts = wolinfo->wolopts & SUPPORTED_WAKE;

	ret = device_set_wakeup_enable(&dev->udev->dev, pdata->wolopts);
	if (ret < 0)
		netdev_warn(dev->net, "device_set_wakeup_enable error %d\n", ret);

	return ret;
}

static const struct ethtool_ops smsc95xx_ethtool_ops = {
	.get_link	= usbnet_get_link,
	.nway_reset	= usbnet_nway_reset,
	.get_drvinfo	= usbnet_get_drvinfo,
	.get_msglevel	= usbnet_get_msglevel,
	.set_msglevel	= usbnet_set_msglevel,
	.get_settings	= usbnet_get_settings,
	.set_settings	= usbnet_set_settings,
	.get_regs_len	= smsc95xx_ethtool_getregslen,
	.get_regs	= smsc95xx_ethtool_getregs,
	.get_wol	= smsc95xx_ethtool_get_wol,
	.set_wol	= smsc95xx_ethtool_set_wol,
};

static int smsc95xx_ioctl(struct net_device *netdev, struct ifreq *rq, int cmd)
{
	struct usbnet *dev = netdev_priv(netdev);

	if (!netif_running(netdev))
		return -EINVAL;

	return generic_mii_ioctl(&dev->mii, if_mii(rq), cmd, NULL);
}

static int smsc95xx_reset(struct usbnet *dev)
{
	u32 burst_cap = 0;
	int ret = 0;

	ret = smsc95xx_cfile_init(dev, __smsc95xx_read_reg,__smsc95xx_write_reg);
	if (ret < 0) {
		netdev_warn(dev->net, "parse of config file failed: %d", ret);
		return -EIO;
	}

	if (!turbo_mode) {
		burst_cap = 0;
		dev->rx_urb_size = MAX_SINGLE_PACKET_SIZE;
	} else if (dev->udev->speed == USB_SPEED_HIGH) {
		burst_cap = DEFAULT_HS_BURST_CAP_SIZE / HS_USB_PKT_SIZE;
		dev->rx_urb_size = DEFAULT_HS_BURST_CAP_SIZE;
	} else {
		burst_cap = DEFAULT_FS_BURST_CAP_SIZE / FS_USB_PKT_SIZE;
		dev->rx_urb_size = DEFAULT_FS_BURST_CAP_SIZE;
	}

	netif_dbg(dev, ifup, dev->net, "rx_urb_size=%ld\n",
		  (ulong)dev->rx_urb_size);

	/* Initialize MII structure */
	dev->mii.dev        = dev->net;
	dev->mii.mdio_read  = smsc95xx_mdio_read;
	dev->mii.mdio_write = smsc95xx_mdio_write;

	ret = smsc95xx_cfile_register_control(dev, "A", NULL);
	if (ret < 0) {
		netdev_warn(dev->net, "failed to register control.\n");
		return ret;
	}

	netif_dbg(dev, ifup, dev->net, "smsc95xx_reset, return 0\n");
	return 0;
}

static const struct net_device_ops smsc95xx_netdev_ops = {
	.ndo_open		= usbnet_open,
	.ndo_stop		= usbnet_stop,
	.ndo_start_xmit		= usbnet_start_xmit,
	.ndo_tx_timeout		= usbnet_tx_timeout,
	.ndo_change_mtu		= usbnet_change_mtu,
	.ndo_set_mac_address 	= eth_mac_addr,
	.ndo_validate_addr	= eth_validate_addr,
	.ndo_do_ioctl 		= smsc95xx_ioctl,
};

static int smsc95xx_bind(struct usbnet *dev, struct usb_interface *intf)
{
	struct smsc95xx_priv *pdata = NULL;
	int ret;

	printk(KERN_INFO SMSC_CHIPNAME " v" SMSC_DRIVER_VERSION "\n");

	tmp_dev = dev;
	ret = usbnet_get_endpoints(dev, intf);
	if (ret < 0) {
		netdev_warn(dev->net, "usbnet_get_endpoints failed: %d\n", ret);
		return ret;
	}

	dev->data[0] = (unsigned long)kzalloc(sizeof(struct smsc95xx_priv),
					      GFP_KERNEL);

	pdata = (struct smsc95xx_priv *)(dev->data[0]);
	if (!pdata)
		return -ENOMEM;

	spin_lock_init(&pdata->mac_cr_lock);
	mutex_init(&pdata->access_mutex);
	mutex_init(&disconnect_mutex);

	if (DEFAULT_TX_CSUM_ENABLE)
		dev->net->features |= NETIF_F_HW_CSUM;
	if (DEFAULT_RX_CSUM_ENABLE)
		dev->net->features |= NETIF_F_RXCSUM;

	dev->net->hw_features = NETIF_F_HW_CSUM | NETIF_F_RXCSUM;

	dev->net->netdev_ops = &smsc95xx_netdev_ops;
	dev->net->ethtool_ops = &smsc95xx_ethtool_ops;
	dev->net->flags |= IFF_MULTICAST;
	dev->net->hard_header_len += SMSC95XX_TX_OVERHEAD_CSUM;
	dev->hard_mtu = dev->net->mtu + dev->net->hard_header_len;
	return 0;
}

static void smsc95xx_unbind(struct usbnet *dev, struct usb_interface *intf)
{
	struct smsc95xx_priv *pdata = (struct smsc95xx_priv *)(dev->data[0]);

	pr_info("%s\n", __func__);

	mutex_lock(&disconnect_mutex);

	if (pdata) {
		netif_dbg(dev, ifdown, dev->net, "free pdata\n");
		kfree(pdata);
		pdata = NULL;
		dev->data[0] = 0;
	}
	smsc95xx_cfile_exit();

	mutex_unlock(&disconnect_mutex);
}

static int smsc95xx_suspend(struct usb_interface *intf, pm_message_t message)
{
	return 0;
}

static int smsc95xx_resume(struct usb_interface *intf)
{
	return 0;
}

static int smsc95xx_reset_resume(struct usb_interface *intf)
{
	struct usbnet *dev = usb_get_intfdata(intf);
	int ret;

	ret = smsc95xx_reset(dev);
	if (ret < 0)
		return ret;

	return smsc95xx_resume(intf);
}

static void smsc95xx_rx_csum_offload(struct sk_buff *skb)
{
	skb->csum = *(u16 *)(skb_tail_pointer(skb) - 2);
	skb->ip_summed = CHECKSUM_COMPLETE;
	skb_trim(skb, skb->len - 2);
}

static int smsc95xx_rx_fixup(struct usbnet *dev, struct sk_buff *skb)
{
	/* This check is no longer done by usbnet */
	if (skb->len < dev->net->hard_header_len)
		return 0;

	while (skb->len > 0) {
		u32 header, align_count;
		struct sk_buff *ax_skb;
		unsigned char *packet;
		u16 size;

		memcpy(&header, skb->data, sizeof(header));
		le32_to_cpus(&header);
		packet = skb_pull(skb, 4 + NET_IP_ALIGN);
		if (packet == NULL) {
			pr_info("%s: failed to skb_pull.\n", __func__);
			return 0;
		}

		/* get the packet length */
		size = (u16)((header & RX_STS_FL_) >> 16);
		align_count = (4 - ((size + NET_IP_ALIGN) % 4)) % 4;

		if (unlikely(header & RX_STS_ES_)) {
			netif_dbg(dev, rx_err, dev->net,
				  "Error header=0x%08x\n", header);
			dev->net->stats.rx_errors++;
			dev->net->stats.rx_dropped++;

			if (header & RX_STS_CRC_) {
				dev->net->stats.rx_crc_errors++;
			} else {
				if (header & (RX_STS_TL_ | RX_STS_RF_))
					dev->net->stats.rx_frame_errors++;

				if ((header & RX_STS_LE_) &&
					(!(header & RX_STS_FT_)))
					dev->net->stats.rx_length_errors++;
			}
		} else {
			/* ETH_FRAME_LEN + 4(CRC) + 2(COE) + 4(Vlan) */
			if (unlikely(size > (ETH_FRAME_LEN + 12))) {
				netif_dbg(dev, rx_err, dev->net,
					  "size err header=0x%08x\n", header);
				return 0;
			}

			/* last frame in this batch */
			if (skb->len == size) {
				if (dev->net->features & NETIF_F_RXCSUM)
					smsc95xx_rx_csum_offload(skb);
				skb_trim(skb, skb->len - 4); /* remove fcs */
				skb->truesize = size + sizeof(struct sk_buff);

				return 1;
			}

			ax_skb = skb_clone(skb, GFP_ATOMIC);
			if (unlikely(!ax_skb)) {
				netdev_warn(dev->net, "Error allocating skb\n");
				return 0;
			}

			ax_skb->len = size;
			ax_skb->data = packet;
			skb_set_tail_pointer(ax_skb, size);

			if (dev->net->features & NETIF_F_RXCSUM)
				smsc95xx_rx_csum_offload(ax_skb);
			skb_trim(ax_skb, ax_skb->len - 4); /* remove fcs */
			ax_skb->truesize = size + sizeof(struct sk_buff);

			usbnet_skb_return(dev, ax_skb);
		}

		packet = skb_pull(skb, size);
		if (packet == NULL) {
			pr_info("%s: failed to skb_pull.\n", __func__);
			return 0;
		}

		/* padding bytes before the next frame starts */
		if (skb->len) {
			packet = skb_pull(skb, align_count);
			if (packet == NULL) {
				pr_info("%s: failed to skb_pull.\n", __func__);
				return 0;
			}
		}
	}

	if (unlikely(skb->len < 0)) {
		netdev_warn(dev->net, "invalid rx length<0 %d\n", skb->len);
		return 0;
	}

	return 1;
}

static u32 smsc95xx_calc_csum_preamble(struct sk_buff *skb)
{
	u16 low_16 = (u16)skb_checksum_start_offset(skb);
	u16 high_16 = low_16 + skb->csum_offset;
	return (high_16 << 16) | low_16;
}

static struct sk_buff *smsc95xx_tx_fixup(struct usbnet *dev,
					 struct sk_buff *skb, gfp_t flags)
{
	bool csum = skb->ip_summed == CHECKSUM_PARTIAL;
	int overhead = csum ? SMSC95XX_TX_OVERHEAD_CSUM : SMSC95XX_TX_OVERHEAD;
	u32 tx_cmd_a, tx_cmd_b;

	/* We do not advertise SG, so skbs should be already linearized */
	BUG_ON(skb_shinfo(skb)->nr_frags);

	if (skb_headroom(skb) < overhead) {
		struct sk_buff *skb2 = skb_copy_expand(skb,
			overhead, 0, flags);
		dev_kfree_skb_any(skb);
		skb = skb2;
		if (!skb)
			return NULL;
	}

	if (csum) {
		if (skb->len <= 45) {
			/* workaround - hardware tx checksum does not work
			 * properly with extremely small packets */
			long csstart = skb_checksum_start_offset(skb);
			__wsum calc = csum_partial(skb->data + csstart,
				skb->len - csstart, 0);
			*((__sum16 *)(skb->data + csstart
				+ skb->csum_offset)) = csum_fold(calc);

			csum = false;
		} else {
			u32 csum_preamble = smsc95xx_calc_csum_preamble(skb);
			skb_push(skb, 4);
			cpu_to_le32s(&csum_preamble);
			memcpy(skb->data, &csum_preamble, 4);
		}
	}

	skb_push(skb, 4);
	tx_cmd_b = (u32)(skb->len - 4);
	if (csum)
		tx_cmd_b |= TX_CMD_B_CSUM_ENABLE;
	cpu_to_le32s(&tx_cmd_b);
	memcpy(skb->data, &tx_cmd_b, 4);

	skb_push(skb, 4);
	tx_cmd_a = (u32)(skb->len - 8) | TX_CMD_A_FIRST_SEG_ |
		TX_CMD_A_LAST_SEG_;
	cpu_to_le32s(&tx_cmd_a);
	memcpy(skb->data, &tx_cmd_a, 4);

	return skb;
}

static int smsc95xx_manage_power(struct usbnet *dev, int on)
{
	struct smsc95xx_priv *pdata = (struct smsc95xx_priv *)(dev->data[0]);

	dev->intf->needs_remote_wakeup = on;

	if (pdata->features & FEATURE_REMOTE_WAKEUP)
		return 0;

	/* this chip revision isn't capable of remote wakeup */
	netdev_info(dev->net, "hardware isn't capable of remote wakeup\n");

	if (on)
		usb_autopm_get_interface_no_resume(dev->intf);
	else
		usb_autopm_put_interface(dev->intf);

	return 0;
}

static int smsc95xx_set_reg_id(const char *val, struct kernel_param *kp)
{
	int ret;
	unsigned long reg_id;

	ret = kstrtoul(val, 0, &reg_id);
	if (ret) {
		pr_err("%s: failed to kstrtoul, ret=%d\n", __func__, ret);
		return ret;
	}

	if (debug_device == 0x00000001UL) {
		if (0x000001FCUL < reg_id) {
			pr_err("%s:Error reg_id=0x%03X, Not Support\n", __func__, (unsigned int)reg_id);
			return -EINVAL;
		}
	} else if (debug_device == 0x00000002UL) {
		if (0x0000001FUL < reg_id) {
			pr_err("%s:Error reg_id=0x%03X, Not Support\n", __func__, (unsigned int)reg_id);
			return -EINVAL;
		}
	} else if (debug_device == 0x00000003UL) {
		if (0x00000AFFUL < reg_id) {
			pr_err("%s:Error reg_id=0x%03X, Not Support\n", __func__, (unsigned int)reg_id);
			return -EINVAL;
		}
	} else {
		pr_err("%s:Error Device=0x%03X, Unknown device\n", __func__, (unsigned int)debug_device);
		return -EINVAL;
	}

	ret = param_set_ulong(val, kp);
	if (ret < 0) {
		pr_err("%s:err(%d)\n",__func__,ret);
		return ret;
	}

	pr_info("%s SET__REGISTER_ID,ID[h]:%03x\n", SMSC_CHIPNAME, (unsigned int)reg_id);

	return 0;
}

static int smsc95xx_set_reg_data(const char *val, struct kernel_param *kp)
{
	int ret = 0;
	struct smsc95xx_priv *pdata = NULL;

	ret = param_set_ulong(val, kp);
	if (ret) {
		pr_err("%s:err(%d)\n",__func__,ret);
		return ret;
	}

	mutex_lock(&disconnect_mutex);
	pdata = (struct smsc95xx_priv *)(tmp_dev->data[0]);
	if (pdata == NULL) {
		mutex_unlock(&disconnect_mutex);
		return -EIO;
	}

	mutex_lock(&pdata->access_mutex);

	if (debug_device == 0x00000001UL) {
		ret = smsc95xx_write_reg(tmp_dev, debug_reg_id, debug_reg_data);
	} else if (debug_device == 0x00000002UL) {
		ret = smsc95xx_mdio_write(tmp_dev->net, tmp_dev->mii.phy_id, debug_reg_id, debug_reg_data);
	} else if (debug_device == 0x00000003UL) {
		ret = smsc95xx_mdio_write(tmp_dev->net, tmp_dev->mii.phy_id, 0x0000001E, debug_reg_id);
		if (0 <= ret) {
			ret = smsc95xx_mdio_write(tmp_dev->net, tmp_dev->mii.phy_id, 0x0000001F, debug_reg_data);
		}
	} else {
		pr_err("%s:Error Device=0x%03X, Unknown device\n", __func__, (unsigned int)debug_device);
		ret = -EINVAL;
	}

	mutex_unlock(&pdata->access_mutex);
	mutex_unlock(&disconnect_mutex);

	if (ret < 0) {
		pr_err("%s:Error writing(%d)\n", __func__, ret);
		return ret;
	}
	return 0;
}

static int smsc95xx_get_reg_data(char *buffer, struct kernel_param *kp)
{
	int ret = 0;
	u32 reg_data = 0;
	struct smsc95xx_priv *pdata = NULL;

	mutex_lock(&disconnect_mutex);
	pdata = (struct smsc95xx_priv *)(tmp_dev->data[0]);
	if (pdata == NULL) {
		mutex_unlock(&disconnect_mutex);
		return -EIO;
	}

	mutex_lock(&pdata->access_mutex);

	if (debug_device == 0x00000001UL) {
		ret = smsc95xx_read_reg(tmp_dev, debug_reg_id, &reg_data);
	} else if (debug_device == 0x00000002UL) {
		ret = smsc95xx_mdio_read(tmp_dev->net, tmp_dev->mii.phy_id, debug_reg_id);
		if (0 <= ret) {
			reg_data = ret;
		}
	} else if (debug_device == 0x00000003UL) {
		ret = smsc95xx_mdio_write(tmp_dev->net, tmp_dev->mii.phy_id, 0x0000001E, debug_reg_id);
		if (0 <= ret) {
			ret = smsc95xx_mdio_read (tmp_dev->net, tmp_dev->mii.phy_id, 0x0000001F);
		}
		if (0 <= ret) {
			reg_data = ret;
		}
	} else {
		pr_err("%s:Error Device=0x%03X, Unknown device\n", __func__, (unsigned int)debug_device);
		ret = -EINVAL;
	}

	mutex_unlock(&pdata->access_mutex);
	mutex_unlock(&disconnect_mutex);

	if (ret < 0) {
		pr_err("%s:Error cannot read(%d)\n",__func__,ret);
		return ret;
	}
	return sprintf(buffer, "0x%08X", (unsigned int)reg_data);
}

static const struct driver_info smsc95xx_info = {
	.description	= "smsc95xx USB 2.0 Ethernet",
	.bind		= smsc95xx_bind,
	.unbind		= smsc95xx_unbind,
	.link_reset	= smsc95xx_link_reset,
	.reset		= smsc95xx_reset,
	.rx_fixup	= smsc95xx_rx_fixup,
	.tx_fixup	= smsc95xx_tx_fixup,
	.status		= smsc95xx_status,
	.manage_power	= smsc95xx_manage_power,
	.flags		= FLAG_ETHER | FLAG_SEND_ZLP | FLAG_LINK_INTR,
};

static const struct usb_device_id products[] = {
	{
		/* SMSC9500 USB Ethernet Device */
		USB_DEVICE(0x0424, 0x9500),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9505 USB Ethernet Device */
		USB_DEVICE(0x0424, 0x9505),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9500A USB Ethernet Device */
		USB_DEVICE(0x0424, 0x9E00),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9505A USB Ethernet Device */
		USB_DEVICE(0x0424, 0x9E01),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9512/9514 USB Hub & Ethernet Device */
		USB_DEVICE(0x0424, 0xec00),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9500 USB Ethernet Device (SAL10) */
		USB_DEVICE(0x0424, 0x9900),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9505 USB Ethernet Device (SAL10) */
		USB_DEVICE(0x0424, 0x9901),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9500A USB Ethernet Device (SAL10) */
		USB_DEVICE(0x0424, 0x9902),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9505A USB Ethernet Device (SAL10) */
		USB_DEVICE(0x0424, 0x9903),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9512/9514 USB Hub & Ethernet Device (SAL10) */
		USB_DEVICE(0x0424, 0x9904),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9500A USB Ethernet Device (HAL) */
		USB_DEVICE(0x0424, 0x9905),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9505A USB Ethernet Device (HAL) */
		USB_DEVICE(0x0424, 0x9906),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9500 USB Ethernet Device (Alternate ID) */
		USB_DEVICE(0x0424, 0x9907),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9500A USB Ethernet Device (Alternate ID) */
		USB_DEVICE(0x0424, 0x9908),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC9512/9514 USB Hub & Ethernet Device (Alternate ID) */
		USB_DEVICE(0x0424, 0x9909),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC LAN9530 USB Ethernet Device */
		USB_DEVICE(0x0424, 0x9530),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC LAN9730 USB Ethernet Device */
		USB_DEVICE(0x0424, 0x9730),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{
		/* SMSC LAN89530 USB Ethernet Device */
		USB_DEVICE(0x0424, 0x9E08),
		.driver_info = (unsigned long) &smsc95xx_info,
	},
	{ },		/* END */
};
MODULE_DEVICE_TABLE(usb, products);

static struct usb_driver smsc95xx_driver = {
	.name		= "smsc95xx",
	.id_table	= products,
	.probe		= usbnet_probe,
	.suspend	= smsc95xx_suspend,
	.resume		= smsc95xx_resume,
	.reset_resume	= smsc95xx_reset_resume,
	.disconnect	= usbnet_disconnect,
	.disable_hub_initiated_lpm = 1,
	.supports_autosuspend = 1,
};

module_usb_driver(smsc95xx_driver);

MODULE_AUTHOR("Nancy Lin");
MODULE_AUTHOR("Steve Glendinning <steve.glendinning@shawell.net>");
MODULE_DESCRIPTION("SMSC95XX USB 2.0 Ethernet Devices");
MODULE_LICENSE("GPL");
