/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2019 KYOCERA Corporation
 * (C) 2023 KYOCERA Corporation
 */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */
/*===========================================================================*/


/*----------------------------------------------------------------------------*/
/* INCLUDE FILES                                                              */
/*----------------------------------------------------------------------------*/
#include <linux/errno.h>
#include <linux/fs.h>       /* inode, file, file_operations */
#include <linux/kernel.h>   /* printk */
#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/wakelock.h>
#include <crypto/hash.h>
#include <crypto/sha.h>
#include <crypto/public_key.h>
#include <crypto/kc_public_key.h>
#include <soc/qcom/smem.h>
#include <linux/kclsm_sysfs.h>
/*----------------------------------------------------------------------------*/
/* Define data                                                                */
/*----------------------------------------------------------------------------*/

#define RCC_BUFLEN                 16384        /* RCC Module Work Area Size */
#define PATH_LEN                   32
#define ERR_STR_LEN                64

#define DEBUG_BUFLEN               1024

/* 4ByteSUM Calculation */
#define DEFAULT_MASK               0xFFFFFFFFUL
#define SIZE_OF_DWORD              0x00000004UL
#define BIT_OF_BYTE                0x00000008UL
#define BIT_OF_WORD                0x00000010UL
#define BYTE_ALIGN_MASK            0x00000001UL
#define WORD_ALIGN_MASK            0x00000002UL

/* Meta related information */
#define META_MAGIC_NUM             0x4154454DUL     /* MagicNumber("META")  */
#define META_INFO_SIZE             2160             /* Meta information size */
#define META_SUM_SIZE              4                /* Sum size to be given to Meta  */

/* Digital signature related information */
#define ALGO_NAME_LENGTH           16
#define KC_PUBLIC_KEY_PATH         "/etc/kc_public.key"    /* Public Key of KC_META */
#define DN_PUBLIC_KEY_PATH         "/etc/dn_public.key"    /* Public Key of DN_APP_META */
#define ACDB_PUBLIC_KEY_PATH       "/etc/acdb_public.key"  /* Public Key of ACDB_META */
#define ALGO_INFO_PATH             "/etc/algo_info.bin"    /* Algorithm information */

/* SMEM save result code */
/* If you add an error code, also add it to rcc_save_result */
#define RCC_RESULT_OK              0x00         /* Check OK */
#define RCC_RESULT_NG              0x01         /* Check NG */

/* Thread Delay Time */
#define RCC_THREAD_DELAY_TIME      1            /* second */
#define RCC_THREAD_STANDBY_TIME    20           /* second */

#define RCC_FLAG_DISABLE           0x44434352UL /* "RCCD" */
#define RCC_FLAG_ENABLE            0x45545352UL /* "RSTE" */

#define RCC_CHECK_MAX_COUNT        (uint32_t)2  /* check max count */

#define RCC_CHECK_DNA_STATE_RETRY_MAX (uint32_t)60  /* retry up to 60 sec. */

#define FOTA_RECOVERY_BOOT         (uint8_t)1
#define FOTA_RECOVERY_BACKUP_BOOT  (uint8_t)2
/*----------------------------------------------------------------------------*/
/* Enumeration type                                                           */
/*----------------------------------------------------------------------------*/

/* Meta Kind */
typedef enum {
    RCC_KC_META = 0,
    RCC_DN_APP_META,
    RCC_ACDB_META,
    RCC_META_NUM,
    RCC_META_NONE = 0x7FFFFFFF
} rcc_meta_kind_type;

/* Image check order */
typedef enum {
    RCC_IMG_SYSTEM = 0,
    RCC_IMG_MODEM,
    RCC_IMG_DN_APP,
    RCC_IMG_ACDB,
    RCC_IMG_RECOVERYFS,
    RCC_IMG_NUM,
    RCC_IMG_NONE = 0x7FFFFFFF
} rcc_check_order_type;

/* RCC error code */
typedef enum {
    RCC_ERROR_NONE = 0,
    RCC_ERROR_BUFFER_ALLOC = 1,
    RCC_ERROR_META_OPEN,
    RCC_ERROR_META_HEADER_READ,
    RCC_ERROR_META_MAGIC_CHECK,
    RCC_ERROR_META_DATA_READ,
    RCC_ERROR_META_SUM_READ,
    RCC_ERROR_META_SUM_COMP,
    RCC_ERROR_META_PINFO_READ,
    RCC_ERROR_META_SIGN_MALLOC,
    RCC_ERROR_META_SIGN_READ,
    RCC_ERROR_IMAGE_OPEN,
    RCC_ERROR_IMAGE_READ,
    RCC_ERROR_DIGEST_COMP,
    RCC_ERROR_ALGO_DATA_READ,
    RCC_ERROR_HASH_DESC_ALLOC,
    RCC_ERROR_HASH_INIT,
    RCC_ERROR_HASH_UPDATE,
    RCC_ERROR_HASH_FINAL,
    RCC_ERROR_SMEM_ALLOC,
    RCC_ERROR_SET_PUBKEY,
    RCC_ERROR_SET_SIGN,
    RCC_ERROR_SET_BUFFER,
    RCC_ERROR_PARAMETER,
    RCC_ERROR_NUM
} rcc_error_code_type;

/*----------------------------------------------------------------------------*/
/* STRUCTURE                                                                  */
/*----------------------------------------------------------------------------*/
/* Meta Information */
typedef struct rcc_meta_info_struct {
    char                meta_path[PATH_LEN];        /* Meta Binary path(dev/mtdX)     */
    char                publickey_path[PATH_LEN];   /* Public Key path(etc/xxxxx.key) */
    uint32_t            meta_start_offset;          /* Meta information start offset  */
    uint32_t            sign_start_offset;          /* Digital signature start offset */
} rcc_meta_info_type;

/* Image Information */
typedef struct rcc_image_info_struct {
    rcc_meta_kind_type  meta_kind;                  /* Meta kind include Image          */
    uint32_t            meta_info_offset;           /* Order in meta in the Meta binary */
    char                image_path[PATH_LEN];       /* Image path(dev/mtdX)             */
    uint32_t            image_size;                 /* Image size                       */
    unsigned char       *image_sign_ptr;            /* Signature Image pointer          */
} rcc_image_info_type;

/* Meta Binary Header Information */
typedef struct rcc_meta_header_struct {
    uint32_t        meta_magic;             /* Magic Number                       */
    uint32_t        rom_meta_cnt;           /* Meta Information number(ROM Check) */
    uint32_t        ram_meta_cnt;           /* Meta Information number(RAM Check) */
    uint32_t        other_meta_cnt;         /* Meta Information number(Other)     */
} rcc_meta_header_type;

/* Meta Binary Program Header of Meta Information */
typedef struct rcc_program_info_struct {
    uint32_t        program_size;           /* Program Size         */
    char            program_ver[16];        /* Program Version      */
    uint32_t        program_identifier;     /* Program Identifier   */
    uint32_t        file_type;              /* File Type            */
    char            time_stamp[20];         /* Time Stamp           */
    char            reserved[48];           /* Reserved             */
} rcc_program_info_type;

/* Algorithm information of signature */
typedef struct rcc_algorithm_info_struct {
    uint32_t        public_key_len;
    uint32_t        modulus_len;
    uint32_t        exponent_len;
    uint32_t        modulus_offset;
    uint32_t        exponent_offset;
    uint32_t        key_algo_type;
    uint32_t        signature_len;
    uint32_t        digest_size;
    uint32_t        hash_algo_type;
    unsigned char   algo_name[ALGO_NAME_LENGTH];
} rcc_algorithm_info_type;

typedef struct
{
    struct wake_lock wlock;
} cnfl_env_type;

/*----------------------------------------------------------------------------*/
/* STATIC                                                                     */
/*----------------------------------------------------------------------------*/
/* Input Meta Binary Information of RCC(ROM Completeness Check) */
static rcc_meta_info_type gMetaInfo[RCC_META_NUM] = {
    {"dev/mtd11", KC_PUBLIC_KEY_PATH, 0, 0},    /* kc_meta     */
    {"dev/mtd19", DN_PUBLIC_KEY_PATH, 0, 0},    /* dn_app_meta */
    {"dev/mtd17", ACDB_PUBLIC_KEY_PATH, 0, 0}   /* acdb_meta   */
};

/* Image Information of RCC(ROM Completeness Check) - Arrange in order to check */
static rcc_image_info_type gImageInfo[RCC_IMG_NUM] = {
    {RCC_KC_META,     0, "dev/mtd15", 0, NULL},  /* system     */
    {RCC_KC_META,     1, "dev/mtd14", 0, NULL},  /* modem      */
    {RCC_DN_APP_META, 0, "dev/mtd18", 0, NULL},  /* DN_APP     */
    {RCC_ACDB_META,   0, "dev/mtd16", 0, NULL},  /* acdb       */
    {RCC_KC_META,     2, "dev/mtd9",  0, NULL}   /* recoveryfs */
};

/* RCC Module Work Area */
static unsigned char *gpucRccWorkBuf = NULL;

static struct task_struct *rcc_tsk = NULL;
static rcc_algorithm_info_type *gpRccAlgoInfo = NULL;
static uint32_t gulRetryNum = 0;

static cnfl_env_type cnfl_env;
static cnfl_env_type *pcnfl_env = &cnfl_env;

/*----------------------------------------------------------------------------*/
/* PROTOTYPES                                                                 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/* Function                                                                   */
/*----------------------------------------------------------------------------*/

/*
 *==============================================================================
 *
 * FUNCTION
 *   debug_data_dump
 *
 * DESCRIPTION
 *   Dump data in hexadecimal(Debug)
 *
 * PARAMETERS
 *   unsigned char *pucBuf : Data Storage buffer
 *   uint32_t ulSize       : Dump Size
 *
 * RETURN VALUE
 *   none
 *
 *==============================================================================
 */
static void debug_data_dump(unsigned char *pucBuf, uint32_t ulSize)
{
    uint32_t ulCnt = 0;
    char dbgstr[16] = {0};
    char *pcDbgBuf = NULL;
    
    if (pucBuf == NULL)
    {
        return;
    }
    
    /* 1byte -> 2char + \n + null */
    if (((ulSize * (uint32_t)2) + (uint32_t)1 + (uint32_t)1) > (uint32_t)DEBUG_BUFLEN)
    {
        return;
    }
    
    pcDbgBuf = (unsigned char *)kzalloc((size_t)DEBUG_BUFLEN, GFP_KERNEL);
    if (pcDbgBuf == NULL)
    {
        pr_err("debug buffer malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return;
    }
    
    for (ulCnt = 0; ulCnt < ulSize; ulCnt++)
    {
        (void) sprintf(dbgstr, "%02X", pucBuf[ulCnt]);
        (void) strcat(pcDbgBuf, dbgstr);
    }
    (void) strcat(pcDbgBuf, "\n");
    pr_debug("%s", pcDbgBuf);
    
    kfree(pcDbgBuf);
    
    return;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   calc_4b_sum_on_mem
 *
 * DESCRIPTION
 *   4Byte Sum calculate
 *
 * PARAMETERS
 *   uint32_t _addr : SUM calculation start address
 *   uint32_t _len  : SUM calculation length
 *
 * RETURN VALUE
 *   SUM Value
 *
 *==============================================================================
 */
static uint32_t calc_4b_sum_on_mem(uint32_t _addr, uint32_t _len)
{
    uint32_t addr      = 0;
    uint32_t len       = 0;
    uint32_t sum       = 0;
    uint32_t bit_mask  = 0;
    uint32_t bit_shift = 0;
    
    addr = _addr;
    len  = _len;
    
    if ( len == (uint32_t)0 )
    {
        return sum;
    }
    
    /* odd align, 2 byte align --> 4 byte align */
    if ( (addr & (BYTE_ALIGN_MASK | WORD_ALIGN_MASK)) != (uint32_t)0 )
    {
        /* Get bit shift and bit mask */
        if ((addr & (BYTE_ALIGN_MASK | WORD_ALIGN_MASK)) == (BYTE_ALIGN_MASK | WORD_ALIGN_MASK))
        {
            bit_shift = (BIT_OF_BYTE | BIT_OF_WORD);
        }
        else
        {
            bit_shift = ((addr & BYTE_ALIGN_MASK) != (uint32_t)0) ? BIT_OF_BYTE : BIT_OF_WORD;
        }
        bit_mask  = DEFAULT_MASK << bit_shift;
        
        /* Size renew */
        len -= (SIZE_OF_DWORD - (addr & (BYTE_ALIGN_MASK | WORD_ALIGN_MASK)));
        
        /* Arrange to 4 byte align */
        addr &= ~(BYTE_ALIGN_MASK | WORD_ALIGN_MASK);
        sum += *((uint32_t*)addr) & bit_mask;
        
        /* Renew address */
        addr += SIZE_OF_DWORD;
    }
    
    /* 4 byte align */
    while ( len >= SIZE_OF_DWORD )
    {
        /* DO SUM */
        sum += *((uint32_t*)addr);
        
        /* renew address */
        len  -= SIZE_OF_DWORD;
        addr += SIZE_OF_DWORD;
    }
    
    /* Rest processing */
    if ( len != (uint32_t)0 )
    {
        /* Get bit shift and bit mask */
        bit_shift = ((sizeof(uint32_t) - len) * BIT_OF_BYTE);
        bit_mask  = DEFAULT_MASK >> bit_shift;
        
        sum += *((uint32_t*)addr) & bit_mask;
    }
    
    return sum;
}
/* calc_4b_sum_on_mem */


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_save_result_in_smem
 *
 * DESCRIPTION
 *   Save the result of RCC (ROM Completeness Check) in SMEM
 *
 * PARAMETERS
 *   uint8_t cCheckResult : RCC result
 *
 * RETURN VALUE
 *   (0 == value): Success / (0 != value): Error
 *
 *==============================================================================
 */
static int32_t rcc_save_result_in_smem(uint8_t cCheckResult)
{
    uint8_t *rcc_check_result = NULL;
    
    rcc_check_result = (uint8_t*)kc_smem_alloc(SMEM_AT_KCDIAG_SLEEP_SUM_CHECK_CHK, KCC_SMEM_AT_KCDIAG_SLEEP_SUM_CHECK_CHK_SIZE);
    if (rcc_check_result != NULL)
    {
        *rcc_check_result = cCheckResult;
    }
    else
    {
        pr_err("kc_smem_alloc(SMEM_AT_KCDIAG_SLEEP_SUM_CHECK_CHK) is null. filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_SMEM_ALLOC;
    }
    
    return 0;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_fault_detection
 *
 * DESCRIPTION
 *   reset when an error is detected
 *
 * PARAMETERS
 *   none
 *
 * RETURN VALUE
 *   none
 *
 *==============================================================================
 */
static void rcc_fault_detection(void)
{
    uint32_t *rcc_reset_flag = NULL;
    
    rcc_reset_flag = (uint32_t*)kc_smem_alloc(SMEM_RCC_RESET_CHK, KCC_SMEM_RCC_RESET_CHK_SIZE);
    if (rcc_reset_flag != NULL)
    {
        if (*rcc_reset_flag != RCC_FLAG_DISABLE)
        {
            panic("rcc check fault\n");
        }
    }
    else
    {
        panic("rcc check fault\n");
    }
    
    return;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_save_result
 *
 * DESCRIPTION
 *   Save RCC (ROM Completeness Check) result in SMEM
 *
 * PARAMETERS
 *   rcc_error_code_type  ulErrCode    : Error code
 *   rcc_meta_kind_type   ulMetaKind   : Meta kind
 *   rcc_check_order_type ulCheckImage : Check image type
 *
 * RETURN VALUE
 *   none
 *
 *==============================================================================
 */
static void rcc_save_result(rcc_error_code_type ulErrCode, rcc_meta_kind_type ulMetaKind, rcc_check_order_type ulCheckImage)
{
    int32_t lRet         = 0;
    uint8_t cCheckResult = 0;
    
    pr_debug("------ save rcc result [START] ------\n err_code=0x%X, meta_kind=0x%X, check_image=0x%X\n", ulErrCode, ulMetaKind, ulCheckImage);
    
    switch(ulErrCode)
    {
    case RCC_ERROR_NONE:
        cCheckResult = (uint8_t)RCC_RESULT_OK;
        break;
    default:
        cCheckResult = (uint8_t)RCC_RESULT_NG;
        break;
    }
    
    /* Save results in SMEM */
    if ((ulErrCode == RCC_ERROR_NONE) || (gulRetryNum >= (RCC_CHECK_MAX_COUNT - (uint32_t)1)))
    {
        lRet = rcc_save_result_in_smem(cCheckResult);
        if (lRet < 0)
        {
            pr_err("smem save fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        }
    }
    
    pr_debug("------ save rcc result [END] ------\n");
    
    return;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_check_meta_binary
 *
 * DESCRIPTION
 *   Calculate SUM value of meta binary, confirm the normality of meta information
 *
 * PARAMETERS
 *   none
 *
 * RETURN VALUE
 *   (0 == value): Success / (-1 >= value): Error
 *
 *==============================================================================
 */
static int32_t rcc_check_meta_binary(void)
{
    struct file *filp = NULL;
    uint32_t ulReqSize = 0;
    uint32_t ulHdrSize = 0;
    uint32_t ulSumLen = 0;
    uint32_t ulRemainSize = 0;
    uint32_t ulCalcSum = 0;
    uint32_t ulMetaSum = 0;
    uint32_t ulCnt;
    loff_t oft = 0;
    int rlen = 0;
    rcc_meta_kind_type ulMetaCnt;
    rcc_meta_header_type *pMetaHdr = NULL;
    rcc_meta_info_type *pMetaInfo = NULL;
    
    
    pr_debug("------ check meta binary ------\n");
    
    for (ulMetaCnt = (rcc_meta_kind_type)(ulCnt = 0); ulMetaCnt < RCC_META_NUM; ulMetaCnt = (rcc_meta_kind_type)(++ulCnt))
    {
        pMetaInfo = (rcc_meta_info_type *)&gMetaInfo[ulMetaCnt];
        
        pr_debug("ulMetaCnt=%d, meta_path=%s\n", ulMetaCnt, pMetaInfo->meta_path);
        
        filp = filp_open(pMetaInfo->meta_path, O_RDONLY, 0);
        if (IS_ERR(filp))
        {
            pr_err("open fail, filename=%s, line=%d, func=%s, pathname=%s, err=%ld\n", __FILE__, __LINE__, __func__, pMetaInfo->meta_path, PTR_ERR(filp));
            rcc_save_result(RCC_ERROR_META_OPEN, ulMetaCnt, RCC_IMG_NONE);
            return -(int32_t)RCC_ERROR_META_OPEN;
        }
        
        ulHdrSize = sizeof(rcc_meta_header_type);
        
        rlen = kernel_read(filp, (loff_t)0, (char *)gpucRccWorkBuf, (unsigned long)ulHdrSize);
        if (rlen != (int)ulHdrSize)
        {
            pr_err("read fail, filename=%s, line=%d, func=%s, rlen=%d\n", __FILE__, __LINE__, __func__, rlen);
            (void) filp_close(filp, NULL);
            rcc_save_result(RCC_ERROR_META_HEADER_READ, ulMetaCnt, RCC_IMG_NONE);
            return -(int32_t)RCC_ERROR_META_HEADER_READ;
        }
        
        /* Debug */
        pr_debug("------ meta header ------\n");
        debug_data_dump(gpucRccWorkBuf, ulHdrSize);
        
        pMetaHdr = (rcc_meta_header_type *)gpucRccWorkBuf;
        
        if (pMetaHdr->meta_magic != META_MAGIC_NUM)
        {
            pr_err("meta magic unmatch, filename=%s, line=%d, func=%s, magic=0x%X\n", __FILE__, __LINE__, __func__, pMetaHdr->meta_magic);
            (void) filp_close(filp, NULL);
            rcc_save_result(RCC_ERROR_META_MAGIC_CHECK, ulMetaCnt, RCC_IMG_NONE);
            return -(int32_t)RCC_ERROR_META_MAGIC_CHECK;
        }
        
        /* get sum calc length */
        ulSumLen = ulHdrSize
                + ((pMetaHdr->rom_meta_cnt + pMetaHdr->ram_meta_cnt + pMetaHdr->other_meta_cnt) * (uint32_t)META_INFO_SIZE)
                + (pMetaHdr->rom_meta_cnt * gpRccAlgoInfo->signature_len);
        
        pr_debug("ulSumLen=0x%X\n", ulSumLen);
        
        /* set offset information in meta binary */
        pMetaInfo->meta_start_offset = ulHdrSize;
        pMetaInfo->sign_start_offset = ulHdrSize
                + ((pMetaHdr->rom_meta_cnt + pMetaHdr->ram_meta_cnt + pMetaHdr->other_meta_cnt) * (uint32_t)META_INFO_SIZE);
        
        pr_debug("meta_start_offset=0x%X  sign_start_offset=0x%X\n", pMetaInfo->meta_start_offset, pMetaInfo->sign_start_offset);
        
        
        /* sum calc */
        oft = (loff_t)0;
        ulRemainSize = ulSumLen;
        ulCalcSum = 0;
        
        while (ulRemainSize > (uint32_t)0)
        {
            /* read image */
            ulReqSize = (ulRemainSize > (uint32_t)RCC_BUFLEN) ? (uint32_t)RCC_BUFLEN : ulRemainSize;
            
            rlen = kernel_read(filp, (loff_t)oft, (char *)gpucRccWorkBuf, (unsigned long)ulReqSize);
            if (rlen != (int)ulReqSize)
            {
                pr_err("read fail, filename=%s, line=%d, func=%s, rlen=%d\n", __FILE__, __LINE__, __func__, rlen);
                (void) filp_close(filp, NULL);
                rcc_save_result(RCC_ERROR_META_DATA_READ, ulMetaCnt, RCC_IMG_NONE);
                return -(int32_t)RCC_ERROR_META_DATA_READ;
            }
            
            ulRemainSize -= ulReqSize;
            oft += (loff_t)ulReqSize;
            
            ulCalcSum += calc_4b_sum_on_mem( (uint32_t)(gpucRccWorkBuf), ulReqSize );
        }
        
        /* get sum from meta binary */
        rlen = kernel_read(filp, (loff_t)oft, (char *)gpucRccWorkBuf, (unsigned long)META_SUM_SIZE);
        if (rlen != (int)META_SUM_SIZE)
        {
            pr_err("read fail, filename=%s, line=%d, func=%s, rlen=%d\n", __FILE__, __LINE__, __func__, rlen);
            (void) filp_close(filp, NULL);
            rcc_save_result(RCC_ERROR_META_SUM_READ, ulMetaCnt, RCC_IMG_NONE);
            return -(int32_t)RCC_ERROR_META_SUM_READ;
        }
        
        ulMetaSum = *((uint32_t *)gpucRccWorkBuf);
        
        /* compare calc sum and meta sum */
        if (ulCalcSum != ulMetaSum)
        {
            pr_err("sum unmatch, filename=%s, line=%d, func=%s, calcSum=0x%X, metaSum=0x%X\n", __FILE__, __LINE__, __func__, ulCalcSum, ulMetaSum);
            (void) filp_close(filp, NULL);
            rcc_save_result(RCC_ERROR_META_SUM_COMP, ulMetaCnt, RCC_IMG_NONE);
            return -(int32_t)RCC_ERROR_META_SUM_COMP;
        }
        
        
        pr_debug("ulCalcSum=0x%X  ulMetaSum=0x%X\n", ulCalcSum, ulMetaSum);
        
        (void) filp_close(filp, NULL);
    }
    
    return 0;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_set_image_info
 *
 * DESCRIPTION
 *   Acquire the image information of the RCC(ROM Completeness Check) object
 *   from the meta information, and set up
 *
 * PARAMETERS
 *   none
 *
 * RETURN VALUE
 *   (0 == value): Success / (-1 >= value): Error
 *
 *==============================================================================
 */
static int32_t rcc_set_image_info(void)
{
    struct file *filp = NULL;
    uint32_t ulPrgOffset = 0;
    uint32_t ulSignOffset = 0;
    uint32_t ulPrgInfoSize = 0;
    uint32_t ulCnt;
    int rlen = 0;
    rcc_program_info_type *pPrgInfo = NULL;
    rcc_image_info_type *pImageInfo = NULL;
    rcc_meta_info_type *pMetaInfo = NULL;
    rcc_check_order_type ulImageCnt;
    
    
    pr_debug("------ set image information ------\n");
    
    for (ulImageCnt = (rcc_check_order_type)(ulCnt = 0); ulImageCnt < RCC_IMG_NUM; ulImageCnt = (rcc_check_order_type)(++ulCnt))
    {
        pr_debug("ulImageCnt=%d\n", ulImageCnt);
        
        pImageInfo = (rcc_image_info_type *)&gImageInfo[ulImageCnt];
        pMetaInfo = (rcc_meta_info_type *)&gMetaInfo[pImageInfo->meta_kind];
        
        
        filp = filp_open(pMetaInfo->meta_path, O_RDONLY, 0);
        if (IS_ERR(filp))
        {
            pr_err("open fail, filename=%s, line=%d, func=%s, pathname=%s, errno=%ld\n", __FILE__, __LINE__, __func__, pMetaInfo->meta_path, PTR_ERR(filp));
            rcc_save_result(RCC_ERROR_META_OPEN, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_META_OPEN;
        }
        
        /* get image size from meta information program header */
        ulPrgOffset = pMetaInfo->meta_start_offset + (pImageInfo->meta_info_offset * (uint32_t)META_INFO_SIZE);
        
        pr_debug("ulPrgOffset=0x%X\n", ulPrgOffset);
        
        ulPrgInfoSize = sizeof(rcc_program_info_type);
        
        rlen = kernel_read(filp, (loff_t)ulPrgOffset, (char *)gpucRccWorkBuf, (unsigned long)ulPrgInfoSize);
        if (rlen != (int)ulPrgInfoSize)
        {
            pr_err("read fail, filename=%s, line=%d, func=%s, rlen=%d\n", __FILE__, __LINE__, __func__, rlen);
            (void) filp_close(filp, NULL);
            rcc_save_result(RCC_ERROR_META_PINFO_READ, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_META_PINFO_READ;
        }
        
        pPrgInfo = (rcc_program_info_type *)gpucRccWorkBuf;
        pImageInfo->image_size = pPrgInfo->program_size;
        
        pr_debug("ImageSize=0x%X\n", pImageInfo->image_size);
        
        
        /* get image signature */
        ulSignOffset = pMetaInfo->sign_start_offset + (pImageInfo->meta_info_offset * gpRccAlgoInfo->signature_len);
        
        pr_debug("ulSignOffset=0x%X\n", ulSignOffset);
        
        pImageInfo->image_sign_ptr = (unsigned char *)kzalloc((size_t)gpRccAlgoInfo->signature_len, GFP_KERNEL);
        if (pImageInfo->image_sign_ptr == NULL)
        {
            pr_err("buffer malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
            (void) filp_close(filp, NULL);
            rcc_save_result(RCC_ERROR_META_SIGN_MALLOC, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_META_SIGN_MALLOC;
        }
        
        rlen = kernel_read(filp, (loff_t)ulSignOffset, (char *)pImageInfo->image_sign_ptr, (unsigned long)gpRccAlgoInfo->signature_len);
        if (rlen != (int)gpRccAlgoInfo->signature_len)
        {
            pr_err("read fail, filename=%s, line=%d, func=%s, rlen=%d\n", __FILE__, __LINE__, __func__, rlen);
            (void) filp_close(filp, NULL);
            rcc_save_result(RCC_ERROR_META_SIGN_READ, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_META_SIGN_READ;
        }
        
        (void) filp_close(filp, NULL);
        
        /* Debug */
        pr_debug("------ image signature ------\n");
        debug_data_dump(pImageInfo->image_sign_ptr, gpRccAlgoInfo->signature_len);
    }
    
    return 0;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_set_algo_info
 *
 * DESCRIPTION
 *   Set algorithm information
 *
 * PARAMETERS
 *   none
 *
 * RETURN VALUE
 *   (0 == value): Success / (-1 >= value): Error
 *
 *==============================================================================
 */
static int32_t rcc_set_algo_info(void)
{
    struct file *filp = NULL;
    int rlen = 0;
    
    /* alloc algo info buffer */
    gpRccAlgoInfo = (rcc_algorithm_info_type *)kzalloc((size_t)sizeof(rcc_algorithm_info_type), GFP_KERNEL);
    if (gpRccAlgoInfo == NULL)
    {
        pr_err("rcc algo info buffer malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        rcc_save_result(RCC_ERROR_BUFFER_ALLOC, RCC_META_NONE, RCC_IMG_NONE);
        return -(int32_t)RCC_ERROR_BUFFER_ALLOC;
    }
    
    filp = filp_open(ALGO_INFO_PATH, O_RDONLY, 0);
    if (IS_ERR(filp))
    {
        pr_err("open fail, filename=%s, line=%d, func=%s, pathname=%s, errno=%ld\n", __FILE__, __LINE__, __func__, ALGO_INFO_PATH, PTR_ERR(filp));
        rcc_save_result(RCC_ERROR_IMAGE_OPEN, RCC_META_NONE, RCC_IMG_NONE);
        return -(int32_t)RCC_ERROR_IMAGE_OPEN;
    }
    
    rlen = kernel_read(filp, (loff_t)0, (char *)gpRccAlgoInfo, (unsigned long)sizeof(rcc_algorithm_info_type));
    if (rlen != (int)sizeof(rcc_algorithm_info_type))
    {
        pr_err("read fail, filename=%s, line=%d, func=%s, rlen=%d\n", __FILE__, __LINE__, __func__, rlen);
        (void) filp_close(filp, NULL);
        rcc_save_result(RCC_ERROR_ALGO_DATA_READ, RCC_META_NONE, RCC_IMG_NONE);
        return -(int32_t)RCC_ERROR_ALGO_DATA_READ;
    }
    
    (void) filp_close(filp, NULL);
    
    return 0;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_alloc_shash_desc
 *
 * DESCRIPTION
 *   Set hash information
 *
 * PARAMETERS
 *   shash_desc    **desc      : Hash structure
 *
 * RETURN VALUE
 *   (0 == value): Success / (-1 >= value): Error
 *
 *==============================================================================
 */
static int32_t rcc_alloc_shash_desc(struct shash_desc **desc)
{
    struct crypto_shash *tfm = NULL;
    size_t desc_size = 0;
    
    tfm = crypto_alloc_shash(gpRccAlgoInfo->algo_name, 0, 0);
    if (IS_ERR(tfm))
    {
        pr_err("crypto_alloc_shash fail, filename=%s, line=%d, func=%s, errno=%ld\n", __FILE__, __LINE__, __func__, PTR_ERR(tfm));
        return -(int32_t)RCC_ERROR_BUFFER_ALLOC;
    }
    
    desc_size = crypto_shash_descsize(tfm) + sizeof(**desc);
    
    *desc = kzalloc(desc_size, GFP_KERNEL);
    if (*desc == NULL)
    {
        pr_err("buffer malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        crypto_free_shash(tfm);
        return -(int32_t)RCC_ERROR_BUFFER_ALLOC;
    }
    
    (*desc)->tfm   = tfm;
    (*desc)->flags = 0;
    
    return 0;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_free_shash_desc
 *
 * DESCRIPTION
 *   Clear hash information
 *
 * PARAMETERS
 *   shash_desc    *desc      : Hash structure
 *
 * RETURN VALUE
 *   none
 *
 *==============================================================================
 */
static void rcc_free_shash_desc(struct shash_desc *desc)
{
    if (desc == NULL)
    {
        pr_err("data is null, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return;
    }
    
    crypto_free_shash(desc->tfm);
    kfree(desc);
    
    return;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_set_public_key_info
 *
 * DESCRIPTION
 *   Set public key information
 *
 * PARAMETERS
 *   public_key            *pub_key          : Public key structure
 *   unsigned char         *public_key_data  : Public Key Data
 *
 * RETURN VALUE
 *   (0 == value): Success / (-1 >= value): Error
 *
 *==============================================================================
 */
static int32_t rcc_set_public_key_info(struct public_key *pub_key, unsigned char *public_key_data)
{
    int ret = 0;
    
    if ((pub_key == NULL) || (public_key_data == NULL))
    {
        pr_err("data is null, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_PARAMETER;
    }
    
    pub_key->algo      = NULL;
    pub_key->pkey_algo = PKEY_ALGO_RSA;
    
    pub_key->rsa.n = mpi_alloc(gpRccAlgoInfo->modulus_len);
    if (pub_key->rsa.n == NULL)
    {
        pr_err("buffer malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_BUFFER_ALLOC;
    }
    
    pub_key->rsa.e = mpi_alloc(gpRccAlgoInfo->exponent_len);
    if (pub_key->rsa.e == NULL)
    {
        pr_err("buffer malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_BUFFER_ALLOC;
    }
    
    ret = mpi_set_buffer(pub_key->rsa.n, &public_key_data[gpRccAlgoInfo->modulus_offset], gpRccAlgoInfo->modulus_len, 0);
    if (ret != 0)
    {
        pr_err("mpi_set_buffer fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_SET_BUFFER;
    }
    
    ret = mpi_set_buffer(pub_key->rsa.e, &public_key_data[gpRccAlgoInfo->exponent_offset], gpRccAlgoInfo->exponent_len, 0);
    if (ret != 0)
    {
        pr_err("mpi_set_buffer fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_SET_BUFFER;
    }
    
    return 0;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_set_sig_info
 *
 * DESCRIPTION
 *   Set the signature information
 *
 * PARAMETERS
 *   public_key_signature    *sig         : Signature structure
 *   unsigned char           *hash_data   : Hash Data
 *   unsigned char           *sig_data    : Signature Data
 *
 * RETURN VALUE
 *   (0 == value): Success / (-1 >= value): Error
 *
 *==============================================================================
 */
static int32_t rcc_set_sig_info(struct public_key_signature *sig, unsigned char *hash_data, unsigned char *sig_data)
{
    int ret = 0;
    
    if ((sig == NULL) || (sig_data == NULL))
    {
        pr_err("data is null, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_PARAMETER;
    }
    
    sig->digest         = hash_data;
    sig->digest_size    = gpRccAlgoInfo->digest_size;
    sig->nr_mpi         = 1;
    sig->pkey_hash_algo = gpRccAlgoInfo->hash_algo_type;
    
    sig->rsa.s = mpi_alloc(gpRccAlgoInfo->signature_len);
    if (sig->rsa.s == NULL)
    {
        pr_err("buffer malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_BUFFER_ALLOC;
    }
    
    ret = mpi_set_buffer(sig->rsa.s, sig_data, gpRccAlgoInfo->signature_len, 0);
    if (ret != 0)
    {
        pr_err("mpi_set_buffer fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return -(int32_t)RCC_ERROR_SET_BUFFER;
    }
    
    return 0;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_free_public_key_info
 *
 * DESCRIPTION
 *   Clear public key information
 *
 * PARAMETERS
 *   public_key            *pub_key    : Public key structure
 *
 * RETURN VALUE
 *   none
 *
 *==============================================================================
 */
static void rcc_free_public_key_info(struct public_key *pub_key)
{
    if (pub_key == NULL)
    {
        pr_err("data is null, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return;
    }
    
    if (pub_key->rsa.e != NULL)
    {
        mpi_free(pub_key->rsa.e);
    }
    
    if (pub_key->rsa.n != NULL)
    {
        mpi_free(pub_key->rsa.n);
    }
    
    return;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_free_sig_info
 *
 * DESCRIPTION
 *   Clear signature information
 *
 * PARAMETERS
 *   public_key_signature    *sig      : Signature structure
 *
 * RETURN VALUE
 *   none
 *
 *==============================================================================
 */
static void rcc_free_sig_info(struct public_key_signature *sig)
{
    if (sig == NULL)
    {
        pr_err("data is null, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        return;
    }
    
    if (sig->rsa.s != NULL)
    {
        mpi_free(sig->rsa.s);
    }
    
    return;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_execute
 *
 * DESCRIPTION
 *   Run RCC(ROM Completeness Check)
 *
 * PARAMETERS
 *   none
 *
 * RETURN VALUE
 *   (0 == value): Success / (-1 >= value): Error
 *
 *==============================================================================
 */
static int32_t rcc_execute(void)
{
    struct file *filp = NULL;
    unsigned char *pucHash = NULL;
    uint32_t ulReqSize = 0;
    uint32_t ulRemainSize = 0;
    int rlen = 0;
    loff_t oft = 0;
    
    int ret = 0;
    int32_t result = 0;
    struct shash_desc *desc = NULL;
    struct public_key pub_key;
    struct public_key_signature sig;
    
    rcc_image_info_type *pImageInfo = NULL;
    rcc_meta_info_type *pMetaInfo = NULL;
    rcc_check_order_type ulImageCnt;
    uint32_t ulCnt;
    
    pr_debug("------ ROM Completeness Check execute ------\n");
    
    for (ulImageCnt = (rcc_check_order_type)(ulCnt = 0); ulImageCnt < RCC_IMG_NUM; ulImageCnt = (rcc_check_order_type)(++ulCnt))
    {
        pImageInfo = (rcc_image_info_type *)&gImageInfo[ulImageCnt];
        pMetaInfo = (rcc_meta_info_type *)&gMetaInfo[pImageInfo->meta_kind];
        
        pr_debug("ulImageCnt=%d, image_path=%s\n", ulImageCnt, pImageInfo->image_path);
        
        /*-----------------------------------*/
        /* calculate digest from flash image */
        /*-----------------------------------*/
        filp = filp_open(pImageInfo->image_path, O_RDONLY, 0);
        if (IS_ERR(filp))
        {
            pr_err("open fail, filename=%s, line=%d, func=%s, pathname=%s, errno=%ld\n", __FILE__, __LINE__, __func__, pImageInfo->image_path, PTR_ERR(filp));
            rcc_save_result(RCC_ERROR_IMAGE_OPEN, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_IMAGE_OPEN;
        }
        
        result = rcc_alloc_shash_desc(&desc);
        if ((result != 0) || (desc == NULL))
        {
            pr_err("rcc_alloc_shash_desc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
            (void) filp_close(filp, NULL);
            rcc_save_result(RCC_ERROR_HASH_DESC_ALLOC, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_HASH_DESC_ALLOC;
        }
        
        ret = crypto_shash_init(desc);
        if (ret != 0)
        {
            pr_err("crypto_shash_init fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
            (void) filp_close(filp, NULL);
            rcc_free_shash_desc(desc);
            rcc_save_result(RCC_ERROR_HASH_INIT, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_HASH_INIT;
        }
        
        pr_debug("ImageSize=0x%X\n", pImageInfo->image_size);
        
        oft = (loff_t)0;
        ulRemainSize = pImageInfo->image_size;
        while (ulRemainSize > (uint32_t)0)
        {
            /* read image */
            ulReqSize = (ulRemainSize > (uint32_t)RCC_BUFLEN) ? (uint32_t)RCC_BUFLEN : ulRemainSize;
            
            rlen = kernel_read(filp, (loff_t)oft, (void *)gpucRccWorkBuf, (size_t)ulReqSize);
            if (rlen != (int)ulReqSize)
            {
                pr_err("read fail, filename=%s, line=%d, func=%s, rlen=%d\n", __FILE__, __LINE__, __func__, rlen);
                (void) filp_close(filp, NULL);
                rcc_free_shash_desc(desc);
                rcc_save_result(RCC_ERROR_IMAGE_READ, pImageInfo->meta_kind, ulImageCnt);
                return -(int32_t)RCC_ERROR_IMAGE_READ;
            }
            
            ulRemainSize -= ulReqSize;
            oft += (loff_t)ulReqSize;
            
            /* calculate hash */
            ret = crypto_shash_update(desc, (unsigned char *)gpucRccWorkBuf, (unsigned int)ulReqSize);
            if (ret != 0)
            {
                pr_err("crypto_shash_update fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
                (void) filp_close(filp, NULL);
                rcc_free_shash_desc(desc);
                rcc_save_result(RCC_ERROR_HASH_UPDATE, pImageInfo->meta_kind, ulImageCnt);
                return -(int32_t)RCC_ERROR_HASH_UPDATE;
            }
        }
        
        /* alloc Hash buffer */
        pucHash = (unsigned char *)kzalloc((size_t)gpRccAlgoInfo->digest_size, GFP_KERNEL);
        if (pucHash == NULL)
        {
            pr_err("rcc Hash malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
            (void) filp_close(filp, NULL);
            rcc_free_shash_desc(desc);
            rcc_save_result(RCC_ERROR_BUFFER_ALLOC, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_BUFFER_ALLOC;
        }
        
        ret = crypto_shash_final(desc, pucHash);
        if (ret != 0)
        {
            pr_err("crypto_shash_final fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
            (void) filp_close(filp, NULL);
            rcc_free_shash_desc(desc);
            kfree(pucHash);
            rcc_save_result(RCC_ERROR_HASH_FINAL, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_HASH_FINAL;
        }
        
        (void) filp_close(filp, NULL);
        rcc_free_shash_desc(desc);
        
        /* Debug */
        pr_debug("------ image digest of flash image ------\n");
        debug_data_dump(pucHash, gpRccAlgoInfo->digest_size);
        
        filp = filp_open(pMetaInfo->publickey_path, O_RDONLY, 0);
        if (IS_ERR(filp))
        {
            pr_err("open fail, filename=%s, line=%d, func=%s, pathname=%s, errno=%ld\n", __FILE__, __LINE__, __func__, pMetaInfo->publickey_path, PTR_ERR(filp));
            kfree(pucHash);
            rcc_save_result(RCC_ERROR_IMAGE_OPEN, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_IMAGE_OPEN;
        }
        
        rlen = kernel_read(filp, (loff_t)0, (void *)gpucRccWorkBuf, (size_t)gpRccAlgoInfo->public_key_len);
        if (rlen != (int)gpRccAlgoInfo->public_key_len)
        {
            pr_err("read fail, filename=%s, line=%d, func=%s, rlen=%d\n", __FILE__, __LINE__, __func__, rlen);
            (void) filp_close(filp, NULL);
            kfree(pucHash);
            rcc_save_result(RCC_ERROR_IMAGE_READ, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_IMAGE_READ;
        }
        
        (void) filp_close(filp, NULL);
        
        result = rcc_set_public_key_info(&pub_key, gpucRccWorkBuf);
        if (result < 0)
        {
            pr_err("rcc_set_public_key_info fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
            kfree(pucHash);
            rcc_save_result(RCC_ERROR_SET_PUBKEY, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_SET_PUBKEY;
        }
        
        result = rcc_set_sig_info(&sig, pucHash, pImageInfo->image_sign_ptr);
        if (result < 0)
        {
            pr_err("rcc_set_sig_info fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
            kfree(pucHash);
            rcc_free_public_key_info(&pub_key);
            rcc_save_result(RCC_ERROR_SET_SIGN, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_SET_SIGN;
        }
        
        ret = kc_public_key_verify_signature(&pub_key, &sig);
        if (ret < 0)
        {
            pr_err("digest unmatch, filename=%s, line=%d, func=%s, ImageCnt=%d, image_path=%s, ret=%d\n", __FILE__, __LINE__, __func__, ulImageCnt, pImageInfo->image_path, ret);
            kfree(pucHash);
            rcc_free_public_key_info(&pub_key);
            rcc_free_sig_info(&sig);
            rcc_save_result(RCC_ERROR_DIGEST_COMP, pImageInfo->meta_kind, ulImageCnt);
            return -(int32_t)RCC_ERROR_DIGEST_COMP;
        }
        
        kfree(pucHash);
        rcc_free_public_key_info(&pub_key);
        rcc_free_sig_info(&sig);
        
        pr_info("----- digest match (ImageCnt=%d image_path=%s) -----\n", ulImageCnt, pImageInfo->image_path);
    }
    
    return 0;
}


/*
 *==============================================================================
 *
 * FUNCTION
 *   rcc_thread_main
 *
 * DESCRIPTION
 *   Start the RCC(ROM Completeness Check) thread
 *
 * PARAMETERS
 *   none
 *
 * RETURN VALUE
 *   none
 *
 *==============================================================================
 */
static int32_t rcc_thread_main( void )
{
    int32_t ret = 0;
    int32_t check_result = 0;
    uint32_t ulImageCnt = 0;
    
    pr_info("rcc - Starting ROM Completeness Check\n");
    
    /* alloc rcc work buffer */
    gpucRccWorkBuf = (unsigned char *)kzalloc((size_t)RCC_BUFLEN, GFP_KERNEL);
    if (gpucRccWorkBuf == NULL)
    {
        pr_err("rcc work buffer malloc fail, filename=%s, line=%d, func=%s\n", __FILE__, __LINE__, __func__);
        rcc_save_result(RCC_ERROR_BUFFER_ALLOC, RCC_META_NONE, RCC_IMG_NONE);
        goto RCC_END;
    }
    
    /* set algorithm information */
    ret = rcc_set_algo_info();
    if (ret < 0)
    {
        pr_err("rcc_set_algo_info fail, filename=%s, line=%d, func=%s, ret=%d\n", __FILE__, __LINE__, __func__, ret);
        goto RCC_END;
    }
    
    /* check meta binary */
    ret = rcc_check_meta_binary();
    if (ret < 0)
    {
        pr_err("rcc_check_meta_binary fail, filename=%s, line=%d, func=%s, ret=%d\n", __FILE__, __LINE__, __func__, ret);
        goto RCC_END;
    }
    
    /* set image information */
    ret = rcc_set_image_info();
    if (ret < 0)
    {
        pr_err("rcc_set_image_info fail, filename=%s, line=%d, func=%s, ret=%d\n", __FILE__, __LINE__, __func__, ret);
        goto RCC_END;
    }
    
    /* RCC(ROM Completeness Check) execute */
    ret = rcc_execute();
    if (ret < 0)
    {
        pr_err("rcc_execute fail, filename=%s, line=%d, func=%s, ret=%d\n", __FILE__, __LINE__, __func__, ret);
        goto RCC_END;
    }
    
RCC_END: /* end */
    
    /* reset check */
    if ((ret < 0) || (gpucRccWorkBuf == NULL))
    {
        if (gulRetryNum >= (RCC_CHECK_MAX_COUNT - (uint32_t)1))
        {
            rcc_fault_detection();
        }
        
        check_result = -1;
    }
    
    /* free rcc work buffer */
    if (gpucRccWorkBuf != NULL)
    {
        kfree(gpucRccWorkBuf);
        gpucRccWorkBuf = NULL;
    }
    
    /* free algorithm info buffer */
    if (gpRccAlgoInfo != NULL)
    {
        kfree(gpRccAlgoInfo);
        gpRccAlgoInfo = NULL;
    }
    
    /* free signature store buffer */
    for (ulImageCnt = 0; ulImageCnt < (uint32_t)RCC_IMG_NUM; ulImageCnt++)
    {
        if (gImageInfo[ulImageCnt].image_sign_ptr != NULL)
        {
            kfree(gImageInfo[ulImageCnt].image_sign_ptr);
            gImageInfo[ulImageCnt].image_sign_ptr = NULL;
        }
    }
    
    pr_info("rcc - End ROM Completeness Check\n");
    
    return check_result;
}


static int rcc_thread(void *arg)
{
    int32_t ret = 0;
    int32_t check_result = 0;
    uint32_t dna_state_check_retry_cnt = 0;
    
    pr_info("start rcc-thread\n");
    
    wake_lock_init(&(pcnfl_env->wlock), WAKE_LOCK_SUSPEND, "rcc_driver");
    wake_lock(&(pcnfl_env->wlock));
    
    /* save smem */
    ret = rcc_save_result_in_smem((uint8_t)RCC_RESULT_OK);
    if (ret < 0)
    {
        pr_err("smem save error, filename=%s, line=%d, func=%s, ret=%d\n", __FILE__, __LINE__, __func__, ret);
        wake_unlock(&(pcnfl_env->wlock));
        wake_lock_destroy(&(pcnfl_env->wlock));
        return -1;
    }

    /* Wait until DNA is ready. */
    while (kclsm_sysfs_is_dna_ready() == (bool)false) {
        dna_state_check_retry_cnt++;

        if (dna_state_check_retry_cnt == RCC_CHECK_DNA_STATE_RETRY_MAX) {
            panic("Failed to get DNA status!!");
        }

        set_current_state(TASK_INTERRUPTIBLE);
        (void) schedule_timeout((RCC_THREAD_DELAY_TIME * HZ));
    }

    (void) pr_info("Now, DNA is ready. cnt = %d\n", dna_state_check_retry_cnt);

    set_current_state(TASK_INTERRUPTIBLE);
    (void) schedule_timeout((RCC_THREAD_STANDBY_TIME * HZ));
    (void) pr_info("DNA initialize done\n");

    while (gulRetryNum < RCC_CHECK_MAX_COUNT)
    {
        check_result = rcc_thread_main();
        if (check_result == 0)
        {
            break;
        }
        gulRetryNum++;
    }
    
    wake_unlock(&(pcnfl_env->wlock));
    wake_lock_destroy(&(pcnfl_env->wlock));
    pr_info("end rcc-thread\n");
    
    return 0;
}


static int __init rcc_init( void )
{
    bool rcc_kfbm_enabled = (bool)false;
    uint8_t *kfbm_read_ptr = NULL;
    bool rcc_fota_enabled = (bool)false;
    uint8_t *fota_smem_ptr = NULL;

    pr_info("rcc driver loaded\n");

    /* Get KFBM status from SMEM */
    kfbm_read_ptr = (uint8_t *)kc_smem_alloc_get(SMEM_FACTORY_OPTIONS,4);
    if (kfbm_read_ptr != NULL) {
        if ((uint8_t)kfbm_read_ptr[3] == (uint8_t)0x01) {
            rcc_kfbm_enabled = (bool)true;
        }
    }

    /* Get develop mode from SMEM */
    if (rcc_kfbm_enabled == (bool)false) {
        kfbm_read_ptr = (uint8_t *)kc_smem_alloc_get(SMEM_USB_DEVELOPERS_FUNC_DISABLE, KCC_SMEM_USB_DEVELOPERS_FUNC_DISABLE_SIZE);
        if (kfbm_read_ptr != NULL) {
            if (*kfbm_read_ptr == (uint8_t)0x03) {
                rcc_kfbm_enabled = (bool)true;
            }
        }
    }

    /* Get FOTA boot mode from SMEM */
    fota_smem_ptr = (uint8_t *)kc_smem_alloc_get(SMEM_FOTA_BOOT_TYPE, 1);
    if (fota_smem_ptr != NULL) {
        if ((uint8_t)*fota_smem_ptr == FOTA_RECOVERY_BOOT || (uint8_t)*fota_smem_ptr == FOTA_RECOVERY_BACKUP_BOOT) {
            rcc_fota_enabled = (bool)true;
        }
    }

    if (rcc_kfbm_enabled == (bool)false && rcc_fota_enabled == (bool)false) {
        rcc_tsk = kthread_run(rcc_thread, NULL, "rcc execute");

        if (IS_ERR(rcc_tsk)) {
            (void) pr_err("error! rcc_thread ret = %d\n", IS_ERR(rcc_tsk));
            return -1;
        }
    } else {
        (void) pr_info("Disable rcc_thread for KFBM/FOTA enabled.");
    }

    return 0;
}


static void __exit rcc_exit( void )
{
    (void) kthread_stop(rcc_tsk);
    pr_info("rcc driver unloaded\n");
    
    return;
}


MODULE_LICENSE("GPL");

module_init(rcc_init);
module_exit(rcc_exit);
