/*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*
                              <kcjlog.c>
DESCRIPTION

EXTERNALIZED FUNCTIONS

This software is contributed or developed by KYOCERA Corporation.
(C) 2015 KYOCERA Corporation
(C) 2016 KYOCERA Corporation
(C) 2018 KYOCERA Corporation

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License version 2 and
only version 2 as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.

*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*/
#define pr_fmt(fmt) "KCJLOG: " fmt
/*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*
 * INCLUDE FILES
*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/rtc.h>
#include <linux/io.h>
#include <linux/pstore.h>
#include <linux/pstore_ram.h>
#include <linux/proc_fs.h>
#include <linux/kc_phymap.h>
#include <linux/kcjlog.h>
#include <linux/slab.h>
#include <soc/qcom/smsm.h>
#include "resetlog.h"

/*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*
 * CONSTATNT / DEFINE DECLARATIONS
*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*/
static const char crash_system[SYSTEM_MAX][CRASH_SYSTEM_SIZE] = {
	CRASH_SYSTEM_UNKNOWN,
	CRASH_SYSTEM_KERNEL,
	CRASH_SYSTEM_MODEM,
	CRASH_SYSTEM_ADSP,
	CRASH_SYSTEM_USERLAND,
	CRASH_SYSTEM_BOOT
};

static const char crash_kind[KIND_MAX][CRASH_KIND_SIZE] = {
	CRASH_KIND_UNKNOWN,
	CRASH_KIND_PANIC,
	CRASH_KIND_FATAL,
	CRASH_KIND_EXCEPTION,
	CRASH_KIND_WDOG_HW,
	CRASH_KIND_WDOG_SW
};

static uint32_t	bark_regaddr = 0;
static struct persistent_ram_zone *ram_console_zone;

static addr_offset_type addr_info;
static void __iomem *uninit_ram_virt_addr = NULL;

#define BOOT_LOG_FLAG_OFFSET       0x003FF010UL
#define BOOT_LOG_FLAG_ADDR         (MSM_UNINIT_RAM_BASE + BOOT_LOG_FLAG_OFFSET)
#define BOOT_FLAG_MAGIC            0x21545352UL
#define BOOT_FLAG_CLEAR            0x00000000UL

#define ID_POCKET_CONNECT          " "

/*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*
 * PROTOTYPE DECLARATION
*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*/

/*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*
 * FUNCTION DEFINITIONS
*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*====*/
/*
 * check_smem_kcjlog()
 *
 * Note: Check kcjlog from smem.
 */
static unsigned char check_smem_kcjlog( ram_log_info_type *pSmemLogInfo )
{
	unsigned char ret = 0;
	unsigned char cmp_system[ CRASH_SYSTEM_SIZE ];
	unsigned char cmp_kind[ CRASH_KIND_SIZE ];

	memcpy( cmp_system, pSmemLogInfo->crash_system, CRASH_SYSTEM_SIZE );

	/* system check */
	if ( (0 == strncmp(cmp_system, CRASH_SYSTEM_KERNEL   , strlen(CRASH_SYSTEM_KERNEL ))) ||
		 (0 == strncmp(cmp_system, CRASH_SYSTEM_MODEM    , strlen(CRASH_SYSTEM_MODEM  ))) ||
		 (0 == strncmp(cmp_system, CRASH_SYSTEM_ADSP     , strlen(CRASH_SYSTEM_ADSP   ))) ||
		 (0 == strncmp(cmp_system, CRASH_SYSTEM_USERLAND , strlen(CRASH_SYSTEM_USERLAND))) )
	{
		ret = 1;
	}

	memcpy( cmp_kind, pSmemLogInfo->crash_kind, CRASH_KIND_SIZE );

	/* kind check */
	if ( (0 == strncmp(cmp_kind, CRASH_KIND_PANIC      , strlen(CRASH_KIND_PANIC     ))) ||
		 (0 == strncmp(cmp_kind, CRASH_KIND_FATAL      , strlen(CRASH_KIND_FATAL     ))) ||
		 (0 == strncmp(cmp_kind, CRASH_KIND_EXCEPTION  , strlen(CRASH_KIND_EXCEPTION ))) ||
		 (0 == strncmp(cmp_kind, CRASH_KIND_WDOG_HW    , strlen(CRASH_KIND_WDOG_HW   ))) ||
		 (0 == strncmp(cmp_kind, CRASH_KIND_WDOG_SW    , strlen(CRASH_KIND_WDOG_SW   ))) )
	{
		ret = 1;
	}
	return ret;
}

/*
 * set_smem_kcjlog()
 *
 * Note: Set kcjlog to smem.
 */
void set_smem_kcjlog(crash_system_type system_type, crash_kind_type kind)
{
	ram_log_info_type *pSmemLogInfo;

	mb();
	pSmemLogInfo = (ram_log_info_type *)kc_smem_alloc( SMEM_CRASH_LOG, SIZE_SMEM_ALLOC );

	if ((pSmemLogInfo == NULL) || (system_type >= SYSTEM_MAX) || (kind >= KIND_MAX)) {
		pr_info( "[%s] pSmemLogInfo:%p\n", __func__, pSmemLogInfo );
		return;
	}
	if (check_smem_kcjlog(pSmemLogInfo) == 1) {
		return;
	}
	memcpy( &pSmemLogInfo->crash_system[0], crash_system[system_type], strlen(crash_system[system_type]) );
	memcpy( &pSmemLogInfo->crash_kind[0],   crash_kind[kind],          strlen(crash_kind[kind])          );
	mb();
}

/*
 * set_kcj_fixed_info()
 *
 * Note: Set kcj fixed info to uninit ram.
 */
static void set_kcj_fixed_info(void)
{
	ram_log_info_type *pPanicInfo;
	ram_log_info_type *pSmemLogInfo;
	unsigned char   temp_magic[MAGIC_CODE_SIZE];
	unsigned int build_id_len, pocket_ver_len, connect_len, buffer_len;
	unsigned char   temp_buf[VERSION_SIZE];

	mb();
	pPanicInfo   = (ram_log_info_type *)addr_info.addr_control_info;
	pSmemLogInfo = (ram_log_info_type *)kc_smem_alloc( SMEM_CRASH_LOG, SIZE_SMEM_ALLOC );

	if ( (pPanicInfo == NULL) || (pSmemLogInfo == NULL) ) {
		pr_info( "[%s] pPanicInfo:%p pSmemLogInfo:%p\n", __func__, pPanicInfo, pSmemLogInfo );
		return;
	}

	build_id_len = strlen(BUILD_DISPLAY_ID);
	connect_len = strlen(ID_POCKET_CONNECT);
	pocket_ver_len = strlen(POCKET_VERSION);
	buffer_len = sizeof(((ram_log_info_type *)NULL)->linux_ver);

	memcpy_toio( &pPanicInfo->magic_code[0]  , CRASH_MAGIC_CODE   , strlen( CRASH_MAGIC_CODE ) );

	/* set linux_banner to uninit ram & smem crash log */
	memset_io(&pPanicInfo->linux_ver[0], 0, buffer_len);
	memset(&pSmemLogInfo->linux_ver[0], 0, buffer_len);
	memcpy_toio(&pPanicInfo->linux_ver[0],
				BUILD_DISPLAY_ID,
				build_id_len);
	memcpy_toio( &pPanicInfo->model[0]     , PRODUCT_MODEL_NAME , strlen( PRODUCT_MODEL_NAME ));
	memcpy(&pSmemLogInfo->linux_ver[0] , BUILD_DISPLAY_ID   , build_id_len);
	memcpy(&pSmemLogInfo->model[0]     , PRODUCT_MODEL_NAME , strlen( PRODUCT_MODEL_NAME ));
	if (buffer_len > (build_id_len+connect_len+pocket_ver_len)) {
		if (buffer_len == sizeof(temp_buf)) {
			memset(temp_buf, 0, sizeof(temp_buf));
			memcpy(temp_buf, ID_POCKET_CONNECT, connect_len);
			strlcat(temp_buf,
					POCKET_VERSION,
					sizeof(temp_buf)-connect_len-1UL);
			memcpy_toio(&pPanicInfo->linux_ver[build_id_len],
						temp_buf,
						strlen(temp_buf));
			memcpy(&pSmemLogInfo->linux_ver[build_id_len],
					temp_buf,
					strlen(temp_buf));
		}
	}

	memcpy_fromio( temp_magic  , &pPanicInfo->magic_code[0]   , MAGIC_CODE_SIZE );

}

/*
* set_kcj_fixed_info_modem()
 *
 * Note: Set kcj modem fixed info to uninit ram.
 */
void set_kcj_fixed_info_modem(void)
{
    ram_log_info_type *pPanicInfo;
    ram_log_info_type *pSmemLogInfo;

    mb();
    pPanicInfo   = (ram_log_info_type *)addr_info.addr_control_info;
    pSmemLogInfo = (ram_log_info_type *)kc_smem_alloc( SMEM_CRASH_LOG, SIZE_SMEM_ALLOC );

    if ( (pPanicInfo == NULL) || (pSmemLogInfo == NULL) ) {
		    pr_info( "[%s] pPanicInfo:%p pSmemLogInfo:%p\n", __func__, pPanicInfo, pSmemLogInfo );
            return;
    }

    /* set modem info on smem to uninit ram */
    memcpy_toio( &pPanicInfo->modem_ver[0], &pSmemLogInfo->modem_ver[0], VERSION_SIZE );
}

/*
 * set_kcj_usrland_log()
 *
 * Note: Set usrland log to uninit ram.
 */
void set_kcj_usrland_log(const char *log_data)
{
	unsigned char *userland_log;

	userland_log= (unsigned char *)addr_info.addr_userland_log;

	if ((log_data == NULL) || (userland_log == NULL)){
		return;
	}

	memset_io(userland_log, 0x00, SIZE_USERLAND_LOG);
	memcpy_toio(userland_log, log_data, (SIZE_USERLAND_LOG-1));
}

/*
 * get_crash_time()
 *
 * Note: Get utc time.
 */
static void get_crash_time(unsigned char *buf, unsigned int bufsize)
{
	struct timespec ts_now;
	struct rtc_time tm;
	char   tmp_buf[CRASH_TIME_SIZE];
	unsigned int copy_size;

	ts_now = current_kernel_time();
	rtc_time_to_tm(ts_now.tv_sec, &tm);

	snprintf( tmp_buf, sizeof(tmp_buf), "%d-%02d-%02d %02d:%02d:%02d.%09lu UTC",
				tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday,
				tm.tm_hour, tm.tm_min, tm.tm_sec, ts_now.tv_nsec);

	memset_io( buf, 0x00, bufsize );

	copy_size = min(bufsize, (unsigned int)CRASH_TIME_SIZE);
	memcpy_toio( buf, tmp_buf, copy_size );

}

/*
 * set_kcj_crash_info()
 *
 * Note: Set crash info to uninit ram.
 */
void set_kcj_crash_info(void)
{
	ram_log_info_type *pPanicInfo;
	ram_log_info_type *pSmemLogInfo;

	mb();
	pPanicInfo   = (ram_log_info_type *)addr_info.addr_control_info;
	pSmemLogInfo = (ram_log_info_type *)kc_smem_alloc( SMEM_CRASH_LOG, SIZE_SMEM_ALLOC );

	if ( (pPanicInfo == NULL) || (pSmemLogInfo == NULL) ) {
		pr_info( "[%s] pPanicInfo:%p pSmemLogInfo:%p\n", __func__, pPanicInfo, pSmemLogInfo );
		return;
	}
	if (check_smem_kcjlog(pSmemLogInfo) == 0) {
		clear_kcj_crash_info();
		return;
	}
	set_kcj_fixed_info();
	set_modemlog_info();

	memset_io( &pPanicInfo->crash_system[0], 0x00, CRASH_SYSTEM_SIZE );
	memset_io( &pPanicInfo->crash_kind[0]  , 0x00, CRASH_KIND_SIZE );

	memcpy_toio( &pPanicInfo->crash_system[0], &pSmemLogInfo->crash_system[0], CRASH_SYSTEM_SIZE );
	memcpy_toio( &pPanicInfo->crash_kind[0]  , &pSmemLogInfo->crash_kind[0]  , CRASH_KIND_SIZE );

	get_crash_time( &pPanicInfo->crash_time[0], CRASH_TIME_SIZE );
}

/*
 * set_kcj_regsave_addr()
 *
 * Note: Set regsave addr to uninit ram.
 */
void set_kcj_regsave_addr(unsigned int regsave_addr)
{
	ram_log_info_type *pPanicInfo;

	pPanicInfo   = (ram_log_info_type *)addr_info.addr_control_info;
	pPanicInfo->regsave_addr = regsave_addr;
	pPanicInfo->regsave_addr_check = ~regsave_addr;
	bark_regaddr = regsave_addr;
}

/*
 * set_kcj_unknown_info()
 *
 * Note: Set crash unknown info to uninit ram.
 */
static void set_kcj_unknown_info(void)
{
	ram_log_info_type *pPanicInfo;

	pPanicInfo = (ram_log_info_type *)addr_info.addr_control_info;

	if ( pPanicInfo == NULL ) {
		pr_info( "[%s] pPanicInfo:%p\n", __func__, pPanicInfo );
		return;
	}

	memcpy_toio( &pPanicInfo->crash_system[0], CRASH_SYSTEM_UNKNOWN, strlen(CRASH_SYSTEM_UNKNOWN) );
	memcpy_toio( &pPanicInfo->crash_kind[0]  , CRASH_KIND_UNKNOWN  , strlen(CRASH_KIND_UNKNOWN  ) );
}

/*
 * clear_kcj_crash_info()
 *
 * Note: Clear crash info from uninit ram.
 */
void clear_kcj_crash_info( void )
{
	ram_log_info_type *pPanicInfo;

	pPanicInfo = (ram_log_info_type *)addr_info.addr_control_info;

	if ( pPanicInfo == NULL ) {
		pr_info( "[%s] pPanicInfo:%p\n", __func__, pPanicInfo );
		return;
	}

	memset_io( &pPanicInfo->crash_system[0], 0x00, CRASH_SYSTEM_SIZE );
	memset_io( &pPanicInfo->crash_kind[0]  , 0x00, CRASH_KIND_SIZE );
	memset_io( &pPanicInfo->crash_time[0]  , 0x00, CRASH_TIME_SIZE );
}

/*
 * set_modemlog_info()
 *
 * Note: Set modem log info from smem to uninit ram.
 */
void set_modemlog_info(void)
{
	int i;
	ram_log_info_type *pSmemLogInfo;
	ram_log_info_type *pPanicInfo;

	mb();
	pSmemLogInfo = (ram_log_info_type *)kc_smem_alloc( SMEM_CRASH_LOG, SIZE_SMEM_ALLOC );
	pPanicInfo   = (ram_log_info_type *)addr_info.addr_control_info;

	if ( (pSmemLogInfo == NULL) || (pPanicInfo == NULL) ) {
		pr_info( "[%s] pPanicInfo:%p pSmemLogInfo:%p\n", __func__, pPanicInfo, pSmemLogInfo );
		return;
	}

	for( i = 0 ; i < MINFO_SIZE ; i++ )
	{
		pPanicInfo->m_info[i]  = pSmemLogInfo->m_info[i];
	}

    memcpy_toio( &pPanicInfo->modem_ver[0], &pSmemLogInfo->modem_ver[0], VERSION_SIZE );

}

static int kcj_uninit_ram_ioremap( void )
{
	/* uninit ram  */
	if ( !request_mem_region(MSM_KCJLOG_BASE, MSM_KCJLOG_SIZE, "kc_uninit") )
	{
		printk( "memory region request fail for uninit ram." );
		return -1;
	}

	uninit_ram_virt_addr = ioremap_nocache( MSM_KCJLOG_BASE, MSM_KCJLOG_SIZE );
	if( NULL == uninit_ram_virt_addr )
	{
		printk( "ioremap fail for uninit ram." );
		return -1;
	}

	addr_info.addr_control_info       = uninit_ram_virt_addr;
	addr_info.addr_kcj_ram_console    = uninit_ram_virt_addr + SIZE_CONTROL_INFO;
	addr_info.addr_kernel_log         = addr_info.addr_kcj_ram_console;
	addr_info.addr_logcat_main        = addr_info.addr_kernel_log         + SIZE_KERNEL_LOG;
	addr_info.addr_logcat_system      = addr_info.addr_logcat_main        + SIZE_LOGCAT_MAIN;
	addr_info.addr_logcat_events      = addr_info.addr_logcat_system      + SIZE_LOGCAT_SYSTEM;
	addr_info.addr_logcat_radio       = addr_info.addr_logcat_events      + SIZE_LOGCAT_EVENTS;
    addr_info.addr_userland_log       = addr_info.addr_logcat_radio       + SIZE_LOGCAT_RADIO;
	addr_info.addr_smem_event_log     = addr_info.addr_userland_log       + SIZE_USERLAND_LOG;
	addr_info.addr_modem_f3_log       = addr_info.addr_smem_event_log     + SIZE_SMEM_EVENT_LOG;
	addr_info.addr_modem_err_data_log = addr_info.addr_modem_f3_log       + SIZE_MODEM_F3_LOG;

	printk( "[Y.Y] kcj_uninit_ram_ioremap: uninit_ram_virt_addr = 0x%x\n", (uint32_t)uninit_ram_virt_addr );

	return 0;
}

static void __iomem *kcj_get_uninit_ram_base_addr( void )
{
	if ( NULL == uninit_ram_virt_addr )
	{
		if ( -1 == kcj_uninit_ram_ioremap() )
		{
			return NULL;
		}
	}

	return( uninit_ram_virt_addr );
}

/*
 * kcj_last_kmsg_read()
 *
 * Note: read last_kmsg.
 */
static ssize_t kcj_last_kmsg_read(struct file *file, char __user *buf, size_t len, loff_t *offset )
{
	loff_t pos = *offset;
	ssize_t count;
	struct persistent_ram_zone *prz = ram_console_zone;
	size_t old_log_size = persistent_ram_old_size(prz);
	const char *old_log = persistent_ram_old(prz);
	char *str;
	int ret;

	if (dmesg_restrict && !capable(CAP_SYSLOG))
		return -EPERM;

	/* Main last_kmsg log */
	if (pos < old_log_size) {
		count = min(len, (size_t)(old_log_size - pos));
		if (copy_to_user(buf, old_log + pos, count))
			return -EFAULT;
		goto out;
	}

	/* ECC correction notice */
	pos -= old_log_size;
	count = persistent_ram_ecc_string(prz, NULL, 0);
	if (pos < count) {
		str = kmalloc(count, GFP_KERNEL);
		if (!str)
			return -ENOMEM;
		persistent_ram_ecc_string(prz, str, count + 1);
		count = min(len, (size_t)(count - pos));
		ret = copy_to_user(buf, str + pos, count);
		kfree(str);
		if (ret)
			return -EFAULT;
		goto out;
	}
	/* EOF */
	return 0;

out:
	*offset += count;
	return count;
}

static const struct file_operations kcj_last_kmsg_file_ops = {
	.owner = THIS_MODULE,
	.read = kcj_last_kmsg_read,
};

/*
 * kcj_last_kmsg_late_init()
 *
 * Note: last_kmsg init.
 */
static int __init kcj_last_kmsg_late_init( void )
{
	struct proc_dir_entry *entry;

	if ( ram_console_zone == NULL )
		goto out;

	if (persistent_ram_old_size(ram_console_zone) == 0)
	{
		pr_err("last_kmsg is not found.\n");
		goto out;
	}

	entry = proc_create( "last_kmsg", S_IFREG | S_IRUGO, NULL, &kcj_last_kmsg_file_ops);
	if ( !entry )
	{
		printk( "%s: failed to create proc entry\n", __func__ );
	    persistent_ram_free_old(ram_console_zone);
		goto out;
	}
out:
	return 0;
}

/*
 * kcj_pstore_open()
 *
 * Note: callback from pstore. no operation.
 */
static int kcj_pstore_open(struct pstore_info *psi)
{
	/* NOP */
	return 0;
}

/*
 * kcj_pstore_close()
 *
 * Note: callback from pstore. no operation.
 */
static int kcj_pstore_close(struct pstore_info *psi)
{
	/* NOP */
	return 0;
}

/*
 * kcj_pstore_read()
 *
 * Note: callback from pstore. no operation.
 */
static ssize_t kcj_pstore_read(u64 *id, enum pstore_type_id *type,
							int *count, struct timespec *timespec,
							char **buf, bool *compressed,
							struct pstore_info *psi)
{
	/* NOP */
	return 0;
}

/*
 * kcj_pstore_write()
 *
 * Note: callback from pstore. write console to uninit ram.
 */
static int kcj_pstore_write(enum pstore_type_id type,
							enum kmsg_dump_reason reason, u64 *id,
							unsigned int part, int count, bool compressed,
							size_t size, struct pstore_info *psi)
{
	/* Write console to uninit ram */
    persistent_ram_write(ram_console_zone, psi->buf, size);
	return 0;
}

/*
 * kcj_pstore_erase()
 *
 * Note: callback from pstore. no operation.
 */
static int kcj_pstore_erase(enum pstore_type_id type, u64 id, int count,
							struct timespec time_sp, struct pstore_info *psi)
{
	/* NOP */
	return 0;
}

static struct pstore_info kcj_pstore_info = {
	.owner		= THIS_MODULE,
	.name		= "kcjlog",
	.open		= kcj_pstore_open,
	.close		= kcj_pstore_close,
	.read		= kcj_pstore_read,
	.write		= kcj_pstore_write,
	.erase		= kcj_pstore_erase,
};

/*
 * kcj_ram_console_init()
 *
 * Note: kcj ram console init.
 */
static int __init kcj_ram_console_init(void)
{
	struct persistent_ram_zone *prz;
	struct persistent_ram_ecc_info ecc_info;

	unsigned int memtype = 1;

	if ( NULL == kcj_get_uninit_ram_base_addr() ) return -1;

    /* set default ecc */
	ecc_info.block_size =   128;
	ecc_info.ecc_size   =    16;
	ecc_info.symsize    =     8;
	ecc_info.poly       = 0x11d;

	/* create persistent ram */
	prz = persistent_ram_new((phys_addr_t)addr_info.addr_kcj_ram_console, SIZE_KERNEL_LOG, 0, &ecc_info, memtype);

	if (IS_ERR(prz)) {
		int err = PTR_ERR(prz);
		pr_err( "failed kcjlog persistent_ram err:%d\n", err);
		goto out;
	}
	ram_console_zone = prz;
	persistent_ram_zap(ram_console_zone);

	/* register  pstore */
	kcj_pstore_info.bufsize = 1024; /* LOG_LINE_MAX */
	kcj_pstore_info.buf     = kmalloc(kcj_pstore_info.bufsize, GFP_KERNEL);
	if (kcj_pstore_info.buf == NULL)
	{
		pr_err( "failed kcjlog kmalloc:\n");
		goto out;
	}
	spin_lock_init(&kcj_pstore_info.buf_lock);
	pstore_register(&kcj_pstore_info);

out:
    return 0;
}

static void kcj_clear_boot_magic( void )
{
	void __iomem *boot_flag = NULL;

	if (!request_mem_region(BOOT_LOG_FLAG_ADDR, sizeof(uint32_t), "boot_flag")) {
		pr_err("[%s] ERROR: request_mem_region()\n", __func__);
		return;
	}

	boot_flag  = ioremap_nocache(BOOT_LOG_FLAG_ADDR, sizeof(uint32_t));
	if (boot_flag == NULL) {
		release_mem_region(BOOT_LOG_FLAG_ADDR, sizeof(uint32_t));
		pr_err("[%s] NULL == ioremap_nocache()\n", __func__);
		return;
	}

	writel(BOOT_FLAG_CLEAR, boot_flag);
	printk("[%s] Clear boot_flag SUCCESS", __func__ );

	iounmap(boot_flag);
	release_mem_region(BOOT_LOG_FLAG_ADDR, sizeof(uint32_t));

	return;
}

/*
 * kcjlog_init()
 *
 * Note: kcjlog init.
 */
static int __init kcjlog_init(void)
{
	kcj_clear_boot_magic();

	if ( NULL == kcj_get_uninit_ram_base_addr() ) return -1;

	memset_io( addr_info.addr_control_info, 0x00, SIZE_CONTROL_INFO );
	if(bark_regaddr != 0)
	{
		set_kcj_regsave_addr(bark_regaddr);
	}
	set_kcj_fixed_info();
	set_kcj_unknown_info();

	return 0;
}

/*
 * get_uninit_control_info_ram_addr()
 *
 * Note: get uninit ram address.
 */
void __iomem *get_uninit_control_info_ram_addr( void )
{
    if( NULL != kcj_get_uninit_ram_base_addr() )
    {
        return ( addr_info.addr_control_info );
    }

    return NULL;
}

/*
 * get_uninit_logcat_ram_addr()
 *
 * Note: get uninit ram address.
 */
void __iomem *get_uninit_logcat_main_ram_addr( void )
{
    if( NULL != kcj_get_uninit_ram_base_addr() )
    {
        return ( addr_info.addr_logcat_main );
    }

    return NULL;
}

/*
 * get_uninit_logcat_system_ram_addr()
 *
 * Note: get uninit ram address.
 */
void __iomem *get_uninit_logcat_system_ram_addr( void )
{
    if( NULL != kcj_get_uninit_ram_base_addr() )
    {
        return ( addr_info.addr_logcat_system );
    }

    return NULL;
}

/*
 * get_uninit_logcat_events_ram_addr()
 *
 * Note: get uninit ram address.
 */
void __iomem *get_uninit_logcat_events_ram_addr( void )
{
    if( NULL != kcj_get_uninit_ram_base_addr() )
    {
        return ( addr_info.addr_logcat_events );
    }

    return NULL;
}

/*
 * get_uninit_logcat_radio_ram_addr()
 *
 * Note: get uninit ram address.
 */
void __iomem *get_uninit_radio_events_ram_addr( void )
{
    if( NULL != kcj_get_uninit_ram_base_addr() )
    {
        return ( addr_info.addr_logcat_radio );
    }

    return NULL;
}

static int __init kcj_uninitram_init( void )
{
	addr_info.addr_control_info       = NULL;
	addr_info.addr_kcj_ram_console    = NULL;
	addr_info.addr_kernel_log         = NULL;
	addr_info.addr_logcat_main        = NULL;
	addr_info.addr_logcat_system      = NULL;
	addr_info.addr_logcat_events      = NULL;
	addr_info.addr_logcat_radio       = NULL;
	addr_info.addr_userland_log       = NULL;
	addr_info.addr_smem_event_log     = NULL;
	addr_info.addr_modem_f3_log       = NULL;
	addr_info.addr_modem_err_data_log = NULL;

	return 0;
}

pure_initcall(kcj_uninitram_init);
device_initcall(kcjlog_init);
device_initcall(kcj_ram_console_init);
late_initcall(kcj_last_kmsg_late_init);
