/*
 * f_whcm.c -- USB WHCM function driver
 *
 * Copyright (C) 2010 Fabien Chouteau <fabien.chouteau@barco.com>
 * Copyright (C) 2014, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * The Linux Foundation chooses to take subject only to the GPLv2 license
 * terms, and distributes only under these terms.
 */

/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2019 KYOCERA Corporation
 * (C) 2020 KYOCERA Corporation
 */

#include <linux/kernel.h>
#include <linux/module.h>

struct f_whcm {
	struct usb_function func;
};

struct usb_cdc_whcm_union_desc {
	__u8 bLength;
	__u8 bDescriptorType;
	__u8 bDescriptorSubType;
	__u8 bControlInterface;
	__u8 bSubordinateInterface0;
	__u8 bSubordinateInterface1;
	__u8 bSubordinateInterface2;
	__u8 bSubordinateInterface3;
	__u8 bSubordinateInterface4;
} __attribute__ ((packed));

struct usb_cdc_whcm_union_develop_desc {
	__u8 bLength;
	__u8 bDescriptorType;
	__u8 bDescriptorSubType;
	__u8 bControlInterface;
	__u8 bSubordinateInterface0;
	__u8 bSubordinateInterface1;
	__u8 bSubordinateInterface2;
	__u8 bSubordinateInterface3;
	__u8 bSubordinateInterface4;
	__u8 bSubordinateInterface5;
	__u8 bSubordinateInterface6;
} __attribute__ ((packed));

struct usb_cdc_whcm_union_audio_desc {
	__u8 bLength;
	__u8 bDescriptorType;
	__u8 bDescriptorSubType;
	__u8 bControlInterface;
	__u8 bSubordinateInterface0;
	__u8 bSubordinateInterface1;
	__u8 bSubordinateInterface2;
	__u8 bSubordinateInterface3;
	__u8 bSubordinateInterface4;
	__u8 bSubordinateInterface5;
	__u8 bSubordinateInterface6;
	__u8 bSubordinateInterface7;
} __attribute__ ((packed));

struct usb_cdc_whcm_union_audio_develop_desc {
	__u8 bLength;
	__u8 bDescriptorType;
	__u8 bDescriptorSubType;
	__u8 bControlInterface;
	__u8 bSubordinateInterface0;
	__u8 bSubordinateInterface1;
	__u8 bSubordinateInterface2;
	__u8 bSubordinateInterface3;
	__u8 bSubordinateInterface4;
	__u8 bSubordinateInterface5;
	__u8 bSubordinateInterface6;
	__u8 bSubordinateInterface7;
	__u8 bSubordinateInterface8;
	__u8 bSubordinateInterface9;
} __attribute__ ((packed));

static struct usb_interface_descriptor whcm_interface_desc = {
	.bLength            = sizeof(whcm_interface_desc),
	.bDescriptorType    = USB_DT_INTERFACE,
	.bAlternateSetting  = 0,
	.bNumEndpoints      = 0,
	.bInterfaceClass    = USB_CLASS_COMM,
	.bInterfaceSubClass = 8,
	.bInterfaceProtocol = 0,
};

static struct usb_cdc_header_desc whcm_header_desc = {
	.bLength            = sizeof(whcm_header_desc),
	.bDescriptorType    = USB_DT_CS_INTERFACE,
	.bDescriptorSubType = USB_CDC_HEADER_TYPE,
	.bcdCDC             = cpu_to_le16(0x0110),
};

static struct usb_cdc_obex_desc whcm_obex_desc = {
	.bLength            = sizeof(whcm_obex_desc),
	.bDescriptorType    = USB_DT_CS_INTERFACE,
	.bDescriptorSubType = USB_CDC_WHCM_TYPE,
	.bcdVersion         = cpu_to_le16(0x0100),
};

static struct usb_cdc_whcm_union_desc whcm_union_desc_constitution = {
	.bLength            = sizeof(whcm_union_desc_constitution),
	.bDescriptorType    = USB_DT_CS_INTERFACE,
	.bDescriptorSubType = USB_CDC_UNION_TYPE,
	.bControlInterface  = 0,
	.bSubordinateInterface0 = 1,
	.bSubordinateInterface1 = 2,
	.bSubordinateInterface2 = 3,
	.bSubordinateInterface3 = 4,
	.bSubordinateInterface4 = 5,
};

static struct usb_cdc_whcm_union_develop_desc whcm_union_desc_constitution_develop = {
	.bLength            = sizeof(whcm_union_desc_constitution_develop),
	.bDescriptorType    = USB_DT_CS_INTERFACE,
	.bDescriptorSubType = USB_CDC_UNION_TYPE,
	.bControlInterface  = 0,
	.bSubordinateInterface0 = 1,
	.bSubordinateInterface1 = 2,
	.bSubordinateInterface2 = 3,
	.bSubordinateInterface3 = 4,
	.bSubordinateInterface4 = 5,
	.bSubordinateInterface5 = 6,
	.bSubordinateInterface6 = 7,
};

static struct usb_cdc_whcm_union_audio_desc whcm_union_desc_constitution_audio = {
	.bLength            = sizeof(whcm_union_desc_constitution_audio),
	.bDescriptorType    = USB_DT_CS_INTERFACE,
	.bDescriptorSubType = USB_CDC_UNION_TYPE,
	.bControlInterface  = 0,
	.bSubordinateInterface0 = 1,
	.bSubordinateInterface1 = 2,
	.bSubordinateInterface2 = 3,
	.bSubordinateInterface3 = 4,
	.bSubordinateInterface4 = 5,
	.bSubordinateInterface5 = 6,
	.bSubordinateInterface6 = 7,
	.bSubordinateInterface7 = 8,
};

static struct usb_cdc_whcm_union_audio_develop_desc whcm_union_desc_constitution_audio_develop = {
	.bLength            = sizeof(whcm_union_desc_constitution_audio_develop),
	.bDescriptorType    = USB_DT_CS_INTERFACE,
	.bDescriptorSubType = USB_CDC_UNION_TYPE,
	.bControlInterface  = 0,
	.bSubordinateInterface0 = 1,
	.bSubordinateInterface1 = 2,
	.bSubordinateInterface2 = 3,
	.bSubordinateInterface3 = 4,
	.bSubordinateInterface4 = 5,
	.bSubordinateInterface5 = 6,
	.bSubordinateInterface6 = 7,
	.bSubordinateInterface7 = 8,
	.bSubordinateInterface8 = 9,
	.bSubordinateInterface9 = 10,
};

static struct usb_descriptor_header *whcm_descriptors[] = {
	(struct usb_descriptor_header *)&whcm_interface_desc,
	(struct usb_descriptor_header *)&whcm_header_desc,
	(struct usb_descriptor_header *)&whcm_obex_desc,
	(struct usb_descriptor_header *)&whcm_union_desc_constitution,
	NULL,
};

static struct usb_string whcm_func_string_defs[] = {
	[0].s = "WHCM Interface",
	{},
};

static struct usb_gadget_strings whcm_func_string_table = {
	.language = 0x0409, /* en-US */
	.strings  = whcm_func_string_defs,
};

static struct usb_gadget_strings *whcm_func_strings[] = {
	&whcm_func_string_table,
	NULL,
};

#define USB_CONSTITUTION_NORMAL       0x00U
#define USB_CONSTITUTION_DEVELOP      0x10U
#define USB_CONSTITUTION_AUDIO        0x20U
#define USB_CONSTITUTION_AUDIO_DEVLOP (USB_CONSTITUTION_DEVELOP|USB_CONSTITUTION_AUDIO)

static __u8 constitution = 0;

static void set_constitution(__u8 no)
{
	constitution = no;
}

static void whcm_free_descriptors(struct usb_function *f)
{
	if (f->hs_descriptors)
		usb_free_descriptors(f->hs_descriptors);
	if (f->fs_descriptors)
		usb_free_descriptors(f->fs_descriptors);
}

static void whcm_disable(struct usb_function *f)
{
	return;
}

static int whcm_set_alt(struct usb_function *f, unsigned intf, unsigned alt)
{
	return 0;
}

static int whcm_bind(struct usb_configuration *c, struct usb_function *f)
{
	int status;

	/* String Descriptor always rebuilds. */
	whcm_func_string_defs[0].id = 0;

	if (whcm_func_string_defs[0].id == 0) {
		status = usb_string_id(c->cdev);
		if (status < 0) {
			pr_err("%s: failed to set offset\n", __func__);
			return status;
		}
		whcm_func_string_defs[0].id = status;
		whcm_interface_desc.iInterface = status;
	}

	status = usb_interface_id(c, f);
	if (status < 0) {
		pr_err("%s: failed to get Interface Number\n", __func__);
		return status;
	}

	whcm_interface_desc.bInterfaceNumber = status;

	/* Decide whcm_descriptors from constitution */
	switch (constitution) {
	case USB_CONSTITUTION_NORMAL:
		whcm_descriptors[3] = (struct usb_descriptor_header *)&whcm_union_desc_constitution;
		break;
	case USB_CONSTITUTION_DEVELOP:
		whcm_descriptors[3] = (struct usb_descriptor_header *)&whcm_union_desc_constitution_develop;
		break;
	case USB_CONSTITUTION_AUDIO:
		whcm_descriptors[3] = (struct usb_descriptor_header *)&whcm_union_desc_constitution_audio;
		break;
	case USB_CONSTITUTION_AUDIO_DEVLOP:
		whcm_descriptors[3] = (struct usb_descriptor_header *)&whcm_union_desc_constitution_audio_develop;
		break;
	default:
		pr_err("%s: invalid constitution(0x%x).\n", __func__, constitution);
		return -ENOMEM;
	}

	f->fs_descriptors = usb_copy_descriptors(whcm_descriptors);
	if (!f->fs_descriptors) {
		pr_err("%s: failed to copy descriptors.\n", __func__);
		return -ENOMEM;
	}

	if (gadget_is_dualspeed(c->cdev->gadget)) {
		f->hs_descriptors = usb_copy_descriptors(whcm_descriptors);
		if (!f->hs_descriptors) {
			pr_err("%s: failed to copy descriptors.\n", __func__);
			whcm_free_descriptors(f);
			return -ENOMEM;
		}
	}

	return 0;
}

static void whcm_unbind(struct usb_configuration *c, struct usb_function *f)
{
	struct f_whcm *whcm = container_of(f, struct f_whcm, func);

	whcm_free_descriptors(f);
	kfree(whcm);
}

static int whcm_bind_config(struct android_usb_function *f, struct usb_configuration *c)
{
	struct f_whcm *whcm;
	int status;

	whcm = kzalloc(sizeof(*whcm), GFP_KERNEL);
	if (!whcm) {
		pr_err("%s: failed to allocate memory.\n", __func__);
		return -ENOMEM;
	}

	whcm->func.name    = "whcm";
	whcm->func.strings = whcm_func_strings;
	whcm->func.bind    = whcm_bind;
	whcm->func.unbind  = whcm_unbind;
	whcm->func.set_alt = whcm_set_alt;
	whcm->func.disable = whcm_disable;

	status = usb_add_function(c, &whcm->func);
	if (status) {
		pr_err("%s: failed to add function.\n", __func__);
		kfree(whcm);
	}

	return status;
}
