/*
 * carrier boards communicatons core.
 * demultiplexes the cbc protocol.
 *
 * Copryright (C) 2014 Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 */
#include "link/cbc_kmod_load_monitoring.h"
#include <linux/string.h>
#include <linux/sysfs.h>
#include <linux/stat.h>

struct cbc_load_monitoring {
	u8 tx_queue_average;
	u8 tx_queue_max;
	u8 tx_queue_overload;
	u8 tx_uart_average;
	u8 tx_uart_max;
	u8 rx_uart_average;
	u8 rx_uart_max;
};

static struct cbc_load_monitoring load_monitoring;


/* callbacks for the cbc-core */

#if (CBC_LOAD_MONITOR_ENABLE_FRAME_QUEUE_UTILIZATION == 1U)
static void on_cbc_load_queue_utilization_cb(enum  cbc_frame_queue_utilization tx_average_utilization
		, enum cbc_frame_queue_utilization tx_max_utilization)
{
	load_monitoring.tx_queue_average = tx_average_utilization;
	load_monitoring.tx_queue_max = tx_max_utilization;
}


static void on_cbc_load_queue_overload_cb(void)
{
	if (load_monitoring.tx_queue_overload < 0xFF)
		load_monitoring.tx_queue_overload++;
}

#endif

#if (CBC_LOAD_MONITOR_ENABLE_UART_TRANSMISSION_MONITOR == 1U)
static void on_cbc_load_tx_uart_cb(u8 tx_average_load, u8 tx_max_load)
{
	load_monitoring.tx_uart_average = tx_average_load;
	load_monitoring.tx_uart_max = tx_max_load;
}
#endif

#if (CBC_LOAD_MONITOR_ENABLE_UART_RECEPTION_MONITOR == 1U)
static void on_cbc_load_rx_uart_cb(u8 rx_average_load, u8 rx_max_load)
{
	load_monitoring.rx_uart_average = rx_average_load;
	load_monitoring.rx_uart_max = rx_max_load;
}
#endif

/* sys files for the values */

static ssize_t load_monitoring_reset_store(struct class *cls,
					   struct class_attribute *attr,
					   const char *buf, size_t count)
{
	pr_info("reset load monitoring counters\n");
	/* reset counters */
#if (CBC_LOAD_MONITOR_ENABLE_UART_RECEPTION_MONITOR == 1U)
	cbc_load_monitor_reset_checksum_errors_on_receive();
	cbc_load_monitor_reset_sequence_counter_errors_on_receive();
#endif /* (CBC_LOAD_MONITOR_ENABLE_UART_RECEPTION_MONITOR == 1U) */
	load_monitoring.tx_queue_overload = 0;
	return count;
}

static ssize_t tx_queue_average_show(struct class *cls,
				     struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", load_monitoring.tx_queue_average);
}

static ssize_t tx_queue_max_show(struct class *cls,
				 struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", load_monitoring.tx_queue_max);
}

static ssize_t tx_overloads_show(struct class *cls,
				 struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", load_monitoring.tx_queue_overload);
}

static ssize_t tx_uart_average_show(struct class *cls,
				    struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", load_monitoring.tx_uart_average);
}

static ssize_t tx_uart_max_show(struct class *cls,
				struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", load_monitoring.tx_uart_max);
}

static ssize_t rx_uart_average_show(struct class *cls,
				    struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", load_monitoring.rx_uart_average);
}

static ssize_t rx_uart_max_show(struct class *cls,
				struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", load_monitoring.rx_uart_max);
}

static ssize_t rx_checksum_errors_show(struct class *cls, struct
				       class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n",
		       cbc_load_monitor_get_checksum_errors_on_receive());
}

static ssize_t rx_sequence_counter_errors_show(struct class *cls,
					       struct class_attribute *attr,
					       char *buf)
{
	return sprintf(buf, "%d\n",
		       cbc_load_monitor_get_sequence_counter_errors_on_receive());
}

static CLASS_ATTR(load_monitoring_reset, (S_IWUSR|S_IWGRP),
		  NULL, load_monitoring_reset_store);
static CLASS_ATTR_RO(tx_queue_average);
static CLASS_ATTR_RO(tx_queue_max);
static CLASS_ATTR_RO(tx_overloads);
static CLASS_ATTR_RO(tx_uart_average);
static CLASS_ATTR_RO(tx_uart_max);
static CLASS_ATTR_RO(rx_uart_average);
static CLASS_ATTR_RO(rx_uart_max);
static CLASS_ATTR_RO(rx_checksum_errors);
static CLASS_ATTR_RO(rx_sequence_counter_errors);

int cbc_kmod_load_monitoring_init(struct class *cls)
{
	/* register callbacks */
	int res = 0;
	enum cbc_error cbc_res = e_cbc_error_ok;

	cbc_res = cbc_load_monitor_set_callbacks(
#if (CBC_LOAD_MONITOR_ENABLE_FRAME_QUEUE_UTILIZATION == 1U)
				  &on_cbc_load_queue_utilization_cb,
				  &on_cbc_load_queue_overload_cb
# if ((CBC_LOAD_MONITOR_ENABLE_UART_TRANSMISSION_MONITOR == 1U)\
|| (CBC_LOAD_MONITOR_ENABLE_UART_RECEPTION_MONITOR == 1U))
				  ,
# endif
#endif
#if (CBC_LOAD_MONITOR_ENABLE_UART_TRANSMISSION_MONITOR == 1U)
				  &on_cbc_load_tx_uart_cb
# if (CBC_LOAD_MONITOR_ENABLE_UART_RECEPTION_MONITOR == 1U)
				  ,
# endif
#endif
#if (CBC_LOAD_MONITOR_ENABLE_UART_RECEPTION_MONITOR == 1U)
				  &on_cbc_load_rx_uart_cb
#endif
			  );

	if (cbc_res != e_cbc_error_ok)
		res = -EINVAL;

	if (!res)
		res = class_create_file(cls, &class_attr_load_monitoring_reset);
	if (!res)
		res = class_create_file(cls, &class_attr_tx_queue_average);
	if (!res)
		res = class_create_file(cls, &class_attr_tx_queue_max);
	if (!res)
		res = class_create_file(cls, &class_attr_tx_overloads);
	if (!res)
		res = class_create_file(cls, &class_attr_tx_uart_average);
	if (!res)
		res = class_create_file(cls, &class_attr_tx_uart_max);
	if (!res)
		res = class_create_file(cls, &class_attr_rx_uart_average);
	if (!res)
		res = class_create_file(cls, &class_attr_rx_uart_max);
	if (!res)
		res = class_create_file(cls, &class_attr_rx_checksum_errors);
	if (!res)
		res = class_create_file(cls, &class_attr_rx_sequence_counter_errors);

	return res;
}




