/*
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 */
#define pr_fmt(fmt) "errmem doorbells: " fmt

#include <linux/device.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_irq.h>

#include <ghs_vmm/ghs_vmm_guest_control.h>

#include "shared_memory.h"

static struct doorbell_endpoint {
	/* endpoint_id is only valid for a generator */
	u32 endpoint_id;
	/* irq is only valid for a listener */
	int irq;
} db_data;

/**
 * errmem_wake_up_reader - Wake up the reader
 * @wq: The wait queue for local wake-up
 *
 * The reader may be distant (from another VM) or local. This is determined by
 * checking the errmem_can_read output.
 */
void errmem_wake_up_reader(wait_queue_head_t *wq)
{
	if (errmem_can_read()) {
		/* Erroneous configuration ? */
		WARN_ON_ONCE(!wq);
		if (wq)
			wake_up(wq);
	} else {
		GHS_VMM_Doorbell_Signal(db_data.endpoint_id);
	}
}

/**
 * errmem_doorbell_interrupt - Interrupt handler for doorbells events
 * @irq: The interrupt request
 * @id: The wait queue to wake up
 *
 * If the current VM is configured to be able to read the error memory content,
 * it may also receive event from other VMs. This handler is therefore attached
 * to a doorbell. BUG is raised if this VM was not expected to handle such
 * an event.
 *
 * Return: IRQ_HANDLED
 */
static irqreturn_t errmem_doorbell_interrupt(int irq, void *id)
{
	wait_queue_head_t *wq = (wait_queue_head_t *)id;

	WARN_ONCE(!errmem_can_read(), "Erroneous errmem configuration.");

	errmem_wake_up_reader(wq);

	return IRQ_HANDLED;
}

/**
 * errmem_prepare_event_irq - Prepare the IRQ handling
 * @of_node The device node used to look for irq
 * @wq: The wait queue to be attached to be passed to the interrupt handler
 *
 * Look for the doorbell interrupt to be used as notifier for new messages
 * from other VM.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int errmem_prepare_event_irq(struct device_node *of_node,
				    wait_queue_head_t *wq)
{
	int ret = 0;

	db_data.irq = irq_of_parse_and_map(of_node, 0);
	if (!db_data.irq) {
		pr_err("Missing or invalid interrupts\n");
		return -ENODEV;
	}

	ret = request_irq(db_data.irq, errmem_doorbell_interrupt, 0,
			  "errmem_doorbell", wq);

	if (ret)
		irq_dispose_mapping(db_data.irq);

	return ret;
}

/**
 * errmem_unprepare_event_irq - Release the event handling resources
 * @wq: The wait queue argument used to request the IRQ.
 *
 * We need to dispose the IRQ we requested.
 *
 */
static void errmem_unprepare_event_irq(wait_queue_head_t *wq)
{
	if (!db_data.irq)
		return;

	free_irq(db_data.irq, wq);

	irq_dispose_mapping(db_data.irq);
}

/**
 * errmem_prepare_read_event - Prepare the read event handling
 * @pdev: The platform device where to find the data we need
 * @wq: The wait queue to be used on events
 *
 * We need to get the doorbell and the IRQ data, then double check that the data
 * we found is consistent with our current configuration.
 *
 * Return: 0 on success, an error code otherwise.
 */
int errmem_prepare_read_event(struct platform_device *pdev,
			      wait_queue_head_t *wq)
{
	struct device_node *of_node;
	u32 db_phdl;
	bool can_notify;
	int ret = -EINVAL;

	if (!pdev)
		return ret;

	of_node = pdev->dev.of_node;

	/* Now let's retrieve the doorbell specific node */
	if (of_property_read_u32_index(of_node, "ghs,doorbell", 0, &db_phdl)) {
		pr_err("Can't get phandle - aborting");
		return ret;
	}

	of_node = of_find_node_by_phandle(db_phdl);
	can_notify = of_property_read_bool(of_node, "ghs,generator");

	if (can_notify == of_property_read_bool(of_node, "ghs,listener")) {
		pr_err("generator/listener not specified - aborting\n");
		goto put_node;
	}

	if (errmem_can_read() == can_notify) {
		pr_err("Wrong couple errmem_can_read/can_notify");
		goto put_node;
	}

	if (can_notify) {
		if (of_property_read_u32_index(of_node,
					       "ghs,endpoint-id",
					       0,
					       &db_data.endpoint_id))
			pr_err("Missing ghs,endpoint-id - aborting\n");
		else
			ret = 0;

	} else {
		if (errmem_prepare_event_irq(of_node, wq))
			pr_err("IRQ not available - aborting\n");
		else
			ret = 0;
	}

put_node:
	of_node_put(of_node);

	return ret;
}

/**
 * errmem_unprepare_read_event - Helper to release IRQ resources
 * @wq: The wait queue used to prepare the IRQ.
 *
 * This function is called while the driver is unloaded.
 *
 */
void errmem_unprepare_read_event(wait_queue_head_t *wq)
{
	return errmem_unprepare_event_irq(wq);
}

/**
 * is_owner_alive - Check whether if a specific VM is alive
 * @owner: The owner to check aliveness.
 *
 * On system with virtualization, we need to be able to know if a slot is in
 * LLRB_WRITE state on purpose. If the VM that owned the slot crashed, we may
 * want to recover it before it restarts.
 *
 * Return: True if the VM is alive, false otherwise.
 */
bool is_owner_alive(unsigned int owner)
{
	int vm = 0;

	if ((owner == errmem_get_owner_id()) ||
	    (owner < ERRMEM_OWNER_BASE))
		return true;

	vm = (int)(owner - ERRMEM_OWNER_BASE);

	if (ghs_get_vm_state(vm) == GHS_VM_RUNNING)
		return true;

	return false;
}
