/*
 * linux/drivers/char/errmem/errmem.c
 *
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * Based on the original driver from:
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *          Markus Kretschmann (mkretschmann@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 */
#define pr_fmt(fmt) "errmem: " fmt

#include <linux/console.h>
#include <linux/errmem.h>
#include <linux/kernel.h>
#include <linux/io.h>
#include <linux/memblock.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/platform_device.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/stddef.h>
#include <linux/string.h>
#include <linux/vmalloc.h>

#include "internal.h"
#include "llrb.h"
#include "shared_memory.h"
#include "hypervisor_logs.h"

#ifndef CONFIG_OF
static struct platform_device *errmem_device;
#endif

static unsigned long errmem_address = CONFIG_ERRMEM_ADDRESS;
module_param(errmem_address, ulong, 0444);
MODULE_PARM_DESC(errmem_address, "Default address to remap.");
static unsigned long errmem_size = CONFIG_ERRMEM_SIZE;
module_param(errmem_size, ulong, 0444);
MODULE_PARM_DESC(errmem_size, "Default size to remap.");

static int errmem_fatal;
static atomic_t nr_read_session;/* Read sessions, must not exceed 1 */
static struct llrb *llrb;	/* Lock-less ring buffer */
static unsigned int console_wr;	/* Console write index */
/* Console log level filter */
static unsigned int loglevel = CONFIG_ERRMEM_DEFAULT_LOG_FILTER;
static unsigned int kernel_wr;	/* Kernel API write index */
static wait_queue_head_t rd_wq;	/* Reader wait queue */

/* Whether if the external read event trigger is initialized */
static int read_event_initialized;

static int remap_failed;

/**
 * errmem_wake_up - Wakes up the reader wait queue
 */
void errmem_wake_up(void)
{
	errmem_wake_up_reader(&rd_wq);
}

/**
 * errmem_console_write - Write console messages in the ring buffer
 * @cons: The console structure
 * @str: The string to store
 * @size: The size of the string to be stored
 */
static void errmem_console_write(struct console *cons,
				 const char *str,
				 unsigned int size)
{
	char *tmp = NULL;
	char *new_line = NULL;
	unsigned int level = 0;
	unsigned long long seq = 0;
	unsigned long long ts_usec = 0;
	unsigned long long ts_sec = 0;
	char cont = '\0';

	if (!str || !size)
		return;

	if (sscanf(str, "%u,%llu,%llu,%c;", &level, &seq, &ts_usec, &cont) != 4)
		return;

	/* Get rid of facility */
	level &= 7;

	if (!((level <= loglevel) ||
	      /* If we are going to crash, panic or oops has put the log level
	       * to verbose.
	       */
	      (console_loglevel == CONSOLE_LOGLEVEL_MOTORMOUTH)))
		return;

	ts_sec = ts_usec / USEC_PER_SEC;
	ts_usec = ts_usec % USEC_PER_SEC;

	/* Look for the actual string start */
	tmp = strchr(str, ';');

	if (!tmp)
		return;

	tmp++;

	/* Ignore empty logs */
	if ((*tmp == '\0') || (*tmp == '\n'))
		return;

	new_line = strchr(tmp, '\n');
	if (new_line) {
		/* Suppress new line if we are not in emergency */
		if (level)
			*new_line = '\0';
		/* Force NULL termination */
		*(new_line + 1) = '\0';
		size = new_line + 1 - tmp;
	}

	if (cont == '-') {
		char buff[64] = { '\0' };
		int sz = snprintf(buff, 64, "[%5llu.%06llu] ", ts_sec, ts_usec);

		if (console_wr)
			console_wr = llrb_store(llrb, "\n", 1, console_wr);

		console_wr = llrb_store(llrb, buff, sz, console_wr);
	}

	console_wr = llrb_store(llrb, tmp, size, console_wr);
}

static struct console _errmem_console = {
	.name = "errmem",
	.write = errmem_console_write,
	.flags = CON_ENABLED | CON_CONSDEV | CON_ANYTIME | CON_EXTENDED,
};

int errmem_get_console_wr(void)
{
	return console_wr;
}

int errmem_get_console_lvl(void)
{
	return loglevel;
}

void errmem_set_console_lvl(int lvl)
{
	if (lvl > CONSOLEMAXLEVEL)
		lvl = CONSOLEMAXLEVEL;

	loglevel = lvl;
}

/**
 * errmem_write - Write a user space message to the ring buffer
 * @filp: The user's file descriptor, where private data is stored
 * @data: The data to write into the ring buffer
 * @length: The length of the data, can be 0 for strings
 *
 * Return: Written length on success, an error code otherwise
 */
ssize_t errmem_write(struct file *filp, char *data, unsigned int length)
{
	struct errmem_private_data *private = filp->private_data;

	if (!data)
		return -EINVAL;

	if (!(filp->f_mode & FMODE_WRITE))
		return -EACCES;

	/* store user space message in ring buffer */
	private->wr = llrb_store(llrb, data, length, private->wr);

	return (ssize_t)length;
}

/**
 * __errmem_read - Read a message from the ring buffer
 * @msg: The structure were to put the message data to be returned
 * @rd: The read index to be retrieved
 * @old_rd: The old and user expected read index, used to determine if slot were
 *          lost
 *
 * Internal function to be used once all the user's parameters have been
 * checked. It will try to retrieve a slot with a specific sequence number
 * from the ring buffer. Some sanity checks are then performed. The slot data
 * are finally returned in the errmem_message structure given as argument.
 *
 * Return: The sequence number of the retrieved slot, 0 if an error occurred.
 */
static int __errmem_read(struct errmem_message *msg,
			 unsigned int rd,
			 unsigned int old_rd)
{
	int ret;
	/* slot = { 0 } can't be used here due to GCC Bug 53119 */
	struct llrb_slot slot = { { 0 } };

	/* Read the message from the ring buffer */
	ret = llrb_read(llrb, rd, &slot);

	/* Set default values */
	msg->internal.seqnum = rd;
	msg->type = ERRMEM_TYPE_TRACE;
	msg->length = 0;
	msg->internal.flags = (unsigned int)slot.flags;
	/* offset no longer needed */
	msg->internal.offset = 0;

	if ((ret < 0) || (slot.seqnum > rd)) {
		msg->internal.flags |= ERRMEM_FLAG_DROPPED;
		/* the sequence number must have been acknowledged */
		return 0;
	}

	/* Is that even possible ?
	 * That would mean the message has never been overwritten.
	 * That must be impossible with the new state machine as
	 * no slot are ever skipped. Anyway, keep the test in.
	 */
	if (unlikely(slot.seqnum < rd)) {
		msg->internal.flags |= ERRMEM_FLAG_OUTDATED;
		/* sequence number */
		msg->internal.seqnum = slot.seqnum;
	}

	/* We lost some messages due to wrap around and the reader slowness */
	if (unlikely(old_rd && (old_rd < rd)))
		msg->internal.flags |= ERRMEM_FLAG_DROPPED;

	memcpy(&msg->message, slot.data.buffer, slot.data.len);

	if (slot.data.buffer[0])
		msg->type = ERRMEM_TYPE_ASCII;

	/* time */
	memcpy(&msg->internal.local_clock, &slot.time, sizeof(slot.time));

	/* length */
	msg->length = slot.data.len;

	return rd;
}

/**
 * errmem_read - Read message from the ring buffer
 * @filp: The file descriptor were the user's read index is stored.
 * @msg: The message structure were the output is stored.
 *
 * After parameters checks, this function retrieves the first available slot's
 * sequence number that can be read starting from the one saved for the current
 * user.
 * On success, the next expected sequence number to be read is stored in
 * the private data.
 *
 * Return: 0 on success, an error code otherwise.
 */
ssize_t errmem_read(struct file *filp, struct errmem_message *msg)
{
	struct errmem_private_data *private = filp->private_data;
	unsigned int rd;

	if (!llrb)
		return -ENODEV;

	if (!msg)
		return -EINVAL;

	if (errmem_fatal || !errmem_can_read())
		return -ENOEXEC;

	if (!(filp->f_mode & FMODE_READ))
		return -EACCES;

	/* Get the index from the next message to be read */
	rd = llrb_can_read(llrb, private->rd);
	if (!rd)
		return -ENOENT;

	if (__errmem_read(msg, rd, private->rd))
		private->rd = rd + 1;

	return 0;
}

/**
 * errmem_fop_open - Open file operator
 * @inode: File's inode.
 * @filp: File structure pointer.
 *
 * Initialize the private data structure for a new client. If the user is
 * allowed to open the file in READ mode, the private read index is initialized
 * with the first readable message or the ring buffer read index if no message
 * is available.
 * Only one reader is allowed.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int errmem_fop_open(struct inode *inode, struct file *filp)
{
	struct errmem_private_data *private;

	if (errmem_fatal)
		return -ENOEXEC;

	if (!llrb)
		return -EIO;

	/* Only one session with read access will be accepted */
	if (filp->f_mode & FMODE_READ) {
		if (atomic_cmpxchg(&nr_read_session, 0, 1))
			return -EMFILE;
	}

	private = kzalloc(sizeof(*private), GFP_KERNEL);
	if (!private)
		return -ENOMEM;

	private->read_size = sizeof(struct errmem_message);

	/* Initialize read pointer only in case of FMODE_READ. */
	if (filp->f_mode & FMODE_READ) {
		unsigned int rd;
		/* Call to llrb_can_read will adjust llrb->rd if necessary */
		rd = llrb_can_read(llrb, atomic_read(&llrb->rd));
		if (rd)
			/* If something is available for reading */
			private->rd = rd;
		else
			/* By default give llrb->rd as initial value */
			private->rd = atomic_read(&llrb->rd);
	}

	filp->private_data = private;

	return 0;
}

/**
 * errmem_fop_release - Release file operator
 * @inode: File's inode.
 * @filp: File structure pointer.
 *
 * Frees the private data, and release the read session lock if needed.
 *
 * Return: 0.
 */
static int errmem_fop_release(struct inode *inode, struct file *file)
{
	struct errmem_private_data *private = file->private_data;

	/* This make incomplete slot to be delivered to user-space */
	llrb_update_ready_counter(llrb, private->wr);

	kfree(file->private_data);
	file->private_data = NULL;

	if (file->f_mode & FMODE_READ)
		atomic_set(&nr_read_session, 0);

	return 0;
}

/**
 * errmem_fop_ioctl - I/O control file operator
 * @filp: The file descriptor, where private data is.
 * @cmd: The command to be executed.
 * @arg: Argument given for this command.
 *
 * Performs the IOCTL described in the documentation. That goes from ring
 * buffer setup, to read/write operation and acknowledgment.
 *
 * Return: 0 on success, an error code otherwise.
 */
static long errmem_fop_ioctl(struct file *filp,
			     unsigned int cmd,
			     unsigned long arg)
{
	if (_IOC_TYPE(cmd) != ERRMEM_MAGIC_IOC)
		return -ENOTTY;

	return errmem_ioctl(llrb, filp, _IOC_NR(cmd), arg);
}

/**
 * errmem_fop_poll - Poll file operator
 * @filp: The file descriptor were private data is.
 * @wait: The poll table were the wait queue is registered.
 *
 * Enable the poll wait queue and returns read and write events if necessary.
 *
 * Return: Event flags if they are relevant.
 */
static unsigned int errmem_fop_poll(struct file *filp, poll_table *wait)
{
	unsigned int rc = 0;
	struct errmem_private_data *private = filp->private_data;

	poll_wait(filp, &rd_wq,  wait);

	if (filp->f_mode & FMODE_READ) {
		/* Check if there is something to be read */
		if (llrb_can_read(llrb, private->rd))
			rc |= POLLIN | POLLRDNORM;
	}

	if (filp->f_mode & FMODE_WRITE) {
		/* Check if there is space to be written */
		if (llrb_can_write(llrb))
			rc |= POLLOUT | POLLWRNORM;
	}

	return rc;
}

/**
 * errmem_fop_read - Read file operator
 * @filp: The file descriptor where the private data is
 * @data: The buffer where to copy the message to
 * @size: The size of the output buffer
 * @offset: Ignored.
 *
 * User interface to read the ring buffer content. The message itself is
 * the only information given through this interface.
 *
 * Return: The copied length on success, an error code otherwise.
 */
static ssize_t errmem_fop_read(struct file *filp,
			       char __user *data,
			       size_t size,
			       loff_t *offset)
{
	struct errmem_private_data *private = filp->private_data;
	struct errmem_message msg = { 0 };
	int ret = 0;

	if (!llrb_blocking_read(llrb, private->rd))
		ret = errmem_read(filp, &msg);

	/* Don't forget the final '\0' not included in length there */
	if (size < msg.length)
		ret = -EIO;

	if (ret >= 0) {
		if (copy_to_user((void __user *)data,
				 msg.message,
				 msg.length))
			ret = -EFAULT;
	}

	if (ret >= 0)
		return msg.length;

	return ret;
}

/**
 * errmem_fop_write - Write file operator
 * @filp: The file descriptor where private data is
 * @data: The data to be stored
 * @size: The size of the data to be stored
 * @offset: Ignored
 *
 * User interface to store data in the ring buffer.
 * Allows storing binary or ASCII data. Binary data must start with '\0'
 * and is limited to 256 bytes. ASCII data has no limit. Checks for binary limit
 * is performed in llrb_data_put.
 *
 * Return: The size of the data written on success, an error core otherwise
 */
static ssize_t errmem_fop_write(struct file *filp,
				const char __user *data,
				size_t size, loff_t *offset)
{
	char *buffer;
	ssize_t rc;

	if (!size)
		return -EINVAL;

	buffer = memdup_user(data, size);
	if (IS_ERR(buffer))
		return PTR_ERR(buffer);

	rc = errmem_write(filp, buffer, size);

	kfree(buffer);

	return rc;
}

/**
 * errmem_store - Store data from kernel space
 * @fatal: Whether if the driver must enter in a fatal state
 * @length: The length of the data to be stored
 * @msg: The data to be stored
 *
 * Kernel API to store data to the ring buffer. If the @fatal is set to 1,
 * user open and read interfaces are closed.
 *
 * Return: The index where the data has been stored if incomplete, 0 otherwise.
 */
unsigned int errmem_store(int fatal, unsigned int length, unsigned char *msg)
{
	errmem_fatal |= fatal;

	if (!length || !msg)
		return 0;

	kernel_wr = llrb_store(llrb, (char *)msg, length, kernel_wr);

	return kernel_wr;
}
EXPORT_SYMBOL(errmem_store);

static const struct file_operations errmem_fops = {
	.open = errmem_fop_open,
	.release = errmem_fop_release,
	.unlocked_ioctl = errmem_fop_ioctl,
	.poll = errmem_fop_poll,
	.read = errmem_fop_read,
	.write = errmem_fop_write,
};

static struct miscdevice errmem_miscdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "errmem",
	.fops = &errmem_fops
};

/**
 * fetch_hyp_logs - Fetch hypervisor specific logs
 * @dev: ignored
 * @attr: ignored
 * @buf: ignored
 * @count: returned
 *
 * On systems with virtualization, the hypervisor may provide an interface
 * to get specific logs. This function allows user to trigger the retrieval of
 * the logs from this interface.
 *
 * Return: count on success, -ENODEV on failure.
 */
static ssize_t fetch_hyp_logs(struct device *dev,
			      struct device_attribute *attr,
			      const char *buf,
			      size_t count)
{
	if (errmem_read_hyp_logs(llrb))
		return -ENODEV;

	return count;
}
static DEVICE_ATTR(fetch_hyp_logs, 0200, NULL, fetch_hyp_logs);

static void *persistent_ram_vmap(phys_addr_t start, size_t size)
{
	struct page **pages;
	phys_addr_t page_start;
	unsigned int page_count;
	pgprot_t prot;
	unsigned int i;
	void *vaddr;

	page_start = start - offset_in_page(start);
	page_count = DIV_ROUND_UP(size + offset_in_page(start), PAGE_SIZE);

	prot = pgprot_noncached(PAGE_KERNEL);

	pages = kmalloc_array(page_count, sizeof(struct page *), GFP_KERNEL);
	if (!pages)
		return NULL;

	for (i = 0; i < page_count; i++) {
		phys_addr_t addr = page_start + i * PAGE_SIZE;

		pages[i] = pfn_to_page(addr >> PAGE_SHIFT);
	}

	vaddr = vmap(pages, page_count, VM_MAP, prot);
	kfree(pages);

	pr_info("Persistent RAM mapped.\n");

	return vaddr;
}

static void *persistent_ram_iomap(phys_addr_t start, size_t size)
{
	void *va;

	if (!request_mem_region(start, size, "persistent_ram")) {
		pr_err("request mem region (0x%llx@0x%llx) failed\n",
		       (unsigned long long)size, (unsigned long long)start);
		return NULL;
	}

	va = ioremap(start, size);

	return va;
}

static int persistent_ram_buffer_map(phys_addr_t start,
				     phys_addr_t size,
				     void **addr)
{
	if (pfn_valid(__phys_to_pfn(start)))
		*addr = persistent_ram_vmap(start, size);
	else
		*addr = persistent_ram_iomap(start, size);

	if (!*addr) {
		pr_err("%s: Failed to map 0x%llx pages at 0x%llx\n", __func__,
		       (unsigned long long)size, (unsigned long long)start);
		return -ENOMEM;
	}

	return 0;
}

static void persistent_ram_free(struct llrb *llrb,
				phys_addr_t paddr,
				phys_addr_t size)
{
	if (!llrb)
		return;

	if (pfn_valid(__phys_to_pfn(paddr))) {
		vunmap(llrb);
	} else {
		iounmap(llrb);
		release_mem_region(paddr, size);
	}
}

/**
 * errmem_probe - Probe the driver
 * @pdev: The platform device structure
 *
 * Once the device is found, the probe will retrieve information about memory
 * region to be used for the ring buffer, and initialize it. Once done,
 * the interfaces are made available for users.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int errmem_probe(struct platform_device *pdev)
{
	int ret = -ENODEV;
	phys_addr_t start = errmem_address;
	resource_size_t size = errmem_size;
	void *addr;

#ifdef CONFIG_OF
	struct resource *res;
	struct resource rmem;
	struct device_node *node;

	node = of_parse_phandle(pdev->dev.of_node, "memory-region", 0);
	if (node) {
		ret = of_address_to_resource(node, 0, &rmem);
		of_node_put(node);
		if (ret) {
			pr_err("errmem: memory-region not found\n");
			goto out;
		}
		res = &rmem;
	} else {
		res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
		if (!res) {
			pr_err("not found in device tree\n");
			goto out;
		}
	}

	start = res->start;
	size = resource_size(res);
#endif

	init_waitqueue_head(&rd_wq);

	read_event_initialized = !errmem_prepare_read_event(pdev, &rd_wq);

	if (start) {
		/* use reserved physical memory*/
		if (persistent_ram_buffer_map(start, size, &addr)) {
			pr_warn("ioremap failed on address=0x%lx size=0x%x\n",
				(unsigned long)start,
				(unsigned int)size);
			ret = -ENOMEM;
			goto out;
		}

		pr_info("Mapped address=0x%p from 0x%p size=0x%p\n",
			addr,
			(void *)start,
			(void *)size);
	} else {
		remap_failed = 1;
		/*
		 * reserved physical memory is not available;
		 * allocate with kmalloc anyway to have a
		 * somewhat functional error memory even if it
		 * doesn't survive a reboot
		 */
		addr = devm_kzalloc(&pdev->dev, size, GFP_ATOMIC);
		if (!addr) {
			ret = -ENOMEM;
			goto out;
		}

		pr_info("kmalloc address=0x%p size=0x%x\n",
			addr,
			(unsigned int)size);
	}

	llrb = llrb_init_mem(addr, size);
	if (!llrb)
		goto out;

	if (misc_register(&errmem_miscdev)) {
		pr_err("failed to register device\n");
		goto out;
	}

	if (!errmem_init_hyp_logs(llrb, &pdev->dev))
		device_create_file(errmem_miscdev.this_device,
				   &dev_attr_fetch_hyp_logs);

	pr_info("version %x.%02x, console log level filter %d\n",
		ERRMEM_VERSION >> 8,
		ERRMEM_VERSION & 0xff,
		CONFIG_ERRMEM_DEFAULT_LOG_FILTER);

	register_console(&_errmem_console);

	return 0;

out:
	if (read_event_initialized)
		errmem_unprepare_read_event(&rd_wq);

	return ret;
}

/**
 * errmem_remove - Remove the driver
 * @pdev: The platform device to be used.
 *
 * Make the interfaces unavailable and release the resources if needed.
 *
 * Return: 0.
 */
static int errmem_remove(struct platform_device *pdev)
{
	phys_addr_t start = errmem_address;
	unsigned int size = errmem_size;

#ifdef CONFIG_OF
	struct resource *res;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		pr_err("not found in device tree\n");
	} else {
		start = res->start;
		size = (unsigned int)resource_size(res);
	}
#endif

	unregister_console(&_errmem_console);

	device_remove_file(errmem_miscdev.this_device,
			   &dev_attr_fetch_hyp_logs);

	errmem_deinit_hyp_logs();

	misc_deregister(&errmem_miscdev);

	if (read_event_initialized)
		errmem_unprepare_read_event(&rd_wq);

	if (!remap_failed)
		persistent_ram_free(llrb, start, size);

	pr_info("removed\n");

	return 0;
}

static const struct of_device_id errmem_ids[] = {
	{ .compatible = "adit,errmem" },
	{ }
};

static struct platform_driver errmem_driver = {
	.driver = {
		.name   = "errmem",
		.of_match_table = of_match_ptr(errmem_ids),
	},
	.probe = errmem_probe,
	.remove = errmem_remove
};

/**
 * errmem_init - Initialize the driver
 *
 * The driver registers itself to be probed if a specific platform device
 * is found.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int __init errmem_init(void)
{
	int ret = platform_driver_register(&errmem_driver);

	if (ret)
		goto fail_platform_device1;

#ifndef CONFIG_OF
	ret = -ENOMEM;
	errmem_device = platform_device_alloc("errmem", -1);

	if (!errmem_device)
		goto fail_platform_device2;

	ret = platform_device_add(errmem_device);

	if (ret)
		goto fail_platform_device3;
#endif
	return 0;

#ifndef CONFIG_OF
fail_platform_device3:
	platform_device_put(errmem_device);

fail_platform_device2:
	platform_driver_unregister(&errmem_driver);
#endif

fail_platform_device1:
	pr_info("failed ret=%d\n", ret);
	return ret;
}

/**
 * errmem_exit - Unregister the driver
 */
static void __exit errmem_exit(void)
{
#ifndef CONFIG_OF
	if (errmem_device)
		platform_device_unregister(errmem_device);
#endif

	platform_driver_unregister(&errmem_driver);
}

subsys_initcall(errmem_init);
module_exit(errmem_exit);

MODULE_AUTHOR("Frederic Berat <fberat@de.adit-jv.com>");
MODULE_DESCRIPTION("Error memory");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("3.02");
