/*
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 */
#define pr_fmt(fmt) "errmem Klogs: " fmt

#include <linux/device.h>
#include <linux/io.h>
#include <linux/memblock.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/slab.h>
#include <linux/stddef.h>

#include "shared_memory.h"
#include "hypervisor_logs.h"
#include "llrb.h"
#include "llrb_data.h"

#define KLOG_MAGIC 0x474f4c4b676f6c6bULL /* "klogKLOG" => 0xGOLKgolk */

static int hyp_logs_wr;
static atomic_t reading;

static unsigned long klog_size;
static phys_addr_t klog_base;

static struct klog {
	unsigned long long klogmagic;
	atomic_t offset;
	char buffer[];
} *klog_buffer;

/**
 * errmem_write_from_hyp_logs - Stores messages from Klog in the ring buffer
 * @llrb: The ring buffer where to store the messages
 * @data: The data to be stored
 * @length: The length of the data to be stored
 *
 * Similarly to the other API, this function will prepare the @it iterator and
 * finally call llrb_store.
 *
 * Return: 0
 */
static int errmem_write_from_hyp_logs(struct llrb *llrb,
				      char *data,
				      unsigned int length)
{
	hyp_logs_wr = llrb_store_owner(llrb,
				       data,
				       length,
				       hyp_logs_wr,
				       ERRMEM_OWNER_HYPERVISOR);

	return 0;
}

/**
 * errmem_read_hyp_logs - Get messages from Klog and store them
 * @llrb: The ring buffer where to store the messages
 *
 * Once we read the offset, we extract the messages from the klog buffer
 * and call errmem_write_from_hyp_logs with them as argument.
 * The offset as to be reset if no more messages are available.
 *
 * Return: 0
 */
int errmem_read_hyp_logs(struct llrb *llrb)
{
	int old_offset = 0;
	int mod_offset = 0;
	int start = 0;
	int to_read = 0;
	atomic_t *offset = NULL;

	if (!klog_buffer || !llrb)
		return 0;

	offset = &klog_buffer->offset;

	if (!klog_buffer || (atomic_cmpxchg(&reading, 0, 1) != 0)) {
		pr_info("Skipping Klogs read.\n");
		return 0;
	}

	do {
		start = old_offset;
		old_offset = atomic_read(offset);
		mod_offset = old_offset;

		if (old_offset == 0)
			break;

		to_read = old_offset - start;

		if (old_offset > klog_size)
			mod_offset = old_offset % klog_size;

		if (start > klog_size)
			start = start % klog_size;

		if (start > mod_offset) {
			int len = klog_size - start;

			errmem_write_from_hyp_logs(llrb,
						   klog_buffer->buffer + start,
						   len);
			to_read -= len;
			start = 0;
		}

		if (to_read)
			errmem_write_from_hyp_logs(llrb,
						   klog_buffer->buffer + start,
						   mod_offset - start);

	} while (atomic_cmpxchg(offset, old_offset, 0) != old_offset);

	atomic_set(&reading, 0);

	return 0;
}

static struct errmem_worker {
	/* work_struct on top so we don't need container_of */
	struct work_struct work;
	struct llrb *llrb;
} em_work;

/**
 * deffered_read_hyp_logs - Read klog deferred
 * @work: The work structure which is part of an errmem_worker structure
 *
 * As it may be quite long to read Klog, we defer it instead of doing it
 * during init.
 *
 */
static void deffered_read_hyp_logs(struct work_struct *work)
{
	struct errmem_worker *ew = (struct errmem_worker *)work;

	if (!ew)
		return;

	errmem_read_hyp_logs(ew->llrb);
}

/**
 * check_klog - Check Klog buffer integrity and prepare the deferred read
 * @llrb: The ring buffer used to store messages
 *
 * The Klog buffer is expected to start with a specific magic. Check if we are
 * able to find it. Then defer the read and give the hand back to the kernel.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int check_klog(struct llrb *llrb)
{
	if (!klog_buffer || !llrb)
		return -EINVAL;

	if (klog_buffer->klogmagic != KLOG_MAGIC) {
		pr_err("Magic is wrong. Found %llx expecting %llx\n",
		       klog_buffer->klogmagic,
		       KLOG_MAGIC);
		iounmap(klog_buffer);
		klog_buffer = NULL;
		return -ENODEV;
	}

	INIT_WORK(&em_work.work, deffered_read_hyp_logs);

	em_work.llrb = llrb;

	schedule_work(&em_work.work);

	return 0;
}

/**
 * errmem_init_hyp_logs - Initialize klog handling
 * @llrb: The ring buffer to be used to store messages.
 *
 * Retrieve the Klog buffer which is expected to be initialized and filled by
 * the hypervisor. Once done, check the integrity of this buffer and read its
 * content.
 *
 * Return: 0 on success, an error code otherwise.
 */
int errmem_init_hyp_logs(struct llrb *llrb, struct device *dev)
{
	struct device_node *of_node;
	int num_reg = 0, i;
	struct resource *res, temp_res;
	struct resource *org_res;
	u32 shm_phandle;
	unsigned long size;

	if (!llrb) {
		pr_err("Wrong parameter\n");
		goto exit;
	}

	atomic_set(&reading, 0);

	of_node = of_find_compatible_node(NULL, NULL, "ghs,ghserrmem");

	/* Now let's retrieve the doorbell specific node */
	if (of_property_read_u32_index(of_node,
				       "ghs,shared-memory-resource",
				       0,
				       &shm_phandle)) {
		pr_err("Can't get phandle - aborting");
		goto exit;
	}

	of_node_put(of_node);
	of_node = of_find_node_by_phandle(shm_phandle);

	while (of_address_to_resource(of_node, num_reg, &temp_res) == 0)
		num_reg++;

	if (num_reg != 1) {
		pr_err("Wrong number of resources (%d expected 1).", num_reg);
		goto exit;
	}

	res = kzalloc(sizeof(*res), GFP_KERNEL);
	org_res = res;

	if (!res)
		goto exit;

	if (of_address_to_resource(of_node, i, res)) {
		pr_info("of_address_to_resource failed\n");
		goto cleanup;
	}

	size = resource_size(res);

	/* This memory must be out of kernel space */
	if (memblock_is_memory(res->start) ||
	    memblock_reserve(res->start, size)) {
		pr_info("Unable to reserve Klog memory\n");
		goto cleanup;
	}

	klog_size = size - offsetof(struct klog, buffer);
	klog_base = res->start;

	klog_buffer = (struct klog *)devm_ioremap(dev, res->start, klog_size);
	/* We are done. We can now properly clean-up and check the result.
	 * check_klog() will validate the klog_buffer pointer and return the
	 * appropriate value.
	 */
cleanup:
	kfree(org_res);
exit:
	/* Release node */
	of_node_put(of_node);
	return check_klog(llrb);
}

void errmem_deinit_hyp_logs(void)
{
	if (klog_base)
		memblock_free(klog_base,
			      klog_size + offsetof(struct klog, buffer));
}
