/*
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 */
#define pr_fmt(fmt) "errmem ioctl: " fmt

#include <linux/uaccess.h>

#include <uapi/linux/errmem.h>

#include "internal.h"
#include "llrb.h"
#include "llrb.h"

/**
 * ioctl_acknowledge - Acknowledge messages as properly read
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * All the slots up to the value given in @arg will be considered as
 * acknowledged. @arg as to be different from ~0 for backward compatibility
 * reasons.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_acknowledge(struct llrb *llrb,
			     struct file *filp,
			     unsigned long arg)
{
	unsigned int ack = 0;

	if (!(filp->f_mode & FMODE_READ))
		return -EACCES;

	if (get_user(ack, (unsigned int __user *)arg))
		return -EFAULT;

	if (~ack)
		llrb_ack(llrb, ack);

	return 0;
}

/**
 * ioctl_get_kernel_msg_lvl - Gives the user the console log level
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_get_kernel_msg_lvl(struct llrb *llrb,
				    struct file *filp,
				    unsigned long arg)
{
	int lvl = errmem_get_console_lvl();

	return put_user(lvl, (int __user *)arg);
}

/**
 * ioctl_set_kernel_msg_lvl - Sets the console log level
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_set_kernel_msg_lvl(struct llrb *llrb,
				    struct file *filp,
				    unsigned long arg)
{
	int lvl = 0;

	if (get_user(lvl, (int __user *)arg))
		return -EFAULT;

	errmem_set_console_lvl(lvl);

	pr_info("Log level filter set to %d.\n", errmem_get_console_lvl());
	return 0;
}

/**
 * ioctl_get_watermark_low - Gives the user the low watermark value
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_get_watermark_low(struct llrb *llrb,
				   struct file *filp,
				   unsigned long arg)
{
	return put_user(llrb_get_watermark_low(llrb),
			(unsigned int __user *)arg);
}

/**
 * ioctl_get_watermark_high - Gives the user the high watermark value
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_get_watermark_high(struct llrb *llrb,
				    struct file *filp,
				    unsigned long arg)
{
	return put_user(llrb_get_watermark_high(llrb),
			(unsigned int __user *)arg);
}

/**
 * ioctl_set_watermark_low - Sets low watermark value
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_set_watermark_low(struct llrb *llrb,
				   struct file *filp,
				   unsigned long arg)
{
	unsigned int watermark = 0;

	if (get_user(watermark, (unsigned int __user *)arg))
		return -EFAULT;

	return llrb_set_watermark_low(llrb, watermark);
}

/**
 * ioctl_set_watermark_high - Sets high watermark value
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_set_watermark_high(struct llrb *llrb,
				    struct file *filp,
				    unsigned long arg)
{
	unsigned int watermark = 0;

	if (get_user(watermark, (unsigned int __user *)arg))
		return -EFAULT;

	return llrb_set_watermark_high(llrb, watermark);
}

/**
 * ioctl_flush - Flushes the console to the ring buffer
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * This function is kept for backward compatibility but do not have real
 * meaning anymore. It simply invalidate the current console wr counter.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_flush(struct llrb *llrb,
		       struct file *filp,
		       unsigned long arg)
{
	llrb_update_ready_counter(llrb, errmem_get_console_wr());

	return 0;
}

/**
 * ioctl_get_slots_number - Gives the user the amount of available slots
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_get_slots_number(struct llrb *llrb,
				  struct file *filp,
				  unsigned long arg)
{
	return put_user(llrb_get_size(llrb), (unsigned int __user *)arg);
}

/**
 * ioctl_get_slotsize - Gives the user the size of the slot's buffer
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_get_slotsize(struct llrb *llrb,
			      struct file *filp,
			      unsigned long arg)
{
	return put_user(LLRB_DATA_SIZE, (unsigned int __user *)arg);
}

/**
 * ioctl_get_used_slots - Gives the user the amount of used slots
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_get_used_slots(struct llrb *llrb,
				struct file *filp,
				unsigned long arg)
{
	unsigned int used = atomic_read(&llrb->wr) - atomic_read(&llrb->rd);

	return put_user(used, (unsigned int __user *)arg);
}

/**
 * ioctl_get_version - Gives the user the version of the driver
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_get_version(struct llrb *llrb,
			     struct file *filp,
			     unsigned long arg)
{
	return put_user(ERRMEM_VERSION, (unsigned long __user *)arg);
}

/**
 * ioctl_write - Writes user's provided data to the ring buffer
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_write(struct llrb *llrb,
		       struct file *filp,
		       unsigned long arg)
{
	struct errmem_message msg = { 0 };

	if (copy_from_user(&msg, (const char __user *)arg, sizeof(msg)))
		return -EFAULT;

	if (!msg.length)
		return -EINVAL;

	return errmem_write(filp, msg.message, msg.length);
}

/**
 * ioctl_read - Provide the oldest non-read message to the user
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * The message is provided to the user along with some slot data like the
 * current flags, the slot ID, the clock, the CRC, the type of the data ...
 * See the struct errmem_message for more information.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_read(struct llrb *llrb,
		      struct file *filp,
		      unsigned long arg)
{
	struct errmem_message msg = { 0 };
	ssize_t rc = 0;
	struct errmem_private_data *private = filp->private_data;

	if (!(filp->f_mode & FMODE_READ))
		return -EACCES;

	rc = errmem_read(filp, &msg);
	if (rc < 0)
		return rc;

	if (copy_to_user((void __user *)arg, &msg, private->read_size))
		return -EFAULT;

	return msg.length;
}

/**
 * ioctl_set_read_size - Sets the user's reading size
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * The read size is used when the ioctl_read is called to determine if we can
 * copy a complete struct errmem_message in the user's argument.
 * This could be removed as no other size is expected and considered anyway.
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_set_read_size(struct llrb *llrb,
			       struct file *filp,
			       unsigned long arg)
{
	struct errmem_private_data *private = filp->private_data;

	if (!(filp->f_mode & FMODE_READ))
		return -EACCES;

	return get_user(private->read_size, (unsigned int __user *)arg);
}

/**
 * ioctl_set_epoch - Set's the real world time
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
static int ioctl_set_epoch(struct llrb *llrb,
			   struct file *filp,
			   unsigned long arg)
{
	unsigned long long epoch_time = 0;

	if (copy_from_user(&epoch_time,
			   (void __user *)arg,
			   sizeof(unsigned long long)))
		return -EFAULT;

	epoch_time *= NSEC_PER_MSEC;

	return llrb_set_real_world_time(epoch_time);
}

static int (*ioctl_funcs[IOCTL_ERRMEM_COUNT])(struct llrb *,
					      struct file *,
					      unsigned long) = {
	[_IOCTL_ERRMEM_VERSION] = ioctl_get_version,
	[_IOCTL_ERRMEM_GET_NUMBER_OF_SLOTS] = ioctl_get_slots_number,
	[_IOCTL_ERRMEM_GET_SLOTSIZE] = ioctl_get_slotsize,
	[_IOCTL_ERRMEM_GET_USED_SLOTS] = ioctl_get_used_slots,
	[_IOCTL_ERRMEM_ACKNOWLEDGE] = ioctl_acknowledge,
	[_IOCTL_ERRMEM_FLUSH] = ioctl_flush,
	[_IOCTL_ERRMEM_GET_WATERMARK_LOW] = ioctl_get_watermark_low,
	[_IOCTL_ERRMEM_SET_WATERMARK_LOW] = ioctl_set_watermark_low,
	[_IOCTL_ERRMEM_GET_WATERMARK_HIGH] = ioctl_get_watermark_high,
	[_IOCTL_ERRMEM_SET_WATERMARK_HIGH] = ioctl_set_watermark_high,
	[_IOCTL_ERRMEM_GET_KERNEL_MSG_LEVEL] = ioctl_get_kernel_msg_lvl,
	[_IOCTL_ERRMEM_SET_KERNEL_MSG_LEVEL] = ioctl_set_kernel_msg_lvl,
	[_IOCTL_ERRMEM_WRITE] = ioctl_write,
	[_IOCTL_ERRMEM_READ] = ioctl_read,
	[_IOCTL_ERRMEM_SET_READ_SIZE] = ioctl_set_read_size,
	[_IOCTL_ERRMEM_SET_EPOCH_MS] = ioctl_set_epoch
};

/**
 * errmem_ioctl - Checks the generic ioctl parameters and executes the
 *		  corresponding ioctl function.
 * @llrb:	The ring buffer
 * @filp:	The file descriptor pointer
 * @arg:	Users argument
 *
 * Return: 0 on success, an error code otherwise.
 */
int errmem_ioctl(struct llrb *llrb,
		 struct file *filp,
		 unsigned int cmd,
		 unsigned long arg)
{
	if (!llrb)
		return -ENODEV;

	if (cmd > IOCTL_ERRMEM_COUNT)
		return -ENOTTY;

	return ioctl_funcs[cmd](llrb, filp, arg);
}
