/*
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * Based on the original driver from:
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *          Markus Kretschmann (mkretschmann@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 */
#define pr_fmt(fmt) "llrb: " fmt

#include <linux/crc16.h>
#include <linux/delay.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/string.h>

#include "llrb.h"
#include "shared_memory.h"

/* Lazy init */
static int mem_init_type = CONFIG_ERRMEM_MEM_INIT_TYPE;
module_param(mem_init_type, int, 0444);
MODULE_PARM_DESC(mem_init_type,
		 "Type of memory init: 0 - full, 1 - partial, 2 - lazy");
#define LLRB_FULL_INIT		0
#define LLRB_PARTIAL_INIT	1
#define LLRB_LAZY_INIT		2

/* Internal flags - +0 to +6 available */
#define ERRMEM_NO_STATE_BIT		ERRMEM_RESERVED_START /* Deprecated */
/* ERRMEM_RESERVED_START + 1 can be recycled */
#define ERRMEM_VALIDATED_BIT		(ERRMEM_RESERVED_START + 2)
#define ERRMEM_MUST_COMPLETE_BIT	(ERRMEM_RESERVED_START + 3)
#define ERRMEM_MUST_VALIDATE_BIT	(ERRMEM_RESERVED_START + 4)

#define ERRMEM_LAST_BIT			ERRMEM_MUST_COMPLETE_BIT

#define ERRMEM_FLAG_NO_STATE		BIT(ERRMEM_NO_STATE_BIT)
#define ERRMEM_FLAG_VALIDATED		BIT(ERRMEM_VALIDATED_BIT)
#define ERRMEM_FLAG_MUST_COMPLETE	BIT(ERRMEM_MUST_COMPLETE_BIT)
#define ERRMEM_FLAG_MUST_VALIDATE	BIT(ERRMEM_MUST_VALIDATE_BIT)

#define ERRMEM_FLAG_INIT_FLAGS		ERRMEM_FLAG_OK

/* Redefine ALL_FLAGS to include internals which user must not care of */
#ifdef ERRMEM_ALL_FLAGS
#undef ERRMEM_ALL_FLAGS
#endif
#define ERRMEM_ALL_FLAGS	GENMASK(ERRMEM_LAST_BIT, ERRMEM_DROPPED_BIT)

/* magic numbers */
#define LLRB_MAGIC_START	0xdeadbeefU
#define LLRB_MAGIC_INIT		0xdeadbeefbeefdeadULL
#define LLRB_MAGIC_END		0xfeebdaedU

/* initial maximum threshold calculated as size divided by this (25%) */
#define INITIAL_THRES	4

/* Atomic states of slots in the ring buffer */
#define LLRB_FREE	0 /* Initial state */
#define LLRB_RESERVED	1 /* Some data has been written. Slot may be reused */
#define LLRB_WRITING	2 /* Slot locked in writing mode */
#define LLRB_READ	3 /* Slot is being read and must be freed before use */

/* This value is local to each instance of the driver */
static unsigned long vhigh;

/* Slot accessors */
static inline struct llrb_slot *get_slot(struct llrb *llrb, unsigned int seqnum)
{
	struct llrb_slot *out = llrb->slots;

	return &out[seqnum % llrb->size];
}

/* Slot state accessors */
#define state_init(slot, value)						\
		(atomic_set(&slot->state, value))

#define get_slot_state(slot)						\
		(atomic_read(&slot->state))

#define slot_state_cmpxchg(slot, old, new)				\
		(atomic_cmpxchg(&slot->state, old, new) == old)

#define is_valid_slot(slot)						\
		((unsigned int)atomic_read(&(slot)->state) <= LLRB_READ)

/* llrb_data_completed - Check whether if the data is completed */
static inline int llrb_data_completed(struct llrb_data *data)
{
	/* Binary data OR last character == '\n' */
	return !data->buffer[0] || (data->buffer[data->len - 1] == '\n');
}

/* Variable used to store real world time in nanoseconds*/
static unsigned long long real_world_time;

/**
 * llrb_set_watermark_low - set the ring buffer's low water mark
 * @llrb: The ring buffer
 * @low: The new value to be used
 *
 * Indicates the minimum number of slot on top of the read pointer to be written
 * before notifying the reader.
 * Defaults to 0.
 *
 * Return: 0 on success, an error code otherwise.
 */
int llrb_set_watermark_low(struct llrb *llrb, unsigned int low)
{
	if (!llrb)
		return -ENODEV;

	/* If negative, count backwards from the end.
	 * This looks a bit ugly but it's kept for backward compatibility.
	 */
	if ((int)low < 0)
		low += llrb_get_size(llrb);

	if (low > llrb_get_size(llrb))
		return -EINVAL;

	llrb->low = low;

	return 0;
}

/**
 * llrb_set_watermark_high - set the ring buffer's high water mark
 * @llrb: The ring buffer
 * @high: The new value to be used
 *
 * Any slot between the write index and this watermark is indicated as
 * "soon overwritten". The owner of the slot must therefore release it.
 * Defaults to a quarter of the size of the ring buffer.
 *
 * Return: 0 on success, an error code otherwise.
 */
int llrb_set_watermark_high(struct llrb *llrb, unsigned int high)
{
	unsigned int min = 0;

	if (!llrb)
		return -ENODEV;

	/* If negative, count backwards from the end
	 * This looks a bit ugly but it's kept for backward compatibility.
	 */
	if ((int)high < 0)
		high += llrb_get_size(llrb);

	if (high > llrb_get_size(llrb))
		return -EINVAL;

	min = llrb_get_size(llrb) / INITIAL_THRES;

	llrb->high = (high > min) ? high : min;

	return 0;
}

/**
 * llrb_recover_slot - Tries to recover slot data buffer
 * @slot: The slot to recover.
 *
 * If a slot is in LLRB_WRITING state while the memory is initialized,
 * the buffer content is not trusty. Therefore, one needs to try to recover
 * any useful ASCII data from it and recalculate the corresponding string
 * length.
 * Not expected to be used on non-ASCII slots.
 */
static void llrb_recover_slot(struct llrb_slot *slot)
{
	int i = 0;

	if (unlikely(slot->data.buffer[0] == '\0')) {
		slot->data.len = LLRB_DATA_SIZE;
		return;
	}

	while (i < LLRB_DATA_SIZE - 1) {
		if (slot->data.buffer[i] == '\0') {
			slot->data.buffer[i++] = '\n';
			slot->data.buffer[i] = '\0';
			break;
		}

		if (slot->data.buffer[i] == '\n') {
			slot->data.buffer[++i] = '\0';
			break;
		}

		i++;
	}

	/* Include '\0' in length for legacy reasons */
	slot->data.len = i + 1;
}

/**
 * llrb_init_slot - Initialize a slot
 * @slot: The slot to initialize
 *
 * Default values are set for the given slot.
 */
static inline void llrb_init_slot(struct llrb_slot *slot)
{
	state_init(slot, LLRB_FREE);
	slot->seqnum = 0U;
	slot->time = 0ULL;
	slot->data.len = 0;
	slot->flags = ERRMEM_FLAG_INIT_FLAGS;
}

/**
 * llrb_calculate_crc - calculate a slot's CRC
 * @slot: The slot to calculate the CRC of.
 *
 * Copy the slot on stack to make the CRC calculation faster.
 * We only copy what we need for calculation. The gain may be up to 75%,
 * whereas the time lost on the memcpy is less than 10% of the original
 * duration.
 * CRC on memory: ~45us on SabreSD board.
 * CRC on stack: ~5us on SabreSD board.
 * memcpy: ~2us on SabreSD board.
 *
 * Return: The CRC.
 */
static u16 llrb_calculate_crc(struct llrb_slot *slot)
{
	struct llrb_slot temp;
	size_t len = offsetof(struct llrb_slot, crc) -
		offsetof(struct llrb_slot, seqnum);

	cpy_from_llrb(&temp.seqnum, &slot->seqnum, len);

	/* CRC over sequence number, time, and data */
	return crc16(0, (char *)&temp.seqnum, len);
}

/**
 * __llrb_validate_slot - Effectively validate the content of a slot
 * @slot: The slot to be validated.
 *
 * If the slot is marked as incomplete, it will put the length to the max.
 * Then if the data is an ASCII string, it will try to recover the buffer.
 * Finally, the CRC is recalculated.
 */
static inline void __llrb_validate_slot(struct llrb_slot *slot)
{
	int len = slot->data.len;

	/* slot is incomplete */
	set_bit(ERRMEM_VALIDATED_BIT, &slot->flags);

	if (test_bit(ERRMEM_INCOMPLETE_BIT, &slot->flags))
		slot->data.len = LLRB_DATA_SIZE;

	/* Try to recover a valid string */
	if (*slot->data.buffer)
		llrb_recover_slot(slot);

	if (len != slot->data.len)
		set_bit(ERRMEM_INCOMPLETE_BIT, &slot->flags);

	slot->crc = llrb_calculate_crc(slot);
}

/**
 * llrb_must_validate - Validate a ring buffer slot
 * @slot: The slot to be validated.
 *
 * Makes a decision on how to take care of a slot depending on its state.
 * It may call llrb_init_slot. __llrb_validate_slot is executed on the copied
 * slot while reading.
 */
static inline void llrb_must_validate(struct llrb_slot *slot)
{
	unsigned int owner = ERRMEM_GET_VM(slot->flags);

	switch (get_slot_state(slot)) {
	case LLRB_WRITING:
		if ((owner != errmem_get_owner_id()) &&
		    is_owner_alive(owner))
			/* Not interested */
			break;
		set_bit(ERRMEM_INCOMPLETE_BIT, &slot->flags);
		if (mem_init_type != LLRB_LAZY_INIT) {
			__llrb_validate_slot(slot);
			atomic_cmpxchg(&slot->state,
				       LLRB_WRITING,
				       LLRB_RESERVED);
		} else {
			set_bit(ERRMEM_MUST_VALIDATE_BIT, &slot->flags);
		}
		break;
	case LLRB_RESERVED:
		/* FALL THROUGH */
	case LLRB_READ:
		if (!errmem_can_read())
			/* Not interested */
			break;
		if (mem_init_type != LLRB_LAZY_INIT)
			__llrb_validate_slot(slot);
		else
			set_bit(ERRMEM_MUST_VALIDATE_BIT, &slot->flags);
		break;
	case LLRB_FREE:
		break;
	default:
		if (!errmem_can_read())
			/* Not interested */
			break;
		llrb_init_slot(slot);
	}
}

/**
 * llrb_get_timestamp - Gives time stamp of local clock + real world time
 *
 * Gives the time to be stored in the ring buffer slots when newly taken.
 *
 * Return: The requested time stamp
 */
static unsigned long long llrb_get_timestamp(void)
{
	return real_world_time + local_clock();
}

/**
 * __llrb_read - Read data from the specified slot in ring buffer
 * @llrb: The ring buffer
 * @slot_in: The slot to copy
 * @slot_out: The slot to copy to
 *
 * Copies locally a slot which is expected to be in LLRB_READ state
 * and makes sanity checks regarding the CRC, the flags and the effective state.
 *
 * Return: 0.
 */
static int __llrb_read(struct llrb *llrb,
		       struct llrb_slot *slot_in,
		       struct llrb_slot *slot_out)
{
	unsigned long flags = 0;

	local_irq_save(flags);
	/* Let's first extract the data from the ring buffer */
	cpy_from_llrb(slot_out, slot_in, sizeof(struct llrb_slot));
	local_irq_restore(flags);

	atomic_cmpxchg(&slot_in->state, LLRB_RESERVED, LLRB_READ);

	/* Then double check the content */
	if (!is_valid_slot(slot_out) ||
	    (slot_out->data.len > LLRB_DATA_SIZE)) {
		pr_info("Invalid slot, forcing validation.\n");
		goto crc_error;
	}

	if (!llrb_data_completed(&slot_out->data) ||
	    !is_owner_alive(ERRMEM_GET_VM(slot_out->flags)))
		llrb_must_validate(slot_out);

	if (test_and_clear_bit(ERRMEM_MUST_VALIDATE_BIT, &slot_out->flags))
		__llrb_validate_slot(slot_out);

	if (llrb_calculate_crc(slot_out) != slot_out->crc) {
crc_error:
		slot_out->flags |= ERRMEM_FLAG_CRC_ERROR;
		__llrb_validate_slot(slot_out);
	}

	if (unlikely((get_slot_state(slot_out) == LLRB_WRITING) ||
		     (get_slot_state(slot_out) == LLRB_FREE))) {
		slot_out->flags |= ERRMEM_FLAG_INCOMPLETE;
		llrb_recover_slot(slot_out);
	}

	if (test_and_clear_bit(ERRMEM_DROPPED_BIT, &llrb->flags))
		slot_out->flags |= ERRMEM_FLAG_DROPPED;

	if (test_and_clear_bit(ERRMEM_INITIALIZED_BIT, &llrb->flags))
		slot_out->flags |= ERRMEM_FLAG_INITIALIZED;

	slot_out->flags &= ~ERRMEM_USER_RESERVED_MASK;

	return 0;
}

/**
 * llrb_read - Reads a block of data from the ring buffer
 * @llrb: The ring buffer
 * @rb: The expected sequence number
 * @slot_out: The output slot
 *
 * Get and read the slot corresponding to the given @rd sequence number.
 * If the expected slot has a too high sequence number, the output slot is
 * marked as dropped and an error code is returned.
 *
 * Return: 0 on success, an error code otherwise.
 */
int llrb_read(struct llrb *llrb,
	      unsigned int rd,
	      struct llrb_slot *slot_out)
{
	struct llrb_slot *slot = NULL;

	if (!llrb || !rd || !slot_out)
		return -ENOENT;

	slot = get_slot(llrb, rd);

	if (slot->seqnum > rd) {
		/* Read index is not accurate anymore */
		slot_out->flags = ERRMEM_FLAG_DROPPED;
		return -ENOENT;
	}

	return __llrb_read(llrb, slot, slot_out);
}

/**
 * __llrb_ack - Acknowledge slots as read
 * @llrb: The ring buffer
 * @to_ack: The last slot's sequence number to acknowledge
 *
 * Slots from llrb->rd to @to_ack will be considered as acknowledged.
 */
static void __llrb_ack(struct llrb *llrb, unsigned int to_ack)
{
	/* Write counter */
	unsigned int wr = atomic_read(&llrb->wr);
	unsigned int min_rd = 1;

	min_rd = (wr < llrb_get_size(llrb)) ? 1 : wr - llrb_get_size(llrb);

	min_rd = max(min_rd, to_ack + 1);
	min_rd = min(min_rd, wr);

	if (min_rd > (unsigned int)atomic_read(&llrb->rd))
		atomic_set(&llrb->rd, min_rd);
}

/**
 * llrb_ack - Acknowledge slots as read
 * @llrb: The ring buffer
 * @to_ack: The slot's sequence number to acknowledge
 *
 * If @to_ack is 0, then the next message to be acknowledged is acknowledged.
 */
void llrb_ack(struct llrb *llrb, unsigned int to_ack)
{
	if (!llrb)
		return;

	if (!to_ack)
		to_ack = atomic_read(&llrb->rd);

	__llrb_ack(llrb, to_ack);
}

/**
 * __llrb_can_read - Determine whether if a slot can be read
 * @llrb: The ring buffer
 * @rd: The minimum sequence number expected
 *
 * Starting from the @rd sequence number, this function look for the first
 * readable slot, up to @llrb write index.
 * Returns 0 if no slot is available to read.
 *
 * Return: The sequence number of the first readable slot, 0 if none is ready.
 */
static unsigned int __llrb_can_read(struct llrb *llrb,
				    unsigned int wr,
				    unsigned int rd,
				    unsigned int rdy)
{
	struct llrb_slot *slot = NULL;
	/* Low water mark */
	unsigned int low = llrb_get_watermark_low(llrb);

	while ((rdy < wr) && (wr - rdy >= low) && (rdy >= rd)) {
		slot = get_slot(llrb, rd);

		if (unlikely(slot->seqnum != rd)) {
			/* rd is far late ... */
			rd++;
			continue;
		}

		/* determine the state of the oldest slot to be read */
		switch (get_slot_state(slot)) {
		case LLRB_READ:
			/* Forgot to ack ? Or failed to read on first attempt */
			return rd;
		case LLRB_WRITING:
			/* The slot is currently accessed in write mode.
			 * As the ready counter indicates this slot as readable,
			 * it must be one that as been lost due to a reset.
			 */
			llrb_must_validate(slot);
			return rd;
		case LLRB_RESERVED:
			return rd;
		case LLRB_FREE:
			/* FALL THROUGH */
		default:
			break;
		}
	}

	return 0;
}

/**
 * llrb_can_read - Return the slot number that can be read
 * @llrb: The ring buffer
 * @rd: The minimum sequence number expected
 *
 * Checks inputs and calls __llrb_can_read.
 *
 * Return: The sequence number of the first readable slot, 0 if none is ready.
 */
unsigned int llrb_can_read(struct llrb *llrb, unsigned int rd)
{
	unsigned int min_rd = 0;
	/* Write counter */
	unsigned int wr = 0;
	/* Slot ready counter */
	unsigned int rdy = 0;

	if (!llrb)
		return 0;

	wr = atomic_read(&llrb->wr);
	rdy = atomic_read(&llrb->rdy);

	min_rd = (wr < llrb_get_size(llrb)) ? 1 : wr - llrb_get_size(llrb);

	/* Re-adjust llrb->rd only on open if no ack ever occurred */
	if (!rd && ((unsigned int)atomic_read(&llrb->rd) < min_rd))
		atomic_set(&llrb->rd, min_rd);

	/* Adjust RD to the minimum value it can have */
	if (rd < min_rd) {
		rd = min_rd;
		set_bit(ERRMEM_DROPPED_BIT, &llrb->flags);
	}

	return __llrb_can_read(llrb, wr, rd, rdy);
}

/**
 * high_watermark_handling - Mark next slot to be released
 * @llrb: The ring buffer
 * @wr: Current position of the write index
 *
 * The high watermark points to the latest slot that will be overwritten soon.
 * This marker notifies the original owner of a slot that it's time to take
 * a new one as this slot will be discarded soon if not read.
 */
static void high_watermark_handling(struct llrb *llrb, unsigned int wr)
{
	unsigned int high = 0;
	struct llrb_slot *slot = NULL;

	high = wr + llrb_get_watermark_high(llrb);

	slot = get_slot(llrb, high);
	set_bit(ERRMEM_MUST_COMPLETE_BIT, &slot->flags);
}

/**
 * llrb_update_ready_counter - Update the ready counter and notify the reader
 * @llrb: The ring buffer
 * @wr: The write index to be marked as ready
 *
 * The ready counter indicates the latest slot that is available for reading.
 * If a reader is waiting for messages, it will be notified that new messages
 * are available.
 */
void llrb_update_ready_counter(struct llrb *llrb, unsigned int wr)
{
	unsigned int rdy = atomic_read(&llrb->rdy);

	if (unlikely(!wr))
		return;

	while (rdy < wr) {
		/* We can't use atomic_set here as there can be concurrent
		 * access to the rdy counter.
		 */
		atomic_cmpxchg(&llrb->rdy, rdy, wr);
		rdy = atomic_read(&llrb->rdy);
	}

	/* As the ready counter as been updated, the reader may be interested */
	errmem_wake_up();
}

/**
 * __llrb_free - Free a slot and mark next slot to be completed soon
 * @slot: The slot to free
 *
 * The function tries to put the slot pointed out by @slot in the free state.
 * The final state as to be checked as the operation may fail if concurrent
 * operation occurred.
 */
static void __llrb_free(struct llrb_slot *slot)
{
	if (get_slot_state(slot) == LLRB_FREE)
		return;

	/* If slot is in writing state while calling free,
	 * the high water mark is not properly configured.
	 */
	if (unlikely((get_slot_state(slot) == LLRB_WRITING) &&
		     is_owner_alive(ERRMEM_GET_VM(slot->flags)) &&
		     !test_and_clear_bit(ERRMEM_MUST_VALIDATE_BIT,
					 &slot->flags))) {
		/* This bit has normally been set by the Water Mark handler
		 * to avoid spinning. Therefore, warn if the previous writer
		 * didn't took it in account.
		 */
		WARN_ON(!test_and_set_bit(ERRMEM_MUST_COMPLETE_BIT,
					  &slot->flags));
		return;
	}

	/* If owner is dead, there was no other choice. */
	atomic_cmpxchg(&slot->state, LLRB_WRITING, LLRB_RESERVED);

	/* Slot may already be in reading state. Ignore then. */
	atomic_cmpxchg(&slot->state, LLRB_RESERVED, LLRB_READ);

	if (WARN_ON(!is_valid_slot(slot)))
		state_init(slot, LLRB_FREE);

	/* No need to continue if we are not in reading state at this point.
	 * llrb_take_slot will make another try.
	 */
	if (unlikely(get_slot_state(slot) != LLRB_READ))
		return;

	/* No concurrent Free expected */
	WARN_ON(!slot_state_cmpxchg(slot, LLRB_READ, LLRB_FREE));
}

/**
 * llrb_free - Free a slot and mark next slot to be completed soon
 * @llrb: The ring buffer
 * @id: The slot's sequence number to be freed
 *
 * Performs sanity checks and call __llrb_free.
 */
static void llrb_free(struct llrb *llrb, unsigned int id)
{
	struct llrb_slot *slot = NULL;

	if (!llrb || !id)
		return;

	slot = get_slot(llrb, id);

	__llrb_free(slot);

	if (get_slot_state(slot) == LLRB_FREE)
		high_watermark_handling(llrb, id);
}

/**
 * llrb_take_slot - Take a new slot for write operation
 * @llrb: The ring buffer to take the slot from.
 *
 * Must be called with disabled IRQs.
 * The slot sequence number comes from the atomic increment of the @llrb write
 * index. The slot is being freed until it can be put in the LLRB_WRITING state,
 * which "locks" it.
 * The while loop is there by design, even if the failure can be prevented by
 * properly setting the high watermark.
 * Once in LLRB_WRITING state, the slot is properly initialized.
 *
 * Return: The taken slot.
 */
static struct llrb_slot *llrb_take_slot(struct llrb *llrb)
{
	/* WR is the current index available to write. Then we want
	 * to increment it and get the value before the increment.
	 */
	unsigned int wr = atomic_inc_return(&llrb->wr) - 1;
	struct llrb_slot *slot = get_slot(llrb, wr);

	if (unlikely(wr <= vhigh))
		llrb_init_slot(slot);

	do {
		/* Ensure the slot is free */
		llrb_free(llrb, wr);
	} while (!slot_state_cmpxchg(slot, LLRB_FREE, LLRB_WRITING));

	slot->seqnum = wr;
	slot->data.len = 0;

	slot->time = llrb_get_timestamp();

	slot->flags = ERRMEM_FLAG_INIT_FLAGS;

	if (test_bit(ERRMEM_RESTART_BIT, &llrb->flags) &&
	    (wr == llrb->restart_idx)) {
		clear_bit(ERRMEM_RESTART_BIT, &llrb->flags);
		set_bit(ERRMEM_RESTART_BIT, &slot->flags);
	}

	return slot;
}

/**
 * llrb_can_write - Determine if and how many slots can be written
 * @llrb: The ring buffer
 *
 * Note: Useless as the ring buffer overwrite oldest data on wrap around.
 *
 * Return: The amount of writable slots.
 */
unsigned int llrb_can_write(struct llrb *llrb)
{
	unsigned int rd = 0;
	unsigned int wr = 0;

	if (!llrb)
		return 0;

	rd = atomic_read(&llrb->rd);
	wr = atomic_read(&llrb->wr);

	if (wr - rd < llrb->size)
		/* some slots are available */
		return llrb->size - (wr - rd);

	/* the ring buffer is full */
	return 0;
}

/**
 * __llrb_store - Stores data to the ring buffer
 * @llrb: The ring buffer
 * @slot: The initial slot where to store the data
 * @it: The iterator where the data is temporarily stored
 * @flags: The IRQ flags
 *
 * IRQs are expected to have been disabled while the operation are performed.
 * As many slot as needed will be used. IRQ are re-enabled if the slot is
 * released and data still need to be stored.
 *
 * Return: The last sequence number where data has been stored
 */
static unsigned int __llrb_store(struct llrb *llrb,
				 struct llrb_slot *slot,
				 struct llrb_iterator *it,
				 unsigned long *flags)
{
	/* llrb_data_put return 0 if a new slot is needed.
	 * That only happens if the message is not fully copied.
	 */
	while (!llrb_data_put(&slot->data, it)) {
		/* Force completion */
		if (unlikely(!llrb_data_completed(&slot->data)))
			slot->data.len++;

		/* Include '\0' in length for legacy reasons */
		if (slot->data.buffer[0])
			slot->data.len++;

		slot->crc = llrb_calculate_crc(slot);

		atomic_cmpxchg(&slot->state, LLRB_WRITING, LLRB_RESERVED);

		/* Re-allow interrupts as we released a slot from writing */
		local_irq_restore(*flags);

		/* Update the slot ready counter. We might sleep here. */
		llrb_update_ready_counter(llrb, slot->seqnum);

		/* Calculate next iteration length for ASCII data.
		 * strlen can be called as we prepared the string to be able to
		 * do so.
		 */
		if (*it->data)
			it->pos_len = strlen(it->pos);

		/* Back on track. */
		local_irq_save(*flags);

		/* We copied whatever we could. As the message has not been
		 * fully copied, we need another slot.
		 */
		slot = llrb_take_slot(llrb);

		ERRMEM_SET_VM(slot->flags, it->owner);
	}

	return slot->seqnum;
}

/**
 * _llrb_store - Store data in the last used slot if possible
 * @llrb: The ring buffer
 * @wr: Last slot sequence number used (0 if none)
 * @it: The data iterator
 *
 * The last slot used will be taken as a first slot to be used to store data
 * if possible. If the slot sequence number is unexpected, if the slot is asked
 * to be released, or if the slot state is incorrect, a new slot will be taken.
 * As far as a slot is put in LLRB_WRITE state and as the operation must be
 * in an atomic way, the IRQ are disabled.
 *
 * Return: The last slot sequence number if it can be reused, 0 either.
 */
static unsigned int _llrb_store(struct llrb *llrb,
				unsigned int wr,
				struct llrb_iterator *it)
{
	unsigned long flags = 0;
	struct llrb_slot *slot = get_slot(llrb, wr);
	unsigned int wake_wr = 0;

	/* Prevent interrupts during the write operation */
	local_irq_save(flags);

	/* We get a new slot if:
	 * - We didn't have one
	 * - The one we have must be completed
	 * - The slot sequence number is not the one we expected to be
	 * - The current state is not LLRB_RESERVED
	 */
	if ((wr == 0) ||
	    /* The following flag must have been set at least by the
	     * high watermark handler if there is no reader alive.
	     * It allows avoiding back-and-forth state switch and spinning.
	     * That's a guard not a guaranty.
	     */
	    test_bit(ERRMEM_MUST_COMPLETE_BIT, &slot->flags) ||
	    /* If the sequence number has changed, the race is already lost. */
	    (slot->seqnum != wr) ||
	    /* Finally, let's try to win the race against the reader */
	    !slot_state_cmpxchg(slot, LLRB_RESERVED, LLRB_WRITING)) {
		/* We'll update the ready counter if wr was non-zero. */
		wake_wr = wr;

		/* Reserve a new slot. Can't fail. */
		slot = llrb_take_slot(llrb);

		ERRMEM_SET_VM(slot->flags, it->owner);
	}

	wr = __llrb_store(llrb, slot, it, &flags);

	slot = get_slot(llrb, wr);

	/* The buffer is full but the incoming data string
	 * was not '\n' terminated. Forcing completion.
	 */
	if (unlikely(!llrb_data_completed(&slot->data) &&
		     (slot->data.len >= LLRB_DATA_SIZE - 2)))
		slot->data.len = LLRB_DATA_SIZE - 1;

	if (likely(llrb_data_completed(&slot->data))) {
		/* Include '\0' in length for legacy reasons */
		if (slot->data.buffer[0])
			slot->data.len++;

		slot->crc = llrb_calculate_crc(slot);
		wake_wr = wr;

		/* Slot is not needed for us anymore, return 0 */
		wr = 0;
	}

	/* We are done with this message */
	atomic_cmpxchg(&slot->state, LLRB_WRITING, LLRB_RESERVED);

	local_irq_restore(flags);

	llrb_update_ready_counter(llrb, wake_wr);

	return wr;
}

/**
 * llrb_store_owner - Store data with a specific owner
 * @llrb: The ring buffer
 * @data: The data to store
 * @length: The length of the data (can be 0 for ASCII)
 * @wr: Last slot sequence number used (0 if none)
 * @owner: The owner of the message (Current VM or Hypervisor)
 *
 * Parameter are checked and iterator is prepared for the store operation.
 *
 * Return: The last slot sequence number if it can be reused, 0 either.
 */
unsigned int llrb_store_owner(struct llrb *llrb,
			      char *data,
			      unsigned int length,
			      unsigned int wr,
			      unsigned int owner)
{
	struct llrb_iterator it;

	if (!llrb || !data)
		return 0;

	/* Prepare the iterator used during storage */
	llrb_iterator_init(&it, data, length, owner);

	if (!it.len)
		goto cleanup;

	/* Binary data ? */
	if (!*data)
		wr = 0;

	/* store user space message in ring buffer */
	wr = _llrb_store(llrb, wr, &it);

cleanup:
	llrb_iterator_clean(&it);

	return wr;
}

/**
 * llrb_store - Store data for current VM
 * @llrb: The ring buffer
 * @data: The data to store
 * @length: The length of the data (can be 0 for ASCII)
 * @wr: Last slot sequence number used (0 if none)
 *
 * Parameter are checked and iterator is prepared for the store operation.
 * If the ring buffer is being reinitialized, this function actively wait for
 * it to terminate.
 *
 * Return: The last slot sequence number if it can be reused, 0 either.
 */
unsigned int llrb_store(struct llrb *llrb,
			char *data,
			unsigned int length,
			unsigned int wr)
{
	return llrb_store_owner(llrb, data, length, wr, errmem_get_owner_id());
}

/**
 * llrb_set_epoch - Set the epoch value in ns
 * @epoch_time: The epoch time
 *
 * Return: 0 on success, -EINVAL otherwise.
 */
long llrb_set_real_world_time(unsigned long long epoch_time)
{
	/* Get elapsed time by local_clock() */
	unsigned long long elapsed_time = local_clock();

	/* check elapsed is less than or equal to epoch time */
	if (time_before64(elapsed_time, epoch_time)) {
		real_world_time = epoch_time - elapsed_time;
		return 0;
	}

	/* error epoch time is less than elapsed_time */
	return -EINVAL;
}

/**
 * llrb_init - Initialize the ring buffer
 * @llrb: Pointer to the ring buffer to initialize
 * @size: Expected size
 *
 * Initialize a piece of memory to be used as ring buffer.
 *
 * Return: The ring buffer pointer.
 */
static struct llrb *llrb_init(struct llrb *llrb, unsigned int size)
{
	unsigned long long init = atomic64_read(&llrb->initializing);

	if ((init == LLRB_MAGIC_INIT) ||
	    ((unsigned long long)atomic64_cmpxchg(&llrb->initializing,
			      init,
			      LLRB_MAGIC_INIT) != init)) {
		/* Someone else is already doing the initialization.
		 * msleep not allowed during driver init.
		 */
		mdelay(1);

		init = atomic64_read(&llrb->initializing);
		if (likely(init != LLRB_MAGIC_INIT))
			return llrb; /* Init is done */
		/* else force init, we may have been killed while doing it */
		pr_err("Forcing init.");
	}

	/* initialize magic numbers and globals */
	llrb->magic_start = LLRB_MAGIC_START;
	llrb->size = size;

	atomic_set(&llrb->rd, 1);   /* next index to read */
	atomic_set(&llrb->wr, 1);   /* next index to write */
	atomic_set(&llrb->rdy, 0);  /* index of last ready for reading slot */

	/* Lazy init, the writer will initialize the slot on first access */
	vhigh = llrb->size;

	llrb->low = 0;
	llrb->high = (unsigned int)(size / INITIAL_THRES);

	llrb->restart_idx = 0;

	llrb->flags = ERRMEM_FLAG_INITIALIZED;

	*((unsigned int *)(llrb->slots + size)) = LLRB_MAGIC_END;

	/* Init done. */
	atomic64_set(&llrb->initializing, 0ULL);

	pr_debug("Memory re-initialized.\n");

	return llrb;
}

/**
 * llrb_init_mem - Validate or initialize the memory to be used as ring buffer
 * @mem: The memory pointer.
 * @amount: The size of memory to be used.
 *
 * If the memory as already been used, there may be something to recover.
 * Then, first validate the memory if possible, else re-initialize the complete
 * area.
 *
 * Return: A pointer to the newly initialized ring buffer, NULL on failure.
 */
struct llrb *llrb_init_mem(void *mem, unsigned int amount)
{
	/* calculate the size of the ring buffer from the */
	/* aligned size of the ring buffer slots fitting */
	/* into the given piece of memory */
	struct llrb *llrb = (struct llrb *)mem;
	unsigned int slots = 0;
	unsigned int single = sizeof(struct llrb_slot);
	unsigned int total = sizeof(struct llrb);

	unsigned int rd = 0;
	unsigned int wr = 0;
	unsigned int rdy = 0;

	if (amount < total)
		return NULL;

	/* Number slots fitting into the actual memory */
	slots = LLRB_SIZE + (unsigned int)((amount - total) / single);

	pr_debug("Amount of slots: %u.\n", slots);

	/* Sanity checks. Look for the magic numbers to find the end of the
	 * ring buffer in memory. If these are not found, we need to fully
	 * initialize the memory.
	 */
	if ((llrb->magic_start != LLRB_MAGIC_START) ||
	    (llrb->size > slots) ||
	    (*((unsigned int *)(llrb->slots + llrb->size)) != LLRB_MAGIC_END))
		goto new_init;

	/* It looks like we already have a ring buffer in this area */
	slots = llrb->size;

	/* Now performing integrity checks. There might be a valid ring buffer
	 * in memory.
	 */
	rd = atomic_read(&llrb->rd);
	wr = atomic_read(&llrb->wr);
	rdy = atomic_read(&llrb->rdy);

	pr_debug("Validating memory (%#x %#x %u %u %#lx).\n",
		 rd,
		 wr,
		 llrb->low,
		 llrb->high,
		 llrb->flags);

	if (rd && wr && (rd <= wr) && (rdy < wr) &&
	    (llrb->low <= slots) &&
	    (llrb->high <= slots) &&
	    !(llrb->flags & ~ERRMEM_ALL_FLAGS)) {
		/* All the slot that are (partially) written are ready */
		if (errmem_can_read()) {
			atomic_set(&llrb->rdy, wr - 1);
			llrb->restart_idx = wr;
		}

		/* the ring buffer appears to be intact */
		set_bit(ERRMEM_RESTART_BIT, &llrb->flags);

		/* Lazy init, reader and writers will do real initialization or
		 * validation.
		 */
		if (mem_init_type == LLRB_FULL_INIT) {
			rd = 0;
			wr = llrb->size;
		}

		vhigh = wr + llrb->size;
		while (rd < wr)
			/* We only mark the slot for later validation here */
			llrb_must_validate(get_slot(llrb, rd++));

		return llrb;
	}

new_init:
	/* Initialize a new ring buffer */
	return llrb_init(llrb, slots);
}
