/*
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * Based on the original driver from:
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *          Markus Kretschmann (mkretschmann@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * DOC: llrb - lock-less ring buffer
 *
 * llrb provides a lock-less, thread-safe ring buffer with a fixed
 * number of slots. Atomics are used to modify the read and write
 * index into the buffer as well as each slot's state.
 *
 * Indexes increase monotonically and the read index is always equal
 * to or smaller than the write index. Access to the ring buffer's
 * slots is done index modulo number of slots.
 *
 * Only one reader is allowed to have access to the slots at the same
 * time while an arbitrary number of writers can access the driver in parallel.
 * If multiple readers are interested by the data a component in an upper layer
 * shall distribute the data.
 *
 * Each slot can be in any of the following states:
 *  FREE     - the slot is not in use
 *  WRITING  - the slot is currently accessed for write operation
 *  RESERVED - Data has been written in the slot
 *  READ     - the slot has been read and must be freed before reuse
 *
 * FREE is that what it say, the slot is not in use. But this slot can
 * already be allocated and therefore associated to a writing context.
 * WRITING is an non-interruptible and transient state. The time spent in this
 * state must then be as short as possible. That's the only state where the
 * sequence number and the buffer content are altered.
 * If in RESERVED state, a slot as been used by a writer and may be reused on
 * later access, unless the state as been changed meanwhile, by either a reader
 * or another writer.
 * While in READ state, a buffer cannot be used by its original owner anymore
 * in order to append new data. The reader making a local copy of this slot
 * to be able to check for integrity, and let the content of the slot untouched.
 *
 * If a user has data to write to the ring buffer, the driver will try to reuse
 * previous slot if possible, or put the new data in a newly taken slot.
 * Previous slot may not be used anymore if the sequence number associated with
 * it as changed, if it is marked for completion, or if the state is not
 * RESERVED anymore.
 * A new slot is taken by atomically incrementing the buffer write index and
 * taking its previous value. The slot pointed out by this counter will then
 * first be freed and put into the WRITING state. Once the new sequence
 * number and the data are stored, the slot is put into RESERVED state.
 * In the worst case, transitions are as followed in order to get a new slot:
 *                RESERVED => READ => FREE => WRITING
 *
 * The reader is notified when a slot is considered as completed, either because
 * it's filled with BINARY data, or because an ASCII data has been terminated
 * with "\n\0".
 * Then, it will try to extract data from each slot starting from its internal
 * read index, until it reaches the last slot to be completed.
 */

#ifndef _LLRB_H
#define _LLRB_H

#include <linux/atomic.h>
#include <linux/bitops.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/time.h>

#include <uapi/linux/errmem.h>

#include "llrb_data.h"

/* Minimum number of slots in the ring buffer */
#define LLRB_SIZE                16

/* A slot in the ring buffer */
struct llrb_slot {
	atomic_t state;		/* state of the slot */
	/* --- crc start --- */
	unsigned int seqnum;	/* unique sequence number of the slot */
	unsigned long long time;/* write kernel time */
	struct llrb_data data;	/* data */
	/* --- crc end --- */
	unsigned int crc;	/* crc over seqnum,time and data */
	unsigned long flags;	/* slot status flags */
};

/* The ring buffer */
struct llrb {
	unsigned int magic_start;	/* Start of ring buffer */
	atomic64_t initializing;	/* Whether init is ongoing */
	unsigned int size;		/* Number of slots in the ring buffer */
	unsigned int low, high;		/* Read water marks */
	unsigned int restart_idx;	/* The index when the reboot occurred */
	atomic_t rd;			/* Read index */
	atomic_t wr;			/* Write index, rd<=wr */
	atomic_t rdy;			/* Newest slot ready for reading */
	unsigned long flags;		/* Status flags */
	struct llrb_slot slots[LLRB_SIZE]; /* Slots in the ring buffer */
	unsigned int magic_end;		/* End of ring buffer */
};

/* llrb_init_mem - Initialize the ring buffer in a piece of memory */
struct llrb *llrb_init_mem(void *mem, unsigned int amount);

/* llrb_update_ready_counter - Update the ready counter and notify the reader */
void llrb_update_ready_counter(struct llrb *llrb, unsigned int wr);

/* llrb_store_owner - Store data for a specific owner */
unsigned int llrb_store_owner(struct llrb *llrb,
			      char *data,
			      unsigned int length,
			      unsigned int wr,
			      unsigned int owner);

/* llrb_store - Store data in the last used slot if possible */
unsigned int llrb_store(struct llrb *llrb,
			char *data,
			   unsigned int length,
			   unsigned int wr);

/* llrb_read - retrieve a block of data from the ring buffer */
int llrb_read(struct llrb *llrb,
	      unsigned int rd,
	      struct llrb_slot *slot_out);

/* llrb_ack - Acknowledge slots as read */
void llrb_ack(struct llrb *llrb, unsigned int to_ack);

/* llrb_can_write - determine if and how many slots can be written */
unsigned int llrb_can_write(struct llrb *llrb);

/* llrb_can_read - determine if and which slot can be read */
unsigned int llrb_can_read(struct llrb *llrb, unsigned int rd);

/* llrb_get_size - get the ring buffer's size */
static inline unsigned int llrb_get_size(struct llrb *llrb)
{
	if (llrb)
		return llrb->size;
	else
		return 0;
}

/* llrb_get_watermark_low - get the ring buffer's low water mark */
static inline unsigned int llrb_get_watermark_low(struct llrb *llrb)
{
	if (llrb)
		return llrb->low;
	else
		return 0;
}

/* llrb_get_watermark_high - get the ring buffer's high water mark */
static inline unsigned int llrb_get_watermark_high(struct llrb *llrb)
{
	if (llrb)
		return llrb->high;
	else
		return 0;
}

/* llrb_set_watermark_low - Set the ring buffer's low water mark */
int llrb_set_watermark_low(struct llrb *llrb, unsigned int low);

/* llrb_set_watermark_high - Set the ring buffer's high water mark */
int llrb_set_watermark_high(struct llrb *llrb, unsigned int high);

/* llrb_blocking_read - check for blocking read */
static inline int llrb_blocking_read(struct llrb *llrb, unsigned int rd)
{
	if (llrb)
		return rd >= (unsigned int)atomic_read(&llrb->wr);
	else
		return true;
}

/* llrb_set_real_world_time - set the RWT value in ns */
long llrb_set_real_world_time(unsigned long long epoch_time);

#endif /* _LLRB_H */
