/*
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * Based on the original driver from:
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *          Markus Kretschmann (mkretschmann@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 */
#define pr_fmt(fmt) "llrb_data: " fmt

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/uaccess.h>

#include "llrb_data.h"
#include "shared_memory.h"

/**
 * llrb_iterator_prepare_data - Prepare input strings for llrb_data_put
 * @it: Iterator to be prepared
 *
 * ASCII strings need to be prepared to avoid unnecessary processing in
 * the critical path. Each '\n' character indicates a new sub-string.
 *
 * "abc\0\0def\nghi\n\0klm" is converted in "abcdef\n\0ghi\n\0klm".
 * "abcdef\n\0" will be put in one slot of the ring buffer, "ghi\n\0" in another
 * one, and finally "klm" in the last one. On future input, new string may be
 * appended to "klm". The length of "abcdef\n\0" is saved in len_pos.
 */
static inline void llrb_iterator_prepare_data(struct llrb_iterator *it)
{
	unsigned int len = 0;
	unsigned int new_len = 0;
	unsigned char *p = NULL;
	unsigned char *c = NULL;

	it->pos_len = 0;
	len = it->len;
	p = it->data;
	c = it->pos;

	while (len-- && p) {
		/* Discarding '\0' */
		if (unlikely(*p != '\0')) {
			*c++ = *p;
			new_len++;
		}

		/* Pre-cutting the strings */
		if (unlikely(*p++ == '\n')) {
			if (!it->pos_len)
				it->pos_len = new_len;
			*c++ = '\0';
			new_len++;
		}
	}

	/* Properly terminating the string */
	*c = '\0';
	it->len = new_len;

	/* As pos is moved during iteration,
	 * remind pos address for later Kfree.
	 * We don't need the original data address anymore.
	 */
	it->data = it->pos;

	/* Calculate the length of the first string to store */
	if (!it->pos_len)
		it->pos_len = new_len;
}

/**
 * llrb_iterator_init - Initialize the iterator
 * @it: The iterator to be initialized
 * @data: The data to be processed
 * @len: The length of the data
 *
 * Iterator is used to ease the processing of incoming data.
 *
 * Return: The initialized iterator
 */
struct llrb_iterator *llrb_iterator_init(struct llrb_iterator *it,
					 char *data,
					 unsigned int len,
					 unsigned int owner)
{
	it->data = data;
	it->len = 0;
	it->pos = data;
	it->pos_len = 0;
	it->owner = owner;

	if (!data)
		return it;

	it->len = len ? len : strlen(data);
	it->pos_len = it->len;

	/* Only prepare data for ASCII messages */
	if (!*data)
		return it;

	it->pos = kzalloc(len * 2 + 1, GFP_ATOMIC);
	if (it->pos)
		llrb_iterator_prepare_data(it);
	else
		/* We are not able to work, force BUG here ? */
		it->len = 0;

	return it;
}

/**
 * llrb_iterator_clean - Cleanup iterator structure
 * @it: The iterator to cleanup
 *
 * Free previously allocated memory for the iterator.
 * Memory has only been allocated for ASCII string.
 */
void llrb_iterator_clean(struct llrb_iterator *it)
{
	if (!it || !it->data || !*it->data)
		return;

	kfree(it->data);
}

/**
 * llrb_pre_complete - Pre-complete ASCII data
 * @buffer: The buffer to check
 * @len: The current length of the buffer
 *
 * ASCII data must always be terminated with "\n\0" in a slot buffer.
 * Here we check if it's already the case and proceed if not.
 * The @buffer must have a maximum size of @len + 1 minimum.
 */
static inline void llrb_pre_complete(char *buffer, unsigned int len)
{
	if (buffer[len - 1] != '\n')
		buffer[len] = '\n';
	else
		buffer[len] = '\0';

	buffer[len + 1] = '\0';
}

/* llrb_data_put - Copies the data into the slots of the ring buffer.
 * @dst: The destination for incoming data
 * @src: The iterator containing the data to be stored
 *
 * Binary data is limited to 256 bytes, extra bytes are discarded. This
 * data is copied one by one into one slot. Binary data always starts with '\0'.
 *
 * ASCII strings are considered terminated with the '\n' character. Non-'\n'
 * terminated strings from a common source are appended one to each other in
 * the same slot, as far as space is available. Only one '\n' terminated string
 * can be put in a slot as some user space application are not able to handle
 * it otherwise.
 *
 * Therefore, the @src iterator gives several information:
 * - data is the original pointer to the input
 * - len is the complete length of the input data
 * - pos is the pointer to the current iterator start in data
 * - pos_len is the length of the '\n' terminated string pointed to by pos
 *
 * Return: 0 if the data is not fully copied and an other slot is needed,
 *         non-0 otherwise.
 */
unsigned int llrb_data_put(struct llrb_data *dst, struct llrb_iterator *src)
{
	unsigned int len = 0;
	unsigned int max_len = LLRB_DATA_SIZE;
	unsigned char *p = NULL;

	if (!src || !src->data || !dst || !src->len)
		/* nothing to do return non-0 */
		return 1;

	/* Get the start */
	p = dst->buffer + dst->len;

	/* Calculate remaining space */
	max_len -= dst->len;

	/* data[0] != 0 => ASCII message. Space is needed for final "\n\0" */
	if (*src->data) {
		max_len -= 2;
	} else {
		if (dst->len != 0)
			/* A new slot is needed for the binary data */
			return 0;
	}

	/* Calculating remaining data length */
	len = min(src->pos_len, max_len);

	/* We don't have enough space in the current slot for the string, and we
	 * already used it a lot. Let's request a new slot. Expecting the actual
	 * log to be stopped between two words, that makes the log more
	 * readable. The idea is not to have a proper cut for sure though, just
	 * hope for it. The threshold is therefore purely arbitrary.
	 */
#define ARBITRARY_THRESHOLD 20
	if (*src->data &&
	    (len < src->pos_len) &&
	    (dst->len > LLRB_DATA_SIZE - ARBITRARY_THRESHOLD))
		return 0;
#undef ARBITRARY_THRESHOLD

	cpy_to_llrb(p, src->pos, len);

	dst->len += len;

	/* We are done with binary data */
	if (!*src->data)
		return dst->len;

	/* Adjusting new source position */
	src->pos += len;
	src->len -= len;

	/* Skip '\0' if we are not done with the input */
	if (src->len && !*src->pos) {
		src->pos++;
		src->len--;
	}

	/* Always complete buffers, we calculated the length
	 * to be able to do it.
	 */
	llrb_pre_complete(dst->buffer, dst->len);

	/* Maybe were we on the limit */
	if ((src->len <= 2) &&
	    (*src->pos == '\n') &&
	    (dst->len >= LLRB_DATA_SIZE - 2)) {
		src->len = 0;
	}

	if (likely(src->len == 0))
		return len;

	return 0;
}
