/*
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * Based on the original driver from:
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *          Markus Kretschmann (mkretschmann@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * DOC: llrb_data - data to store in a lock-less ring buffer
 *
 * Data in the ring buffer is either binary or ASCII. The maximum data
 * size is LLRB_DATA_SIZE.
 *
 * By convention, binary data starts with \0. Binary data is stored as
 * is, without adding or removing any bytes, one entry per chunk of
 * data. When the maximum data size is reached, remaining data is discarded.
 *
 * ASCII data is split into separate entries at newlines. Each new line
 * indicates a new string.
 *
 * Entries not ending with a newline are merged with successive
 * entries. When the maximum data size is reached without a newline,
 * an entry with a newline is stored and remaining data put into
 * another entry. Entries are "\n\0" terminated.
 */

#ifndef _LLRB_DATA_H
#define _LLRB_DATA_H

#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/string.h>

#include <uapi/linux/errmem.h>

/* maximum size of data in the ring buffer */
#define LLRB_DATA_SIZE		ERRMEM_MAX_ENTRY_LENGTH

#ifdef __aarch64__
#define cpy_to_llrb(dst, src, len)			\
	memcpy_toio((volatile void __iomem *)(dst),	\
		    (src),				\
		    (len))
#define cpy_from_llrb(dst, src, len)			\
	memcpy_fromio((dst),				\
		      (volatile void __iomem *)(src),	\
		      (len))

#else
#define cpy_to_llrb(dst, src, len) memcpy((dst), (src), (len))
#define cpy_from_llrb(dst, src, len) memcpy((dst), (src), (len))
#endif

/* Iterator for external data */
struct llrb_iterator {
	unsigned char *data;	/* data */
	unsigned int len;	/* length of data */
	unsigned char *pos;	/* position of data processing */
	unsigned int pos_len;	/* Length of the data starting from pos */
	unsigned int owner;	/* The VM id that owns the message */
};

/* data to store in the ring buffer */
struct llrb_data {
	int len;			/* Length of data in buffer */
	char buffer[LLRB_DATA_SIZE];
};

/* llrb_iterator_init - initialize an iterator */
struct llrb_iterator *llrb_iterator_init(struct llrb_iterator *it,
					 char *data,
					 unsigned int len,
					 unsigned int owner);

/* llrb_iterator_clean - Cleanup iterator structure */
void llrb_iterator_clean(struct llrb_iterator *it);

/* llrb_data_put - copy data to ring buffer
 * returns the amount copied or 0 if a new slot is needed
 */
unsigned int llrb_data_put(struct llrb_data *dst,      /* Destination  */
			   struct llrb_iterator *src); /* Source */

#endif /* _LLRB_DATA_H */
