/*
 * (c) 2016 Advanced Driver Information Technology GmbH
 *          Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 */

#ifndef __SHARED_MEMORY_H__
#define __SHARED_MEMORY_H__

#include <linux/platform_device.h>
#include <linux/sched.h>

/* As described in Kconfig, owner 0 and 1 are reserved.
 * The VM0 gets owner ID 2 and VM1 owner ID 3.
 */
#define ERRMEM_OWNER_HYPERVISOR 1
#define ERRMEM_OWNER_BASE 2

#ifdef CONFIG_ERRMEM_USE_SHARED_MEMORY

int errmem_prepare_read_event(struct platform_device *pdev,
			      wait_queue_head_t *wq);
void errmem_unprepare_read_event(wait_queue_head_t *wq);

static inline unsigned int errmem_get_owner_id(void)
{
	return CONFIG_ERRMEM_OWNER_ID;
};

#ifdef CONFIG_ERRMEM_CAN_READ
static inline bool errmem_can_read(void) { return true; };
#else
static inline bool errmem_can_read(void) { return false; };
#endif /* CONFIG_ERRMEM_CAN_READ */

void errmem_wake_up_reader(wait_queue_head_t *q);

bool is_owner_alive(unsigned int owner);

#else /* CONFIG_ERRMEM_USE_SHARED_MEMORY */

static inline int
errmem_prepare_read_event(__maybe_unused struct platform_device *pdev,
			  __maybe_unused wait_queue_head_t *wq)
{
	return 0;
};

static inline void
errmem_unprepare_read_event(__maybe_unused wait_queue_head_t *wq)
{
	return;
};

static inline unsigned int errmem_get_owner_id(void) { return 0; };
static inline bool errmem_can_read(void) { return true; };
static inline void errmem_wake_up_reader(wait_queue_head_t *q) { wake_up(q); };

static inline bool is_owner_alive(__maybe_unused unsigned int owner)
{
	return true;
};

#endif /* CONFIG_ERRMEM_USE_SHARED_MEMORY */

void errmem_wake_up(void);

#endif /* __SHARED_MEMORY_H__ */
