/*
 * linux/drivers/misc/exchnd/arch/modules/arm.c
 *
 * Arch specific exception handler internal module handling.
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Exception Handler Modules
 *
 * The exception handler modules collect data of the exception or trigger
 * specific action. They allow to configured what data is collected and what
 * actions are done per exception type.
 */
#include <linux/stacktrace.h>
#include <asm/stacktrace.h>
#include <linux/exchnd.h>
#include "../../internal.h"

#define EHM_BACKTRACE_HEADER "====== backtrace:"

/*
 * Exception handler trace recorder.
 *
 * Implements save_trace for exchnd purpose.
 */
static int exchnd_save_trace(struct stackframe *frame, void *t)
{
	struct stack_trace *trace = t;
	unsigned long addr = frame->pc;

	trace->entries[trace->nr_entries++] = addr;

	return trace->nr_entries >= trace->max_entries;
}

/*
 * Exception handler kernel backtrace collector.
 *
 * Walk and record stack.
 */
int exchnd_kbacktrace(struct pt_regs *regs,
		      struct task_struct *tsk,
		      unsigned char *data)
{
	unsigned int len = 0;
	struct stack_trace trace;
	unsigned long entries[EXCHND_MAX_TRACE_ENTRIES];
	unsigned int i = 0;
	struct stackframe frame;

	register unsigned long current_sp asm ("sp");

	/* Init trace */
	trace.nr_entries = 0;
	trace.max_entries = ARRAY_SIZE(entries);
	trace.entries = entries;
	trace.skip = 0;

	if (!tsk)
		tsk = current;

	if (regs) {
		frame.fp = regs->ARM_fp;
		frame.sp = regs->ARM_sp;
		frame.lr = regs->ARM_lr;
		/* PC might be corrupted, use LR in that case. */
		if (exchnd_ktext_add)
			frame.pc = exchnd_ktext_add(regs->ARM_pc)
				? regs->ARM_pc : regs->ARM_lr;
		else
			frame.pc = regs->ARM_pc;
	} else if (tsk == current) {
		frame.fp = (unsigned long)__builtin_frame_address(0);
		frame.sp = current_sp;
		frame.lr = (unsigned long)__builtin_return_address(0);
		frame.pc = (unsigned long)exchnd_kbacktrace;
	} else {
		/* task blocked in __switch_to */
		frame.fp = thread_saved_fp(tsk);
		frame.sp = thread_saved_sp(tsk);
		/*
		 * The function calling __switch_to cannot be a leaf function
		 * so LR is recovered from the stack.
		 */
		frame.lr = 0;
		frame.pc = thread_saved_pc(tsk);
	}

	len += snprintf(data + len,
			EXCHND_MAX_TRACE_LEN - len,
			EHM_BACKTRACE_HEADER);

	/* Walking */
	walk_stackframe(&frame, exchnd_save_trace, &trace);

	/* Now trace is stored, dump it into data. */
	for (i = 0; i < trace.nr_entries; i++) {
		len += snprintf(data + len,
				EXCHND_MAX_TRACE_LEN - len,
				"\n [<%p>] %pS",
				(void *)trace.entries[i],
				(void *)trace.entries[i]);
		if (len >= EXCHND_MAX_TRACE_LEN) {
			len--;
			break;
		}
	}

	data[len] = '\0';

	return len;
}

unsigned long exchnd_get_fault(struct task_struct *task)
{
	return task ? task->thread.address : 0;
}

#define EHM_PROCESSOR_REGISTERS_HEADER "====== processor registers:\n"

#define REGS_L1 "R0: %08lx R1: %08lx R2: %08lx R3: %08lx R4: %08lx R5: %08lx\n"
#define REGS_L2 "R6: %08lx R7: %08lx R8: %08lx R9: %08lx R10: %08lx\n"
#define REGS_L3 "FP: %08lx IP: %08lx SP: %08lx LR: %08lx PC: %08lx CPSR: %08lx"
/*
 * exception handler register dumper for arm
 *
 * Architecture dependent.
 */
int exchnd_dump_regs(unsigned char *data,
		     struct pt_regs *regs,
		     struct task_struct *task)
{
	return snprintf(data,
			EXCHND_MAX_TRACE_LEN,
			EHM_PROCESSOR_REGISTERS_HEADER
			REGS_L1 REGS_L2 REGS_L3,
			regs->ARM_r0,
			regs->ARM_r1,
			regs->ARM_r2,
			regs->ARM_r3,
			regs->ARM_r4,
			regs->ARM_r5,
			regs->ARM_r6,
			regs->ARM_r7,
			regs->ARM_r8,
			regs->ARM_r9,
			regs->ARM_r10,
			regs->ARM_fp,
			regs->ARM_ip,
			regs->ARM_sp,
			regs->ARM_lr,
			regs->ARM_pc,
			regs->ARM_cpsr);
}

unsigned long exchnd_get_sp(struct task_struct *task)
{
	return thread_saved_sp(task);
}
