/*
 * linux/drivers/misc/exchnd/arch/modules/arm64.c
 *
 * aarch64 specific exception handler internal module handling.
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Exception Handler Modules
 *
 * The exception handler modules collect data of the exception or trigger
 * specific action. They allow to configured what data is collected and what
 * actions are done per exception type.
 */
#include <linux/stacktrace.h>
#include <asm/stacktrace.h>
#include <linux/exchnd.h>
#include "../../internal.h"

#define EHM_BACKTRACE_HEADER "====== backtrace:"

/*
 * Exception handler trace recorder.
 *
 * Implements save_trace for exchnd purpose.
 */
static int exchnd_save_trace(struct stackframe *frame, void *t)
{
	struct stack_trace *trace = t;
	unsigned long addr = frame->pc;

	trace->entries[trace->nr_entries++] = addr;

	return trace->nr_entries >= trace->max_entries;
}

/*
 * Exception handler kernel backtrace collector.
 *
 * Walk and record stack.
 */
int exchnd_kbacktrace(struct pt_regs *regs,
		      struct task_struct *tsk,
		      unsigned char *data)
{
	unsigned int len = 0;
	struct stack_trace trace;
	unsigned long entries[EXCHND_MAX_TRACE_ENTRIES];
	unsigned int i = 0;
	struct stackframe frame;

	register unsigned long current_sp asm ("sp");

	/* Init trace */
	trace.nr_entries = 0;
	trace.max_entries = ARRAY_SIZE(entries);
	trace.entries = entries;
	trace.skip = 0;

	if (!tsk)
		tsk = current;

	if (regs) {
		frame.fp = regs->regs[29];
		frame.sp = regs->sp;
		/* PC might be corrupted, use LR in that case. */
		if (exchnd_ktext_add)
			frame.pc = exchnd_ktext_add(regs->pc)
				? regs->pc : regs->regs[30];
		else
			frame.pc = regs->pc;
	} else if (tsk == current) {
		frame.fp = (unsigned long)__builtin_frame_address(0);
		frame.sp = current_sp;
		frame.pc = (unsigned long)exchnd_kbacktrace;
	} else {
		/* task blocked in __switch_to */
		frame.fp = thread_saved_fp(tsk);
		frame.sp = thread_saved_sp(tsk);
		frame.pc = thread_saved_pc(tsk);
	}

	len += snprintf(data + len,
			EXCHND_MAX_TRACE_LEN - len,
			EHM_BACKTRACE_HEADER);

	/* Walking */
	walk_stackframe(tsk, &frame, exchnd_save_trace, &trace);

	/* Now trace is stored, dump it into data. */
	for (i = 0; i < trace.nr_entries; i++) {
		len += snprintf(data + len,
				EXCHND_MAX_TRACE_LEN - len,
				"\n [<%p>] %pS",
				(void *)trace.entries[i],
				(void *)trace.entries[i]);
		if (len >= EXCHND_MAX_TRACE_LEN) {
			len--;
			break;
		}
	}

	data[len] = '\0';

	return len;
}

unsigned long exchnd_get_fault(struct task_struct *task)
{
	return task ? task->thread.fault_address : 0;
}

#define EHM_PROCESSOR_REGISTERS_HEADER "====== processor registers:\n"

#define REGS_L1 "x0-x9:   %.16llx %.16llx %.16llx %.16llx %.16llx"
#define REGS_L1_1 " %.16llx %.16llx %.16llx %.16llx %.16llx\n"
#define REGS_L2 "x10-x19: %.16llx %.16llx %.16llx %.16llx %.16llx"
#define REGS_L2_1 " %.16llx %.16llx %.16llx %.16llx %.16llx\n"
#define REGS_L3 "x20-x29: %.16llx %.16llx %.16llx %.16llx %.16llx"
#define REGS_L3_1 " %.16llx %.16llx %.16llx %.16llx %.16llx\n"
#define REGS_L4 "x30: %.16llx sp: %.16llx pc: %.16llx pstate: %.16llx"
#define REGS_L4_1 " orig_x0: %.16llx syscallno: %.16llx\n"

/*
 * exception handler register dumper for arm
 *
 * Architecture dependent.
 */
int exchnd_dump_regs(unsigned char *data,
		     struct pt_regs *regs,
		     struct task_struct *task)
{
	return snprintf(data,
			EXCHND_MAX_TRACE_LEN,
			EHM_PROCESSOR_REGISTERS_HEADER
			REGS_L1 REGS_L1_1
			REGS_L2 REGS_L2_1
			REGS_L3 REGS_L3_1
			REGS_L4 REGS_L4_1,
			regs->regs[0],
			regs->regs[1],
			regs->regs[2],
			regs->regs[3],
			regs->regs[4],
			regs->regs[5],
			regs->regs[6],
			regs->regs[7],
			regs->regs[8],
			regs->regs[9],
			regs->regs[10],
			regs->regs[11],
			regs->regs[12],
			regs->regs[13],
			regs->regs[14],
			regs->regs[15],
			regs->regs[16],
			regs->regs[17],
			regs->regs[18],
			regs->regs[19],
			regs->regs[20],
			regs->regs[21],
			regs->regs[22],
			regs->regs[23],
			regs->regs[24],
			regs->regs[25],
			regs->regs[26],
			regs->regs[27],
			regs->regs[28],
			regs->regs[29],
			regs->regs[30],
			regs->sp,
			regs->pc,
			regs->pstate,
			regs->orig_x0,
			regs->syscallno);
}

unsigned long exchnd_get_sp(struct task_struct *task)
{
	return task_pt_regs(task)->sp;
}
