/*
 * linux/drivers/misc/exchnd/arch/modules/x86.c
 *
 * x86 specific exception handler internal module handling.
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by:
 *      Frederic Berat (fberat@de.adit-jv.com)
 *      Shoji Morita (smorita@jp.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Exception Handler Modules
 *
 * The exception handler modules collect data of the exception or trigger
 * specific action. They allow to configured what data is collected and what
 * actions are done per exception type.
 */

#include <linux/stacktrace.h>
#include <asm/stacktrace.h>
#include <linux/exchnd.h>
#include "../../internal.h"

#define EHM_BACKTRACE_HEADER "====== backtrace:"

/*
 * Exception handler kernel backtrace collector.
 *
 * Walk and record stack.
 */
int exchnd_kbacktrace(struct pt_regs *regs,
		      struct task_struct *tsk,
		      unsigned char *data)
{
	unsigned int len = 0;
	struct stack_trace trace;
	unsigned long entries[EXCHND_MAX_TRACE_ENTRIES];
	int i = 0;

	/* Init trace */
	trace.nr_entries = 0;
	trace.max_entries = ARRAY_SIZE(entries);
	trace.entries = entries;
	trace.skip = 0;

	if (!tsk)
		tsk = current;

	save_stack_trace_tsk(tsk, &trace);

	/* Now trace is stored, dump it into data. */
	for (i = 0; i < trace.nr_entries; i++) {
		len += snprintf(data + len,
				EXCHND_MAX_TRACE_LEN - len,
				"\n [<%p>] %pS",
				(void *)trace.entries[i],
				(void *)trace.entries[i]);
		if (len >= EXCHND_MAX_TRACE_LEN) {
			len--;
			break;
		}
	}

	data[len] = '\0';

	return len;
}

unsigned long exchnd_get_fault(struct task_struct *task)
{
	return task ? task->thread.cr2 : 0;
}

#define EHM_PROCESSOR_REGISTERS_HEADER "====== processor registers:\n"

#define REGS_L1 "ax: %08lx bx: %08lx cx: %08lx dx: %08lx si: %08lx di: %08lx\n"
#define REGS_L2 "cs: %08lx ds: %08lx es: %08lx fs: %08lx gs: %08lx bp: %08lx\n"
#define REGS_L3 "ip: %08lx orig_ax: %08lx flags: %08lx sp: %08lx ss: %08lx"

/*
 * exception handler register dumper for i386
 *
 * Architecture dependent.
 */
int exchnd_dump_regs(unsigned char *data,
		     struct pt_regs *regs,
		     __attribute__ ((unused))struct task_struct *task)
{
	return snprintf(data,
			EXCHND_MAX_TRACE_LEN,
			EHM_PROCESSOR_REGISTERS_HEADER
			REGS_L1 REGS_L2 REGS_L3,
			regs->ax,
			regs->bx,
			regs->cx,
			regs->dx,
			regs->si,
			regs->di,
			regs->cs,
			regs->ds,
			regs->es,
			regs->fs,
			regs->gs,
			regs->bp,
			regs->ip,
			regs->orig_ax,
			regs->flags,
			regs->sp,
			regs->ss);
}

unsigned long exchnd_get_sp(struct task_struct *task)
{
	return task_pt_regs(task)->sp;
}
