/*
 * linux/drivers/misc/exchnd/arch/triggers/arm.c
 *
 * arm do_exit handling when KERNEL_PROFILING option is disabled.
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * exception trigger PROCESS_EXIT
 *
 * The exception hooks on a probe on do_exit to trigger when a process ends.
 * Aim is to temporarily modify the call stack, adding a function that will
 * stop the task we want to watch.
 */

#include <linux/ptrace.h>
#include <linux/exchnd.h>

#include "../../internal.h"

void exchnd_restore_registers(struct pt_regs *regs)
{
	__asm__ __volatile__ (
		/*
		 * Return to the context saved on stack.
		 */
		"mov    sp, %0\n\t"

#ifdef CONFIG_THUMB2_KERNEL
		"ldr	lr, [sp, %1]\n\t"	/* lr = saved sp */
		"ldrd	r0, r1, [sp, %4]\n\t"	/* r0,r1 = saved lr,pc */
		"ldr	r2, [sp, %3]\n\t"	/* r2 = saved psr */
		"stmdb	lr!, {r0, r1, r2}\n\t"	/* push saved lr and */
						/* rfe context */
		"ldmia	sp, {r0 - r12}\n\t"
		"mov	sp, lr\n\t"
		"ldr	lr, [sp], #4\n\t"
		"rfeia	sp!\n\t"
#else
		"ldr	r0, [sp, %3]\n\t"
		"msr	cpsr_cxsf, r0\n\t"
		"ldmia	sp, {r0 - pc}\n\t"
#endif
		:
		: "p" (regs),
		  "J" (offsetof(struct pt_regs, ARM_sp)),
		  "J" (offsetof(struct pt_regs, ARM_pc)),
		  "J" (offsetof(struct pt_regs, ARM_cpsr)),
		  "J" (offsetof(struct pt_regs, ARM_lr))
		: "memory", "cc");
}

#ifdef CONFIG_THUMB2_KERNEL
static inline void exchnd_prepare_cpsr(struct pt_regs *regs)
{
	long cpsr = regs->ARM_cpsr;

	/* Set correct Thumb state in cpsr */
	if (regs->ARM_pc & 1)
		cpsr |= PSR_T_BIT;
	else
		cpsr &= ~PSR_T_BIT;

	regs->ARM_cpsr = cpsr;
}
#else
static inline void exchnd_prepare_cpsr(struct pt_regs *regs) {}
#endif

void exchnd_arch_do_exit(struct pt_regs *regs)
{
	regs->ARM_pc = (long)exchnd_stop_exit;

	exchnd_prepare_cpsr(regs);
}
