/*
 * linux/drivers/misc/exchnd/arch/triggers/x86.c
 *
 * x86 do_exit handling when KERNEL_PROFILING option is disabled.
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by:
 *      Frederic Berat (fberat@de.adit-jv.com)
 *      Shoji Morita (smorita@jp.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * exception trigger PROCESS_EXIT
 *
 * The exception hooks on a probe on do_exit to trigger when a process ends.
 * Aim is to temporarily modify the call stack, adding a function that will
 * stop the task we want to watch.
 */

#include <linux/ptrace.h>
#include <linux/exchnd.h>

#include "../../internal.h"

void exchnd_restore_registers(struct pt_regs *regs)
{
	asm volatile (
		"movl %0, %%esp\n\t" /* Load regs' top to stack pointer */
		"movl %1, %%eax\n\t" /* Load saved stack point - 12 */
		/* Copy "ax", "flags" and "ip" to the top of original stack as
		 * below.
		 * -12 +---------+
		 *     + ax      |
		 *  -8 +---------+
		 *     | flags   |
		 *  -4 +---------+
		 *     | ip      +
		 *   0 +---------+ <= top of original stack
		 *     | ....    |
		 */
		"movl 48(%%esp), %%ebx\n\t" /* Get do_exit's 2nd instruction
					     * from regs + 48.
					     */
		"movl %%ebx, 8(%%eax)\n\t"  /* Store the IP on the top of
					     * original stack.
					     */
		"movl 56(%%esp), %%ebx\n\t" /* Get status reg from regs + 56. */
		"movl %%ebx, 4(%%eax)\n\t"  /* Store the status register on the
					     * top of IP above.
					     */
		"movl 24(%%esp), %%ebx\n\t" /* Get eax reg from regs + 24. */
		"movl %%ebx, 0(%%eax)\n\t"  /* Store the eax register on the top
					     * of the status register above.
					     * This eax is used to save the
					     * (esp - 12) until last three
					     * instructions of resume procedure
					     * shall be started.
					     */
		/* Recover registers except for ones above. */
		"popl %%ebx\n\t"
		"popl %%ecx\n\t"
		"popl %%edx\n\t"
		"popl %%esi\n\t"
		"popl %%edi\n\t"
		"popl %%ebp\n\t"
		"addl $4, %%esp\n\t" /* already moved */
		"popl %%ds\n\t"
		"popl %%es\n\t"
		"popl %%fs\n\t"
		"popl %%gs\n\t"
		/* Set stack pointer to the original - 12
		 * and return to do_exit()
		 */
		"movl %%eax, %%esp\n\t"
		"popl %%eax\n\t"
		"popf\n\t"
		"ret\n\t"
		:
		: "g" (regs),
		  "g" (regs->sp - 12)
		: "memory", "cc");
}

/**
 * exchnd_do_exit - Function notified by the probe for the process exit
 * @p: probe information
 * @regs: registers at the time of the call
 *
 * This function is called by the probe for "do_exit". The function triggers
 * the analysis of the process exit.
 * We save regs on a list as we are not able to do it on stack on every cases.
 */
void exchnd_arch_do_exit(struct pt_regs *regs)
{
	regs->ip = (long)exchnd_stop_exit;
}
