/*
 * linux/drivers/misc/exchnd/arch/triggers/x86_64.c
 *
 * x86_64 do_exit handling when KERNEL_PROFILING option is disabled.
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by:
 *      Frederic Berat (fberat@de.adit-jv.com)
 *      Shoji Morita (smorita@jp.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * exception trigger PROCESS_EXIT
 *
 * The exception hooks on a probe on do_exit to trigger when a process ends.
 * Aim is to temporarily modify the call stack, adding a function that will
 * stop the task we want to watch.
 */

#include <linux/ptrace.h>
#include <linux/exchnd.h>

#include "../../internal.h"

void exchnd_restore_registers(struct pt_regs *regs)
{
	asm volatile (
		"mov %0, %%rsp\n\t" /* Load regs' top to stack pointer */
		"mov %1, %%rax\n\t" /* Load saved stack point - 12 */
		/* Copy "ax", "flags" and "ip" to the top of original stack as
		 * below.
		 * -12 +---------+
		 *     + ax      |
		 *  -8 +---------+
		 *     | flags   |
		 *  -4 +---------+
		 *     | ip      +
		 *   0 +---------+ <= top of original stack
		 *     | ....    |
		 */
		"mov 48(%%rsp), %%rbx\n\t" /* Get do_exit's 2nd instruction
					    * from regs + 48.
					    */
		"mov %%rbx, 8(%%rax)\n\t"  /* Store the IP on the top of
					    * original stack.
					    */
		"mov 56(%%rsp), %%rbx\n\t" /* Get status reg from regs + 56. */
		"mov %%rbx, 4(%%rax)\n\t"  /* Store the status register on the
					    * top of IP above.
					    */
		"mov 24(%%rsp), %%rbx\n\t" /* Get eax reg from regs + 24. */
		"mov %%rbx, 0(%%rax)\n\t"  /* Store the eax register on the top
					    * of the status register above.
					    * This eax is used to save the
					    * (esp - 12) until last three
					    * instructions of resume procedure
					    * shall be started.
					    */
		/* Recover registers except for ones above. */
		"pop %%rbx\n\t"
		"pop %%rcx\n\t"
		"pop %%rdx\n\t"
		"pop %%rsi\n\t"
		"pop %%rdi\n\t"
		"pop %%rbp\n\t"
		"add $4, %%rsp\n\t" /* already moved */
		"pop %%fs\n\t"
		"pop %%gs\n\t"
		/* Set stack pointer to the original - 12
		 * and return to do_exit()
		 */
		"mov %%rax, %%rsp\n\t"
		"pop %%rax\n\t"
		"popf\n\t"
		"ret\n\t"
		:
		: "g" (regs),
		  "g" (regs->sp - 12)
		: "memory", "cc");
}

/**
 * exchnd_do_exit - Function notified by the probe for the process exit
 * @p: probe information
 * @regs: registers at the time of the call
 *
 * This function is called by the probe for "do_exit". The function triggers
 * the analysis of the process exit.
 * We save regs on a list as we are not able to do it on stack on every cases.
 */
void exchnd_arch_do_exit(struct pt_regs *regs)
{
	regs->ip = (long)exchnd_stop_exit;
}
