/*
 * linux/drivers/misc/exchnd/exchnd.c
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Kai Tomerius (ktomerius@de.adit-jv.com)
 *            Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Exception Handler
 *
 * The exception handler is designed to collect data and do predefined actions
 * when certain exceptional events occur. The collected data is either provided
 * by the character device to the user space or stored in the error memory in
 * case of a fatal exception. The collected data shall support the analysis
 * of the exception. The predefined actions shall allow emergency actions in
 * case of an exception, e.g. stop flash chips from further writing,
 * structured restart, etc.
 *
 * The occurrence of an event is detected through a notification callbacks or
 * kprobes in the Linux kernel, or by request from the user land.
 *
 * The exception handler module provides a character device that provides
 * data and allows control. The data collected by the exception handler is
 * via a character device (/dev/exchnd). The exception can be controlled and
 * configured by the user space by ioctl commands from user space.
 */
#define pr_fmt(fmt) "exchnd: " fmt

#include <linux/exchnd.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/poll.h>
#include <linux/slab.h>
#include <linux/kthread.h>
#include <linux/kallsyms.h>
#include <linux/init.h>

#include "internal.h"

/*
 * Definitions
 */
static struct exchnd_private {
	/* Indicates if the one handle to the device is already open */
	atomic_t device_locked;
	/* Daemon last supposed pid */
	pid_t exchndd_pid;
	/* Indicates if OOM is ongoing */
	unsigned int oom;
} exchnd_p = {
	.device_locked = ATOMIC_INIT(0),
	.exchndd_pid = 0,
	.oom = 0
};

/* Dynamic debug */
static int debug;
module_param(debug, int, 0600);

#define DEFAULT_EC_CONF CONFIG_EXCHND_TRIGGER_SET
#define DEFAULT_ESC_CONF CONFIG_EXCHND_SIGNAL_SET
#define DEFAULT_EPEC_CONF CONFIG_EXCHND_PEXIT_SET
/* All triggers are enabled by default. */
#define DEFAULT_ENABLED_TRIGGERS CONFIG_EXCHND_ENABLED_TRIGGERS
#define DEFAULT_DISABLED_MODULES CONFIG_EXCHND_DISABLED_MODULES

/* Module params */
int default_trigger_set = DEFAULT_EC_CONF;
module_param(default_trigger_set, int, 0644);
MODULE_PARM_DESC(default_trigger_set,
		 "Default set of configuration for triggers.");
int default_signal_set = DEFAULT_ESC_CONF;
module_param(default_signal_set, int, 0644);
MODULE_PARM_DESC(default_sgnal_set,
		 "Default set of configuration for signals.");
static int default_pexit_conf = DEFAULT_EPEC_CONF;
module_param(default_pexit_conf, int, 0644);
MODULE_PARM_DESC(default_pexit_conf,
		 "Default configuration for process exit.");
static int enabled_triggers = DEFAULT_ENABLED_TRIGGERS;
module_param(enabled_triggers, int, 0400);
MODULE_PARM_DESC(enabled_triggers,
		 "Select triggers to enable.");
int disabled_modules = DEFAULT_DISABLED_MODULES;
module_param(disabled_modules, int, 0400);
MODULE_PARM_DESC(disabled_modules,
		 "Select collector modules to disable.");

/*
 * Private structure accessors
 */

/**
 * exchnd_get_locker - Return the atomic lock flag address
 */
atomic_t *exchnd_get_locker(void)
{
	return &exchnd_p.device_locked;
}

/**
 * exchnd_set_pid - Sets the daemon pid
 */
void exchnd_set_pid(pid_t pid)
{
	if (exchnd_p.exchndd_pid == 0) {
		if (exchnd_debug(EXCHND_DEBUG_DAEMON))
			pr_info("Daemon pid set (%u).\n", pid);
		exchnd_p.exchndd_pid = pid;
	}
}

/**
 * exchnd_get_pid - Get the daemon pid
 */
pid_t exchnd_get_pid(void)
{
	return exchnd_p.exchndd_pid;
}

/**
 * exchnd_unset_pid - Unset the daemon pid
 */
void exchnd_unset_pid(void)
{
	if (exchnd_p.exchndd_pid != 0) {
		if (exchnd_debug(EXCHND_DEBUG_DAEMON))
			pr_info("Daemon pid unset.\n");
		exchnd_p.exchndd_pid = 0;
		rb_read_recover();
	}
}

static unsigned long last_oom;
/**
 * exchnd_oom_start - Set OOM flag indicating start of OOM handling
 *
 * No need for atomic operation here. We only want to avoid most of exception
 * during OOM handling.
 */
void exchnd_oom_start(void)
{
	/* That may be a afterglow of previous OOM */
	if (last_oom < (jiffies - msecs_to_jiffies(100)))
		exchnd_p.oom = 1;
}

/**
 * exchnd_oom_end - Unset flag indicating OOM handling
 */
void exchnd_oom_end(void)
{
	exchnd_p.oom = 0;
	last_oom = jiffies;
}

/**
 * exchnd_oom_ongoing - Indicate if OOM handling is ongoing
 *
 * Return 1 if OOM is ongoing, 0 either.
 */
int exchnd_oom_ongoing(void)
{
	return exchnd_p.oom;
}

int exchnd_debug(int mask)
{
	return debug & mask;
}

/*
 * Character device driver callback functions
 */

/**
 * exchnd_fop_open - open the exception handler device
 * @inode: unused
 * @filp: file handle of the requesting process
 *
 * Creates a file handle for exception character device for the user space.
 * Only one process (the exception handler daemon) is allowed to read from
 * the device and will get a read permit. Other file handle may only do ioctl.
 *
 * Return: 0 for success, error code otherwise
 */
static int exchnd_fop_open(struct inode *inode, struct file *filp)
{
	struct exchnd_file_private *private =
		kmalloc(sizeof(struct exchnd_file_private), GFP_KERNEL);
	atomic_t *device_locked = exchnd_get_locker();

	if (!private)
		return -ENOMEM;

	filp->private_data = (void *)private;
	/* Assure that only one file handle has read access */
	if ((filp->f_mode & FMODE_READ) &&
	    (atomic_cmpxchg(device_locked, 0, 1) == 0))
		private->read_permit = true;
	else
		private->read_permit = false;

	return 0;
}

/**
 * exchnd_fop_release - close exception handler device
 * @inode: unused
 * @filp: file handle of the requesting process
 *
 * Destroys the file handle of the device for the user space. It also frees
 * the lock so that the device can be opened again. If the file handle had
 * a read permit, the exclusive access is returned and will be acquired by
 * the next open.
 *
 * Return: 0 for success, error code otherwise
 */
static int exchnd_fop_release(struct inode *inode, struct file *filp)
{
	struct exchnd_file_private *private = filp->private_data;
	atomic_t *device_locked = exchnd_get_locker();

	/* Allow read again if read permit is returned */
	if (private->read_permit) {
		atomic_set(device_locked, 0);
		if (exchnd_debug(EXCHND_DEBUG_DAEMON))
			pr_info("Daemon exits.\n");
		/* Daemon is not there anymore */
		exchnd_unset_pid();
	}

	kfree(filp->private_data);

	return 0;
}

/**
 * exchnd_fop_poll - allow poll/select on exception handler character device
 * @filp: file handle of the requesting process
 * @wait: wait table for this file
 *
 * The function checks the if data is available and waits if no data is
 * available. The poll access is only allowed with the exclusive read access.
 *
 * Return: 0 for success, error code otherwise
 */
static unsigned int exchnd_fop_poll(struct file *filp, poll_table *wait)
{
	unsigned int rc = 0;
	struct exchnd_file_private *private = filp->private_data;

	if (!private->read_permit)
		return -EACCES;
	/* From now on, file was necessarily opened in read mode. */

	poll_wait(filp, rb_get_read_wait_queue(),  wait);

	exchnd_set_pid(current->group_leader->pid);

	/* check if there is something to be read */
	if (!rb_empty())
		/* can read */
		rc |= POLLIN | POLLRDNORM;

	return rc;
}

/**
 * exchnd_fop_read - reads from the ring buffer of the exception handler device
 * @filp: file handle of the requesting process
 * @data: buffer to read the data to
 * @size: number of bytes to read
 * @offset: offset in file (not supported)
 *
 * Reads the exception data from the exception handler ring buffer and provides
 * it to the user space. If ring buffer is empty the method blocks the read,
 * unless non-blocking I/O is activated. Only the file handle with the read
 * permit is allowed to read. All others will get an error.
 *
 * Return: the number of read bytes, error code otherwise
 */
static ssize_t exchnd_fop_read(struct file *filp,
			       char __user *out,
			       size_t size,
			       loff_t *offset)
{
	struct exchnd_file_private *private = filp->private_data;
	ssize_t ret = 0;
	char *data = NULL;

	if (!out)
		return -EINVAL;

	if (!(filp->f_mode & FMODE_READ))
		return -EACCES;

	if (!private->read_permit)
		return -EACCES;

	data = kzalloc(size, GFP_KERNEL);
	if (!data)
		return -ENOMEM;

	do {
		ret = rb_read(data, size);
		if (ret > 0) {
			rb_wakeup_write_wait_queue();
			ret = size;

			if (copy_to_user(out, data, size))
				ret = -EFAULT;
		}

		if ((ret != -EAGAIN) ||
		    ((ret == -EAGAIN) &&
		     (filp->f_flags & O_NONBLOCK))) {
			/* Just return size or -EAGAIN. */
			kfree(data);
			return ret;
		}

		/* blocking I/O: put the task to sleep
		 * size == -EAGAIN && !O_NONBLOCK
		 */
	} while (!rb_wait_read_wait_queue());

	kfree(data);

	/* blocking I/O: task has been interrupted by a signal */
	return -EINTR;
}

/**
 * exchnd_fop_write - write to the ring buffer of the exception handler device
 * @filp: file handle of the requesting process
 * @data: data to write
 * @size: size of the data to write
 * @offset: offset in the file (not supported)
 *
 * Write access from user space is not expected and therefore not allowed. This
 * is a read only device. The control is done via ioctl calls.
 *
 * Return: Number of bytes written, error code otherwise
 */
static ssize_t exchnd_fop_write(struct file *filp, const char __user *data,
				size_t size, loff_t *offset) {
	return -EACCES;
}

/* Platform device without using device tree arch*/
static struct platform_device *exchnd_device;

/* File operation functions for the character device driver */
static const struct file_operations exchnd_fops = {
	.open = exchnd_fop_open,
	.release = exchnd_fop_release,
	.unlocked_ioctl = exchnd_fop_ioctl,
	.poll = exchnd_fop_poll,
	.read = exchnd_fop_read,
	.write = exchnd_fop_write,
};

/* Definition of the exception handler character device */
static struct miscdevice exchnd_miscdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "exchnd",
	.fops = &exchnd_fops
};

/*
 * Module handling functions
 */
static void exchnd_deferred_init(struct work_struct *work)
{
	unsigned int i;
	ktime_t uninitialized_var(calltime), delta, rettime;

	if (IS_BUILTIN(CONFIG_EXCHND) && initcall_debug) {
		/* pr_debug can't be used here as the parser don't expect
		 * something like "exchnd: " in front of the comment.
		 */
		printk(KERN_DEBUG "calling  %pF @ %i\n",
		       exchnd_deferred_init, task_pid_nr(current));
		calltime = ktime_get();
	}

	exchnd_ktext_add = (void *)exchnd_get_symbol(kernel_text_address);

	if (exchnd_init_modules())
		return;

	/* Add module list to signals */
	for (i = 1; i < SIGRTMIN; i++) {
		/* Attach modules */
		exchnd_module_attach(&exchnd_signal_list[i],
				     &signal_conf[default_signal_set][i]);
	}

	/* Attach process exit default */
	exchnd_module_attach(&exchnd_pexit_default,
			     &pexit_conf[default_pexit_conf]);

	/* Setup exception triggers */
	for (i = 0; i < EHT_LAST_ELEMENT; i++) {
		/* Attach modules */
		exchnd_module_attach(&exchnd_trigger_list[i].modules,
				     &trigger_conf[default_trigger_set][i]);
		/* Init triggers*/
		if (exchnd_trigger_list[i].init &&
		    (enabled_triggers & (1 << i)))
			exchnd_trigger_list[i].init(&exchnd_trigger_list[i]);
	}

	if (IS_BUILTIN(CONFIG_EXCHND) && initcall_debug) {
		rettime = ktime_get();
		delta = ktime_sub(rettime, calltime);
		/* pr_debug can't be used here as the parser don't expect
		 * something like "exchnd: " in front of the comment.
		 */
		printk(KERN_DEBUG "initcall %pF returned 0 after %lld usecs\n",
		       exchnd_deferred_init,
			 (long long)ktime_to_ns(delta) >> 10);
	}
}

static DECLARE_WORK(deferred_init_work, exchnd_deferred_init);

int (*exchnd_ktext_add)(unsigned long);
/**
 * exchnd_probe - Initializes the exception handler module
 * @pdev: exchnd device created by the kernel
 *
 * The function is called after the exception handler device is created and
 * initializes the driver.
 *
 * Return: 0 on success, error code otherwise
 */
static int exchnd_probe(struct platform_device *pdev)
{
	int rc = 0;

	/* Initialize the ring buffer for the device */
	rc = rb_init(pdev);
	if (rc < 0)
		return rc;

	if (misc_register(&exchnd_miscdev)) {
		pr_err("Failed to register device\n");
		return  -ENODEV;
	}

	/* Init task watchdog */
	rc = exchnd_init_wd();
	if (rc != 0)
		goto cleanup_wd;

	/* Init exception queue */
	rc = eq_init();
	if (rc != 0)
		goto cleanup_eq;

	/* Trigger init takes 120ms, we need to defer init. */
	schedule_work(&deferred_init_work);

	pr_info("Version %x.%02x\n",
		EXCHND_VERSION >> 8, EXCHND_VERSION & 0xff);

	return 0;

cleanup_eq:
	exchnd_deinit_wd();
cleanup_wd:
	misc_deregister(&exchnd_miscdev);

	return rc;
}

/**
 * exchnd_remove - Deinitializes the exception handler module
 * @pdev: exchnd device created by the kernel
 *
 * The function deinitializes the exception handler module and reverses
 * everything done in exchnd_probe.
 *
 * Return: 0 on success, error code otherwise
 */
static int exchnd_remove(struct platform_device *pdev)
{
	unsigned int i;

	/* Ensure that the deferred init is done. */
	cancel_work_sync(&deferred_init_work);

	/* remove exception triggers */
	for (i = 0; i < EHT_LAST_ELEMENT; i++) {
		if (exchnd_trigger_list[i].deinit)
			exchnd_trigger_list[i].deinit(&exchnd_trigger_list[i]);
	}

	exchnd_deinit_modules();

	/* Deinitialize exception queue */
	eq_deinit();

	/* De-init watchdog */
	exchnd_deinit_wd();

	misc_deregister(&exchnd_miscdev);
	pr_info("Removed\n");
	return 0;
}

/* Exception handler character device driver */
static struct platform_driver exchnd_driver = {
	.driver = {
		.name   = "exchnd",
	},
	.probe = exchnd_probe,
	.remove = exchnd_remove
};

/* Functions to add and unregister the platform device for platforms without a
 * device tree
 */
static int exchnd_add_platform_device(void)
{
	int ret = -ENOMEM;

	exchnd_device = platform_device_alloc("exchnd", -1);
	if (!exchnd_device)
		goto fail_device_alloc;

	ret = platform_device_add(exchnd_device);
	if (ret)
		goto fail_device_add;

	return 0;

fail_device_add:
	platform_device_put(exchnd_device);

fail_device_alloc:
	return ret;
}

static void exchnd_unregister_platform_device(void)
{
	if (exchnd_device)
		platform_device_unregister(exchnd_device);
}

/**
 * exchnd_init - Registers device driver
 *
 * The function registers the device driver
 *
 * Return: 0 on success, error code otherwise
 */
static int __init exchnd_init(void)
{
	int ret = platform_driver_register(&exchnd_driver);

	if (ret)
		goto fail_platform_driver_register;

	ret = exchnd_add_platform_device();
	if (ret)
		goto fail_add_platform_device;

	return 0;

fail_add_platform_device:
	platform_driver_unregister(&exchnd_driver);

fail_platform_driver_register:
	pr_info("Failed ret=%d\n", ret);
	return ret;
}

/**
 * exchnd_exit - Unregisters device driver
 *
 * The function unregisters the device driver
 *
 * Return: 0 on success, error code otherwise
 */
static void __exit exchnd_exit(void)
{
	exchnd_unregister_platform_device();
	platform_driver_unregister(&exchnd_driver);
}

module_init(exchnd_init);
module_exit(exchnd_exit);

MODULE_AUTHOR("Kai Tomerius <ktomerius@de.adit-jv.com>");
MODULE_AUTHOR("Matthias Weise <mweise@de.adit-jv.com>");
MODULE_AUTHOR("Frederic Berat <fberat@de.adit-jv.com>");
MODULE_DESCRIPTION("Exception handler");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("3.02");
