/*
 * linux/drivers/misc/exchnd/modules.c
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Kai Tomerius (ktomerius@de.adit-jv.com)
 *            Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Exception Handler Modules
 *
 * The exception handler modules collect data of the exception or trigger
 * specific action. They allow to configured what data is collected and what
 * actions are done per exception type.
 */
#define pr_fmt(fmt) "exchnd modules: " fmt

#include <linux/exchnd.h>
#include <linux/kernel_stat.h>
#include <linux/reboot.h>
#include <linux/sched.h>
#include <linux/pid_namespace.h>

#include <linux/module.h>
#include <linux/tracepoint.h>
#include <asm/syscall.h>
#include <trace/syscall.h>

#include <linux/slab.h>
#include <linux/uaccess.h>

#include <trace/events/syscalls.h>
#include <trace/events/sched.h>

#include "internal.h"

#ifdef CONFIG_64BIT
#define MEM_TMPL "%016lx"
#else
#define MEM_TMPL "%08lx"
#endif
#define XMEM_TMPL "0x" MEM_TMPL

/* Memory pools for exceptions */
static char *exh_pool;
static char *exh_fatal_pool;

/* Pool accessor */
static char *exchnd_get_pool(enum exchnd_triggers t)
{
	if (exchnd_trigger_list[t].fatality == FT_FATAL)
		return exh_fatal_pool;

	return exh_pool;
}

/*
 * exception handler module BACKTRACE
 *
 * The module collects the backtrace of the current process. If the stack
 * is completely in user space, the generation is done in user space.
 * Shall not be executed in user space in case of OOM.
 */
static int exchnd_backtrace_execute(struct exception_info *info,
				    enum exchnd_modules module)
{
	/* Common stuff. */
	struct exchnd_message_header header;
	unsigned char *data = exchnd_get_pool(info->trigger);
	struct task_struct *task = info->task;

	if (!task) {
		pr_warn("%s skipped as there is no way to retrieve frames.\n",
			__func__);
		goto exit;
	}

	header.type = module;
	header.trigger = info->trigger;
	header.seq_num = 0;
	header.sig = info->sig;
	header.pid = task->pid;

	/* User space */
	if (!(task->flags & PF_KTHREAD)) {
		header.length                = 0;
		header.flags.collected       = 0;
		header.flags.internal        = 0;
		header.flags.addition_needed = 1;
	} else {
		/* Kernel */
		header.flags.collected       = 1;
		header.flags.addition_needed = 0;
		/* On x86 arch, we can no longer fetch task information
		 * from pt_regs. The pointer to task_struct is removed from
		 * struct thread_info. In order to maintain compatibility
		 * between arm & x86 arch, the backtrace feature for kernel
		 * tasks will be back through a patch later
		 */
		pr_warn("Ignoring backtrace for kernel task\n");
		goto exit;
	}

	info->write_func(&header, data);
	memset(data, 0, header.length);

exit:
	return 0;
}

/*
 * exception handler module BACKTRACE_ALL_THREADS
 *
 * The module collects the backtrace of all threads (including itself) of the
 * current process. If the trace is completely in user space, the
 * generation is done in user space.
 * Shall not be executed in user space in case of OOM.
 */

static int exchnd_backtrace_all_threads_execute(struct exception_info *info,
						enum exchnd_modules module)
{
	struct exchnd_message_header header;

	if (!info->task) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		goto exit;
	}

	/* User space */
	if (!(info->task->flags & PF_KTHREAD)) {
		header.type = module;
		header.trigger = info->trigger;
		header.pid = info->task->pid;
		header.seq_num               = 0;
		header.length                = 0;
		header.flags.collected       = 0;
		header.flags.internal        = 0;
		header.flags.addition_needed = 1;
		info->write_func(&header, NULL);
	}

exit:
	return 0;
}

/*
 * exception handler module CGROUPS
 *
 * The module collects the cgroup information of the current process. If the
 * process is completely in user space, the generation is done in user space.
 */
static int exchnd_cgroups_execute(struct exception_info *info,
				  enum exchnd_modules module)
{
	struct exchnd_message_header header;

	header.length = 0;
	header.type = module;
	header.trigger = info->trigger;
	header.pid = 0;
	header.seq_num               = 0;
	header.flags.collected       = 0;
	header.flags.internal        = 0;
	header.flags.addition_needed = 1;

	info->write_func(&header, NULL);

	return 0;
}

/*
 * exception handler module EHM_CPU_USAGE
 *
 * The module collects the workload of the CPUs.
 */

static int exchnd_cpu_usage_execute(struct exception_info *info,
				    enum exchnd_modules module)
{
	struct exchnd_message_header header;

	header.length = 0;
	header.type = module;
	header.trigger = info->trigger;
	header.pid = 0;
	header.seq_num               = 0;
	header.flags.collected       = 0;
	header.flags.internal        = 0;
	header.flags.addition_needed = 1;

	info->write_func(&header, NULL);

	return 0;
}

#define EHM_FAULT_ADDRESS_HEADER "====== fault address:"
/*
 * exception handler module EHM_FAULT_ADDRESS
 *
 * The module collects the address of the code that crashed.
 */

static int exchnd_fault_address_execute(struct exception_info *info,
					enum exchnd_modules module)
{
	struct exchnd_message_header header;
	struct task_struct *task = info->task;
	unsigned char *data = exchnd_get_pool(info->trigger);
	unsigned long address = 0;

	if (!task) {
		pr_warn("%s skipped as there is no way to retrieve task.\n",
			__func__);
		goto exit;
	}

	address = exchnd_get_fault(task);
	header.type = module;
	header.trigger = info->trigger;
	header.pid = task->pid;
	header.seq_num               = 0;
	header.flags.collected       = 1;
	header.flags.internal        = 0;
	header.flags.addition_needed = 0;

	header.length = snprintf(data,
				 EXCHND_MAX_ENTRY_LEN,
				 EHM_FAULT_ADDRESS_HEADER " %p\n",
				 (void *)address);

	info->write_func(&header, data);
	memset(data, 0, header.length);

exit:
	return 0;
}

/*
 * exception handler module EHM_FS_STATE
 *
 * The module collects the state of the file systems.
 */

static int exchnd_fs_state_execute(struct exception_info *info,
				   enum exchnd_modules module)
{
	struct exchnd_message_header header;

	/* User space */
	header.length = 0;
	header.type = module;
	header.trigger = info->trigger;
	header.pid = 0;
	header.seq_num               = 0;
	header.flags.collected       = 0;
	header.flags.internal        = 0;
	header.flags.addition_needed = 1;

	info->write_func(&header, NULL);

	return 0;
}

/*
 * exception handler module EHM_PROCESSOR_REGISTERS
 *
 * The module collects the contents of the processor registers at the time
 * of the crash.
 */
static int exchnd_processor_registers_execute(struct exception_info *info,
					      enum exchnd_modules module)
{
	struct exchnd_message_header header;
	unsigned char *data = exchnd_get_pool(info->trigger);
	struct pt_regs *regs = NULL;

	if (!info->task && !info->regs) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		goto exit;
	}

	/* Init header. */
	header.type = module;
	header.trigger = info->trigger;
	header.seq_num               = 0;
	header.flags.internal        = 0;
	header.flags.collected       = 1;
	header.flags.addition_needed = 0;

	if (info->regs) {
		regs = info->regs;
		header.pid = 0;
	} else {
		/* Should work even if we are in current task. */
		regs = task_pt_regs(info->task);
		header.pid = info->task->pid;
	}

	header.length = exchnd_dump_regs(data, regs, info->task);

	info->write_func(&header, data);
	memset(data, 0, header.length);

exit:
	return 0;
}

/*
 * exception handler module EHM_SYSTEM_INFO
 *
 * The module collects information about the state of the system.
 */
static int exchnd_system_info_execute(struct exception_info *info,
				      enum exchnd_modules module)
{
	return 0;
}

/*
 * exception handler module EHM_MEMORY_MAP
 *
 * The module collects the memory map of the process that crashed.
 */
static int exchnd_memory_map_execute(struct exception_info *info,
				     enum exchnd_modules module)
{
	if (!info->task) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		goto exit;
	}

	/* User space */
	if (!(info->task->flags & PF_KTHREAD)) {
		struct exchnd_message_header header;

		header.length = 0;
		header.type = module;
		header.trigger = info->trigger;
		header.pid = info->task->pid;
		header.seq_num               = 0;
		header.flags.collected       = 0;
		header.flags.internal        = 0;
		header.flags.addition_needed = 1;

		info->write_func(&header, NULL);
	}
	/* Kernel */
	/* TODO: Kernel thread handling */

exit:
	return 0;
}

#define EHM_MEMORY_USAGE_HEADER "====== memory usage:\n"
/*
 * exception handler module EHM_MEMORY_USAGE
 *
 * The module collects the memory usage of the complete system at the time of
 * the crash.
 */
static int exchnd_memory_usage_execute(struct exception_info *info,
				       enum exchnd_modules module)
{
	struct exchnd_message_header header;
	unsigned char *data = exchnd_get_pool(info->trigger);
	struct sysinfo val;

	/* Init header. */
	header.type = module;
	header.trigger = info->trigger;
	header.pid                   = 0;
	header.seq_num               = 0;
	header.flags.collected       = 1;
	header.flags.internal        = 0;
	header.flags.addition_needed = 0;

	/* Getting meminfo. */
	si_meminfo(&val);

	/* Formatting */
#define K(x) ((x) << (PAGE_SHIFT - 10)) /* Display in kilobytes. */
	header.length = snprintf(data,
				 EXCHND_MAX_TRACE_LEN,
				 EHM_MEMORY_USAGE_HEADER
				 "MemTotal:       %8lu kB\n"
				 "MemFree:        %8lu kB\n"
				 "Buffers:        %8lu kB\n"
				 "HighTotal:      %8lu kB\n"
				 "HighFree:       %8lu kB\n",
				 K(val.totalram),
				 K(val.freeram),
				 K(val.bufferram),
				 K(val.totalhigh),
				 K(val.freehigh));

	info->write_func(&header, data);
	memset(data, 0, header.length);

	return 0;
}

#define EHM_PROCESS_LIST_HEADER "===== process list:\n"
/*
 * exception handler module EHM_PROCESS_LIST
 *
 * The module collects a list of all existing processes at the time of the
 * crash.
 */
static int exchnd_process_list_execute(struct exception_info *info,
				       enum exchnd_modules module)
{
	struct exchnd_message_header header;
	unsigned char *data = exchnd_get_pool(info->trigger);

	if (!info->task) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		goto exit;
	}

	header.type = module;
	header.trigger = info->trigger;
	header.pid = info->task->pid;
	header.seq_num = 0;
	header.flags.internal = 0;

	/* User space */
	if (!(info->task->flags & PF_KTHREAD)) {
		header.length                = 0;
		header.flags.collected       = 0;
		header.flags.addition_needed = 1;
	} else {
		/* Kernel */
		unsigned int len = 0;
		struct task_struct *tsk;

		header.flags.collected       = 1;
		header.flags.addition_needed = 0;

		len = snprintf(data,
			       EXCHND_MAX_ENTRY_LEN,
			       EHM_PROCESS_LIST_HEADER);

		rcu_read_lock();
		for_each_process(tsk) {
			pid_t tid = task_tgid_vnr(tsk);

			len += snprintf(data + len,
					5 * EXCHND_MAX_TRACE_LEN - len,
					"%-5d %s\n",
					tid,
					tsk->comm);

			if (len >= (5 * EXCHND_MAX_TRACE_LEN)) {
				data[--len] = '\0';
				break;
			}
		}
		rcu_read_unlock();
		data[len] = '\n';
		header.length = len;
	}

	info->write_func(&header, data);
	memset(data, 0, header.length);

exit:
	return 0;
}

#define EHM_STACK_DUMP_HEADER "===== stack dump:\n"
/*
 * exception handler module EHM_STACK_DUMP
 *
 * The module collects the contents of the active stack at the time of the
 * crash.
 * Shall not be executed in user space in case of OOM.
 */
static int exchnd_stack_dump_execute(struct exception_info *info,
				     enum exchnd_modules module)
{
	struct exchnd_message_header header;
	struct task_struct *task = info->task;
	struct pt_regs *regs = info->regs;
	unsigned char *data = exchnd_get_pool(info->trigger);
	unsigned long sp = 0UL;
	unsigned long start_stack = 0UL;
	int len = 0;

	if (!task && !regs) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		goto exit;
	}

	/* Init header. */
	header.type = module;
	header.trigger = info->trigger;
	header.sig = info->sig;
	header.seq_num               = 0;
	header.flags.internal        = 0;
	header.flags.collected       = 1;
	header.flags.addition_needed = 0;

	header.pid = task->pid;
	if (!regs)
		regs = task_pt_regs(task);

	if (!(task->flags & PF_KTHREAD) && task->mm) {
		sp = kernel_stack_pointer(regs);
		start_stack = task->mm->start_stack;
		len += snprintf(data,
				EXCHND_MAX_TRACE_LEN - len,
				"%lX",
				start_stack);
		header.flags.internal        = 1;
		header.flags.addition_needed = 1;
		goto dump;
	} else
		/* On x86 arch, we can no longer fetch task information
		 * from pt_regs. The pointer to task_struct is removed from
		 * struct thread_info. In order to maintain compatibility
		 * between arm & x86 arch, the stack dump feature for kernel
		 * tasks will be back through a patch later
		 */
		pr_warn("Ignoring stack dump for kernel task\n");

dump:
	header.length = len;
	info->write_func(&header, data);
	memset(data, 0, header.length);

exit:
	return 0;
}

/*
 * exception handler module EHM_THREAD_LIST
 *
 * The module collects the list of threads belonging to the process that
 * crashed.
 */
static int exchnd_thread_list_execute(struct exception_info *info,
				      enum exchnd_modules module)
{
	if (!info->task) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		goto exit;
	}

	/* User space */
	if (!(info->task->flags & PF_KTHREAD)) {
		struct exchnd_message_header header;

		header.length = 0;
		header.type = module;
		header.trigger = info->trigger;
		header.pid = info->task->pid;
		header.seq_num               = 0;
		header.flags.collected       = 0;
		header.flags.internal        = 0;
		header.flags.addition_needed = 1;

		info->write_func(&header, NULL);
	}
	/* Kernel */
	/* TODO: Kernel thread handling */

exit:
	return 0;
}

/**
 * exchnd_sys_restart_execute - module that restarts the system
 * @info: information about the exception to handle
 *
 * The function represent the exception handler module EHM_SYS_RESTART. It
 * is an action module and restarts the system.
 *
 * Return: 0 for success, error code otherwise
 */
static int exchnd_sys_restart_execute(struct exception_info *info,
				      enum exchnd_modules module)
{
	char buf[] = "Restarting the system.\n";
	char cmd[] = "Exchnd SYS_RESTART";
	struct exchnd_message_header header;

	header.length                = strlen(buf);
	header.type                  = EHM_SYS_RESTART;
	header.trigger               = info->trigger;
	if (info->task)
		header.pid           = info->task->pid;
	else
		header.pid           = 0;
	header.seq_num               = 0;
	header.flags.collected       = 1;
	header.flags.internal        = 0;
	header.flags.addition_needed = 0;

	/* Write directly in errmem as we will most likely not return
	 * in user land.
	 */
	em_write(&header, buf);

	if (!in_irq() && !in_softirq())
		kernel_restart(cmd);

	emergency_restart();

	return 0;
}

#define EHM_MEMORY_DUMP_HEADER "====== memory dump around address"
/*
 * exception handler module EHM_MEMORY_DUMP
 *
 * The module collects the memory around the address which access lead to the
 * crash.
 * Shall not be executed in user space in case of OOM.
 */
static int exchnd_memory_dump_execute(struct exception_info *info,
				      enum exchnd_modules module)
{
	struct task_struct *tsk = info->task;
	struct exchnd_message_header header;
	unsigned char *data = exchnd_get_pool(info->trigger);
	int len = 0;
	unsigned long faulty_add = 0;
	int i = 0;

	/* Line to dump should be even */
	int LINES_TO_DUMP = 6;
	int DUMPS_PER_LINE = 4;
	int ITER_START = ((LINES_TO_DUMP - 2) * DUMPS_PER_LINE) / 2;
	int ITER_END = (LINES_TO_DUMP * DUMPS_PER_LINE) / 2;
	int MAX_DUMP_LEN = LINES_TO_DUMP * EXCHND_MAX_ENTRY_LEN;

	if (!tsk) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		goto exit;
	}

	faulty_add = KSTK_EIP(tsk);

	if (unlikely(!faulty_add))
		return 0;

	header.type = module;
	header.trigger = info->trigger;
	header.pid = tsk->pid;
	header.seq_num               = 0;
	header.flags.collected       = 1;
	header.flags.internal        = 0;
	header.flags.addition_needed = 0;

	if (!(tsk->flags & PF_KTHREAD)) {
		len = snprintf(data,
			       MAX_DUMP_LEN,
			       "%lX",
			       faulty_add);
		header.flags.internal        = 1;
		header.flags.addition_needed = 1;
	} else {
		len = snprintf(data,
			       MAX_DUMP_LEN,
			       EHM_MEMORY_DUMP_HEADER " %p:\n",
			       (void *)faulty_add);

		/* Formatting output */
		for (i = -ITER_START; i < ITER_END;) {
			int j = 0;
			/* Getting address from unsigned long */
			unsigned long *add = (unsigned long *)
				(faulty_add + i * sizeof(long));

			len += snprintf(data + len,
					MAX_DUMP_LEN - len,
					"<%p>: ",
					add);

			for (j = 0; j < DUMPS_PER_LINE; j++) {
				len += snprintf(data + len,
						MAX_DUMP_LEN - len,
						MEM_TMPL,
						*add);
				i++;
				add = (unsigned long *)
					(faulty_add + i * sizeof(long));
			}

			len += snprintf(data + len,
					MAX_DUMP_LEN - len,
					"\n");

			if (len >= MAX_DUMP_LEN) {
				len = MAX_DUMP_LEN - 1;
				break;
			}
		}
		data[len] = '\n';
	}

	header.length = len;
	info->write_func(&header, data);
	memset(data, 0, header.length);

exit:
	return 0;
}

/* Library path */
static char exchnd_lib_name[5][EXH_PATH_MAX] = {
	"exchnd_lib.so",
	"exchnd_lib.so",
	"exchnd_lib.so",
	"exchnd_lib.so",
	"exchnd_lib.so"
};

/*
 * exchnd_set_lib_path - Library path setter
 *
 * Allows user to define a specific library name and path for an application
 * specific module.
 *
 * return an error if module value is out of bounds.
 */
int exchnd_set_lib_path(struct exchnd_conf_app_spec *conf)
{
	if ((conf->module < EHM_APPLICATION_SPECIFIC1) ||
	    (conf->module > EHM_APPLICATION_SPECIFIC5))
		return -EINVAL;

	strncpy(exchnd_lib_name[conf->module - EHM_APPLICATION_SPECIFIC1],
		conf->lib_path,
		EXH_PATH_MAX - 1);

	return 0;
}

/*
 * exception handler module EHM_APPLICATION_SPECIFICX
 *
 * This function is the common part to any application specific module.
 */
static int exchnd_application_specific_execute(struct exception_info *info,
					       enum exchnd_modules module)
{
	struct exchnd_message_header header;
	char *lib_name = NULL;
	int ret = -EINVAL;

	if ((module < EHM_APPLICATION_SPECIFIC1) ||
	    (module > EHM_APPLICATION_SPECIFIC5)) {
		pr_err("Invalid value for module: %d.\n", module);
		goto exit;
	}

	ret = 0;

	lib_name = exchnd_lib_name[module - EHM_APPLICATION_SPECIFIC1];

	header.type = module;
	header.trigger = info->trigger;
	header.pid = 0;

	if (info->task)
		header.pid = info->task->pid;

	header.seq_num               = 0;
	header.flags.collected       = 1;
	header.flags.internal        = 1;
	header.flags.addition_needed = 1;

	header.length = strlen(lib_name) + 1;
	info->write_func(&header, (unsigned char *)lib_name);

exit:
	return ret;
}

#ifdef CONFIG_TRACEPOINTS

#define exchnd_register_tracepoint(x, y, z)			\
	({							\
	 tracepoint_probe_register(&__tracepoint_##x, y, z);	\
	 })

#define exchnd_unregister_tracepoint(x, y, z)			\
	({							\
	 tracepoint_probe_unregister(&__tracepoint_##x, y, z);	\
	 })

#endif /* CONFIG_TRACEPOINTS */

#define EH_DEFAULT_HIST_COUNT 4096
#define EH_READ_HIST_COUNT 64
static unsigned int exchnd_read_hist_count = EH_READ_HIST_COUNT;
static pid_t exchnd_trace_pid;

void exchnd_set_trace_pid(pid_t pid)
{
	exchnd_trace_pid = pid;
}

void exchnd_set_hist_size(unsigned int size)
{
	if (size > (EH_DEFAULT_HIST_COUNT / 4))
		size = EH_DEFAULT_HIST_COUNT / 4;

	if (size == 0)
		size = EH_READ_HIST_COUNT;

	pr_info("Modifying history size to %d.\n", size);
	exchnd_read_hist_count = size;
}

#define EHM_HIST_SYSCALL_HEADER "====== syscall history\n"

static unsigned int syscall_index;

unsigned int get_syscall_index(void)
{
	return syscall_index;
}

#if !defined(CONFIG_EXCHND_MODULE) && defined(CONFIG_TRACEPOINTS)
/*
 * exception handler module EHM_HIST_SYSCALLS
 *
 * Provide the list of last hist_count system calls that were made.
 *
 */
static struct exchnd_hist_syscall {
	pid_t		pid;			/* process ID */
	int		nr;			/* Sys call NR */
	unsigned long	args[6];		/* Sys call arguments */
	u64		time;			/* Jiffies */
	char		cpu;			/* CPU id */
	char		comm[TASK_COMM_LEN];	/* task comm */
} *syscall_rb, *pool_syscall, *pool_syscall_fatal;

static DEFINE_SPINLOCK(syscall_rb_lock);
static char syscall_names[NR_syscalls][KSYM_SYMBOL_LEN] = { { '\0' } };
static int syscall_reading;
static unsigned int syscall_pad;

static void exchnd_sys_entry(void *data, struct pt_regs *regs, long id)
{
	int tmp_idx;

	if (id < 0)
		return;

	if (id >= NR_syscalls)
		return;

	/* Bad luck for data loss, but we can live with that.
	 * We don't want to lock during syscall. System is already slowed down
	 * enough by executing this code.
	 */
	if (syscall_reading)
		return;

	if (exchnd_trace_pid &&
	    (exchnd_trace_pid != current->group_leader->pid))
		return;

	tmp_idx = syscall_index++;
	tmp_idx &= (EH_DEFAULT_HIST_COUNT - 1);

	/* Getting basic information */
	syscall_rb[tmp_idx].pid = current->pid;
	syscall_rb[tmp_idx].nr = id;
	preempt_disable();
	syscall_rb[tmp_idx].cpu = smp_processor_id();
	preempt_enable();
	syscall_rb[tmp_idx].time = cpu_clock(syscall_rb[tmp_idx].cpu);
	memcpy(syscall_rb[tmp_idx].comm, current->comm, TASK_COMM_LEN);
	/* Getting args */
	syscall_get_arguments(current, regs, 0, 6, syscall_rb[tmp_idx].args);
}

#define EXCH_SYS_INFO "%20llu: %-*s(%3d) from %-*s(%5d) on CPU %d."
#define EXCH_SYS_ARGS				\
	" Args " XMEM_TMPL XMEM_TMPL XMEM_TMPL	\
	XMEM_TMPL XMEM_TMPL XMEM_TMPL "\n"

static int exchnd_hist_syscall_execute(struct exception_info *info,
				       enum exchnd_modules module)
{
	struct exchnd_message_header header;
	unsigned char *data = exchnd_get_pool(info->trigger);

	unsigned int i = 0;
	unsigned long flags;
	unsigned int start = 0;
	unsigned int length = 0;
	struct exchnd_hist_syscall *local_syscall = NULL;
	int local_size = sizeof(*local_syscall) * exchnd_read_hist_count;
	unsigned int size = 0;
	int cpylen = 0;
	int ret = 0;

	if (!info->task) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		ret = 0;
		goto exit;
	}

	if (exchnd_trigger_list[info->trigger].fatality == FT_FATAL)
		local_syscall = pool_syscall_fatal;
	else
		local_syscall = pool_syscall;

	/* Init header. Data has to be directly sent to outputs. */
	header.type = module;
	header.trigger = info->trigger;
	header.seq_num               = 0;
	header.flags.internal        = 0;
	header.flags.collected       = 1;
	header.flags.addition_needed = 0;
	header.pid                   = 0;

	spin_lock_irqsave(&syscall_rb_lock, flags);
	if (!syscall_rb || !local_syscall) {
		/* Nothing to read anymore */
		spin_unlock_irqrestore(&syscall_rb_lock, flags);
		goto exit;
	}

	size = sizeof(struct exchnd_hist_syscall);

	i = info->syscall_index;

	start = (i - exchnd_read_hist_count) & (EH_DEFAULT_HIST_COUNT - 1);
	if ((start + 1) == EH_DEFAULT_HIST_COUNT)
		start = 0;

	syscall_reading++;
	/* We are somewhere in the middle of the history */
	if ((EH_DEFAULT_HIST_COUNT - start) >= exchnd_read_hist_count)
		memcpy(local_syscall,
		       &syscall_rb[start],
		       size * exchnd_read_hist_count);
	else {
		cpylen = size * (EH_DEFAULT_HIST_COUNT - start);
		memcpy(local_syscall, syscall_rb + start, cpylen);

		cpylen = size * (exchnd_read_hist_count) - cpylen;
		memcpy(local_syscall + (EH_DEFAULT_HIST_COUNT - start),
		       syscall_rb,
		       cpylen);
	}
	syscall_reading--;
	spin_unlock_irqrestore(&syscall_rb_lock, flags);

	length += snprintf(data + length,
			   2 * EXCHND_MAX_TRACE_LEN - length,
			   EHM_HIST_SYSCALL_HEADER);

	for (i = 0; i < exchnd_read_hist_count; i++) {
		/* Skip unused elements */
		if (local_syscall[i].time == 0)
			continue;

		/* Send the message to user space once we have one "block" */
		if (length >= EXCHND_MAX_TRACE_LEN) {
			header.length = length;
			info->write_func(&header, data);
			length = 0;
		}

		length += snprintf(data + length,
				   2 * EXCHND_MAX_TRACE_LEN - length,
				   EXCH_SYS_INFO EXCH_SYS_ARGS,
				   local_syscall[i].time,
				   syscall_pad,
				   syscall_names[local_syscall[i].nr],
				   local_syscall[i].nr,
				   TASK_COMM_LEN,
				   local_syscall[i].comm,
				   local_syscall[i].pid,
				   local_syscall[i].cpu,
				   local_syscall[i].args[0],
				   local_syscall[i].args[1],
				   local_syscall[i].args[2],
				   local_syscall[i].args[3],
				   local_syscall[i].args[4],
				   local_syscall[i].args[5]);
	}

	header.length = length;

	info->write_func(&header, data);

exit:
	memset(data, 0, header.length);
	memset(local_syscall, 0, local_size);

	return ret;
}

static inline void exchnd_set_traceflags(struct task_struct *t)
{
	if (exchnd_trace_pid && (exchnd_trace_pid != t->group_leader->pid) &&
	    test_tsk_thread_flag(t, TIF_SYSCALL_TRACEPOINT))
		clear_tsk_thread_flag(t, TIF_SYSCALL_TRACEPOINT);

	if ((!exchnd_trace_pid || (exchnd_trace_pid == t->group_leader->pid)) &&
	    !test_tsk_thread_flag(t, TIF_SYSCALL_TRACEPOINT))
		set_tsk_thread_flag(t, TIF_SYSCALL_TRACEPOINT);
}

static void enable_trace(void *data,
			 bool preempt,
			 struct task_struct *prev,
			 struct task_struct *next)
{
	if (prev->mm)
		exchnd_set_traceflags(prev);

	if (next->mm)
		exchnd_set_traceflags(next);
}

static void *exchnd_hist_syscall_init(struct exchnd_module *mod)
{
	int res = 0, i = 0;
	unsigned long flags;
	int size = sizeof(*syscall_rb) * EH_DEFAULT_HIST_COUNT;

	if (exchnd_debug(EXCHND_DEBUG_MODULE))
		pr_info("Initializing %s.\n",
			exchnd_mod_names[EHM_HIST_SYSCALLS]);

	if (mod->opaque)
		goto exit;

	/* Initialize the ring buffer */
	syscall_rb = kzalloc(size, GFP_KERNEL);
	if (!syscall_rb)
		goto exit;

	pool_syscall = kzalloc(size / 4, GFP_KERNEL);
	if (!pool_syscall)
		goto free_syscall_rb;

	pool_syscall_fatal = kzalloc(size / 4, GFP_KERNEL);
	if (!pool_syscall_fatal)
		goto free_pool_syscall;

	syscall_index = 0;
	res = exchnd_register_tracepoint(sys_enter,
					 &exchnd_sys_entry,
					 &syscall_rb_lock);
	if (res) {
		pr_err("sys_enter probe registration failed (%d)\n", res);
		goto free_pool_syscall_fatal;
	}

	res = exchnd_register_tracepoint(sched_switch, &enable_trace, NULL);
	if (res) {
		pr_err("sched_switch probe registration failed (%d)\n", res);
		goto unregister_sys_enter;
	}

	syscall_pad = 0;

	for (i = 0; i < NR_syscalls; i++) {
		sprint_symbol_no_offset(syscall_names[i],
					(unsigned long)sys_call_table[i]);

		if (strlen(syscall_names[i]) > syscall_pad)
			syscall_pad = strlen(syscall_names[i]);
	}

	spin_lock_irqsave(&syscall_rb_lock, flags);
	syscall_reading = 0;

	mod->opaque = &exchnd_sys_entry;
	mod->execute = exchnd_hist_syscall_execute;
	spin_unlock_irqrestore(&syscall_rb_lock, flags);

	return mod->opaque;

unregister_sys_enter:
	exchnd_unregister_tracepoint(sys_enter,
				     &exchnd_sys_entry,
				     &syscall_rb_lock);
free_pool_syscall_fatal:
	kfree(pool_syscall_fatal);
	pool_syscall_fatal = NULL;
free_pool_syscall:
	kfree(pool_syscall);
	pool_syscall = NULL;
free_syscall_rb:
	kfree(syscall_rb);
	syscall_rb = NULL;
exit:
	return NULL;
}

static void exchnd_hist_syscall_deinit(struct exchnd_module *mod)
{
	unsigned long flags;
	struct exchnd_hist_syscall *free_pool_syscall = pool_syscall;
	struct exchnd_hist_syscall *free_syscall_fatal = pool_syscall_fatal;
	struct exchnd_hist_syscall *free_syscall_rb = syscall_rb;

	/* Safely destroy the ring buffer */
	spin_lock_irqsave(&syscall_rb_lock, flags);

	if (!mod->opaque) {
		spin_unlock_irqrestore(&syscall_rb_lock, flags);
		return;
	}

	/* First prevent execution */
	mod->execute = NULL;
	/* Prevent concurrent de-init */
	mod->opaque = NULL;
	/* Ignore ongoing syscalls */
	syscall_reading++;

	spin_unlock_irqrestore(&syscall_rb_lock, flags);

	if (exchnd_debug(EXCHND_DEBUG_MODULE))
		pr_info("De-initializing %s.\n",
			exchnd_mod_names[EHM_HIST_SYSCALLS]);

	/* Unregister the probe */
	exchnd_unregister_tracepoint(sys_enter,
				     &exchnd_sys_entry,
				     &syscall_rb_lock);

	exchnd_unregister_tracepoint(sched_switch, &enable_trace, NULL);

	spin_lock_irqsave(&syscall_rb_lock, flags);
	/* Safely destroy the ring buffer */
	syscall_rb = NULL;
	pool_syscall = NULL;
	pool_syscall_fatal = NULL;
	spin_unlock_irqrestore(&syscall_rb_lock, flags);

	kfree(free_pool_syscall);
	kfree(free_syscall_rb);
	kfree(free_syscall_fatal);
}
#else
static void *exchnd_hist_syscall_init(struct exchnd_module *mod)
{
	pr_info("%s is not available if not built-in.\n",
		exchnd_mod_names[EHM_HIST_SYSCALLS]);
	return NULL;
}

static void exchnd_hist_syscall_deinit(struct exchnd_module *mod) { }
#endif /* !CONFIG_TRACEPOINTS || CONFIG_EXCHND_MODULE */

#define EHM_HIST_TASKSWITCHES_HEADER "====== task switch history:\n"
#define TSWITCH_TPL "%llu: switch from %s(%d) to %s(%d) on CPU %d\n"

static unsigned int tswitch_index;

unsigned int get_tswitch_index(void)
{
	return tswitch_index;
}

#if !defined(CONFIG_EXCHND_MODULE) && defined(CONFIG_TRACEPOINTS)
/*
 * exception handler module EHM_HIST_TASKSWITCHES
 *
 * Provide the list of last EH_DEFAULT_HIST_COUNT task switches that were made.
 *
 */
static struct exchnd_hist_tswitch {
	pid_t		ppid;			/* prev process ID */
	pid_t		npid;			/* next process ID */
	u64		time;			/* Jiffies */
	char		cpu;			/* CPU id */
	char		pcomm[TASK_COMM_LEN];	/* prev task comm */
	char		ncomm[TASK_COMM_LEN];	/* next task comm */
} *tswitch_rb, *pool_tswitch, *pool_tswitch_fatal;

static DEFINE_SPINLOCK(tswitch_rb_lock);
static int taskswitch_reading;

static int exchnd_hist_tswitch_execute(struct exception_info *info,
				       enum exchnd_modules module)
{
	struct exchnd_message_header header;
	unsigned char *data = exchnd_get_pool(info->trigger);
	int size = 0;
	int cpylen = 0;
	int ret = 0;

	unsigned int i = 0;
	unsigned long flags;
	unsigned int start;
	unsigned int length = 0;

	struct exchnd_hist_tswitch *local_tswitch = NULL;
	int local_size = sizeof(*tswitch_rb) * exchnd_read_hist_count;

	if (!info->task) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		ret = 0;
		goto exit;
	}

	if (exchnd_trigger_list[info->trigger].fatality == FT_FATAL)
		local_tswitch = pool_tswitch_fatal;
	else
		local_tswitch = pool_tswitch;

	/* Init header. Data has to be directly sent to outputs. */
	header.type = module;
	header.trigger = info->trigger;
	header.seq_num               = 0;
	header.flags.internal        = 0;
	header.flags.collected       = 1;
	header.flags.addition_needed = 0;
	header.pid                   = 0;

	spin_lock_irqsave(&tswitch_rb_lock, flags);
	if (!tswitch_rb || !local_tswitch) {
		/* Nothing to read anymore */
		spin_unlock_irqrestore(&tswitch_rb_lock, flags);
		goto free_local_tswitch;
	}

	size = sizeof(struct exchnd_hist_tswitch);

	/* Search for task in the buffer */
	i = info->tswitch_index;

	start = (i - exchnd_read_hist_count) & (EH_DEFAULT_HIST_COUNT - 1);
	if ((start + 1) == EH_DEFAULT_HIST_COUNT)
		start = 0;

	taskswitch_reading++;
	/* We are somewhere in the middle of the history */
	if ((EH_DEFAULT_HIST_COUNT - start) >= exchnd_read_hist_count)
		memcpy(local_tswitch,
		       &tswitch_rb[start],
		       size * exchnd_read_hist_count);
	else {
		cpylen = size * (EH_DEFAULT_HIST_COUNT - start);
		memcpy(local_tswitch, tswitch_rb + start, cpylen);

		cpylen = size * (exchnd_read_hist_count) - cpylen;
		memcpy(local_tswitch + (EH_DEFAULT_HIST_COUNT - start),
		       tswitch_rb,
		       cpylen);
	}
	taskswitch_reading--;

	spin_unlock_irqrestore(&tswitch_rb_lock, flags);

	length += snprintf(data + length,
			   2 * EXCHND_MAX_TRACE_LEN - length,
			   EHM_HIST_TASKSWITCHES_HEADER);

	for (i = 0; i < exchnd_read_hist_count; i++) {
		/* Skip unused elements */
		if (local_tswitch[i].time == 0)
			continue;

		/* Send the message to user space once we have one "block" */
		if (length >= EXCHND_MAX_TRACE_LEN) {
			header.length = length;
			info->write_func(&header, data);
			length = 0;
		}

		length += snprintf(data + length,
				   2 * EXCHND_MAX_TRACE_LEN - length,
				   TSWITCH_TPL,
				   local_tswitch[i].time,
				   local_tswitch[i].pcomm,
				   local_tswitch[i].ppid,
				   local_tswitch[i].ncomm,
				   local_tswitch[i].npid,
				   local_tswitch[i].cpu);
	}

	header.length = length;

	info->write_func(&header, data);

free_local_tswitch:
	memset(local_tswitch, 0, local_size);
	memset(data, 0, header.length);
exit:
	return ret;
}

static void exchnd_tswitch(void *data,
			   bool preempt,
			   struct task_struct *prev,
			   struct task_struct *next)
{
	int tmp_idx;

	/* Bad luck for data loss, but we can live with that */
	if (taskswitch_reading)
		return;

	if (exchnd_trace_pid &&
	    (exchnd_trace_pid != prev->group_leader->pid) &&
	    (exchnd_trace_pid != next->group_leader->pid))
		return;

	tmp_idx = tswitch_index++;
	tmp_idx &= (EH_DEFAULT_HIST_COUNT - 1);

	/* Getting basic information */
	tswitch_rb[tmp_idx].ppid = prev->pid;
	tswitch_rb[tmp_idx].npid = next->pid;

	preempt_disable();
	tswitch_rb[tmp_idx].cpu = smp_processor_id();
	preempt_enable();
	tswitch_rb[tmp_idx].time = cpu_clock(tswitch_rb[tmp_idx].cpu);

	memcpy(tswitch_rb[tmp_idx].pcomm, prev->comm, TASK_COMM_LEN);
	memcpy(tswitch_rb[tmp_idx].ncomm, next->comm, TASK_COMM_LEN);
}

static void *exchnd_hist_tswitch_init(struct exchnd_module *mod)
{
	int res = 0;
	int size = sizeof(*tswitch_rb) * EH_DEFAULT_HIST_COUNT;

	if (exchnd_debug(EXCHND_DEBUG_MODULE))
		pr_info("Initializing %s.\n",
			exchnd_mod_names[EHM_HIST_TASKSWITCHES]);

	if (mod->opaque)
		goto exit;

	/* Initialize the ring buffer */
	tswitch_rb = kzalloc(size, GFP_KERNEL);
	if (!tswitch_rb)
		goto exit;

	pool_tswitch = kzalloc(size / 4, GFP_KERNEL);
	if (!pool_tswitch)
		goto free_tswitch_rb;

	pool_tswitch_fatal = kzalloc(size / 4, GFP_KERNEL);
	if (!pool_tswitch_fatal)
		goto free_pool_tswitch;

	tswitch_index = 0;
	taskswitch_reading = 0;

	res = exchnd_register_tracepoint(sched_switch, &exchnd_tswitch, NULL);
	if (res) {
		pr_err("sched_switch probe registration failed (%d)\n", res);
		goto free_pool_tswitch_fatal;
	}

	mod->opaque = &exchnd_tswitch;
	mod->execute = exchnd_hist_tswitch_execute;

	return mod->opaque;

free_pool_tswitch_fatal:
	kfree(pool_tswitch_fatal);
	pool_tswitch_fatal = NULL;
free_pool_tswitch:
	kfree(pool_tswitch);
	pool_tswitch = NULL;
free_tswitch_rb:
	kfree(tswitch_rb);
	tswitch_rb = NULL;
exit:
	return NULL;
}

static void exchnd_hist_tswitch_deinit(struct exchnd_module *mod)
{
	unsigned long flags;
	struct exchnd_hist_tswitch *free_tswitch_rb = tswitch_rb;
	struct exchnd_hist_tswitch *free_tmp_tswitch = pool_tswitch;
	struct exchnd_hist_tswitch *free_tswitch_fatal = pool_tswitch_fatal;

	/* Prevent execution of the collector. */
	spin_lock_irqsave(&tswitch_rb_lock, flags);

	if (!mod->opaque) {
		spin_unlock_irqrestore(&tswitch_rb_lock, flags);
		return;
	}

	/* First prevent execution */
	mod->execute = NULL;
	/* Prevent concurrent de-init */
	mod->opaque = NULL;
	/* Ignore ongoing task switches */
	taskswitch_reading++;

	spin_unlock_irqrestore(&tswitch_rb_lock, flags);

	if (exchnd_debug(EXCHND_DEBUG_MODULE))
		pr_info("De-initializing %s.\n",
			exchnd_mod_names[EHM_HIST_TASKSWITCHES]);

	/* First unregister the probe */
	exchnd_unregister_tracepoint(sched_switch, &exchnd_tswitch, NULL);

	/* Safely destroy the ring buffer */
	spin_lock_irqsave(&tswitch_rb_lock, flags);
	tswitch_rb = NULL;
	pool_tswitch = NULL;
	pool_tswitch_fatal = NULL;
	spin_unlock_irqrestore(&tswitch_rb_lock, flags);

	kfree(free_tswitch_rb);
	kfree(free_tmp_tswitch);
	kfree(free_tswitch_fatal);
}
#else
static void *exchnd_hist_tswitch_init(struct exchnd_module *mod)
{
	pr_info("%s is not available if not built-in.\n",
		exchnd_mod_names[EHM_HIST_TASKSWITCHES]);
	return NULL;
}

static void exchnd_hist_tswitch_deinit(struct exchnd_module *mod) { }
#endif /* !CONFIG_TRACEPOINTS || CONFIG_EXCHND_MODULE */

#define EHM_SCHED_INFO_HEADER "====== scheduler information:\n"
/*
 * exception handler module EHM_SCHED_INF
 *
 * The module collects scheduling related information that are available in
 * task structure.
 */
#if defined(CONFIG_SCHEDSTATS) || defined(CONFIG_TASK_DELAY_ACCT)
static int exchnd_sched_inf_execute(struct exception_info *info,
				    enum exchnd_modules module)
{
	struct exchnd_message_header header;
	unsigned char *data = exchnd_get_pool(info->trigger);
	unsigned int len = 0;

	if (!info->task) {
		pr_warn("%s skipped as task is a NULL pointer.\n", __func__);
		return 0;
	}

	header.type = module;
	header.trigger = info->trigger;
	header.pid = info->task->pid;
	header.seq_num               = 0;
	header.flags.collected       = 1;
	header.flags.internal        = 0;
	header.flags.addition_needed = 0;

	len = snprintf(data, EXCHND_MAX_TRACE_LEN, EHM_SCHED_INFO_HEADER);

	len += snprintf(data + len,
			EXCHND_MAX_TRACE_LEN - len,
			"pcount:       %20lu\n"
			"run_delay:    %20llu\n"
			"last arrival: %20llu\n"
			"last_queued:  %20llu\n",
			info->task->sched_info.pcount,
			info->task->sched_info.run_delay,
			info->task->sched_info.last_arrival,
			info->task->sched_info.last_queued);

	header.length = len;
	info->write_func(&header, data);
	memset(data, 0, header.length);

	return 0;
}
#else
static int exchnd_sched_inf_execute(struct exception_info *info,
				    enum exchnd_modules module)
{
	pr_warn("%s can't be executed as there is no data available.\n",
		__func__);

	return 0;
}
#endif

/** exchnd_init_modules - Prepares the modules
 *
 * This will prepare the memory pool to be used by the modules at runtime, and
 * then initialize them one by one.
 *
 * Return: 0 on success, an error code otherwise.
 */
int exchnd_init_modules(void)
{
	int i;

	exh_pool = kzalloc(5 * EXCHND_MAX_TRACE_LEN, GFP_KERNEL);
	if (!exh_pool)
		return -ENOMEM;

	exh_fatal_pool = kzalloc(5 * EXCHND_MAX_TRACE_LEN, GFP_KERNEL);
	if (!exh_fatal_pool) {
		kfree(exh_pool);
		return -ENOMEM;
	}

	/* initialize exception handler modules */
	for (i = 0; i < EHM_LAST_ELEMENT; i++) {
		if (exchnd_module_list[i].init &&
		    !(disabled_modules & (1 << i)))
			exchnd_module_list[i].init(&exchnd_module_list[i]);
	}

	return 0;
}

/** exchnd_deinit_modules - Deinit modules
 *
 * All modules are deinitialized, and the pool destroyed.
 */
void exchnd_deinit_modules(void)
{
	int i;

	/* remove exception handler modules */
	for (i = 0; i < EHM_LAST_ELEMENT; i++) {
		if (exchnd_module_list[i].deinit)
			exchnd_module_list[i].deinit(&exchnd_module_list[i]);
	}

	kfree(exh_pool);
	kfree(exh_fatal_pool);
}

/* Defines the callbacks and data of the available EH nodules */
struct exchnd_module exchnd_module_list[EHM_LAST_ELEMENT] = {
	[EHM_BACKTRACE] = {
		.execute = exchnd_backtrace_execute },
	[EHM_BACKTRACE_ALL_THREADS] = {
		.execute = exchnd_backtrace_all_threads_execute },
	[EHM_CGROUPS] = {
		.execute = exchnd_cgroups_execute },
	[EHM_CORE_DUMP] = {
		.execute = NULL },
	[EHM_CPU_USAGE] = {
		.execute = exchnd_cpu_usage_execute },
	[EHM_FAULT_ADDRESS] = {
		.execute = exchnd_fault_address_execute },
	[EHM_FS_STATE] = {
		.execute = exchnd_fs_state_execute },
	[EHM_HIST_SYSCALLS] = {
		.execute = NULL,
		.init = exchnd_hist_syscall_init,
		.deinit = exchnd_hist_syscall_deinit },
	[EHM_HIST_TASKSWITCHES] = {
		.execute = NULL,
		.init = exchnd_hist_tswitch_init,
		.deinit = exchnd_hist_tswitch_deinit },
	[EHM_LRU_MEM_PAGES] = {
		.execute = NULL },
	[EHM_MEMORY_DUMP] = {
		.execute = exchnd_memory_dump_execute },
	[EHM_MEMORY_MAP] = {
		.execute = exchnd_memory_map_execute },
	[EHM_MEMORY_USAGE] = {
		.execute = exchnd_memory_usage_execute },
	[EHM_PROCESSOR_REGISTERS] = {
		.execute = exchnd_processor_registers_execute },
	[EHM_SYSTEM_INFO] = {
		.execute = exchnd_system_info_execute },
	[EHM_PROCESS_LIST] = {
		.execute = exchnd_process_list_execute },
	[EHM_SCHED_INF] = {
		.execute = exchnd_sched_inf_execute },
	[EHM_STACK_DUMP] = {
		.execute = exchnd_stack_dump_execute },
	[EHM_THREAD_LIST] = {
		.execute = exchnd_thread_list_execute },
	[EHM_ACTION_START] = {
		.execute = NULL },
	[EHM_SYS_RESTART] = {
		.execute = exchnd_sys_restart_execute },
	[EHM_APPLICATION_SPECIFIC1] = {
		.execute = exchnd_application_specific_execute },
	[EHM_APPLICATION_SPECIFIC2] = {
		.execute = exchnd_application_specific_execute },
	[EHM_APPLICATION_SPECIFIC3] = {
		.execute = exchnd_application_specific_execute },
	[EHM_APPLICATION_SPECIFIC4] = {
		.execute = exchnd_application_specific_execute },
	[EHM_APPLICATION_SPECIFIC5] = {
		.execute = exchnd_application_specific_execute },
};
