/*
 * linux/drivers/misc/exchnd/noprofiling.c
 *
 * do_exit handling when KERNEL_PROFILING option is not set.
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 */

#define pr_fmt(fmt) "exchnd no profiling: " fmt

#include <linux/slab.h>
#include <linux/exchnd.h>

#include "internal.h"

static DEFINE_SPINLOCK(exit_regs_lck);
static LIST_HEAD(exit_regs);

struct exchnd_exit_list {
	struct list_head list;
	unsigned long task;
	struct pt_regs regs;
};

/**
 * exchnd_stop_exit - Effectively stop task on do_exit
 *
 * This function is added in call stack by exchnd_do_exit. The function triggers
 * the analysis of the process exit.
 */
void exchnd_stop_exit(void)
{
	unsigned long flags;
	struct exchnd_exit_list *el, *n;
	struct pt_regs save_regs;
	struct pt_regs *regs = NULL;

	/* First of all recover registers. */
	spin_lock_irqsave(&exit_regs_lck, flags);
	list_for_each_entry_safe(el, n, &exit_regs, list) {
		if (el->task != (unsigned long)current)
			continue;
		memcpy(&save_regs,
		       &el->regs,
		       sizeof(struct pt_regs));
		regs = &save_regs;
		list_del(&el->list);
		kfree(el);
		if (exchnd_debug(EXCHND_DEBUG_PEXIT))
			pr_info("Registers loaded for process %d.\n",
				current->pid);
	}
	spin_unlock_irqrestore(&exit_regs_lck, flags);

	BUG_ON(!regs);

	exchnd_handle_exit(current);

	exchnd_restor_registers(regs);
}

/**
 * exchnd_do_exit - Function notified by the probe for the process exit
 * @p: probe information
 * @regs: registers at the time of the call
 *
 * This function is called by the probe for "do_exit". The function triggers
 * the analysis of the process exit.
 * We save regs on a list as we are not able to do it on stack on every cases.
 */
static void exchnd_do_exit(struct kprobe *p, struct pt_regs *regs,
			   unsigned long flags)
{
	struct exchnd_exit_list *new;

	/* No-one is there to handle this event */
	/* TODO: Mechanism to handle this event internally anyway ? */
	spin_lock(&rb_get_read_wait_queue()->lock);
	/* waitqueue_active checked under wq lock. */
	if (!waitqueue_active(rb_get_read_wait_queue())) {
		if (exchnd_debug(EXCHND_DEBUG_PEXIT))
			pr_info("Daemon is not active while %d exits.\n",
				current->pid);
		exchnd_unset_pid();

		if (test_thread_flag(TIF_MEMDIE))
			exchnd_oom_end();

		spin_unlock(&rb_get_read_wait_queue()->lock);
		goto exit;
	}
	spin_unlock(&rb_get_read_wait_queue()->lock);

	/* Just ignore kernel threads. No chance for us to trace them. */
	if (unlikely(current->flags & PF_KTHREAD)) {
		if (exchnd_debug(EXCHND_DEBUG_PEXIT))
			pr_info("Ignoring kernel thread.\n");

		if (test_thread_flag(TIF_MEMDIE))
			exchnd_oom_end();

		goto exit;
	}

	new = kzalloc(sizeof(*new), GFP_ATOMIC);
	if (!new) {
		if (test_thread_flag(TIF_MEMDIE))
			exchnd_oom_end();

		goto exit;
	}

	new->task = (unsigned long)current;

	memcpy(&new->regs, regs, sizeof(struct pt_regs));

	/* On some arch the regs->sp value is not correctly updated. */
	new->regs->sp = kernel_stack_pointer(regs);

	spin_lock_irqsave(&exit_regs_lck, flags);
	if (exchnd_debug(EXCHND_DEBUG_PEXIT))
		pr_info("Saving registers on list for process %d.\n",
			current->pid);
	list_add_tail(&new->list, &exit_regs);
	spin_unlock_irqrestore(&exit_regs_lck, flags);

	exchnd_arch_do_exit(regs);
exit:
	return;
}

/* probe set to do exit */
static struct kprobe kp_exit;

/**
 * exchnd_exit_init - Initializes out of process exit trigger
 * @trigger: a pointer to the process exit trigger structure
 *
 * This function will will register a probe with the "do_exit" kernel
 * function.
 *
 * Return: probe structure
 */
void *exchnd_exit_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	kp_exit.addr = (kprobe_opcode_t *)exchnd_get_symbol(do_exit);
	kp_exit.post_handler = exchnd_do_exit;

	if (register_kprobe(&kp_exit) == 0)
		trigger->opaque = &kp_exit;

	if (exchnd_debug(EXCHND_DEBUG_PEXIT))
		pr_info("Process exit initialized.\n");

	return trigger->opaque;
}

/**
 * exchnd_exit_deinit - Deinitializes process exit trigger
 * @trigger: a pointer to the process exit structure
 *
 * This function will deinitialize the process exit trigger by unregistering
 * the registered probe.
 */
void exchnd_exit_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		unregister_kprobe(trigger->opaque);

	if (exchnd_debug(EXCHND_DEBUG_PEXIT))
		pr_info("Process exit deinitialized.\n");
}
