/*
 * linux/drivers/misc/exchnd/profiling.c
 *
 * do_exit handling when KERNEL_PROFILING option is set.
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */
#define pr_fmt(fmt) "exchnd profiling: " fmt

#include <linux/thread_info.h>
#include <linux/notifier.h>
#include <linux/profile.h>
#include <linux/sched.h>

#include "internal.h"

/**
 * exchnd_do_exit - Stop task on do_exit
 *
 * This function is called early in do_exit once the PROFILING is enabled.
 * That's the simplest way to monitor tasks exit.
 */
static int exchnd_do_exit(struct notifier_block *nb,
			  unsigned long e,
			  void *data)
{
	struct task_struct *task = data;

	return exchnd_handle_exit(task);
}

static struct notifier_block nb_exit = {
	.notifier_call = exchnd_do_exit
};

/**
 * exchnd_exit_init - Initializes out of process exit trigger
 * @trigger: a pointer to the process exit trigger structure
 *
 * This function will will register a callback for the "do_exit" profiling.
 *
 * Return: pro structure
 */
void *exchnd_exit_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	if (!profile_event_register(PROFILE_TASK_EXIT, &nb_exit)) {
		trigger->opaque = &nb_exit;
		if (exchnd_debug(EXCHND_DEBUG_PEXIT))
			pr_info("Process exit initialized.\n");
	} else {
		if (exchnd_debug(EXCHND_DEBUG_PEXIT))
			pr_info("Process exit initialization failed.\n");
	}

	return trigger->opaque;
}

/**
 * exchnd_exit_deinit - Deinitializes process exit trigger
 * @trigger: a pointer to the process exit structure
 *
 * This function will deinitialize the process exit trigger by unregistering
 * the registered probe.
 */
void exchnd_exit_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		profile_event_unregister(PROFILE_TASK_EXIT, trigger->opaque);

	if (exchnd_debug(EXCHND_DEBUG_PEXIT))
		pr_info("Process exit deinitialized.\n");
}
