/*
 * linux/drivers/misc/exchnd/triggers.c
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Kai Tomerius (ktomerius@de.adit-jv.com)
 *            Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Exception Triggers
 *
 * The exception trigger catch exceptions of the system. Having the triggers
 * as separate components it is possible to configure which exceptions are
 * caught and what is done in case of a specific exception. The trigger hook
 * into the system and let the exception handler handle the exception.
 */

#define pr_fmt(fmt) "exchnd triggers: " fmt

#include <linux/errmem.h>
#include <linux/exchnd.h>
#include <linux/kdebug.h>
#include <linux/kprobes.h>
#include <linux/notifier.h>
#include <linux/module.h>
#include <linux/oom.h>
#include <linux/reboot.h>
#include <linux/slab.h>
#include <linux/task_work.h>

#include "internal.h"

static LIST_HEAD(signal_list);
static DEFINE_SPINLOCK(signal_list_lck);
/* Signal timeout has to be > to 2*watchdog timeout to limit concurrency.
 * Note: Watchdog kicks the task once it reached the timeout the second time.
 */
#define EXCHND_SIGNAL_TIMEOUT	(2 * EXCHND_WD_MSDELAY + 100)
#define EXCHND_SIGNAL_QUEUED	0
#define EXCHND_SIGNAL_WAITING	1
#define EXCHND_SIGNAL_RUNNING	2
#define EXCHND_SIGNAL_SLEEPING	3

struct exchnd_signal_list {
	struct list_head list;
	unsigned long task;
	int sig;
	int found_sigs;
	int state;
	wait_queue_head_t wq;
};

/* Define trigger strings */
static const char * const exchnd_trig_names[] = EXCHND_TRIGGER_STRINGS;
/* Define signal strings */
static const char * const exchnd_sig_names[] = EXCHND_SIGNAL_STRINGS;
/* Handle unhandled signals ? */
unsigned int exchnd_sighdl_mask = (UINT_MAX & ~(1 << SIGILL));

/*
 * Helper functions for generic exception handling.
 *
 * The functions handle the parts that are generic to the exception handling
 * and are the same independent of the configuration.
 */

/**
 * exc_write_string - Write a string to an exception message
 * @string: string to write
 * @func: output function to use
 *
 * This is a helper function that allows easy "printing" string to an
 * exception message.
 */
void exc_write_string(unsigned char *string, struct exception_info *info)
{
	struct exchnd_message_header header;

	header.length                = strlen(string);
	header.type                  = EHM_NONE;
	header.trigger               = info->trigger;
	if (info->task)
		header.pid           = info->task->pid;
	else
		header.pid           = 0;
	header.seq_num               = 0;
	header.flags.collected       = 1;
	header.flags.internal        = 0;
	header.flags.addition_needed = 0;

	info->write_func(&header, string);
}

#define FORMAT_SIGINFO		\
	"Signal info:\n"	\
	"\tsi_signo:%d\n"	\
	"\tsi_errno:%d\n"	\
	"\tsi_code:%#x\n"

#define FORMAT_SIGINFO_SIGKILL	\
	"\tsi_pid:%d\n"		\
	"\tsi_uid:%d\n"

#define FORMAT_SIGINFO_SIGCHLD	\
	"\tsi_pid:%d\n"		\
	"\tsi_uid:%d\n"		\
	"\tsi_status:%d\n"	\
	"\tsi_utime:%ld\n"	\
	"\tsi_stime:%ld\n"

#define FORMAT_SIGINFO_SIGFAULT	\
	"\tsi_addr:%p\n"

#define FORMAT_SIGINFO_SIGPOLL	\
	"\tsi_band:%ld\n"	\
	"\tsi_fd:%d\n"

#define FORMAT_SIGINFO_SIGSYS	\
	"\tsi_call_addr:%p\n"	\
	"\tsi_syscall:%d\n"	\
	"\tsi_arch:%#x\n"

static void write_signal_info(struct exception_info *info)
{
	char msg_buf[256];
	int i;

	if (!info || (info->sig == 0))
		return;

	for (i = 1; i < SIGRTMIN; i++) {
		if ((info->sig & (1 << i)) == 0)
			continue;
		snprintf(msg_buf,
			 sizeof(msg_buf),
			 "signal \"%s\"\n",
			 exchnd_sig_names[i]);
		exc_write_string(msg_buf, info);
		if (exchnd_debug(EXCHND_DEBUG_HEADER))
			pr_info("%s", msg_buf);
	}

	if (info->siginfo) {
		memset(msg_buf, 0, sizeof(msg_buf));
		snprintf(msg_buf,
			 sizeof(msg_buf),
			 FORMAT_SIGINFO,
			 info->siginfo->si_signo,
			 info->siginfo->si_errno,
			 info->siginfo->si_code);

		switch (info->siginfo->si_signo) {
		case SIGKILL:
			snprintf(msg_buf + strlen(msg_buf),
				 sizeof(msg_buf) - strlen(msg_buf),
				 FORMAT_SIGINFO_SIGKILL,
				 info->siginfo->si_pid,
				 info->siginfo->si_uid);
			break;
		case SIGCHLD:
			snprintf(msg_buf + strlen(msg_buf),
				 sizeof(msg_buf) - strlen(msg_buf),
				 FORMAT_SIGINFO_SIGCHLD,
				 info->siginfo->si_pid,
				 info->siginfo->si_uid,
				 info->siginfo->si_status,
				 info->siginfo->si_utime,
				 info->siginfo->si_stime);
			break;
		case SIGILL:
		case SIGFPE:
		case SIGSEGV:
		case SIGBUS:
			snprintf(msg_buf + strlen(msg_buf),
				 sizeof(msg_buf) - strlen(msg_buf),
				 FORMAT_SIGINFO_SIGFAULT,
				 info->siginfo->si_addr);
			break;
		case SIGPOLL:
			snprintf(msg_buf + strlen(msg_buf),
				 sizeof(msg_buf) - strlen(msg_buf),
				 FORMAT_SIGINFO_SIGPOLL,
				 info->siginfo->si_band,
				 info->siginfo->si_fd);
			break;
		case SIGSYS:
			snprintf(msg_buf + strlen(msg_buf),
				 sizeof(msg_buf) - strlen(msg_buf),
				 FORMAT_SIGINFO_SIGSYS,
				 info->siginfo->si_call_addr,
				 info->siginfo->si_syscall,
				 info->siginfo->si_arch);
			break;
		default:
			break;
		}

		exc_write_string(msg_buf, info);
	}
}

/**
 * write_exc_header - Write the standard header of an exception
 * @info: structure containing the exception information
 *
 * This is a helper function that handles the writing of the default exception
 * header to be used in the trigger functions.
 */
void write_exc_header(struct exception_info *info)
{
	char msg_buf[64];

	if (info->msg[0] != '\0')
		snprintf(msg_buf, sizeof(msg_buf),
			 "****** %s *******\n", info->msg);
	else if (info->task && info->task->pid)
		snprintf(msg_buf, sizeof(msg_buf),
			 "****** EXCEPTION in process PID=%u (%s) *******\n",
			 info->task->pid, info->task->comm);
	else
		snprintf(msg_buf, sizeof(msg_buf),
			 "****** EXCEPTION *******\n");

	exc_write_string(msg_buf, info);

	if (exchnd_debug(EXCHND_DEBUG_HEADER))
		pr_info("%s", msg_buf);

	snprintf(msg_buf, sizeof(msg_buf), "exception trigger %s\n",
		 exchnd_trig_names[info->trigger]);

	exc_write_string(msg_buf, info);

	if (exchnd_debug(EXCHND_DEBUG_HEADER))
		pr_info("%s", msg_buf);

	write_signal_info(info);
}

#define errmem_store_str(x) errmem_store(1, strlen(x), x)

/**
 * em_write - write exception directly to the error memory
 * @header: header of the exception contains metadata such a length, etc.
 * @data: the data of the exception message
 *
 * Writes one exception message directly to the error memory. It is assumed
 * that this function is only called in case of a fatal exception. Otherwise
 * the message will be written to the ring buffer.
 */
void em_write(struct exchnd_message_header *header, unsigned char *data)
{
	/* Exception is fatal -> therefore fatal is always set to 1 */
	char *pch = NULL;

	pch = strsep((char **)&data, "\n");

	while (pch) {
		errmem_store_str("EXH FATAL ");
		errmem_store_str(pch);
		errmem_store_str("\n");
		pch = strsep((char **)&data, "\n");
	}
}

/*
 * exception trigger HALT
 *
 * The exception hooks to the halt of the kernel
 */

/**
 * exchnd_halt - Function notified by the kernel reboot notifier
 * @self: a notification structure of kernel reboot notifier
 * @val: type of reboot
 * @data: additional information of the reboot
 *
 * This callback is registered for the kernel reboot notifier and is triggered
 * when the system will be halted.
 *
 * Return: Result of the notification resp. a request to the chain processing
 */
static int exchnd_halt(struct notifier_block *self,
		       unsigned long val,
		       void *data)
{
	/* Just react on a kernel halt */
	if (val == SYS_HALT) {
		struct exception_info *info = NULL;

		if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
			pr_emerg("Halt triggered.");

		info = eq_get_info(EHT_HALT);

		info->ready = 1;
		eq_start_handling(EHT_HALT);

		/* To make sure that daemon completes the current
		 *  signal handling before system restart.
		 *  Shall not be done in IRQ context as it sleeps.
		 */
		if (!in_interrupt())
			eq_deinit();
	}

	return NOTIFY_OK;
}

static struct notifier_block exchnd_halt_notifier = {
	.notifier_call = exchnd_halt,
};

/**
 * exchnd_halt_init - Initializes halt trigger
 * @trigger: a pointer to the halt trigger structure
 *
 * This function will initialize the halt trigger by registering the reboot
 * notifier.
 *
 * Return: Notifier structure
 */
static void *exchnd_halt_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	if (register_reboot_notifier(&exchnd_halt_notifier) == 0)
		trigger->opaque = &exchnd_halt_notifier;

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Halt initialized.");

	return trigger->opaque;
}

/**
 * exchnd_halt_deinit - Deinitializes halt trigger
 * @trigger: a pointer to the halt trigger structure
 *
 * This function will deinitialize the halt trigger by unregistering a reboot
 * notifier.
 */
static void exchnd_halt_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		unregister_reboot_notifier(trigger->opaque);

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Halt deinitialized.");
}

/*
 * exception trigger KERNEL_OOPS
 *
 * The exception hooks to the kernel oops notifier
 */

/**
 * exchnd_kernel_oops - Function notified by the kernel OOPS notifier
 * @self: a notification structure of kernel die notifier
 * @val: type of death
 * @data: additional information of the death
 *
 * This callback is registered for the kernel die notifier and is triggered
 * when the system will be in an OOPS state.
 *
 * Return: Result of the notification resp. a request to the chain processing
 */
static int exchnd_kernel_oops(struct notifier_block *self,
			      unsigned long val,
			      void *data)
{
	if (((*exchnd_trigger_list[EHT_KERNEL_OOPS].modules)[0] != EHM_NONE) &&
	    /* Just react on a kernel OOPS */
	    ((enum die_val)val == DIE_OOPS)) {
		struct exception_info *info = NULL;
		struct die_args *args = data;

		if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
			pr_emerg("Oops triggered.");

		/* Even if we don't call restart directly,
		 * allow system to restart by itself from now.
		 */
		exchnd_trigger_list[EHT_RESTART]
			.deinit(&exchnd_trigger_list[EHT_RESTART]);

		info = eq_get_info(EHT_KERNEL_OOPS);

		/* Fatal exception, regs will stay available
		 * during handling.
		 */
		info->regs = args->regs;

		info->ready = 1;
		eq_start_handling(EHT_KERNEL_OOPS);
	}

	return NOTIFY_OK;
}

static struct notifier_block exchnd_kernel_oops_notifier = {
	.notifier_call = exchnd_kernel_oops,
};

/**
 * exchnd_kernel_oops_init - Initializes kernel OOPS trigger
 * @trigger: a pointer to the kernel OOPS trigger structure
 *
 * This function will initialize the kernel oops trigger by registering a die
 * notifier.
 *
 * Return: Notifier structure
 */
static void *exchnd_kernel_oops_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	if (register_die_notifier(&exchnd_kernel_oops_notifier) == 0)
		trigger->opaque =  &exchnd_kernel_oops_notifier;

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Oops initialized.");

	return trigger->opaque;
}

/**
 * exchnd_halt_deinit - Deinitializes kernel OOPS trigger
 * @trigger: a pointer to the kernel OOPS trigger structure
 *
 * This function will deinitialize the kernel OOPS trigger by unregistering the
 * kernel OOPS notifier.
 */
static void exchnd_kernel_oops_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		unregister_die_notifier(trigger->opaque);

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Oops deinitialized.");
}

/*
 * exception trigger KERNEL_PANIC
 *
 * The exception hooks to the kernel panic notifier
 */

/**
 * exchnd_kernel_panic - Function notified by the kernel panic notifier
 * @self: a notification structure of kernel panic notifier
 * @val: unused
 * @data: additional information of the panic
 *
 * This callback is registered for the kernel panic notifier and is triggered
 * when the system will be in an panic state.
 *
 * Return: Result of the notification resp. a request to the chain processing
 */
static int exchnd_kernel_panic(struct notifier_block *self,
			       unsigned long val,
			       void *data)
{
	struct exception_info *info = NULL;

	if ((*exchnd_trigger_list[EHT_KERNEL_PANIC].modules)[0] != EHM_NONE) {
		if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
			pr_emerg("Panic triggered.");

		/* Even if we don't call restart directly,
		 * allow system to restart by itself from now.
		 */
		exchnd_trigger_list[EHT_RESTART]
			.deinit(&exchnd_trigger_list[EHT_RESTART]);

		info = eq_get_info(EHT_KERNEL_PANIC);

		info->ready = 1;
		eq_start_handling(EHT_KERNEL_PANIC);
	}

	return NOTIFY_OK;
}

static struct notifier_block exchnd_kernel_panic_notifier = {
	.notifier_call = exchnd_kernel_panic,
};

/**
 * exchnd_kernel_panic_init - Initializes kernel panic trigger
 * @trigger: a pointer to the kernel panic trigger structure
 *
 * This function will initialize the panic trigger by registering a kernel
 * panic notifier.
 *
 * Return: Notifier structure
 */
static void *exchnd_kernel_panic_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	if (atomic_notifier_chain_register(&panic_notifier_list,
					   &exchnd_kernel_panic_notifier) == 0)
		trigger->opaque = &exchnd_kernel_panic_notifier;

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Panic initialized.");

	return trigger->opaque;
}

/**
 * exchnd_halt_deinit - Deinitializes kernel panic trigger
 * @trigger: a pointer to the kernel panic trigger structure
 *
 * This function will deinitialize the kernel panic trigger by unregistering the
 * kernel panic notifier.
 */
static void exchnd_kernel_panic_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		atomic_notifier_chain_unregister(&panic_notifier_list,
						 trigger->opaque);

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Panic deinitialized.");
}

/*
 * exception trigger KERNEL_OOM
 *
 * The exception hooks to the kernel out of memory notifier
 */

/**
 * exchnd_oom - Function notified by the out of memory notifier
 * @self: a notification structure of OOM notifier
 * @val: unused
 * @data: additional information of the out of memory situation
 *
 * This callback is registered for the kernel OOM and is triggered when the
 * system will be in an out of memory state.
 *
 * Return: Result of the notification resp. a request to the chain processing
 */
static int exchnd_oom(struct notifier_block *self,
		      unsigned long val,
		      void *data)
{
	struct exception_info *info = NULL;

	/* Ignore exchndd errors.
	 * We don't want to enter in an infinite loop.
	 */
	if ((exchnd_get_pid() == current->group_leader->pid) ||
	    exchnd_oom_ongoing()) {
		if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
			pr_info("OOM ignored as already ongoing.\n");
		goto exit;
	}

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("OOM triggered by %d.\n", current->pid);

	/* Ignore subsequent OOM notifications till a process is killed. */
	exchnd_oom_start();

	info = eq_get_info(EHT_OOM);

	/* By default, consider current task as the initial cause of OOM even if
	 * it can just be unlucky.
	 */
	info->ready = 1;
	eq_start_handling(EHT_OOM);

exit:
	return NOTIFY_OK;
}

static struct notifier_block exchnd_oom_notifier = {
	.notifier_call = exchnd_oom,
};

/**
 * exchnd_oom_init - Initializes out of memory trigger
 * @trigger: a pointer to the OOM trigger structure
 *
 * This function will initialize the OOM trigger by registering a OOM
 * notifier.
 *
 * Return: Notifier structure
 */
static void *exchnd_oom_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	if (register_oom_notifier(&exchnd_oom_notifier) == 0)
		trigger->opaque = &exchnd_oom_notifier;

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("OOM initialized.");

	return trigger->opaque;
}

/**
 * exchnd_oom_deinit - Deinitializes OOM trigger
 * @trigger: a pointer to the OOM trigger structure
 *
 * This function will deinitialize the OOM trigger by unregistering the OOM
 * notifier.
 */
static void exchnd_oom_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		unregister_oom_notifier(trigger->opaque);

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("OOM initialized.");
}

/*
 * exception trigger POWER_OFF
 *
 * The exception hooks to the power off of the kernel
 */

/**
 * exchnd_power_off - Function notified by the kernel reboot notifier
 * @self: a notification structure of kernel reboot notifier
 * @val: type of reboot
 * @data: additional information of the reboot
 *
 * This callback is registered for the kernel reboot notifier and is triggered
 * when the system will be shut down.
 *
 * Return: Result of the notification resp. a request to the chain processing
 */
static int exchnd_power_off(struct notifier_block *self,
			    unsigned long val,
			    void *data)
{
	/* Just react on a kernel power off */
	if (val == SYS_POWER_OFF) {
		struct exception_info *info = NULL;

		if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
			pr_info("SYS_POWER_OFF triggered.\n");

		info = eq_get_info(EHT_POWER_OFF);
		info->ready = 1;
		eq_start_handling(EHT_POWER_OFF);

		/* To make sure that daemon completes the current
		 *  signal handling before system restart.
		 *  Shall not be done in IRQ context as it sleeps.
		 */
		if (!in_interrupt())
			eq_deinit();
	}

	return NOTIFY_OK;
}

static struct notifier_block exchnd_power_off_notifier = {
	.notifier_call = exchnd_power_off,
};

/**
 * exchnd_power_off_init - Initializes out of power off trigger
 * @trigger: a pointer to the power off trigger structure
 *
 * This function will initialize the power off trigger by registering a
 * reboot notifier.
 *
 * Return: Notifier structure
 */
static void *exchnd_power_off_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	if (register_reboot_notifier(&exchnd_power_off_notifier) == 0)
		trigger->opaque = &exchnd_power_off_notifier;

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Trigger power_off initialized.\n");

	return trigger->opaque;
}

/**
 * exchnd_power_off_deinit - Deinitializes power off trigger
 * @trigger: a pointer to the power off trigger structure
 *
 * This function will deinitialize the power off trigger by unregistering a
 * reboot notifier.
 */
static void exchnd_power_off_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		unregister_reboot_notifier(trigger->opaque);

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Trigger power_off deinitialized.\n");
}

/*
 * exception trigger PROCESS_SIGNAL
 *
 * The exception hooks on a jprobe on send_sigqueue to trigger when a process
 * receives a signal
 */

static int (*exchnd_wake_up_state)(struct task_struct *p, unsigned int state);

/**
 * exchnd_wakeup_pid - Helper function to wakeup a waiting signal element
 * @pid: PID of the task to wake up
 *
 * This function may can be used through the IOCTL or by the watchdog.
 * Aim is to synchronize the daemon with scheduled task work.
 */
static void __exchnd_wakeup_pid(pid_t pid, int forced)
{
	unsigned long flags;
	struct exchnd_signal_list *el;
	struct task_struct *t;

	spin_lock_irqsave(&signal_list_lck, flags);
	list_for_each_entry(el, &signal_list, list) {
		t = (struct task_struct *)el->task;
		if (t->pid != pid)
			continue;

		if (forced) {
			/* The watchdog was triggered.
			 * Try to wakeup in any case
			 */
			if (el->state == EXCHND_SIGNAL_SLEEPING)
				el->state = EXCHND_SIGNAL_RUNNING;

			/*Ensure the variable has been set before task wake-up*/
			smp_mb();

			pr_err("Forced wake up for %d\n", pid);

			t->jobctl |= JOBCTL_TRAP_STOP | JOBCTL_TRAPPING;

			if (!exchnd_wake_up_state(t, TASK_WAKEKILL))
				kick_process(t);
		} else if (el->state == EXCHND_SIGNAL_SLEEPING) {
			el->state = EXCHND_SIGNAL_RUNNING;
			if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
				pr_info("Waking up %d\n", pid);
			/*Ensure the variable has been set before task wake-up*/
			smp_mb();
			/* We now expect ptrace_attach to wake-up the task.
			 * That call shall follow the current IOCTL execution.
			 */
		}
		/* We are done with first entry, there may be others
		 * that will be handled during next run.
		 */
		break;
	}
	spin_unlock_irqrestore(&signal_list_lck, flags);
}

void exchnd_wakeup_pid(pid_t pid)
{
	__exchnd_wakeup_pid(pid, 0);
}

void exchnd_wakeup_pid_wd(pid_t pid)
{
	__exchnd_wakeup_pid(pid, 1);
}

/*
 * build_module_list
 *
 */
static void build_module_list(enum exchnd_modules (*modules)[EHM_LAST_ELEMENT],
			      int found_sigs)
{
	enum exchnd_modules temp = EHM_NONE;
	enum exchnd_modules (*temp_list)[EHM_LAST_ELEMENT] = NULL;
	enum exchnd_modules *concat;
	int ordered_list[EHM_LAST_ELEMENT] = { 0 };
	struct k_sigaction *ka = NULL;
	int i = 0;
	int curr = 0;
	int sig = 0;
	int max = 1;
	int restart = 0;

	concat = kzalloc(sizeof(*concat) * EHM_LAST_ELEMENT * SIGRTMIN,
			 GFP_ATOMIC);

	if (!concat) {
		pr_err("No memory for %s\n", __func__);
		return;
	}

	/* Search for all modules to execute and concat in one list */
	for (sig = 1; sig < SIGRTMIN; sig++) {
		if (!(found_sigs & (1 << sig)))
			continue;

		temp = (*exchnd_signal_list[sig])[0];
		if (temp == EHM_NONE) {
			if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
				pr_info("No module for signal %s.\n",
					exchnd_sig_names[sig]);
			continue;
		}

		ka = &current->sighand->action[sig - 1];
		if ((ka->sa.sa_handler != SIG_DFL) &&
		    !(exchnd_sighdl_mask & (1 << sig))) {
			if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
				pr_info("0x%X (%s) masked with 0x%X.\n",
					(1 << sig),
					exchnd_sig_names[sig],
					exchnd_sighdl_mask);
			continue;
		}

		temp_list = exchnd_signal_list[sig];

		while ((*temp_list)[i] != EHM_NONE) {
			concat[curr++] = (*temp_list)[i];
			i++;
		}
		i = 0;
	}

	/* Remove duplicates */
	for (i = 0; i < curr; i++) {
		temp = concat[i];
		if (ordered_list[temp])
			continue;
		ordered_list[temp] = max++;
	}

	curr = 1;

	/* Rebuild the list with original ordering */
	while (curr < max) {
		for (i = 0; i < EHM_LAST_ELEMENT; i++) {
			if (ordered_list[i] != curr)
				continue;
			if (i == EHM_SYS_RESTART) {
				restart = 1;
				curr++;
				continue;
			}
			(*modules)[curr - 1 - restart] = i;
			curr++;
		}
	}

	/* Now put restart at the very end. */
	if (restart)
		(*modules)[curr - 1 - restart] = EHM_SYS_RESTART;

	(*modules)[curr - restart] = EHM_NONE;
	kfree(concat);
}

/**
 * exchnd_stop_task - Function that stops task that has received a signal
 * @head: head of callback list for task add functions
 *
 * This function stops the task that received a signal to allow analyzing the
 * task in the state it received the signal. It removes also the kill state to
 * allow analysis. After the task is prepared the analysis is started.
 * This function is added as callback to task work to be executed before any
 * other task code is executed.
 */
static void exchnd_stop_task(struct callback_head *head)
{
	struct exception_info *info = NULL;
	struct sigpending *cur_pend = &current->pending;
	int found_sigs = 0;
	enum exchnd_modules (*modules)[EHM_LAST_ELEMENT] = NULL;
	struct exchnd_wd_worker *work;
	unsigned int i;
	unsigned long flags;
	struct exchnd_signal_list *el, *n, *el_save = NULL;
	int sig = 0;
	struct list_head *pos;
	struct list_head *plist;
	struct sigqueue *q;
	int el_found = 0;
	wait_queue_head_t *wq;

	if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
		pr_info("Trigger signal catching process %d.\n",
			current->pid);

	/* First of all recover signal infos. */
	spin_lock_irqsave(&signal_list_lck, flags);
	list_for_each_entry(el, &signal_list, list) {
		if (el->task != (unsigned long)current)
			continue;
		if (el->state != EXCHND_SIGNAL_QUEUED) {
			el_found = 1;
			continue;
		}
		if (el_found)
			el->state = EXCHND_SIGNAL_WAITING;
		else
			el->state = EXCHND_SIGNAL_RUNNING;
		wq = &el->wq;
		el_save = el;
		sig = el->sig;
		found_sigs = el->found_sigs;
		found_sigs |= (1 << sig);
		/* We are done with first entry, there may be others
		 * that will be handled during next run.
		 */
		break;
	}
	spin_unlock_irqrestore(&signal_list_lck, flags);

	/* None is there to handle this event */
	/* TODO: Mechanism to handle this event internally anyway ? */
	spin_lock(&rb_get_read_wait_queue()->lock);
	/* waitqueue_active checked under wq lock. */
	if (!waitqueue_active(rb_get_read_wait_queue())) {
		if (exchnd_debug(EXCHND_DEBUG_DAEMON))
			pr_info("Daemon is not available, ignoring signal.\n");
		exchnd_unset_pid();
		spin_unlock(&rb_get_read_wait_queue()->lock);
		goto done;
	}
	spin_unlock(&rb_get_read_wait_queue()->lock);

	if (!sig) {
		if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
			pr_warn("Signal info not found for %d.\n",
				current->pid);
		goto done;
	}

	/* If we have several work which would lead to a stop of
	 * the task we may end up in a dead sleep as ptrace_attach
	 * would never return. Ignore incoming signal if trapping bit
	 * is set.
	 */
	if (current->ptrace && (current->jobctl & JOBCTL_TRAPPING))
		goto done;

	/* Waiting for early signal to be handled. */
	wait_event_timeout(*wq,
			   el_save->state != EXCHND_SIGNAL_WAITING,
			   msecs_to_jiffies(EXCHND_SIGNAL_TIMEOUT));

	if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
		pr_info("Signal info loaded for process %d (%p).\n",
			current->pid,
			el);

	modules = kzalloc(sizeof(*modules) * EHM_LAST_ELEMENT, GFP_ATOMIC);

	if (!modules)
		goto done;

	if (exchnd_is_filtered(current)) {
		memcpy(modules,
		       exchnd_trigger_list[EHT_PROCESS_SIGNAL].modules,
		       sizeof(*modules) * EHM_LAST_ELEMENT);

		if (exchnd_debug(EXCHND_DEBUG_SIGNAL | EXCHND_DEBUG_FILTER))
			pr_info("Process %d filtered.\n", current->pid);
	} else {
		build_module_list(modules, found_sigs);
	}

	/* Early exit ... */
	if (((*modules)[0] == EHM_NONE) || (current->flags & PF_EXITING)) {
		/* The modules var is safe to be freed here as allocated few
		 * lines earlier and not used any longer due to the statement
		 * failure.
		 * In case of allocation failure, this line is never reached.
		 */
		kfree(modules);
done:
		if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
			pr_info("Signal is not to be handled. Ignoring.\n");
		goto exit;
	}

	/* Set task state early so that we don't receive any other signal
	 * for now.
	 */
	set_current_state(TASK_STOPPED);

	plist = &current->pending.list;
	/* Get signal information before stopping task */
	list_for_each(pos, plist) {
		cur_pend = (struct sigpending *)pos;
		for (i = 1; i < SIGRTMIN; i++) {
			if (sigismember(&cur_pend->signal, i))
				found_sigs |= (1 << i);
		}
	}

	plist = &current->signal->shared_pending.list;
	list_for_each(pos, plist) {
		cur_pend = (struct sigpending *)pos;
		for (i = 1; i < SIGRTMIN; i++) {
			if (sigismember(&cur_pend->signal, i))
				found_sigs |= (1 << i);
		}
	}

	work = exchnd_watch_pid(current->pid);
	if (!work)
		pr_warn("Unable to watch pid: %d\n", current->pid);

	info = eq_get_info(EHT_PROCESS_SIGNAL);

	eq_add_modules(info, modules, EXCHND_MOD_DYN);
	info->sig  = found_sigs;
	info->ready = 1;

	plist = &current->pending.list;
	list_for_each_entry(q, plist, list) {
		info->siginfo = &q->info;
		break;
	}

	plist = &current->signal->shared_pending.list;
	if (!info->siginfo)
		list_for_each_entry(q, plist, list) {
			info->siginfo = &q->info;
			break;
		}

	sigdelset(&current->pending.signal, SIGKILL);

	if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
		pr_info("Starting handling of process %d.\n", current->pid);

	spin_lock_irqsave(&signal_list_lck, flags);
	el_save->state = EXCHND_SIGNAL_SLEEPING;
	spin_unlock_irqrestore(&signal_list_lck, flags);

	eq_start_handling(EHT_PROCESS_SIGNAL);

	/* Waiting for early signal to be handled. */
	for (;;) {
		schedule();
		if (el_save->state != EXCHND_SIGNAL_SLEEPING)
			break;
	}

	if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
		pr_info("Process %d back from schedule.\n", current->pid);

	/* Back from schedule, disable watchdog for this thread. */
	exchnd_unwatch(work);

exit:
	spin_lock_irqsave(&signal_list_lck, flags);
	list_for_each_entry_safe(el, n, &signal_list, list) {
		if (!el_save)
			break;
		if (el->task != (unsigned long)current)
			continue;
		if (el == el_save) {
			list_del(&el->list);
			if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
				pr_info("%s info destroyed for PID %d (%p).\n",
					exchnd_sig_names[sig],
					current->pid,
					el);
			kfree(el);
			continue;
		}
		if (el->state == EXCHND_SIGNAL_WAITING) {
			el->state = EXCHND_SIGNAL_RUNNING;
			wake_up(&el->wq);
		}
		/* We are done with first entry, there may be others
		 * that will be handled during next run.
		 */
		break;
	}
	spin_unlock_irqrestore(&signal_list_lck, flags);

	kfree(head);
}

/* Task pointer to original task_add function */
static int (*_task_work_add)(struct task_struct *task,
			     struct callback_head *work, bool notify);

/**
 * exchnd_queue_task - Sends signal receiver task to work queue
 * @sig: number of the signal
 * @info: signal information
 * @t: task that shall receive the signal
 *
 * This function is called if task gets a signal. The functions puts the task
 * to the work queue. The callback routine exchnd_stop_task is then called in
 * the context of the signal receiver task.
 */
static void exchnd_queue_task(int sig, struct siginfo *info,
			      struct task_struct *t)
{
	/* Do not catch signals for exception handler daemon to avoid
	 * deadlocks, because the daemon will be stopped and can therefore
	 * not the exception !
	 */
	if (exchnd_get_pid() != t->group_leader->pid) {
		struct callback_head *stop_cb;
		struct exchnd_signal_list *new;
		struct exchnd_signal_list *el;
		unsigned long flags;
		int el_found = 0;

		/* Sanity check. */
		if (t->flags & PF_EXITING)
			goto exit;

		if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
			pr_info("Process %d receives 0x%X (%s).\n",
				t->pid,
				sig,
				exchnd_sig_names[sig]);

		new = kzalloc(sizeof(*new), GFP_ATOMIC);

		if (!new)
			goto exit;

		new->task = (unsigned long)t;
		new->sig = sig;
		init_waitqueue_head(&new->wq);
		new->state = EXCHND_SIGNAL_QUEUED;

		stop_cb = kzalloc(sizeof(*stop_cb), GFP_ATOMIC);
		if (!stop_cb) {
			kfree(new);
			goto exit;
		}

		spin_lock_irqsave(&signal_list_lck, flags);

		/* Don't add a new element if there is one queued.
		 * Modules collectors to be executed will be merged.
		 */
		list_for_each_entry(el, &signal_list, list) {
			if (el->task != (unsigned long)current)
				continue;
			if (el->state != EXCHND_SIGNAL_QUEUED)
				continue;

			if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
				pr_info("%s info on queued list for %d(%p).\n",
					exchnd_sig_names[sig],
					t->pid,
					el);

			el->found_sigs |= (1 << sig);
			el_found = 1;
			break;
		}

		if (!el_found) {
			if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
				pr_info("Saving %s info on list for %d(%p).\n",
					exchnd_sig_names[sig],
					t->pid,
					new);
			list_add_tail(&new->list, &signal_list);
		} else {
			kfree(stop_cb);
			kfree(new);
			spin_unlock_irqrestore(&signal_list_lck, flags);
			goto exit;
		}

		spin_unlock_irqrestore(&signal_list_lck, flags);

		init_task_work(stop_cb, exchnd_stop_task);

		t->jobctl = (t->jobctl & JOBCTL_STOP_SIGMASK) | sig;
		_task_work_add(t, stop_cb, true);
	} else {
		if (exchnd_debug(EXCHND_DEBUG_DAEMON | EXCHND_DEBUG_SIGNAL))
			pr_info("Process %d receiving 0x%X (%s) ignored.\n",
				t->pid,
				sig,
				exchnd_sig_names[sig]);
	}

exit:
	return;
}

static int (*exchnd_next_signal)(struct sigpending *pending, sigset_t *mask);

/**
 * exchnd_get_signal_to_deliver - JProbe callback for get_signal_to_deliver
 * @sig: number of the signal
 * @info: signal information
 * @p: task that shall receive the signal
 *
 * This function is called by the JProbe set to group_send_sig_info. It calls
 * the generic function for handling signal triggers.
 */
static void exchnd_get_signal_to_deliver(struct siginfo *info,
					 struct k_sigaction *return_ka,
					 struct pt_regs *regs,
					 void *cookie)
{
	struct task_struct *p = current;
	struct sighand_struct *sighand = current->sighand;
	int sig = 0;
	unsigned long flags;

	spin_lock_irqsave(&sighand->siglock, flags);
	sig = exchnd_next_signal(&p->pending, &p->blocked);

	if (sig && (sig < SIGRTMIN))
		exchnd_queue_task(sig, info, p);

	/* Look for the original signal in case of pending SIGKILL */
	if (sig == SIGKILL) {
		sigdelset(&p->pending.signal, SIGKILL);
		sig = exchnd_next_signal(&p->pending, &p->blocked);

		if ((sig > 0) && (sig < SIGRTMIN))
			exchnd_queue_task(sig, info, p);
		sigaddset(&p->pending.signal, SIGKILL);
	}

	/* Only get shared signal from group leader to avoid duplicates. */
	sig = 0;
	if (p->pid == p->group_leader->pid)
		sig = exchnd_next_signal(&p->signal->shared_pending,
					 &p->blocked);

	/* No signal to be delivered or RT signal. */
	if (sig && (sig < SIGRTMIN))
		exchnd_queue_task(sig, info, p);

	spin_unlock_irqrestore(&sighand->siglock, flags);
	jprobe_return();
}

static void exchnd_get_signal(struct ksignal *ksig)
{
	return exchnd_get_signal_to_deliver(&ksig->info, NULL, NULL, NULL);
}

/* jprobe set to send signal */
static struct jprobe jp_sig = {
	.entry = exchnd_get_signal
};

static struct jprobe *jps_sig[1] = {&jp_sig};

/**
 * exchnd_signal_init - Initializes out of process signal triggers
 * @trigger: a pointer to the process signal trigger structure
 *
 * This function will will register JProbe with the "get_signal_to_deliver"
 * kernel function and retrieves the address of the "task_work_add" and
 * "next_signal" functions to allow call chaining in case of triggering.
 *
 * Return: Structure contain the JProbe structures
 */
static void *exchnd_signal_init(struct exchnd_trigger *trigger)
{
	void *jp_addr1 = (void *)exchnd_get_symbol(get_signal);
	int ret;

	trigger->opaque = NULL;

	_task_work_add = (void *)exchnd_get_symbol(task_work_add);
	exchnd_wake_up_state = (void *)exchnd_get_symbol(wake_up_state);
	exchnd_next_signal = (void *)exchnd_get_symbol(next_signal);

	if (jp_addr1 && _task_work_add &&
	    exchnd_wake_up_state && exchnd_next_signal) {
		jp_sig.kp.addr = (kprobe_opcode_t *)jp_addr1;
		ret = register_jprobes(jps_sig, ARRAY_SIZE(jps_sig));
		if (ret != 0) {
			pr_err("Initializing trigger process signal failed\n");
			pr_err("Register_jprobes failed with %d\n", ret);
		} else {
			trigger->opaque = jps_sig;
		}
	}

	if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
		pr_info("Signal handler initialized.\n");

	return trigger->opaque;
}

/**
 * exchnd_signal_deinit - Deinitializes process signal triggers
 * @trigger: a pointer to the process signal trigger structure
 *
 * This function will deinitialize the process signal triggers by unregistering
 * the JProbes.
 */
static void exchnd_signal_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque) {
		struct jprobe **sigs = trigger->opaque;

		unregister_jprobes(sigs, ARRAY_SIZE(jps_sig));
	}

	if (exchnd_debug(EXCHND_DEBUG_SIGNAL))
		pr_info("Signal handler deinitialized.\n");
}

/*
 * exception trigger PROCESS_EXIT
 *
 * The exception hooks on a probe on do_exit to trigger when a process ends.
 * Aim is to temporarily modify the call stack, adding a function that will
 * stop the task we want to watch.
 *
 * Done in exchnd_profiling.c or arch specific files.
 */

int exchnd_handle_exit(struct task_struct *task)
{
	int is_filtered = 0;
	struct exception_info *info = NULL;
	enum exchnd_modules (*modules)[EHM_LAST_ELEMENT] = NULL;
	struct exchnd_wd_worker *work;

	/* No-one is there to handle this event */
	/* TODO: Mechanism to handle this event internally anyway ? */
	spin_lock(&rb_get_read_wait_queue()->lock);
	/* waitqueue_active checked under wq lock. */
	if (!waitqueue_active(rb_get_read_wait_queue())) {
		exchnd_unset_pid();
		spin_unlock(&rb_get_read_wait_queue()->lock);
		goto exit;
	}
	spin_unlock(&rb_get_read_wait_queue()->lock);

	/* Ignore exchndd.
	 * We don't want to stop it !
	 */
	if (exchnd_get_pid() == task->group_leader->pid) {
		if ((exchnd_debug(EXCHND_DEBUG_PEXIT | EXCHND_DEBUG_DAEMON)) &&
		    (task->group_leader->pid == task->pid))
			pr_info("%s: Daemon (%d) exiting.\n",
				__func__,
				task->pid);
		goto exit;
	}

	is_filtered =  exchnd_is_filtered(task);
	if (exchnd_debug(EXCHND_DEBUG_PEXIT | EXCHND_DEBUG_FILTER))
		pr_info("%s: Filter status for process %d: %d.\n",
			__func__,
			task->pid,
			is_filtered);

	/* If process is not in white list, use default. */
	if (!is_filtered)
		modules = exchnd_pexit_default;
	else
		modules = exchnd_trigger_list[EHT_PROCESS_EXIT].modules;

	/* Early exit ...*/
	if ((*modules)[0] == EHM_NONE) {
		if (exchnd_debug(EXCHND_DEBUG_PEXIT))
			pr_info("%s: No modules for exit of %d.\n",
				__func__,
				task->pid);
		goto exit;
	}

	info = eq_get_info(EHT_PROCESS_EXIT);
	info->modules = modules;

	preempt_disable();
	if (exchnd_debug(EXCHND_DEBUG_PEXIT))
		pr_info("%s: Preparing to stop task on exit.\n", __func__);

	set_task_state(task, TASK_TRACED);

	/* Add JOBCTL_STOP_SIGMASK so that ptrace kick this task. */
	task->jobctl |= JOBCTL_STOP_SIGMASK;

	if (exchnd_debug(EXCHND_DEBUG_PEXIT))
		pr_info("%s: Starting handling exit.\n", __func__);

	info->ready = 1;
	eq_start_handling(EHT_PROCESS_EXIT);

	work = exchnd_watch_pid(task->pid);
	if (!work)
		pr_warn("Unable to watch pid: %d\n", task->pid);

	preempt_enable();
	schedule();

	if (exchnd_debug(EXCHND_DEBUG_PEXIT))
		pr_info("%s: Back from schedule.\n", __func__);

	/* Back from schedule, disable watchdog for this thread. */
	exchnd_unwatch(work);

exit:
	if (test_tsk_thread_flag(task, TIF_MEMDIE))
		exchnd_oom_end();

	return NOTIFY_OK;
}

/*
 * exception trigger RESTART
 *
 * The exception hooks to the restart of the kernel
 */

/**
 * exchnd_restart - Function notified by the kernel reboot notifier
 * @self: a notification structure of kernel reboot notifier
 * @val: type of reboot
 * @data: additional information of the reboot
 *
 * This callback is registered for the kernel reboot notifier and is triggered
 * when the system will be restarted.
 *
 * Return: Result of the notification resp. a request to the chain processing
 */
static int exchnd_restart(struct notifier_block *self,
			  unsigned long val,
			  void *data)
{
	/* Just react on a kernel restart */
	/* As the unregister_reboot_notifier function is not safe to be called
	 * while executing exchnd_sys_restart_execute, we need to rely on other
	 * ways in order to ignore incoming restart as far as it's generated
	 * by the exchnd itself. Thus, if a restart is already ongoing, we can
	 * ignore later exchnd_sys_restart_execute calls.
	 */
	if ((val == SYS_RESTART) && !exchnd_test_and_set_restart()) {
		struct exception_info *info = NULL;

		if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
			pr_info("SYS_RESTART triggered.\n");

		info = eq_get_info(EHT_RESTART);
		info->ready = 1;
		eq_start_handling(EHT_RESTART);

		/* To make sure that daemon completes the current
		 *  signal handling before system restart.
		 *  Shall not be done in IRQ context as it sleeps.
		 */
		if (!in_interrupt())
			eq_deinit();
	}

	return NOTIFY_OK;
}

static struct notifier_block exchnd_restart_notifier = {
	.notifier_call = exchnd_restart,
};

/**
 * exchnd_restart_init - Initializes restart trigger
 * @trigger: a pointer to the restart trigger structure
 *
 * This function will initialize the restart trigger by registering a reboot
 * notifier.
 *
 * Return: Notifier structure
 */
static void *exchnd_restart_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	if (register_reboot_notifier(&exchnd_restart_notifier) == 0)
		trigger->opaque = &exchnd_restart_notifier;

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Trigger restart initialized.\n");

	return trigger->opaque;
}

static void exchnd_restart_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		unregister_reboot_notifier(trigger->opaque);

	if (exchnd_debug(EXCHND_DEBUG_TRIGGER))
		pr_info("Trigger restart deinitialized.\n");
}

/**
 * __exchnd_external - Generic external function
 *
 * This function is called by exchnd_on_demand and exchnd_external functions.
 * It can be called in interrupt context and is aimed to create an event, either
 * from kernel request or user land request.
 *
 * @trigger: the exchnd trigger related, ON_DEMAND or EXTERNAL
 * @pid: the pid of the task to collect information of
 * @modules: the list of collector to execute. Default list is used if this
 *           parameter is NULL.
 * @msg: the message to print in the header if the default is not wanted.
 *
 * @return: 0 on success, error otherwise.
 */
static int __exchnd_external(enum exchnd_triggers trigger,
			     pid_t pid,
			     enum exchnd_modules (*modules)[EHM_LAST_ELEMENT],
			     char msg[256])
{
	struct exception_info *info = NULL;
	struct task_struct *task = NULL;
	int mod_type = EXCHND_MOD_STAT;
	/* Attach built-in modules by default */
	enum exchnd_modules (*mods)[EHM_LAST_ELEMENT] =
		exchnd_trigger_list[trigger].modules;

	rcu_read_lock();
	/* find_task_by_vpid can't be used as not exported to modules */
	task = pid_task(find_vpid(pid), PIDTYPE_PID);
	rcu_read_unlock();
	if (!task)
		return -EINVAL;

	/* Nothing to to ? */
	if ((modules && (*modules)[0] == EHM_NONE) ||
	    (!modules && ((*mods)[0] == EHM_NONE)))
		return 0;

	if (modules) {
		mods = kzalloc(sizeof(*mods), GFP_ATOMIC);

		if (!mods)
			return -ENOMEM;

		mod_type = EXCHND_MOD_DYN;
		memcpy(mods, modules, sizeof(*mods));
	}

	info = eq_get_info(trigger);
	info->ready = 1;
	info->task = task;
	eq_add_modules(info, mods, mod_type);
	if (msg)
		strncpy(info->msg, msg, 256);
	/* Ensure message is null terminated */
	info->msg[255] = '\0';

	eq_start_handling(trigger);

	return 0;
}

/**
 * exchnd_on_demand - Event generated on demand from user-land
 * @param: structure including all the needed parameters, pid, modules list,
 *         message.
 *
 * @return: 0 on success, error otherwise.
 */
int exchnd_on_demand(struct exchnd_on_demand *param)
{
	enum exchnd_modules (*mods)[EHM_LAST_ELEMENT] =
		exchnd_trigger_list[EHT_ON_DEMAND].modules;

	if (param->modules[0] != EHM_NONE)
		mods = &param->modules;

	return __exchnd_external(EHT_ON_DEMAND, param->pid, mods, param->msg);
}

/**
 * exchnd_external - Event generated on demand from kernel space
 *
 * This function can be called from interrupt context to trigger the collect of
 * information for a specific process. The collection is done out of the IRQ
 * context.
 *
 * @pid: the pid of the task to collect information of
 * @modules: the list of collector to execute. Default list is used if this
 *           parameter is NULL.
 * @msg: the message to print in the header if the default is not wanted.
 *
 * @return: 0 on success, error otherwise.
 */
int exchnd_external(pid_t pid,
		    enum exchnd_modules (*modules)[EHM_LAST_ELEMENT],
		    char msg[256])
{
	return __exchnd_external(EHT_EXTERNAL, pid, modules, msg);
}
EXPORT_SYMBOL_GPL(exchnd_external);

/**
 * exchnd_module_attach - attach module list to a specific pointer
 *
 * Simple helper.
 * @trigger: trigger's pointer module list should be attached to
 * @modules: modules list pointer
 */
void exchnd_module_attach(enum exchnd_modules (**dest)[EHM_LAST_ELEMENT],
			  enum exchnd_modules (*src)[EHM_LAST_ELEMENT])
{
	*dest = src;
}

enum exchnd_modules
trigger_conf[EC_LAST_ELEMENT][EHT_LAST_ELEMENT][EHM_LAST_ELEMENT] = {
	[EC_NO_MODULES] = { [EHT_NONE ... EHT_RESTART] = { EHM_NONE } },

	[EC_SAFETY] = {
		[EHT_NONE] = { EHM_NONE },

		[EHT_EXTERNAL] = { EHM_NONE },

		[EHT_HALT] = { EHM_HIST_SYSCALLS, EHM_MEMORY_USAGE,
			EHM_NONE },

		[EHT_KERNEL_OOPS] = { EHM_SYS_RESTART, EHM_NONE },

		[EHT_KERNEL_PANIC] = { EHM_SYS_RESTART, EHM_NONE },

		[EHT_ON_DEMAND] = { EHM_NONE },

		[EHT_OOM] = { EHM_CGROUPS, EHM_MEMORY_USAGE, EHM_NONE },

		[EHT_POWER_OFF] = { EHM_NONE },

		[EHT_PROCESS_EXIT] = { EHM_NONE },

		[EHT_PROCESS_SIGNAL] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_MEMORY_MAP, EHM_MEMORY_DUMP,
			EHM_FAULT_ADDRESS, EHM_BACKTRACE,
			EHM_THREAD_LIST, EHM_NONE  },

		[EHT_RESTART] = { EHM_NONE }
	},

	[EC_DEBUG] = {
		[EHT_NONE] = { EHM_NONE },

		[EHT_EXTERNAL] = { EHM_NONE },

		[EHT_HALT] = { EHM_HIST_SYSCALLS, EHM_HIST_TASKSWITCHES,
			EHM_NONE },

		[EHT_KERNEL_OOPS] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_FAULT_ADDRESS,
			EHM_MEMORY_DUMP, EHM_SYS_RESTART,
			EHM_NONE },

		[EHT_KERNEL_PANIC] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_FAULT_ADDRESS,
			EHM_MEMORY_DUMP, EHM_SYS_RESTART,
			EHM_NONE },

		[EHT_ON_DEMAND] = { EHM_NONE },

		[EHT_OOM] = { EHM_CGROUPS, EHM_MEMORY_USAGE, EHM_THREAD_LIST,
			EHM_BACKTRACE_ALL_THREADS, EHM_PROCESS_LIST,
			EHM_STACK_DUMP, EHM_NONE },

		[EHT_POWER_OFF] = { EHM_HIST_SYSCALLS, EHM_NONE },

		[EHT_PROCESS_EXIT] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_MEMORY_MAP, EHM_MEMORY_DUMP,
			EHM_FAULT_ADDRESS, EHM_BACKTRACE,
			EHM_THREAD_LIST, EHM_NONE },

		[EHT_PROCESS_SIGNAL] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_MEMORY_MAP, EHM_MEMORY_DUMP,
			EHM_FAULT_ADDRESS, EHM_BACKTRACE,
			EHM_THREAD_LIST, EHM_NONE  },

		[EHT_RESTART] = { EHM_HIST_SYSCALLS, EHM_HIST_TASKSWITCHES,
			EHM_NONE }
	},

	[EC_VERBOSE] = {
		[EHT_NONE] = { EHM_NONE },

		[EHT_EXTERNAL] = { EHM_NONE },

		[EHT_HALT] = { EHM_FS_STATE, EHM_HIST_SYSCALLS,
			EHM_HIST_TASKSWITCHES, EHM_MEMORY_USAGE,
			EHM_NONE },

		[EHT_KERNEL_OOPS] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_HIST_SYSCALLS,
			EHM_HIST_TASKSWITCHES, EHM_SCHED_INF,
			EHM_FAULT_ADDRESS, EHM_MEMORY_USAGE,
			EHM_MEMORY_DUMP, EHM_SYS_RESTART,
			EHM_NONE },

		[EHT_KERNEL_PANIC] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_HIST_SYSCALLS,
			EHM_HIST_TASKSWITCHES, EHM_SCHED_INF,
			EHM_FAULT_ADDRESS, EHM_MEMORY_USAGE,
			EHM_MEMORY_DUMP, EHM_SYS_RESTART,
			EHM_NONE },

		[EHT_ON_DEMAND] = { EHM_PROCESSOR_REGISTERS, EHM_BACKTRACE,
			EHM_STACK_DUMP, EHM_NONE },

		[EHT_OOM] = { EHM_CGROUPS, EHM_MEMORY_USAGE, EHM_THREAD_LIST,
			EHM_BACKTRACE_ALL_THREADS, EHM_PROCESS_LIST,
			EHM_STACK_DUMP, EHM_NONE },

		[EHT_POWER_OFF] = { EHM_FS_STATE, EHM_HIST_SYSCALLS,
			EHM_HIST_TASKSWITCHES, EHM_MEMORY_USAGE,
			EHM_NONE },

		[EHT_PROCESS_EXIT] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_MEMORY_MAP, EHM_MEMORY_DUMP,
			EHM_FAULT_ADDRESS, EHM_BACKTRACE,
			EHM_THREAD_LIST, EHM_NONE },

		[EHT_PROCESS_SIGNAL] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_MEMORY_MAP, EHM_MEMORY_DUMP,
			EHM_FAULT_ADDRESS, EHM_BACKTRACE,
			EHM_THREAD_LIST, EHM_NONE  },

		[EHT_RESTART] = { EHM_FS_STATE, EHM_HIST_SYSCALLS,
			EHM_HIST_TASKSWITCHES, EHM_MEMORY_USAGE,
			EHM_NONE }
	},

	/* Actual default */
	[EC_RBCM] = {
		[EHT_NONE] = { EHM_NONE },

		[EHT_EXTERNAL] = { EHM_NONE },

		[EHT_HALT] = { EHM_NONE },

		[EHT_KERNEL_OOPS] = { EHM_NONE },

		[EHT_KERNEL_PANIC] = { EHM_NONE },

		[EHT_ON_DEMAND] = { EHM_NONE },

		[EHT_OOM] = { EHM_NONE },

		[EHT_POWER_OFF] = { EHM_NONE },

		[EHT_PROCESS_EXIT] = { EHM_NONE },

		[EHT_PROCESS_SIGNAL] = { EHM_NONE  },

		[EHT_RESTART] = { EHM_NONE }
	},

	[EC_DN_NORESTART] = {
		[EHT_NONE] = { EHM_NONE },

		[EHT_EXTERNAL] = { EHM_NONE },

		[EHT_HALT] = { EHM_NONE },

		[EHT_KERNEL_OOPS] = { EHM_NONE },

		[EHT_KERNEL_PANIC] = { EHM_NONE },

		[EHT_ON_DEMAND] = { EHM_NONE },

		[EHT_OOM] = { EHM_NONE },

		[EHT_POWER_OFF] = { EHM_NONE },

		[EHT_PROCESS_EXIT] = { EHM_NONE },

		[EHT_PROCESS_SIGNAL] = { EHM_NONE  },

		[EHT_RESTART] = { EHM_NONE }
	},

	[EC_DN] = {
		[EHT_NONE] = { EHM_NONE },

		[EHT_EXTERNAL] = { EHM_NONE },

		[EHT_HALT] = { EHM_NONE },

		[EHT_KERNEL_OOPS] = { EHM_SYS_RESTART, EHM_NONE },

		[EHT_KERNEL_PANIC] = { EHM_SYS_RESTART, EHM_NONE },

		[EHT_ON_DEMAND] = { EHM_NONE },

		[EHT_OOM] = { EHM_SYS_RESTART, EHM_NONE },

		[EHT_POWER_OFF] = { EHM_NONE },

		[EHT_PROCESS_EXIT] = { EHM_NONE },

		[EHT_PROCESS_SIGNAL] = { EHM_NONE  },

		[EHT_RESTART] = { EHM_NONE }
	}
};

/* exchnd_trigger - list of exception handler modules per trigger */
struct exchnd_trigger exchnd_trigger_list[EHT_LAST_ELEMENT] = {
	[EHT_NONE] = {
		.fatality = FT_NON_FATAL,
		.init = NULL },

	[EHT_EXTERNAL] = {
		.fatality = FT_NON_FATAL,
		.init = NULL },

	[EHT_HALT] = {
		.fatality = FT_FATAL,
		.init = exchnd_halt_init,
		.deinit = exchnd_halt_deinit },

	[EHT_KERNEL_OOPS] = {
		.fatality = FT_FATAL,
		.init = exchnd_kernel_oops_init,
		.deinit = exchnd_kernel_oops_deinit },

	[EHT_KERNEL_PANIC] = {
		.fatality = FT_FATAL,
		.init = exchnd_kernel_panic_init,
		.deinit = exchnd_kernel_panic_deinit },

	[EHT_ON_DEMAND] = {
		.fatality = FT_NON_FATAL,
		.init = NULL },

	[EHT_OOM] = {
		.fatality = FT_NON_FATAL,
		.init = exchnd_oom_init,
		.deinit = exchnd_oom_deinit },

	[EHT_POWER_OFF] = {
		.fatality = FT_FATAL,
		.init = exchnd_power_off_init,
		.deinit = exchnd_power_off_deinit },

	[EHT_PROCESS_SIGNAL] = {
		.fatality = FT_NON_FATAL,
		.init = exchnd_signal_init,
		.deinit = exchnd_signal_deinit },

	[EHT_PROCESS_EXIT] = {
		.fatality = FT_NON_FATAL,
		.init = exchnd_exit_init,
		.deinit = exchnd_exit_deinit },
	[EHT_RESTART] = {
		.fatality = FT_FATAL,
		.init = exchnd_restart_init,
		.deinit = exchnd_restart_deinit }
};

enum exchnd_modules (*exchnd_signal_list[SIGRTMIN])[EHM_LAST_ELEMENT];

enum exchnd_modules
signal_conf[ESC_LAST_ELEMENT][SIGRTMIN][EHM_LAST_ELEMENT] = {
	[ESC_NO_MODULES] = {
		[1 ... (SIGUSR2 - 1)] = { EHM_NONE },
		[SIGUSR2] = { EHM_THREAD_LIST, EHM_BACKTRACE_ALL_THREADS },
		[(SIGUSR2 + 1) ... SIGSYS] = { EHM_NONE }
	},

	[ESC_SAFETY] = {
		[1 ... (SIGUSR2 - 1)] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_BACKTRACE,
			EHM_NONE },
		[SIGUSR2] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_THREAD_LIST,
			EHM_BACKTRACE_ALL_THREADS,
			EHM_NONE },
		[(SIGUSR2 + 1) ... SIGSYS] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP,
			EHM_BACKTRACE,
			EHM_NONE }
	},

	[ESC_DEBUG] = {
		[1 ... (SIGUSR2 - 1)] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_BACKTRACE,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_NONE },
		[SIGUSR2] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP,  EHM_THREAD_LIST,
			EHM_BACKTRACE_ALL_THREADS,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_NONE },
		[(SIGUSR2 + 1) ... SIGSYS] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP,
			EHM_BACKTRACE,
			EHM_FAULT_ADDRESS,
			EHM_MEMORY_DUMP,
			EHM_NONE }
	},

	[ESC_VERBOSE] = {
		[1 ... SIGSYS] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_THREAD_LIST,
			EHM_BACKTRACE_ALL_THREADS,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_NONE }
	},

	[ESC_ALL_MODULES] = {
		[1 ... SIGSYS] = {
			EHM_BACKTRACE,
			EHM_BACKTRACE_ALL_THREADS,
			EHM_CGROUPS,
			EHM_CORE_DUMP,
			EHM_CPU_USAGE,
			EHM_FAULT_ADDRESS,
			EHM_FS_STATE,
			EHM_HIST_SYSCALLS,
			EHM_HIST_TASKSWITCHES,
			EHM_LRU_MEM_PAGES,
			EHM_MEMORY_DUMP,
			EHM_MEMORY_MAP,
			EHM_MEMORY_USAGE,
			EHM_PROCESSOR_REGISTERS,
			EHM_PROCESS_LIST,
			EHM_SCHED_INF,
			EHM_STACK_DUMP,
			EHM_SYSTEM_INFO,
			EHM_THREAD_LIST,
			EHM_ACTION_START,
			EHM_SYS_RESTART,
			EHM_APPLICATION_SPECIFIC1,
			EHM_APPLICATION_SPECIFIC2,
			EHM_APPLICATION_SPECIFIC3,
			EHM_APPLICATION_SPECIFIC4,
			EHM_APPLICATION_SPECIFIC5,
			EHM_NONE
		}
	},

	/* Actual default */
	[ESC_RBCM_NORESTART] = {
		[SIGHUP ... SIGQUIT] = { EHM_NONE },
		[SIGILL] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP, EHM_NONE },
		[SIGTRAP] = { EHM_NONE },
		[SIGABRT ... SIGFPE] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_BACKTRACE,
			EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_NONE },
		[SIGKILL ... SIGUSR1] = { EHM_NONE },
		[SIGSEGV] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP, EHM_NONE },
		[SIGUSR2] = { EHM_BACKTRACE_ALL_THREADS },
		[SIGPIPE ... SIGTERM] = { EHM_NONE },
		[SIGSTKFLT] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP, EHM_NONE },
		[SIGXCPU ... SIGXFSZ] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_BACKTRACE,
			EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_NONE },
		[SIGVTALRM ... (SIGPWR - 1)] = { EHM_NONE },
		[SIGPWR] = { EHM_SYS_RESTART, EHM_NONE },
		[SIGSYS] = { EHM_NONE }
	},

	[ESC_RBCM_RESTART] = {
		[SIGHUP ... SIGQUIT] = { EHM_NONE },
		[SIGILL] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP, EHM_NONE },
		[SIGTRAP] = { EHM_NONE },
		[SIGABRT ... SIGFPE] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_BACKTRACE,
			EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_SYS_RESTART, EHM_NONE },
		[SIGKILL ... SIGUSR1] = { EHM_NONE },
		[SIGSEGV] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_SYS_RESTART, EHM_NONE },
		[SIGUSR2] = { EHM_BACKTRACE_ALL_THREADS },
		[SIGPIPE ... SIGTERM] = { EHM_NONE },
		[SIGSTKFLT] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_SYS_RESTART, EHM_NONE },
		[SIGXCPU ... SIGXFSZ] = { EHM_PROCESSOR_REGISTERS,
			EHM_STACK_DUMP, EHM_BACKTRACE,
			EHM_MEMORY_MAP, EHM_THREAD_LIST,
			EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			EHM_SYS_RESTART, EHM_NONE },
		[SIGVTALRM ... (SIGPWR - 1)] = { EHM_NONE },
		[SIGPWR] = { EHM_SYS_RESTART, EHM_NONE },
		[SIGSYS] = { EHM_NONE }
	},

	[ESC_DN_NORESTART] = {
		[SIGHUP ... SIGQUIT] = { EHM_NONE },
		[SIGILL] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			      EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			      EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP, EHM_NONE },
		[SIGTRAP] = { EHM_NONE },
		[SIGABRT ... SIGFPE] = { EHM_PROCESSOR_REGISTERS,
					 EHM_STACK_DUMP, EHM_BACKTRACE,
					 EHM_MEMORY_MAP, EHM_THREAD_LIST,
					 EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
					 EHM_NONE },
		[SIGKILL ... SIGUSR1] = { EHM_NONE },
		[SIGSEGV] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			      EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			      EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP, EHM_NONE },
		[SIGUSR2 ... SIGSTKFLT] = { EHM_NONE },
		[SIGXCPU ... SIGXFSZ] = { EHM_PROCESSOR_REGISTERS,
					  EHM_STACK_DUMP, EHM_BACKTRACE,
					  EHM_MEMORY_MAP, EHM_THREAD_LIST,
					  EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
					  EHM_NONE },
		[SIGVTALRM ... SIGPWR] = { EHM_NONE },
		[SIGSYS] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			      EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			      EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP, EHM_NONE }
	},

	[ESC_DN_RESTART] = {
		[SIGHUP ... SIGQUIT] = { EHM_NONE },
		[SIGILL] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			      EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			      EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			      EHM_SYS_RESTART, EHM_NONE },
		[SIGTRAP] = { EHM_NONE },
		[SIGABRT ... SIGFPE] = { EHM_PROCESSOR_REGISTERS,
					 EHM_STACK_DUMP, EHM_BACKTRACE,
					 EHM_MEMORY_MAP, EHM_THREAD_LIST,
					 EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
					 EHM_SYS_RESTART, EHM_NONE },
		[SIGKILL ... SIGUSR1] = { EHM_NONE },
		[SIGSEGV] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			      EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			      EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			      EHM_SYS_RESTART, EHM_NONE },
		[SIGUSR2 ... SIGSTKFLT] = { EHM_NONE },
		[SIGXCPU ... SIGXFSZ] = { EHM_PROCESSOR_REGISTERS,
					  EHM_STACK_DUMP, EHM_BACKTRACE,
					  EHM_MEMORY_MAP, EHM_THREAD_LIST,
					  EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
					  EHM_SYS_RESTART, EHM_NONE },
		[SIGVTALRM ... SIGPWR] = { EHM_NONE },
		[SIGSYS] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
			      EHM_BACKTRACE, EHM_MEMORY_MAP, EHM_THREAD_LIST,
			      EHM_FAULT_ADDRESS, EHM_MEMORY_DUMP,
			      EHM_SYS_RESTART, EHM_NONE }
	}
};

enum exchnd_modules (*exchnd_pexit_default)[EHM_LAST_ELEMENT];

enum exchnd_modules pexit_conf[EPEC_LAST_ELEMENT][EHM_LAST_ELEMENT] = {
	/* Actual default */
	[EPEC_NO_MODULES] = { EHM_NONE },

	[EPEC_SAFETY] = { EHM_PROCESSOR_REGISTERS, EHM_NONE },

	[EPEC_DEBUG] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
		EHM_BACKTRACE, EHM_NONE },

	[EPEC_VERBOSE] = { EHM_PROCESSOR_REGISTERS, EHM_STACK_DUMP,
		EHM_THREAD_LIST, EHM_BACKTRACE_ALL_THREADS,
		EHM_NONE },

	[EPEC_ALL_MODULES] = {
		EHM_BACKTRACE,
		EHM_BACKTRACE_ALL_THREADS,
		EHM_CGROUPS,
		EHM_CORE_DUMP,
		EHM_CPU_USAGE,
		EHM_FAULT_ADDRESS,
		EHM_FS_STATE,
		EHM_HIST_SYSCALLS,
		EHM_HIST_TASKSWITCHES,
		EHM_LRU_MEM_PAGES,
		EHM_MEMORY_DUMP,
		EHM_MEMORY_MAP,
		EHM_MEMORY_USAGE,
		EHM_PROCESSOR_REGISTERS,
		EHM_PROCESS_LIST,
		EHM_SCHED_INF,
		EHM_STACK_DUMP,
		EHM_SYSTEM_INFO,
		EHM_THREAD_LIST,
		EHM_ACTION_START,
		EHM_SYS_RESTART,
		EHM_APPLICATION_SPECIFIC1,
		EHM_APPLICATION_SPECIFIC2,
		EHM_APPLICATION_SPECIFIC3,
		EHM_APPLICATION_SPECIFIC4,
		EHM_APPLICATION_SPECIFIC5,
		EHM_NONE
	}
};
