/******************************************************************************
 *
 * dal_gate: Access controlled interface to Intel TXI via DAL/KDI
 *
 * Author: Philipp Hachtmann (phachtmann@de.adit-jv.com)
 *
 * Copyright (C) 2017 Advanced Driver Information Technology.
 *
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 ******************************************************************************
 *
 * This driver allows userspace applications to use the Intel KDI in kernel
 * interface to the DAL (Dynamic Application Loader).
 *
 * It enforces the convention to pass user and group IDs to the trusted
 * application during session creation. This enables the trusted application
 * code to take decisions based on the current Linux user's identity.
 *
 * The string parameter passed to the trusted application during session
 * creation is formed as follows:
 *
 *    sprintf(buf, "DAL GATE LOGIN 0x%08x 0x%08x", uid, gid)
 *
 *  A sample login string: "DAL GATE LOGIN 0x12345678 0x12345678"
 *  The string length is always 36 bytes. The trailing
 *  \0 character is *not* transferred to the TA.
 *
 ******************************************************************************/

/*
 *  This is a reaction to the -DDEBUG statically set by Intel in the
 *  Makefile in drivers/misc/mei/dal.
 */
#ifndef CONFIG_DAL_GATE_DEBUG
  #ifdef DEBUG
    #undef DEBUG
  #endif
#else
  #ifndef DEBUG
    #define DEBUG
  #endif
#endif

#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/slab.h>
#include <linux/dal_gate.h>
#include <linux/uaccess.h>
#include <linux/firmware.h>
#include <linux/dal.h>
#include <linux/fs.h>

#ifdef CONFIG_DAL_GATE_DEBUG

static int alloc_count;

#define x_kzalloc(...) ({void *p = kzalloc(__VA_ARGS__);		\
	    pr_debug("************* %04i ALLOC: %p (%i)\n",		\
		     __LINE__, p, alloc_count++); p; })

#define x_kfree(ptr) ({pr_debug("************* %04i  FREE: %p (%i)\n",	\
				__LINE__, ptr, --alloc_count);		\
	    kfree(ptr); })
#else

#define x_kzalloc(...) kzalloc(__VA_ARGS__)
#define x_kfree(ptr) kfree(ptr)

#endif /* CONFIG_DAL_GATE_DEBUG */

/******************************************************************************
 *
 * Constant definitions - don't change
 *
 */

#define TA_PACK_EXTENSION ".pack"
#define INIT_PARM_FMT "DAL GATE LOGIN 0x%08x 0x%08x"
#define INIT_PARM_LEN 36  /* final string length, trailing \0 not counted */

/******************************************************************************
 *
 * Local data types
 *
 */

/* data structure representing a cached TA pack file */
struct ta_pack {
	char *ta_id;
	size_t ta_id_len;
	void *data;
	size_t data_len;
};

/* data structure of the device */
struct dal_gate_common {
	/* rw semaphore to avoid concurrent modification of cached_pack */
	struct rw_semaphore cache_lock;
	/* Cached TA pack */
	struct ta_pack *cached_pack;
};

/* data structure associated with each open driver file. */
struct dal_gate_data {
	struct mutex ioctl_lock; /* avoid concurrent ioctls in one file */
	u64 sess_handle;     /* KDI session handle, set to 0 if unused */
};

/******************************************************************************
 *
 * Static data
 *
 */

static struct cdev cdev;   /* Represents the device to the Linux kernel    */
static dev_t devno;        /* Device number, formerly known as major:minor */
static struct dal_gate_common common_data; /* data common to all file handles */

static struct class *dal_gate_sysfs_class;   /* sysfs class     */
static struct device *dal_gate_sysfs_device; /* device in sysfs */

/*
 * Note: The config option only changes the default value of the option!
 */
#ifdef CONFIG_DAL_GATE_DONT_CACHE
static int ta_no_cache = 1;
#else
static int ta_no_cache;
#endif

/******************************************************************************
 *
 * Pack loading and unloading
 *
 */

static int load_ta_file(struct ta_pack *pack, char *ta_id, size_t ta_id_len)
{
	char *filename;
	int res;
	const struct firmware *fw = NULL;

	/*
	 * allocate memory to store the filename of the firmware
	 * ta_id + extension + trailing \0 (part of TA_PACK_EXTENSION)
	 */
	size_t filename_len = ta_id_len + sizeof(TA_PACK_EXTENSION);

	filename = x_kzalloc(filename_len,
			     GFP_KERNEL);
	if (!filename) {
		res = -ENOMEM;
		goto out;
	}

	memcpy(filename, ta_id, ta_id_len);
	memcpy(filename + ta_id_len, TA_PACK_EXTENSION,
	       sizeof(TA_PACK_EXTENSION));

	/* verify that TA ID has expected length (no intermediate \0) */
	if (strlen(filename) != (filename_len - 1)) {
		pr_err("Invalid TA ID\n");
		res = -EINVAL;
		goto out;
	}

	/* Load firmware */
	res = request_firmware(&fw, filename, dal_gate_sysfs_device);
	if (res) {
		pr_err("Could not load TA package: %s\n", filename);
		goto out;
	}
	pr_info("Loaded firmware package file: %s\n", filename);

	pack->data = x_kzalloc(fw->size, GFP_KERNEL);
	if (!pack->data) {
		pr_err("Could not allocated memory!\n");
		res = -ENOMEM;
		goto out_firmware;
	}

	memcpy(pack->data, fw->data, fw->size);
	pack->data_len = fw->size;

out_firmware:
	release_firmware(fw);
out:
	x_kfree(filename);
	return res;
}

static void pack_delete(struct ta_pack **pack)
{
	if (!pack) {
		pr_err("Null pointer in pack_delete!\n");
		return;
	}

	if (!*pack) {
		pr_debug("Pack already cleared\n");
		return;
	}

	if (!(*pack)->ta_id)
		pr_err("TA id is NULL!!\n");
	else
		x_kfree((*pack)->ta_id);

	if (!(*pack)->data)
		pr_err("Firmware pointer is NULL!!\n");
	else
		x_kfree((*pack)->data);

	x_kfree(*pack);
	*pack = NULL;
}

/******************************************************************************
 *
 * Dump function (DEBUG only)
 *
 */
#ifdef CONFIG_DAL_GATE_DEBUG

#define DUMP_BYTES_PER_LINE 16
static void dump_data(const void *d, int len, const char *caption)
{
	int i, line;
	int bytes_left;
	char buf[DUMP_BYTES_PER_LINE * 3 + 6];
	unsigned char *data = (unsigned char *)d;

	(void)caption;
	bytes_left = len;
	pr_debug("%s:\n", caption);
	for (line = 0;
	     line < (len + DUMP_BYTES_PER_LINE - 1) / DUMP_BYTES_PER_LINE;
	     line++) {
		snprintf(buf, sizeof(buf), "%04x:  ",
			 line * DUMP_BYTES_PER_LINE);

		for (i = 0; (i < DUMP_BYTES_PER_LINE) && bytes_left--; i++) {
			snprintf(buf + 3 * i + 6, 5, " %02x",
				 data[i + DUMP_BYTES_PER_LINE * line]);
		}
		pr_debug("%s\n", buf);
	}
}
#else
#define dump_data(...)
#endif

/******************************************************************************
 *
 * KDI error code translation
 *
 */

struct _kdi_err_msg {
	int err;
	const char *msg;
};

static struct _kdi_err_msg kdi_strings[] = {
	{DAL_KDI_SUCCESS, "Success"},
	{DAL_KDI_STATUS_INTERNAL_ERROR,       "internal error"},
	{DAL_KDI_STATUS_INVALID_PARAMS,       "invalid parameter(s)"},
	{DAL_KDI_STATUS_INVALID_HANDLE,       "invalid handle"},
	{DAL_KDI_STATUS_NOT_INITIALIZED,      "not initialized"},
	{DAL_KDI_STATUS_OUT_OF_MEMORY,        "out of memory"},
	{DAL_KDI_STATUS_BUFFER_TOO_SMALL,     "buffer too small"},
	{DAL_KDI_STATUS_OUT_OF_RESOURCE,      "out of resource"},
	{DAL_KDI_STATUS_MAX_SESSIONS_REACHED, "max number of sessions reached"},
	{DAL_KDI_STATUS_UNCAUGHT_EXCEPTION,   "uncaught exception"},
	{DAL_KDI_STATUS_WD_TIMEOUT,           "watchdog timeout"},
	{DAL_KDI_STATUS_APPLET_CRASHED,       "applet crashed"},
	{DAL_KDI_STATUS_TA_NOT_FOUND,         "ta not found"},
	{DAL_KDI_STATUS_TA_EXIST,             "ta exists"},
	{DAL_KDI_STATUS_INVALID_ACP,          "invalid ta package"}
};

static const char *kdi_errstr(int err)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(kdi_strings); i++)
		if (kdi_strings[i].err == err)
			return kdi_strings[i].msg;
	return "<unknown KDI error>";
}

static int transcode_kdi_retval(int err)
{
	switch (err) {
	case DAL_KDI_SUCCESS:
		return 0;
	case DAL_KDI_STATUS_INTERNAL_ERROR:
		return -EIO;

	case DAL_KDI_STATUS_INVALID_PARAMS:
	case DAL_KDI_STATUS_INVALID_HANDLE:
	case DAL_KDI_STATUS_BUFFER_TOO_SMALL:
		return -EINVAL;

	case DAL_KDI_STATUS_NOT_INITIALIZED:
		return -EHOSTDOWN;

	case DAL_KDI_STATUS_OUT_OF_MEMORY:
	case DAL_KDI_STATUS_OUT_OF_RESOURCE:
		return -ENOMEM;

	case DAL_KDI_STATUS_MAX_SESSIONS_REACHED:
		return -EBUSY;
	case DAL_KDI_STATUS_UNCAUGHT_EXCEPTION:
	case DAL_KDI_STATUS_WD_TIMEOUT:
	case DAL_KDI_STATUS_APPLET_CRASHED:
	case DAL_KDI_STATUS_TA_NOT_FOUND:
	case DAL_KDI_STATUS_TA_EXIST:
	case DAL_KDI_STATUS_INVALID_ACP:
		return -EIO;
	default:
		break;
	}
	return -EINVAL;
}

static int __create_session(u64 *sess_handle,
			    struct ta_pack *ta_data,
			    char *init_param,
			    size_t init_len)
{
	int res;

	res = dal_create_session(
		sess_handle,
		ta_data->ta_id,
		ta_data->data,
		ta_data->data_len,
		init_param,
		init_len);

	if (res) {
		pr_err("DAL create session failed: 0x%08x (KDI error string\"%s\")\n",
		       res, kdi_errstr(res));

		res = transcode_kdi_retval(res);
		*sess_handle = 0;
	}

	return res;
}

/*
 * function will return -EAGAIN if TA pack is not cached
 * or loading the cached pack failed as cached content is invalid
 */
static int __try_cached_pack_session(u64 *sess_handle,
				     char *ta_id,
				     size_t ta_id_len,
				     char *init_param,
				     size_t init_len)
{
	struct ta_pack *pack;
	int res = -EAGAIN;

	/* avoid that cache is changed while we access it */
	down_read(&common_data.cache_lock);

	if (!common_data.cached_pack)
		/* nothing in cache */
		goto unlock;

	pack = common_data.cached_pack;

	if ((ta_id_len != pack->ta_id_len) ||
	    (memcmp(ta_id, pack->ta_id, ta_id_len) != 0))
		/* different TA ID in cache */
		goto unlock;

	pr_debug("Load cached pack for TA ID %s\n",
		 pack->ta_id);

	res = dal_create_session(
		sess_handle,
		pack->ta_id,
		pack->data,
		pack->data_len,
		init_param,
		init_len);
	if (res) {
		*sess_handle = 0;

		if (res == DAL_KDI_STATUS_INVALID_ACP) {
			/*
			 * Cached pack file is invalid.
			 * Invalidate it to avoid that others
			 * will use it as well.
			 */

			up_read(&common_data.cache_lock);

			down_write(&common_data.cache_lock);

			/*
			 * Avoid to invalidate already updated
			 * cache content.
			 */
			if (pack == common_data.cached_pack) {
				pr_debug("Invalidate cached pack of TA ID %s\n",
					 pack->ta_id);
				pack_delete(&common_data.cached_pack);
			}

			up_write(&common_data.cache_lock);

			/* trigger loading pack from file-system */
			res = -EAGAIN;

			goto out;
		} else {
			pr_err("DAL create session (cached pack) failed: 0x%08x (KDI error string\"%s\")\n",
			       res, kdi_errstr(res));

			res = transcode_kdi_retval(res);
		}
	}

unlock:
	up_read(&common_data.cache_lock);
out:
	return res;
}

static int __open_session(u64 *sess_handle,
			  const char __user *u_ta_id,
			  size_t ta_id_len,
			  char *init_param,
			  size_t init_len,
			  u32 cache_flags)
{
	struct ta_pack *pack;
	char *ta_id;
	int res;

	ta_id = x_kzalloc(ta_id_len + 1, GFP_KERNEL);
	if (!ta_id) {
		res = -ENOMEM;
		goto out;
	}

	pr_debug("ta_id buffer allocated, %zu bytes at %p\n", ta_id_len, ta_id);
	res = copy_from_user(ta_id, u_ta_id, ta_id_len);
	if (res) {
		res = -EFAULT;
		goto out_free_id;
	}

	/*
	 * Add "secret" zero termination for API call.
	 * Not accounted for in ta_id_len
	 */
	ta_id[ta_id_len] = 0;

	if ((cache_flags & EXT_FLAG__USE_CACHE_MASK)
	    == EXT_FLAG__MAY_USE_CACHE) {
		res = __try_cached_pack_session(sess_handle,
						ta_id,
						ta_id_len,
						init_param,
						init_len);

		/*
		 * check if either successfully loaded cached packed
		 * or retry with pack loaded from file-system not reasonable
		 */
		if (res != -EAGAIN)
			goto out_free_id;
	}

	/*
	 * There is a small chance that two instances will load the
	 * very same pack file from file-system concurrently.
	 * We ignore this rare case.
	 *
	 * In that case the first cached pack data will be freed when
	 * storing the second pack data to cache.
	 */

	/* Create session with TA pack file loaded from file-system */
	pack = x_kzalloc(sizeof(*pack), GFP_KERNEL);
	if (!pack) {
		res = -ENOMEM;
		goto out_free_id;
	}

	res = load_ta_file(pack, ta_id, ta_id_len);
	if (res)
		goto out_free_pack;

	pack->ta_id     = ta_id;
	pack->ta_id_len = ta_id_len;

	res = __create_session(sess_handle,
			       pack,
			       init_param,
			       init_len);

	if ((!res) &&
	    ((cache_flags & EXT_FLAG__UPDATE_CACHE_MASK)
	     == EXT_FLAG__MAY_UPDATE_CACHE)
	   ) {
		/*
		 * We successfully created a session with pack file.
		 * Store to cache
		 */
		down_write(&common_data.cache_lock);

		pack_delete(&common_data.cached_pack);

		pr_debug("Update cached pack for TA ID %s\n",
			 ta_id);
		common_data.cached_pack = pack;

		up_write(&common_data.cache_lock);

		/*
		 * ta_id, pack or pack->data will be
		 * freed when clearing the cache.
		 */
		goto out;
	}

	x_kfree(pack->data);
out_free_pack:
	x_kfree(pack);
out_free_id:
	x_kfree(ta_id);
out:
	return res;
}

static int ioctl_open_session_common(struct dal_gate_sess_args *s_data,
				     u32 cache_flags,
				     struct dal_gate_data *data)
{
	int res;
	char init_param[INIT_PARM_LEN + 1] = {0};

	uid_t login_uid;
	gid_t login_gid;

	/* Login data */
	login_gid = s_data->my_gid;
	login_uid = current_euid().val;

	if (cache_flags & ~EXT_FLAGS_MASK) {
		pr_err("Invalid cache_flags\n");
		return -EINVAL;
	}

	if (ta_no_cache) {
		/* overwrite cache flags if cache is disabled */
		cache_flags &= ~(EXT_FLAG__USE_CACHE_MASK |
				 EXT_FLAG__UPDATE_CACHE_MASK);
		cache_flags |= EXT_FLAG__DONT_USE_CACHE |
			       EXT_FLAG__DONT_UPDATE_CACHE;
	}

	/* Check if we are really member of the group */
	if (!in_egroup_p(KGIDT_INIT(login_gid))) {
		pr_err("Access denied: User %u not in group %u!\n",
		       login_uid, login_gid);
		return -EACCES;
	}

	if (snprintf(init_param,
		     INIT_PARM_LEN + 1,
		     INIT_PARM_FMT,
		     login_uid,
		     login_gid)
	    > INIT_PARM_LEN) {
		pr_err("Creation of init_parm failed for user %u and group %u\n",
		       login_uid, login_gid);
		return -EINVAL;
	}

	res = __open_session(&data->sess_handle,
			     (const char __user *)s_data->ta_id,
			     s_data->ta_id_len,
			     init_param,
			     INIT_PARM_LEN,
			     cache_flags);

	return res;
}

static int ioctl_open_session(struct file *file,
			      unsigned long arg,
			      struct dal_gate_data *data)
{
	int res;
	struct dal_gate_sess_args s_data;

	pr_debug("ioctl: Opening session\n");
	res = copy_from_user(&s_data, (void __user *)arg, sizeof(s_data));
	if (res)
		return -EFAULT;

	res = ioctl_open_session_common(&s_data,
					EXT_FLAGS_DEFAULT,
					data);

	return res;
}

static int ioctl_open_session_ext(struct file *file,
				  unsigned long arg,
				  struct dal_gate_data *data)
{
	int res;
	struct dal_gate_sess_ext_args ext_s_data;

	pr_debug("ioctl: Opening session (extended)\n");
	res = copy_from_user(&ext_s_data,
			     (void __user *)arg,
			     sizeof(ext_s_data));
	if (res)
		return -EFAULT;

	res = ioctl_open_session_common(&ext_s_data.sess_args,
					ext_s_data.ext_flags,
					data);

	return res;
}

static int close_session(struct file *file,
			 struct dal_gate_data *data)
{
	int res = 0;

	if (data) {
		res = dal_close_session(data->sess_handle);
		if (res) {
			pr_err("DAL close session failed: 0x%08x (KDI error string\"%s\")\n",
			       res, kdi_errstr(res));

			res = transcode_kdi_retval(res);
		}

		data->sess_handle = 0;
	}
	return res;
}

static int ioctl_run_command(struct file *file,
			     unsigned long arg,
			     struct dal_gate_data *data)
{
	int res = 0;
	struct dal_gate_cmd cmd;

	void __user *u_cmd_p = (void __user *)arg;

	void *in_data = NULL;
	u8 *out_data = NULL;

	res = copy_from_user(&cmd, u_cmd_p, sizeof(struct dal_gate_cmd));
	if (res)
		return -EFAULT;

	pr_debug("ioctl: Running DAL command %i\n", cmd.cmd);

	if (cmd.in_data_len) {
		in_data = x_kzalloc(cmd.in_data_len, GFP_KERNEL);
		if (!in_data) {
			res = -ENOMEM;
			goto out;
		}
		res = copy_from_user(in_data, cmd.in_data, cmd.in_data_len);
		if (res) {
			res = -EFAULT;
			goto out;
		}
	}

	res = dal_send_and_receive(
		data->sess_handle,
		cmd.cmd,
		in_data,
		cmd.in_data_len,
		&out_data,
		&cmd.out_data_len,
		&cmd.response_code);
	if (res) {
		pr_err("DAL cmd failed: 0x%08x (KDI error string\"%s\")\n",
		       res, kdi_errstr(res));

		res = transcode_kdi_retval(res);
		goto out;
	}

	pr_debug("TA response code: %u\n", cmd.response_code);

	if (cmd.out_data_len) {
		pr_debug("Copying %zu bytes back to userspace\n",
			 cmd.out_data_len);
		dump_data(out_data, cmd.out_data_len, "out data");

		res = copy_to_user(cmd.out_data, out_data, cmd.out_data_len);
		kfree(out_data);
		if (res) {
			pr_err("copy_to_user failed!");
			res = -EFAULT;
			goto out;
		}
	}

	res = copy_to_user(u_cmd_p, &cmd, sizeof(struct dal_gate_cmd));
	if (res)
		res = -EFAULT;
out:
	if (in_data)
		x_kfree(in_data);

	pr_debug("%s leaving with code %i\n", __func__, res);
	return res;
}

/*
 *      ioctl()
 */
static long f_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int res = 0;
	struct dal_gate_data *data = file->private_data;

	/* avoid that ioctls are called concurrently in one file */
	res = mutex_lock_interruptible(&data->ioctl_lock);
	if (res)
		goto out;
	switch (cmd) {
	case DALG_CONNECT_TA:
		res = ioctl_open_session(file, arg, data);
		break;

	case DALG_CONNECT_TA_EXT:
		res = ioctl_open_session_ext(file, arg, data);
		break;

	case DALG_CLOSE_TA:
		res = close_session(file, data);
		break;

	case DALG_TA_CMD:
		res = ioctl_run_command(file, arg, data);
		break;

	default:
		pr_err("Unknown ioctl: 0x%08x\n", cmd);
		res = -ENOTTY;
		break;
	}

	mutex_unlock(&data->ioctl_lock);
out:
	return res;
}

static int f_open(struct inode *inode, struct file *file)
{
	struct dal_gate_data *data;

	data = x_kzalloc(sizeof(*data), GFP_KERNEL);
	if (!data) {
		pr_err("Could not allocate memory!\n");
		return -ENOMEM;
	}

	mutex_init(&data->ioctl_lock);

	file->private_data = data;

	pr_debug("File opened!\n");
	return 0;
}

static int f_release(struct inode *inode, struct file *file)
{
	struct dal_gate_data *data = file->private_data;

	/* Close a KDI session if there is one */
	if (data->sess_handle) {
		pr_debug("Automatically closing KDI session\n");
		dal_close_session(data->sess_handle);
		data->sess_handle = 0;
	}

	x_kfree(data);
	pr_debug("File closed.\n");
	return 0;
}

static const struct file_operations fops = {
	.owner = THIS_MODULE,
	.open = f_open,
	.release = f_release,
	.unlocked_ioctl = f_ioctl,
};

static int __init dal_gate_init(void)
{
	int res;

	pr_info("ADIT DAL gate: Loading\n");

	memset(&common_data, 0, sizeof(common_data));
	init_rwsem(&common_data.cache_lock);

	dal_gate_sysfs_class = class_create(THIS_MODULE, "dal_gate");
	if (IS_ERR(dal_gate_sysfs_class)) {
		pr_err("couldn't create class\n");
		res = PTR_ERR(dal_gate_sysfs_class);
		goto out;
	}

	res = alloc_chrdev_region(&devno, 0, 1, "dal_gate");
	if (res) {
		pr_err("alloc_chrdev_region returned %i\n", res);
		goto out_class;
	}

	cdev_init(&cdev, &fops);
	cdev.owner = THIS_MODULE;
	res = cdev_add(&cdev, devno, 1);
	if (res) {
		pr_err("cdev_add() returned %i\n", res);
		kobject_put(&cdev.kobj);
		goto out_region;
	}

	dal_gate_sysfs_device = device_create(dal_gate_sysfs_class, NULL,
					      devno, NULL, "dal_gate");
	if (IS_ERR(dal_gate_sysfs_device)) {
		pr_err("Could not create sysfs device\n");
		res = PTR_ERR(dal_gate_sysfs_device);
		goto out_cdev;
	}

	/* Success */
	pr_debug("Device registered\n");
	goto out;

out_cdev:
	cdev_del(&cdev);
out_region:
	unregister_chrdev_region(devno, 1);
out_class:
	class_destroy(dal_gate_sysfs_class);
out:
	return res;
}

static void __exit dal_gate_exit(void)
{
	pr_info("ADIT DAL gate: Unloading\n");
	pack_delete(&common_data.cached_pack);
	device_destroy(dal_gate_sysfs_class, devno);
	cdev_del(&cdev);
	unregister_chrdev_region(devno, 1);
	class_destroy(dal_gate_sysfs_class);
}

module_init(dal_gate_init);
module_exit(dal_gate_exit);
module_param(ta_no_cache, int, 0600);

MODULE_AUTHOR("Philipp Hachtmann <phachtmann@de.adit-jv.com>");
MODULE_DESCRIPTION("Discretionary access control for DAL applets");
MODULE_LICENSE("GPL v2");
