/***************************************************************************/ /*
BUSMONI kernel module program
FILE        : busmoni_drv.c
CREATED     : 2015.10.20 by RVC/TrungN
AUTHOR      : Renesas Electronics Corporation
DEVICE      : R-Car H3/M3W(R8A7795/R8A7796)
PLATFORM    : SALVATOR_X
HISTORY     : 
            2015.10.20
              - v0.1 New creator
            2015.10.26
              - v1.0 
              - Update slot_per_ms()
            2016.03.04
              - v1.1
              - Adding DDR monitor (command monitor function)
            2016.03.10
              - Support H3 WS1.1
            2016.06.24
              - v1.2
              - Support M3W WS1.0
			2016.09.01
			  - v1.3
			  - Adding latency monitor function(PerfMon). Support M3W and H3
            
 Copyright (c) 2016 Renesas Electronics Corporation. All rights reserved.
 License        Dual MIT/GPLv2
 
 The contents of this file are subject to the MIT license as set out below.
 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sub license, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 Alternatively, the contents of this file may be used under the terms of
 the GNU General Public License Version 2 ("GPL") in which case the provisions
 of GPL are applicable instead of those above.

 If you wish to allow use of your version of this file only under the terms of
 GPL, and not to allow others to use your version of this file under the terms
 of the MIT license, indicate your decision by deleting the provisions above
 and replace them with the notice and other provisions required by GPL as set
 out in the file called "GPL-COPYING" included in this distribution. If you do
 not delete the provisions above, a recipient may use your version of this file
 under the terms of either the MIT license or GPL.

 This License is also included in this distribution in the file called
 "MIT-COPYING".

 EXCEPT AS OTHERWISE STATED IN A NEGOTIATED AGREEMENT: (A) THE SOFTWARE IS
 PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 PURPOSE AND NONINFRINGEMENT; AND (B) IN NO EVENT SHALL THE AUTHORS OR
 COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


 GPLv2:
 If you wish to use this file under the terms of GPL, following terms are
 effective.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; version 2 of the License.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
***************************************************************************/
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/wait.h>
#include <linux/sched.h>

#define R8A7795
#if defined (R8A7795)
    #include "r8a7795.h"
    #define DEVDESCRIPTION "Support R-Car H3/M3W"
#else
    #error "Device must be defined."
#endif
#include "busmon.h"

#define DEVNAME "rcarbusmoni"

#define APRINT(fmt, args...) \
    printk(KERN_ALERT "busmoni.ko:%d: " fmt, current->pid, ##args)
#define EPRINT(fmt, args...) \
    printk(KERN_ERR "busmoni.ko:%d: " fmt, current->pid, ##args)
#define IPRINT(fmt, args...) \
    printk(KERN_INFO "busmoni.ko:%d: " fmt, current->pid, ##args)
#define NPRINT(fmt, args...) \
    printk(KERN_NOTICE "busmoni.ko:%d: " fmt, current->pid, ##args)
#define DPRINT(fmt, args...) \
    printk(KERN_DEBUG "busmoni.ko:%d: " fmt, current->pid, ##args)

#define diff(a,b)  ((a>=b) ? (a-b) : (a+(uint64_t)(0x10000000000)-b))
    
uint32_t device, device_version; /* device information */
uint32_t master_id_num;
uint32_t ddr_ch_num;

extern BUSMONREG BUSMON; /* BUSMONI registers */
extern DBMONREG DBMON;
extern PERFMONREG PERFMON[MAX_PARA_LAT_MON];

static BUSMONI_DBMON_CMD dbmon_cmd_saved[DDR_CH_NUM_MAX];

static int major_dev = -1;

PVOID pPRRBase = NULL;

BUSMONI_WORK_BUF * buf;

/* This flag to indicate whether the busmoni hardware resouce was relesed */
static uint32_t f_release_busmon = YES;
/* This flag to indicate whether the dbmon hardware resouce was relesed */
static uint32_t f_release_dbmon = YES;
/* This flag to indicate whether the perfmon hardware resouce was relesed */
static uint32_t f_release_perfmon = YES;


/*!
    @brief      Calculate how many slots in 1 ms
    @param[in]  None
    @return     Number of slots in 1 ms
*/
static uint32_t slot_per_ms(uint32_t axi_bus_clk)
{
    uint32_t sslotclock;
    DPRINT("+slot_per_ms\n");
    sslotclock = READREG32(BUSMON.MonitorTimeA) & SSLOTCLOCK_MSK;
    
    DPRINT("-slot_per_ms\n");
    if( sslotclock == 0 ) {
        return(0xFFFFFFFF);
    }
    else {
        return((axi_bus_clk*(1000)) / (sslotclock + 1) );
    }
}

void wait_subslot(uint32_t axi_bus_clk)
{
    uint32_t    dbmonconf2,dbmonconf3;
    //uint32_t    sslot = slot_per_ms(axi_bus_clk);
    uint32_t    uVal;

    /* Backup Registor */
    dbmonconf2 = READREG32(DBMON.DBMONCONF2);
    dbmonconf3 = READREG32(DBMON.DBMONCONF3);
    WRITEREG32(DBMON.DBMONCONF2, 0x00000000);
    WRITEREG32(DBMON.DBMONCONF3, 0xFFFFFFFF);


    /* Save Current MonitorTime */
    uVal = READREG32(BUSMON.MonitorTime);

    /* Start Performance Monitor*/
    WRITEREG32(BUSMON.PerformanceMonitor, PM_PMEN);

    /* Wait until Performance Monitor is started */
    while(uVal == READREG32(BUSMON.MonitorTime));
 
    /* Save Current MonitorTime */
    uVal = READREG32(BUSMON.MonitorTime);

    /* Stop Performance Monitor*/
    WRITEREG32(BUSMON.PerformanceMonitor, 0);

    /* Wait until Performance Monitor is stopped */
    while(uVal == READREG32(BUSMON.MonitorTime));

    /* Resume Registor */
    WRITEREG32(DBMON.DBMONCONF2, dbmonconf2);
    WRITEREG32(DBMON.DBMONCONF3, dbmonconf3);
}

void save_dbmon_cmd_regs(BUSMONI_WORK_BUF * pwork_buf)
{
    int id;
    uint32_t uVal;

    for( id = ddr_ch_num-1; id >= 0; id--)
    {
        uVal  = READREG32(DBMON.DBMONCONF0);
        uVal &= ~(MNDEV_MSK);
        uVal |= (id << MNDEV_SHIFT);
        WRITEREG32(DBMON.DBMONCONF0, uVal);

        /* W/A */
        wait_subslot(pwork_buf->work_paras.axi_bus_clk);

        dbmon_cmd_saved[id].rcmd        = (((uint64_t)(READREG32(DBMON.DBMONRDH)  & DBMON_L08_MSK)) << 32) + (uint64_t)(READREG32(DBMON.DBMONRDL));
        dbmon_cmd_saved[id].wcmd        = (((uint64_t)(READREG32(DBMON.DBMONWRH)  & DBMON_L08_MSK)) << 32) + (uint64_t)(READREG32(DBMON.DBMONWRL));
        dbmon_cmd_saved[id].actcmd      = (((uint64_t)(READREG32(DBMON.DBMONACTH) & DBMON_L08_MSK)) << 32) + (uint64_t)(READREG32(DBMON.DBMONACTL));
        dbmon_cmd_saved[id].refcmd      = (((uint64_t)(READREG32(DBMON.DBMONREFH) & DBMON_L08_MSK)) << 32) + (uint64_t)(READREG32(DBMON.DBMONREFL));
    }
}


/*!
    @brief      Collect traffic bus load of each master_ip
    @param[in]  pwork_buf : a pointer to user memory to store traffic bus load results
    @return     None
*/
void collect_traffic_mon(BUSMONI_WORK_BUF * pwork_buf)
{
    int id;
    DPRINT("+collect_traffic_mon\n");
    for( id = 0; id < master_id_num ; id++) {
        pwork_buf->work_traff_mid[id].total_forw   = READARRAY64(BUSMON.TRAFFICMONITOR_FIX, id) + READARRAY64(BUSMON.TRAFFICMONITOR_BE, id);
    }
    
    pwork_buf->bus_info.monitor_time  = READREG32(BUSMON.MonitorTime);
    pwork_buf->bus_info.slot_per_ms   = slot_per_ms(pwork_buf->work_paras.axi_bus_clk);

    DPRINT("-collect_traffic_mon\n");
}

/*!
    @brief      Collect command monitor results
    @param[in]  pwork_buf : a pointer to user memory to store traffic bus load results
    @return     
*/
void collect_command_mon(BUSMONI_WORK_BUF * pwork_buf)
{
    int id;
    uint32_t uVal;

    DPRINT("+collect_command_mon\n");   
    for( id = 0; id < ddr_ch_num; id++)
    {
        uVal  = READREG32(DBMON.DBMONCONF0);
        uVal &= ~(MNDEV_MSK);
        uVal |= (id << MNDEV_SHIFT);
        WRITEREG32(DBMON.DBMONCONF0, uVal);

        /* W/A */
        wait_subslot(pwork_buf->work_paras.axi_bus_clk);

        pwork_buf->work_dbmon_cmd[id].rcmd       = (((uint64_t)(READREG32(DBMON.DBMONRDH)  & DBMON_L08_MSK)) << 32) + READREG32(DBMON.DBMONRDL);
        pwork_buf->work_dbmon_cmd[id].wcmd       = (((uint64_t)(READREG32(DBMON.DBMONWRH)  & DBMON_L08_MSK)) << 32) + READREG32(DBMON.DBMONWRL);
        pwork_buf->work_dbmon_cmd[id].actcmd     = (((uint64_t)(READREG32(DBMON.DBMONACTH) & DBMON_L08_MSK)) << 32) + READREG32(DBMON.DBMONACTL);
        pwork_buf->work_dbmon_cmd[id].refcmd     = (((uint64_t)(READREG32(DBMON.DBMONREFH) & DBMON_L08_MSK)) << 32) + READREG32(DBMON.DBMONREFL);
        
        pwork_buf->work_dbmon_cmd[id].rcmd       = diff(pwork_buf->work_dbmon_cmd[id].rcmd,       dbmon_cmd_saved[id].rcmd);
        pwork_buf->work_dbmon_cmd[id].wcmd       = diff(pwork_buf->work_dbmon_cmd[id].wcmd,       dbmon_cmd_saved[id].wcmd);
        pwork_buf->work_dbmon_cmd[id].actcmd     = diff(pwork_buf->work_dbmon_cmd[id].actcmd,     dbmon_cmd_saved[id].actcmd);
        pwork_buf->work_dbmon_cmd[id].refcmd     = diff(pwork_buf->work_dbmon_cmd[id].refcmd,     dbmon_cmd_saved[id].refcmd);                      
    }
    DPRINT("-collect_command_mon\n");   
}

/*!
	@brief		Collect latency monitor results
	@param[in]	
	@return		
*/
void collect_latency_mon(BUSMONI_WORK_BUF * pwork_buf)
{
	int i = 0;
	
	DPRINT("+collect_latency_mon\n");
	if (device == R_CAR_M3W) {
		for( i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++) {
			switch (pwork_buf->work_paras.PMON_baseAdd[i]) {
				case 0xFEB8D600://fcpvd0
				case 0xFEB8D500://fcpvd1
				case 0xFEB8D100://fcpvd2 
					//fcpvd read only
					pwork_buf->work_perfmon[i].accum_r      = ((uint64_t)(READREG32(PERFMON[i].ACCUMULATOR_W)) << 32) + READREG32(PERFMON[i].ACCUMULATOR_R);
					pwork_buf->work_perfmon[i].accum_w      = 0;
					pwork_buf->work_perfmon[i].cnt_r        = ((uint32_t)(READREG16(PERFMON[i].TX_COUNTER_W)) << 16) + READREG16(PERFMON[i].TX_COUNTER_R);
					pwork_buf->work_perfmon[i].cnt_w        = 0;
					pwork_buf->work_perfmon[i].max_cycle_r  = READREG32(PERFMON[i].MAX_COUNTER_R);
					pwork_buf->work_perfmon[i].max_cycle_w  = READREG32(PERFMON[i].MAX_COUNTER_W);
					pwork_buf->work_perfmon[i].out_std_over = READREG32(PERFMON[i].OUTSTANDINGOVER);
					break;
				case 0xFEB8D300://vin0
				case 0xFEB8D400://vin1
					//vin write only
					pwork_buf->work_perfmon[i].accum_r      = 0;
					pwork_buf->work_perfmon[i].accum_w      = ((uint64_t)(READREG32(PERFMON[i].ACCUMULATOR_W)) << 32) + READREG32(PERFMON[i].ACCUMULATOR_R);
					pwork_buf->work_perfmon[i].cnt_r        = 0;
					pwork_buf->work_perfmon[i].cnt_w        = ((uint32_t)(READREG16(PERFMON[i].TX_COUNTER_W)) << 16) + READREG16(PERFMON[i].TX_COUNTER_R);
					pwork_buf->work_perfmon[i].max_cycle_r  = READREG32(PERFMON[i].MAX_COUNTER_R);
					pwork_buf->work_perfmon[i].max_cycle_w  = READREG32(PERFMON[i].MAX_COUNTER_W);
					pwork_buf->work_perfmon[i].out_std_over = READREG32(PERFMON[i].OUTSTANDINGOVER);
					break;
				default:
					pwork_buf->work_perfmon[i].accum_r      = READREG32(PERFMON[i].ACCUMULATOR_R);
					pwork_buf->work_perfmon[i].accum_w      = READREG32(PERFMON[i].ACCUMULATOR_W);
					pwork_buf->work_perfmon[i].cnt_r        = READREG32(PERFMON[i].TX_COUNTER_R);
					pwork_buf->work_perfmon[i].cnt_w        = READREG32(PERFMON[i].TX_COUNTER_W);
					pwork_buf->work_perfmon[i].max_cycle_r  = READREG32(PERFMON[i].MAX_COUNTER_R);
					pwork_buf->work_perfmon[i].max_cycle_w  = READREG32(PERFMON[i].MAX_COUNTER_W);
					pwork_buf->work_perfmon[i].out_std_over = READREG32(PERFMON[i].OUTSTANDINGOVER);
					break;
			}
		}	
	} else { //H3
		for( i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++) {
			pwork_buf->work_perfmon[i].accum_r      = READREG32(PERFMON[i].ACCUMULATOR_R);
			pwork_buf->work_perfmon[i].accum_w      = READREG32(PERFMON[i].ACCUMULATOR_W);
			pwork_buf->work_perfmon[i].cnt_r        = READREG32(PERFMON[i].TX_COUNTER_R);
			pwork_buf->work_perfmon[i].cnt_w        = READREG32(PERFMON[i].TX_COUNTER_W);
			pwork_buf->work_perfmon[i].max_cycle_r  = READREG32(PERFMON[i].MAX_COUNTER_R);
			pwork_buf->work_perfmon[i].max_cycle_w  = READREG32(PERFMON[i].MAX_COUNTER_W);
			pwork_buf->work_perfmon[i].out_std_over = READREG32(PERFMON[i].OUTSTANDINGOVER);
			
			
			// DPRINT("Base %d(%08X)\n", i, pwork_buf->work_paras.PMON_baseAdd[i]);
			// DPRINT("ACCUMULATOR_R    %08X\n",pwork_buf->work_perfmon[i].accum_r);
			// DPRINT("ACCUMULATOR_W    %08X\n",pwork_buf->work_perfmon[i].accum_w);
			// DPRINT("TX_COUNTER_R     %08X\n",pwork_buf->work_perfmon[i].cnt_r);
			// DPRINT("TX_COUNTER_W     %08X\n",pwork_buf->work_perfmon[i].cnt_w);
			// DPRINT("MAX_COUNTER_R    %08X\n",pwork_buf->work_perfmon[i].max_cycle_r);
			// DPRINT("MAX_COUNTER_W    %08X\n",pwork_buf->work_perfmon[i].max_cycle_w);
			// DPRINT("OutStandingOver  %08X\n",pwork_buf->work_perfmon[i].out_std_over);
		}
	}
	
	DPRINT("-collect_latency_mon\n");
}


/*!
    @brief     release busmon resource
    @param[in]  
    @return     
*/
void release_busmon_resouce(uint32_t device, uint32_t device_version)
{
    uint32_t pa_traffic_fix;
    uint32_t pa_traffic_be;
    uint32_t pa_monitortime_a;
    uint32_t pa_performance_monitor;
    uint32_t pa_monitortime;
    uint32_t pa_monitor_changetime;
    uint32_t pa_monitor_interrupt;
    uint32_t pa_distribution_threshold;
    
    DPRINT("+release_busmon_resouce\n");  
    pa_traffic_fix = BUSMON_BASE + TRAFFICMONITOR_FIX_OFFSET;
    pa_traffic_be = BUSMON_BASE + TRAFFICMONITOR_BE_OFFSET;
    pa_monitortime_a = BUSMON_BASE + MonitorTimeA_OFFSET;
    pa_performance_monitor = BUSMON_BASE + PerformanceMonitor_OFFSET;
    pa_monitor_changetime = BUSMON_BASE + MonitorChangeTime_OFFSET;
    pa_monitor_interrupt = BUSMON_BASE + MonitorInterrupt_OFFSET;
    pa_monitortime = BUSMON_BASE + MonitorTime_OFFSET;
    pa_distribution_threshold = BUSMON_BASE + DistributionThreshold_OFFSET;
    
    iounmap((void *)BUSMON.TRAFFICMONITOR_FIX);
    iounmap((void *)BUSMON.TRAFFICMONITOR_BE);
    iounmap((void *)BUSMON.MonitorTimeA);
    iounmap((void *)BUSMON.PerformanceMonitor);
    if (device == R_CAR_M3W) {
        iounmap((void *)BUSMON.MonitorChangeTime);
        iounmap((void *)BUSMON.MonitorInterrupt);
    }
    iounmap((void *)BUSMON.MonitorTime);
    iounmap((void *)BUSMON.DistributionThreshold);
    
    release_mem_region(pa_traffic_fix, 0x1000);
    release_mem_region(pa_traffic_be, 0x1000);
    release_mem_region(pa_monitortime_a, sizeof(uint32_t));
    release_mem_region(pa_performance_monitor, sizeof(uint32_t));
    if (device == R_CAR_M3W) {
        release_mem_region(pa_monitor_changetime, sizeof(uint32_t));
        release_mem_region(pa_monitor_interrupt, sizeof(uint32_t));
    }
    release_mem_region(pa_monitortime, sizeof(uint32_t));
    release_mem_region(pa_distribution_threshold, sizeof(uint64_t));
    
    f_release_busmon = YES;
    DPRINT("-release_busmon_resouce\n");
}

/*!
    @brief     release dbmon resource
    @param[in]  
    @return     
*/
void release_dbmon_resouce(uint32_t device, uint32_t device_version)
{   
    uint32_t pa_dbmonconf0; 
    uint32_t pa_dbmonrdl;
    
    DPRINT("+release_dbmon_resouce\n");
    pa_dbmonconf0 = DBMON_BASE;
    pa_dbmonrdl = DBMON_BASE + DBMONRDL_OFFSET;
    
    iounmap((void *)DBMON.DBMONCONF0);
    iounmap((void *)DBMON.DBMONRDL);
    
    release_mem_region(pa_dbmonconf0, 0x1C);
    release_mem_region(pa_dbmonrdl, 0x20);
    
    f_release_dbmon = YES;
    DPRINT("-release_dbmon_resouce\n");
}

/*!
    @brief     release perfmon resource
    @param[in]  
    @return     
*/
void release_perfmon_resouce(uint32_t device, uint32_t device_version, BUSMONI_WORK_BUF * pwork_buf)
{   
	int i;
    
    DPRINT("+release_perfmon_resouce\n");
    for (i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++) {
		iounmap((void *)(PERFMON[i].CCLEAR - CCLEAR_OFFSET));
		release_mem_region(pwork_buf->work_paras.PMON_baseAdd[i], PERFMON_REG_SIZE);
	}
    
    f_release_perfmon = YES;
    DPRINT("-release_perfmon_resouce\n");
}

/*!
 *  @brief      Request bus monitor hardware and initialize them.
 *  @param[in]  pwork_buf : a pointer to the working buffer
 *  @return     0: success 
 *              other: error
 */
int BUSMONI_Init(BUSMONI_WORK_BUF * pwork_buf)
{
    int ret = -ENOMEM;
    DPRINT("+BUSMONI_Init\n");
    
    ret = busmon_init(device, device_version);
    if (ret) {
        EPRINT("Failed to busmon_init()\n");
        return ret;
    }
    f_release_busmon = NO;
    
    ret = dbmon_init(device, device_version);
    if (ret) {
        EPRINT("Failed to dbmon_init()\n");
        release_busmon_resouce(device, device_version);
        return ret;
    }  
    f_release_dbmon = NO;
    
	ret = perfmon_init(device, device_version, pwork_buf);
    if (ret) {
        EPRINT("Failed to perfmon_init()\n");
        release_busmon_resouce(device, device_version);
		release_dbmon_resouce(device, device_version);
        return ret;
    }  
    f_release_perfmon = NO;
	
    DPRINT("-BUSMONI_Init\n");
    return ret;
}
 
/*!
 *  @brief      Identify which SoC is running
 *  @param[in]  None
 *  @return     None
 */
int BUSMONI_Identify_Board_Information(void)
{
    uint64_t uReg;
    uint32_t uVal;
    int ret = 0;
    DPRINT("+BUSMONI_Identify_Board_Information\n");
    
    pPRRBase = ioremap_nocache(PRR_REGBASE, PRR_REGSIZE);
    uReg = (uint64_t)pPRRBase + (PRR - PRR_REGBASE);
    uVal = READREG32(uReg);
    device = uVal & 0x0000ff00;
    device_version = uVal & 0x000000ff;
    switch(device) {
        case R_CAR_H3:
            DPRINT("Device \"R-Car H3");
            master_id_num  = MASTER_ID_NUM_H3;
            ddr_ch_num     = DDR_CH_NUM_H3;
            break;
        case R_CAR_M3W:
            DPRINT("Device \"R-Car M3W");
            master_id_num  = MASTER_ID_NUM_M3W;
            ddr_ch_num     = DDR_CH_NUM_M3W;
            break;  
        default:
            DPRINT("Unsupported Device\r\n");
            ret = -EINVAL;
            break;
    }
    
    switch(device_version) {
        case WS1_0:
            DPRINT(" WS1.0\"\r\n");
            break;

        case WS1_1:
            DPRINT(" WS1.1\"\r\n");
            break;
            
        case WS2_0:
            DPRINT(" WS2.0\"\r\n");
            break;

        case WS3_0:
            DPRINT(" WS3.0\"\r\n");
            break;

        default:
            DPRINT("Unknown WS version\"\r\n");
            ret = -EINVAL;
            break;
    }
    iounmap(pPRRBase);
    
    return ret;
    DPRINT("-BUSMONI_Identify_Board_Information\n");
}

/*!
    @brief      This function is to set necessary parameter which is necessary for the coming measurement.
    @return     0 = success
*/
static int BUSMONI_Set_Parameters(PBUSMONI_WORK_BUF pwork_buf)
{       
	int i;
    DPRINT("+BUSMONI_Set_Parameters\n");            
            
    WRITEREG64(BUSMON.DistributionThreshold, (uint64_t)pwork_buf->work_paras.param_DistributionThreshold);
    WRITEREG32(DBMON.DBMONCONF1, pwork_buf->work_paras.param_DBMONCONF1);
    WRITEREG32(DBMON.DBMONCONF2, pwork_buf->work_paras.param_DBMONCONF2);
    WRITEREG32(DBMON.DBMONCONF3, pwork_buf->work_paras.param_DBMONCONF3);
    
    /* If in case of H3 WS1.1, MNSTATSEL register should be set always '111'. */
    if((device == R_CAR_H3) && (device_version == WS1_1))
    {
        pwork_buf->work_paras.param_DBMONCONF4 |= 0x00700000;       /* DBMONCONF4[22-20]='111' */
    }
    WRITEREG32(DBMON.DBMONCONF4, pwork_buf->work_paras.param_DBMONCONF4);
    
    WRITEREG32(DBMON.DBMONCONF5, pwork_buf->work_paras.param_DBMONCONF5);
    WRITEREG32(DBMON.DBMONCONF6, pwork_buf->work_paras.param_DBMONCONF6);
    
	/* PerfMON setting */
	if (device == R_CAR_M3W) {
		for (i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++) {
			switch (pwork_buf->work_paras.PMON_baseAdd[i]) {
				case 0xFEB8D600://fcpvd0
				case 0xFEB8D500://fcpvd1
				case 0xFEB8D100://fcpvd2
					break;
				case 0xFEB8D300://vin0
				case 0xFEB8D400://vin1
					/* Set Parameter */
					WRITEREG32(PERFMON[i].CENABLE, 0);
					WRITEREG32(PERFMON[i].PRESCALE	, pwork_buf->work_paras.PMON_prescale[i]);
					WRITEREG32(PERFMON[i].IDMSKR1	    , pwork_buf->work_paras.PMON_idmskr1[i]);
					WRITEREG32(PERFMON[i].IDMSKR0	    , pwork_buf->work_paras.PMON_idmskr0[i]);
					WRITEREG32(PERFMON[i].IDMSKW1	    , pwork_buf->work_paras.PMON_idmskw1[i]);
					WRITEREG32(PERFMON[i].IDMSKW0	    , pwork_buf->work_paras.PMON_idmskw0[i]);
					//SELR and SELW not used
					//WRITEREG32(PERFMON[i].SELR		, pwork_buf->work_paras.PMON_selr[i]);
					//WRITEREG32(PERFMON[i].SELW		, pwork_buf->work_paras.PMON_selw[i]);
					break;
				default:
					/* Set Parameter */
					WRITEREG32(PERFMON[i].CENABLE, 0);
					WRITEREG32(PERFMON[i].PRESCALE	, pwork_buf->work_paras.PMON_prescale[i]);
					WRITEREG32(PERFMON[i].IDMSKR1	    , pwork_buf->work_paras.PMON_idmskr1[i]);
					WRITEREG32(PERFMON[i].IDMSKR0	    , pwork_buf->work_paras.PMON_idmskr0[i]);
					WRITEREG32(PERFMON[i].IDMSKW1	    , pwork_buf->work_paras.PMON_idmskw1[i]);
					WRITEREG32(PERFMON[i].IDMSKW0	    , pwork_buf->work_paras.PMON_idmskw0[i]);
					WRITEREG32(PERFMON[i].SELR		, pwork_buf->work_paras.PMON_selr[i]);
					WRITEREG32(PERFMON[i].SELW		, pwork_buf->work_paras.PMON_selw[i]);
					break;
			}
		}
	} else {
		for (i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++) {
			switch (pwork_buf->work_paras.PMON_baseAdd[i]) {
				case 0xFEB8D800://fcpvd0
				case 0xFEB8D700://fcpvd1
				case 0xFEB8D400://fcpvd2
				case 0xFEB8D900://fcpvd3
					break; //Not set paramters for fcpvd*. It should set by uboot 
				default:
					/* Set Parameter */
					WRITEREG32(PERFMON[i].CENABLE, 0);
					WRITEREG32(PERFMON[i].PRESCALE	, pwork_buf->work_paras.PMON_prescale[i]);
					WRITEREG32(PERFMON[i].IDMSKR1	    , pwork_buf->work_paras.PMON_idmskr1[i]);
					WRITEREG32(PERFMON[i].IDMSKR0	    , pwork_buf->work_paras.PMON_idmskr0[i]);
					WRITEREG32(PERFMON[i].IDMSKW1	    , pwork_buf->work_paras.PMON_idmskw1[i]);
					WRITEREG32(PERFMON[i].IDMSKW0	    , pwork_buf->work_paras.PMON_idmskw0[i]);
					WRITEREG32(PERFMON[i].SELR		, pwork_buf->work_paras.PMON_selr[i]);
					WRITEREG32(PERFMON[i].SELW		, pwork_buf->work_paras.PMON_selw[i]);
					break;
			}
		}
	}
	
    DPRINT("-BUSMONI_Set_Parameters\n");
    return 0;
}


/*!
    @brief      opens a device context
    @return     0: success
*/
static int BUSMONI_Open(struct inode *inode, struct file *file)
{
    int ret = 0;
    DPRINT("+BUSMONI_Open\n");
    
    /* Identify the type of board and display it */
    ret = BUSMONI_Identify_Board_Information();

    DPRINT("-BUSMONI_Open\n");
    return ret;
}

/*!
    @brief       closes a device context.
    @return     0: success
*/
static int BUSMONI_Close(struct inode *inode, struct file *file)
{
    DPRINT("+BUSMONI_Close\n");
    DPRINT("-BUSMONI_Close\n");
    return 0;
}

/*!
    @brief      This function sends a command to a device.
    @return     0: success.
                -EINVAL: invalid arguments
                -ENOMEM: no memory or occupied memory
*/
static long BUSMONI_IOControl(struct file *file, unsigned int cmd, unsigned long arg)
{
    int ret = -EINVAL;
	int i;
    void __user *argp = (void __user *)arg;
    PBUSMONI_WORK_BUF pwork_buf;
    DEV_INFO * pdev_info;
    
    DPRINT("+BUSMONI_IOControl\n");
    switch (cmd) {   
        case IOCTL_BUSMONI_SETUP:
            DPRINT("+IOCTL_BUSMONI_SETUP\n");
            pwork_buf = (PBUSMONI_WORK_BUF)argp;
            
			buf = pwork_buf;
            /* Init registers */
            ret = BUSMONI_Init(pwork_buf);
            if (ret) {
                break;
            }
            
            BUSMONI_Set_Parameters(pwork_buf);
            
            pwork_buf->bus_info.device = device;
            pwork_buf->bus_info.device_version = device_version;
            pwork_buf->bus_info.master_id_num = master_id_num;
            pwork_buf->bus_info.ddr_ch_num = ddr_ch_num;
            
            ret = 0;
            DPRINT("-IOCTL_BUSMONI_SETUP\n");
            break;
                
        case IOCTL_BUSMONI_START:
            pwork_buf = (PBUSMONI_WORK_BUF)argp;
            DPRINT("+IOCTL_BUSMONI_START\n");
			
            /* Reset forwarding amount monitor*/
            WRITEREG32(BUSMON.PerformanceMonitor, PM_PRESET);
            
			/* Reset counters of latency monitor*/
			if (device == R_CAR_M3W) {
				for( i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++) {
					switch (pwork_buf->work_paras.PMON_baseAdd[i]) {
						case 0xFEB8D600://fcpvd0
						case 0xFEB8D500://fcpvd1
						case 0xFEB8D100://fcpvd2 
						case 0xFEB8D300://vin0
						case 0xFEB8D400://vin1
							WRITEREG32(PERFMON[i].SELR, 1); //newCCLEAR
							WRITEREG32(PERFMON[i].SELR, 0);
							break;
						default:
							WRITEREG32(PERFMON[i].CCLEAR, 1);
							WRITEREG32(PERFMON[i].CCLEAR, 0);
							break;
					}
				}
			} else {//H3
				for( i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++) {
					WRITEREG32(PERFMON[i].CCLEAR, 1);
					WRITEREG32(PERFMON[i].CCLEAR, 0);
				}	
			}
			
            /* W/A for reading result of command monitor after changing port by DBMONCONF0*/
            wait_subslot(pwork_buf->work_paras.axi_bus_clk);

            /* W/A for reset issue of command monitor */
            save_dbmon_cmd_regs(pwork_buf);
            
            /* Start forwarding amount monitor */
            WRITEREG32(BUSMON.PerformanceMonitor, PM_PMEN);
            
			/* Start Latency Monitor */
			for( i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++) {
				WRITEREG32(PERFMON[i].CENABLE, 1);
			}
			
            ret = 0;
            DPRINT("-IOCTL_BUSMONI_START\n");
            break;
        
        case IOCTL_BUSMONI_STOP:
            pwork_buf = (PBUSMONI_WORK_BUF)argp;
            DPRINT("+IOCTL_BUSMONI_STOP\n");
            
            /* Stop Performance Monitor*/
            WRITEREG32(BUSMON.PerformanceMonitor, 0);
            
			/* Stop Latency Monitor*/
			if (device == R_CAR_M3W) {
				for( i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++){
					switch (pwork_buf->work_paras.PMON_baseAdd[i]) {
						case 0xFEB8D600://fcpvd0
						case 0xFEB8D500://fcpvd1
						case 0xFEB8D100://fcpvd2 
							break; //let fcpvd run continuously
						default:
							WRITEREG32(PERFMON[i].CENABLE, 0);
							break;
					}
				}
			} else { //H3
				for( i = 0; i < pwork_buf->work_paras.PMON_paraNum; i++){
					switch (pwork_buf->work_paras.PMON_baseAdd[i]) {
						case 0xFEB8D800://fcpvd0
						case 0xFEB8D700://fcpvd1
						case 0xFEB8D400://fcpvd2
						case 0xFEB8D900://fcpvd3
							break; //Not stop fcpvd
						default:
							WRITEREG32(PERFMON[i].CENABLE, 0);
							break;
					}
					
				}
			}
			
            /* Collect measurement results*/
            collect_traffic_mon(pwork_buf);
            collect_command_mon(pwork_buf);
            collect_latency_mon(pwork_buf);
			
            ret = 0;
            DPRINT("-IOCTL_BUSMONI_STOP\n");
            break;
		
        case IOCTL_BUSMONI_RELEASE_MEM:
            DPRINT("+IOCTL_BUSMONI_RELEASE_MEM\n");
            pwork_buf = (PBUSMONI_WORK_BUF)argp;
            
            if (f_release_busmon == NO)
                release_busmon_resouce(device, device_version);
            if (f_release_dbmon == NO)
                release_dbmon_resouce(device, device_version);
            if (f_release_perfmon == NO)
                release_perfmon_resouce(device, device_version, pwork_buf);
			
            ret = 0;
            DPRINT("-IOCTL_BUSMONI_RELEASE_MEM\n");
            break;
        
        case IOCTL_BUSMONI_GET_DEVICE_INFO:
            DPRINT("+IOCTL_BUSMONI_GET_DEVICE_INFO\n");
            pdev_info = (DEV_INFO *)argp;
            
            pdev_info->device = device;
            pdev_info->device_version = device_version;
            
            ret = 0;
            DPRINT("-IOCTL_BUSMONI_GET_DEVICE_INFO\n");
            break;
        
        default:
            DPRINT("default:error cmd = 0x%x\n", cmd);
            break;
        }

    DPRINT("-BUSMONI_IOControl\n");
    return ret;
}

static const struct file_operations rcar_axi_fops = {
    .owner      = THIS_MODULE,
    .open       = BUSMONI_Open,
    .release    = BUSMONI_Close,
    .unlocked_ioctl     = BUSMONI_IOControl,
};

/*!
    @brief      This function initialize busmoni driver.
    @return     0: success. 
                -1: indicates failing to register a device.
*/
static int __init rcar_busmoni_init(void)
{
    int i;
    DPRINT("+rcar_busmoni_init\n");
    i = register_chrdev(0, DEVNAME, &rcar_axi_fops);
    if (i < 0) {
        NPRINT("Unable to create chrdev\n");
        return -1;
    }
    major_dev = i;
    IPRINT("Install busmoni driver v1.3\n");
    IPRINT(DEVNAME " major_dev = %d\n", major_dev);
    IPRINT("%s\n", DEVDESCRIPTION);
    DPRINT("-rcar_busmoni_init\n");

    return 0;
}

/*!
    @brief      This function finalize busmoni driver.
    @return     Nothing.
*/
static void __exit rcar_busmoni_exit(void)
{
    if (major_dev != -1) {
        unregister_chrdev(major_dev, DEVNAME);
    }
    if (f_release_busmon == NO)
        release_busmon_resouce(device, device_version);
    if (f_release_dbmon == NO)
        release_dbmon_resouce(device, device_version);
    if (f_release_perfmon == NO)
        release_perfmon_resouce(device, device_version, buf);

    IPRINT("Unloaded busmoni driver sucessfully!!!\n");
}

module_init(rcar_busmoni_init);
module_exit(rcar_busmoni_exit);

MODULE_DESCRIPTION("R-Car GEN3 busload measurement driver");
MODULE_AUTHOR("Renesas Electronics Corporation");
MODULE_LICENSE("Dual MIT/GPL");
