/*
 * USB device quirk handling logic and table
 *
 * Copyright (c) 2007 Oliver Neukum
 * Copyright (c) 2007 Greg Kroah-Hartman <gregkh@suse.de>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, version 2.
 *
 *
 */

#include <linux/usb.h>
#include <linux/usb/quirks.h>
#include <linux/usb/hcd.h>
#include "usb.h"

/* Lists of quirky USB devices, split in device quirks and interface quirks.
 * Device quirks are applied at the very beginning of the enumeration process,
 * right after reading the device descriptor. They can thus only match on device
 * information.
 *
 * Interface quirks are applied after reading all the configuration descriptors.
 * They can match on both device and interface information.
 *
 * Note that the DELAY_INIT and HONOR_BNUMINTERFACES quirks do not make sense as
 * interface quirks, as they only influence the enumeration process which is run
 * before processing the interface quirks.
 *
 * Please keep the lists ordered by:
 * 	1) Vendor ID
 * 	2) Product ID
 * 	3) Class ID
 */
static const struct usb_device_id usb_quirk_list[] = {
	/* CBM - Flash disk */
	{ USB_DEVICE(0x0204, 0x6025), .driver_info = USB_QUIRK_RESET_RESUME },

	/* WORLDE easy key (easykey.25) MIDI controller  */
	{ USB_DEVICE(0x0218, 0x0401), .driver_info =
			USB_QUIRK_CONFIG_INTF_STRINGS },

	/* HP 5300/5370C scanner */
	{ USB_DEVICE(0x03f0, 0x0701), .driver_info =
			USB_QUIRK_STRING_FETCH_255 },

	/* Creative SB Audigy 2 NX */
	{ USB_DEVICE(0x041e, 0x3020), .driver_info = USB_QUIRK_RESET_RESUME },

	/* USB3503 */
	{ USB_DEVICE(0x0424, 0x3503), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Microsoft Wireless Laser Mouse 6000 Receiver */
	{ USB_DEVICE(0x045e, 0x00e1), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Microsoft LifeCam-VX700 v2.0 */
	{ USB_DEVICE(0x045e, 0x0770), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Logitech HD Pro Webcams C920, C920-C and C930e */
	{ USB_DEVICE(0x046d, 0x082d), .driver_info = USB_QUIRK_DELAY_INIT },
	{ USB_DEVICE(0x046d, 0x0841), .driver_info = USB_QUIRK_DELAY_INIT },
	{ USB_DEVICE(0x046d, 0x0843), .driver_info = USB_QUIRK_DELAY_INIT },

	/* Logitech ConferenceCam CC3000e */
	{ USB_DEVICE(0x046d, 0x0847), .driver_info = USB_QUIRK_DELAY_INIT },
	{ USB_DEVICE(0x046d, 0x0848), .driver_info = USB_QUIRK_DELAY_INIT },

	/* Logitech PTZ Pro Camera */
	{ USB_DEVICE(0x046d, 0x0853), .driver_info = USB_QUIRK_DELAY_INIT },

	/* Logitech Quickcam Fusion */
	{ USB_DEVICE(0x046d, 0x08c1), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Logitech Quickcam Orbit MP */
	{ USB_DEVICE(0x046d, 0x08c2), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Logitech Quickcam Pro for Notebook */
	{ USB_DEVICE(0x046d, 0x08c3), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Logitech Quickcam Pro 5000 */
	{ USB_DEVICE(0x046d, 0x08c5), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Logitech Quickcam OEM Dell Notebook */
	{ USB_DEVICE(0x046d, 0x08c6), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Logitech Quickcam OEM Cisco VT Camera II */
	{ USB_DEVICE(0x046d, 0x08c7), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Logitech Harmony 700-series */
	{ USB_DEVICE(0x046d, 0xc122), .driver_info = USB_QUIRK_DELAY_INIT },

	/* Philips PSC805 audio device */
	{ USB_DEVICE(0x0471, 0x0155), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Plantronic Audio 655 DSP */
	{ USB_DEVICE(0x047f, 0xc008), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Plantronic Audio 648 USB */
	{ USB_DEVICE(0x047f, 0xc013), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Artisman Watchdog Dongle */
	{ USB_DEVICE(0x04b4, 0x0526), .driver_info =
			USB_QUIRK_CONFIG_INTF_STRINGS },

	/* Microchip Joss Optical infrared touchboard device */
	{ USB_DEVICE(0x04d8, 0x000c), .driver_info =
			USB_QUIRK_CONFIG_INTF_STRINGS },

	/* CarrolTouch 4000U */
	{ USB_DEVICE(0x04e7, 0x0009), .driver_info = USB_QUIRK_RESET_RESUME },

	/* CarrolTouch 4500U */
	{ USB_DEVICE(0x04e7, 0x0030), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Samsung Android phone modem - ID conflict with SPH-I500 */
	{ USB_DEVICE(0x04e8, 0x6601), .driver_info =
			USB_QUIRK_CONFIG_INTF_STRINGS },

	/* Elan Touchscreen */
	{ USB_DEVICE(0x04f3, 0x0089), .driver_info =
			USB_QUIRK_DEVICE_QUALIFIER },

	{ USB_DEVICE(0x04f3, 0x009b), .driver_info =
			USB_QUIRK_DEVICE_QUALIFIER },

	{ USB_DEVICE(0x04f3, 0x010c), .driver_info =
			USB_QUIRK_DEVICE_QUALIFIER },

	{ USB_DEVICE(0x04f3, 0x0125), .driver_info =
			USB_QUIRK_DEVICE_QUALIFIER },

	{ USB_DEVICE(0x04f3, 0x016f), .driver_info =
			USB_QUIRK_DEVICE_QUALIFIER },

	{ USB_DEVICE(0x04f3, 0x0381), .driver_info =
			USB_QUIRK_NO_LPM },

	{ USB_DEVICE(0x04f3, 0x21b8), .driver_info =
			USB_QUIRK_DEVICE_QUALIFIER },

	/* Roland SC-8820 */
	{ USB_DEVICE(0x0582, 0x0007), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Edirol SD-20 */
	{ USB_DEVICE(0x0582, 0x0027), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Alcor Micro Corp. Hub */
	{ USB_DEVICE(0x058f, 0x9254), .driver_info = USB_QUIRK_RESET_RESUME },

	/* appletouch */
	{ USB_DEVICE(0x05ac, 0x021a), .driver_info = USB_QUIRK_RESET_RESUME },

	/* apple devices */
	{ USB_DEVICE(0x05ac, 0x1200), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1201), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1202), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1203), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1204), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1205), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1206), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1207), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1208), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1209), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x120a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x120b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x120c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x120d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x120e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x120f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1210), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1211), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1212), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1213), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1214), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1215), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1216), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1217), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1218), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1219), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x121a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x121b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x121c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x121d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x121e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x121f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1220), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1221), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1222), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1223), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1224), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1225), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1226), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1227), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1228), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1229), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x122a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x122b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x122c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x122d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x122e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x122f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1230), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1231), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1232), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1233), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1234), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1235), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1236), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1237), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1238), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1239), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x123a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x123b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x123c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x123d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x123e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x123f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1240), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1241), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1242), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1243), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1244), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1245), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1246), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1247), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1248), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1249), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x124a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x124b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x124c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x124d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x124e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x124f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1250), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1251), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1252), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1253), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1254), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1255), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1256), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1257), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1258), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1259), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x125a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x125b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x125c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x125d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x125e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x125f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1260), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1261), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1262), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1263), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1264), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1265), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1266), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1267), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1268), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1269), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x126a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x126b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x126c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x126d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x126e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x126f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1270), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1271), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1272), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1273), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1274), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1275), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1276), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1277), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1278), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1279), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x127a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x127b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x127c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x127d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x127e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x127f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1280), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1281), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1282), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1283), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1284), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1285), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1286), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1287), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1288), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1289), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x128a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x128b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x128c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x128d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x128e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x128f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1290), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1291), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1292), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1293), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1294), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1295), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1296), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1297), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1298), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x1299), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x129a), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x129b), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x129c), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x129d), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x129e), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x129f), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a0), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a1), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a2), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a3), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a4), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a5), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a6), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a7), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a8), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12a9), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12aa), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ab), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ac), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ad), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ae), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12af), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b0), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b1), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b2), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b3), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b4), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b5), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b6), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b7), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b8), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12b9), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ba), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12bb), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12bc), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12bd), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12be), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12bf), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c0), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c1), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c2), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c3), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c4), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c5), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c6), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c7), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c8), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12c9), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ca), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12cb), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12cc), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12cd), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ce), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12cf), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d0), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d1), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d2), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d3), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d4), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d5), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d6), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d7), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d8), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12d9), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12da), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12db), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12dc), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12dd), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12de), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12df), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e0), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e1), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e2), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e3), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e4), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e5), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e6), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e7), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e8), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12e9), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ea), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12eb), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ec), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ed), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ee), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ef), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f0), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f1), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f2), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f3), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f4), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f5), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f6), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f7), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f8), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12f9), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12fa), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12fb), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12fc), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12fd), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12fe), .driver_info = USB_QUIRK_CONFIG_HID },
	{ USB_DEVICE(0x05ac, 0x12ff), .driver_info = USB_QUIRK_CONFIG_HID },
	/* Genesys Logic hub, internally used by Moshi USB to Ethernet Adapter */
	{ USB_DEVICE(0x05e3, 0x0616), .driver_info = USB_QUIRK_NO_LPM },

	/* Avision AV600U */
	{ USB_DEVICE(0x0638, 0x0a13), .driver_info =
	  USB_QUIRK_STRING_FETCH_255 },

	/* Saitek Cyborg Gold Joystick */
	{ USB_DEVICE(0x06a3, 0x0006), .driver_info =
			USB_QUIRK_CONFIG_INTF_STRINGS },

	/* Guillemot Webcam Hercules Dualpix Exchange (2nd ID) */
	{ USB_DEVICE(0x06f8, 0x0804), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Guillemot Webcam Hercules Dualpix Exchange*/
	{ USB_DEVICE(0x06f8, 0x3005), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Midiman M-Audio Keystation 88es */
	{ USB_DEVICE(0x0763, 0x0192), .driver_info = USB_QUIRK_RESET_RESUME },

	/* M-Systems Flash Disk Pioneers */
	{ USB_DEVICE(0x08ec, 0x1000), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Baum Vario Ultra */
	{ USB_DEVICE(0x0904, 0x6101), .driver_info =
			USB_QUIRK_LINEAR_FRAME_INTR_BINTERVAL },
	{ USB_DEVICE(0x0904, 0x6102), .driver_info =
			USB_QUIRK_LINEAR_FRAME_INTR_BINTERVAL },
	{ USB_DEVICE(0x0904, 0x6103), .driver_info =
			USB_QUIRK_LINEAR_FRAME_INTR_BINTERVAL },

	/* Keytouch QWERTY Panel keyboard */
	{ USB_DEVICE(0x0926, 0x3333), .driver_info =
			USB_QUIRK_CONFIG_INTF_STRINGS },

	/* X-Rite/Gretag-Macbeth Eye-One Pro display colorimeter */
	{ USB_DEVICE(0x0971, 0x2000), .driver_info = USB_QUIRK_NO_SET_INTF },

	/* Broadcom BCM92035DGROM BT dongle */
	{ USB_DEVICE(0x0a5c, 0x2021), .driver_info = USB_QUIRK_RESET_RESUME },

	/* MAYA44USB sound device */
	{ USB_DEVICE(0x0a92, 0x0091), .driver_info = USB_QUIRK_RESET_RESUME },

	/* ASUS Base Station(T100) */
	{ USB_DEVICE(0x0b05, 0x17e0), .driver_info =
			USB_QUIRK_IGNORE_REMOTE_WAKEUP },

	/* Action Semiconductor flash disk */
	{ USB_DEVICE(0x10d6, 0x2200), .driver_info =
			USB_QUIRK_STRING_FETCH_255 },

	/* SKYMEDI USB_DRIVE */
	{ USB_DEVICE(0x1516, 0x8628), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Razer - Razer Blade Keyboard */
	{ USB_DEVICE(0x1532, 0x0116), .driver_info =
			USB_QUIRK_LINEAR_UFRAME_INTR_BINTERVAL },

	/* BUILDWIN Photo Frame */
	{ USB_DEVICE(0x1908, 0x1315), .driver_info =
			USB_QUIRK_HONOR_BNUMINTERFACES },

	/* Protocol and OTG Electrical Test Device */
	{ USB_DEVICE(0x1a0a, 0x0200), .driver_info =
			USB_QUIRK_LINEAR_UFRAME_INTR_BINTERVAL },

	/* Corsair Strafe RGB */
	{ USB_DEVICE(0x1b1c, 0x1b20), .driver_info = USB_QUIRK_DELAY_INIT },

	/* Acer C120 LED Projector */
	{ USB_DEVICE(0x1de1, 0xc102), .driver_info = USB_QUIRK_NO_LPM },

	/* Blackmagic Design Intensity Shuttle */
	{ USB_DEVICE(0x1edb, 0xbd3b), .driver_info = USB_QUIRK_NO_LPM },

	/* Blackmagic Design UltraStudio SDI */
	{ USB_DEVICE(0x1edb, 0xbd4f), .driver_info = USB_QUIRK_NO_LPM },

	/* Hauppauge HVR-950q */
	{ USB_DEVICE(0x2040, 0x7200), .driver_info =
			USB_QUIRK_CONFIG_INTF_STRINGS },

	/* INTEL VALUE SSD */
	{ USB_DEVICE(0x8086, 0xf1a5), .driver_info = USB_QUIRK_RESET_RESUME },

	{ }  /* terminating entry must be last */
};

static const struct usb_device_id usb_interface_quirk_list[] = {
	/* Logitech UVC Cameras */
	{ USB_VENDOR_AND_INTERFACE_INFO(0x046d, USB_CLASS_VIDEO, 1, 0),
	  .driver_info = USB_QUIRK_RESET_RESUME },

	{ }  /* terminating entry must be last */
};

static const struct usb_device_id usb_amd_resume_quirk_list[] = {
	/* Lenovo Mouse with Pixart controller */
	{ USB_DEVICE(0x17ef, 0x602e), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Pixart Mouse */
	{ USB_DEVICE(0x093a, 0x2500), .driver_info = USB_QUIRK_RESET_RESUME },
	{ USB_DEVICE(0x093a, 0x2510), .driver_info = USB_QUIRK_RESET_RESUME },
	{ USB_DEVICE(0x093a, 0x2521), .driver_info = USB_QUIRK_RESET_RESUME },
	{ USB_DEVICE(0x03f0, 0x2b4a), .driver_info = USB_QUIRK_RESET_RESUME },

	/* Logitech Optical Mouse M90/M100 */
	{ USB_DEVICE(0x046d, 0xc05a), .driver_info = USB_QUIRK_RESET_RESUME },

	{ }  /* terminating entry must be last */
};

static bool usb_match_any_interface(struct usb_device *udev,
				    const struct usb_device_id *id)
{
	unsigned int i;

	for (i = 0; i < udev->descriptor.bNumConfigurations; ++i) {
		struct usb_host_config *cfg = &udev->config[i];
		unsigned int j;

		for (j = 0; j < cfg->desc.bNumInterfaces; ++j) {
			struct usb_interface_cache *cache;
			struct usb_host_interface *intf;

			cache = cfg->intf_cache[j];
			if (cache->num_altsetting == 0)
				continue;

			intf = &cache->altsetting[0];
			if (usb_match_one_id_intf(udev, intf, id))
				return true;
		}
	}

	return false;
}

static int usb_amd_resume_quirk(struct usb_device *udev)
{
	struct usb_hcd *hcd;

	hcd = bus_to_hcd(udev->bus);
	/* The device should be attached directly to root hub */
	if (udev->level == 1 && hcd->amd_resume_bug == 1)
		return 1;

	return 0;
}

static u32 __usb_detect_quirks(struct usb_device *udev,
			       const struct usb_device_id *id)
{
	u32 quirks = 0;

	for (; id->match_flags; id++) {
		if (!usb_match_device(udev, id))
			continue;

		if ((id->match_flags & USB_DEVICE_ID_MATCH_INT_INFO) &&
		    !usb_match_any_interface(udev, id))
			continue;

		quirks |= (u32)(id->driver_info);
	}

	return quirks;
}

/*
 * Detect any quirks the device has, and do any housekeeping for it if needed.
 */
void usb_detect_quirks(struct usb_device *udev)
{
	udev->quirks = __usb_detect_quirks(udev, usb_quirk_list);

	/*
	 * Pixart-based mice would trigger remote wakeup issue on AMD
	 * Yangtze chipset, so set them as RESET_RESUME flag.
	 */
	if (usb_amd_resume_quirk(udev))
		udev->quirks |= __usb_detect_quirks(udev,
				usb_amd_resume_quirk_list);

	if (udev->quirks)
		dev_dbg(&udev->dev, "USB quirks for this device: %x\n",
			udev->quirks);

#ifdef CONFIG_USB_DEFAULT_PERSIST
	if (!(udev->quirks & USB_QUIRK_RESET))
		udev->persist_enabled = 1;
#else
	/* Hubs are automatically enabled for USB-PERSIST */
	if (udev->descriptor.bDeviceClass == USB_CLASS_HUB)
		udev->persist_enabled = 1;
#endif	/* CONFIG_USB_DEFAULT_PERSIST */
}

void usb_detect_interface_quirks(struct usb_device *udev)
{
	u32 quirks;

	quirks = __usb_detect_quirks(udev, usb_interface_quirk_list);
	if (quirks == 0)
		return;

	dev_dbg(&udev->dev, "USB interface quirks for this device: %x\n",
		quirks);
	udev->quirks |= quirks;
}
