/*
 * linux/include/linux/exchnd.h
 *
 * Copyright (C) 2016 Advanced Driver Information Technology GmbH
 * Written by Matthias Weise (mweise@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   See the
 * GNU General Public License for more details.
 *
 */

#ifndef __EXCHND_H__
#define __EXCHND_H__

#include <linux/module.h>
#include <linux/mutex.h>

#include <uapi/linux/exchnd.h>

/* Function that can be used to access safely the external trigger from
 * another module.
 * @param mod: (struct module *) global parameter provided by user.
 * @param pid: (pid_t) pid of the process to get information of
 * @param mod_list: (enum exchnd_modules (*modules)[EHM_LAST_ELEMENT])
 *			list of collector's modules to be executed
 *			if different form default.
 * @param msg: (char msg[256]) Message to put in log's header instead of
 *			"EXCEPTION in[...]". Can be empty to not change the
 *			original header.
 */
int exchnd_external(pid_t,
		    enum exchnd_modules (*)[EHM_LAST_ELEMENT],
		    char [EXH_ON_DEMAND_MSG]);

#ifdef CONFIG_EXCHND_MODULE /* If not built-in ... */
static inline int exchnd_call_external_trigger(
		struct module *mod,
		pid_t pid,
		enum exchnd_modules (*mod_list)[EHM_LAST_ELEMENT],
		char *msg) {
	int __ret = 0;

	mutex_lock(&module_mutex);
	if (!mod)
		mod = find_module("exchnd");
	mutex_unlock(&module_mutex);

	if (!mod) {
		pr_warn("exchnd module not found.\n");
		__ret = -ENODEV;
	} else if (try_module_get(mod)) {
		__ret = exchnd_external(pid, mod_list, msg);
		module_put(mod);
	} else {
		pr_warn("exchnd ref not taken.\n");
		__ret = -ENODEV;
	}

	return __ret;
};
#else
static inline int exchnd_call_external_trigger(
		struct module *mod,
		pid_t pid,
		enum exchnd_modules (*mod_list)[EHM_LAST_ELEMENT],
		char *msg)
{
	return exchnd_external(pid, mod_list, msg);
};
#endif /* CONFIG_EXCHND_MODULE */

#endif /* __EXCHND_H__ */
