/******************************************************************************
 *
 * dal_gate: Access controlled interface to Intel TXI via DAL/KDI
 *
 * Author: Philipp Hachtmann (phachtmann@de.adit-jv.com)
 *
 * Copyright (C) 2017 Advanced Driver Information Technology.
 *
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 ******************************************************************************
 *
 * This driver allows userspace applications to use the Intel KDI in kernel
 * interface to the DAL (Dynamic Application Loader).
 *
 * It enforces the convention to pass user and group IDs to the trusted
 * application during session creation. This enables the trusted application
 * code to take decisions based on the current Linux user's identity.
 *
 ******************************************************************************/

#ifndef DAL_GATE_UAPI_H
#define DAL_GATE_UAPI_H

#include <asm/ioctl.h>
#include <linux/types.h>

/*
 * defines used for DALG_CONNECT_TA_EXT ioctl
 */

/* if ta data is already cached it will be used */
#define EXT_FLAG__MAY_USE_CACHE		(1 << 0)
/* don't use cached ta data */
#define EXT_FLAG__DONT_USE_CACHE	(0 << 0)
/* after successful operation the ta data may be stored to cache */
#define EXT_FLAG__MAY_UPDATE_CACHE	(1 << 1)
/* don't change the cached ta data */
#define EXT_FLAG__DONT_UPDATE_CACHE	(0 << 1)

#define EXT_FLAGS_DEFAULT	(EXT_FLAG__MAY_USE_CACHE | \
				 EXT_FLAG__MAY_UPDATE_CACHE)

/* define corresponding masks */
#define EXT_FLAG__USE_CACHE_MASK	(1 << 0)
#define EXT_FLAG__UPDATE_CACHE_MASK	(1 << 1)
#define EXT_FLAGS_MASK		(EXT_FLAG__USE_CACHE_MASK | \
				 EXT_FLAG__UPDATE_CACHE_MASK)

/*
 * Data types used for the ioctl functions
 *
 */

struct dal_gate_sess_args {
	const char *ta_id;	/* ID of the trusted application to use */
	const size_t ta_id_len;
	gid_t my_gid;		/* GID to be communicated to TA		*/
				/* --> Kernel driver checks your membership! */
};

struct dal_gate_sess_ext_args {
	struct dal_gate_sess_args sess_args; /* generic session arguments */
	uint32_t ext_flags;     /* additional flags to control e.g. caching */
};

struct dal_gate_cmd {
	uint32_t cmd;	       /* The command code that is sent to the TA */

	void __user *in_data;  /* Data buffer sent to the TA	*/
	size_t in_data_len;    /* Number of bytes in the buffer */

	void __user *out_data; /* Buffer to be filled by the TA */
	size_t out_data_len;   /* This length will be adjusted to actual */
			       /* amount of bytes received from the TA!	 */
	int response_code;     /* Response from the TA */
};

/*
 *    IOCTL definitions
 */

#define DG_IOCTL 177

#define DALG_CONNECT_TA	    _IOW(DG_IOCTL, 1, struct dal_gate_sess_args)
#define DALG_CONNECT_TA_EXT _IOW(DG_IOCTL, 1, struct dal_gate_sess_ext_args)
#define DALG_CLOSE_TA	    _IO(DG_IOCTL, 2)
#define DALG_TA_CMD	    _IOWR(DG_IOCTL, 3, struct dal_gate_cmd)

#endif
