/*
 * uapi/linux/errmem.h
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 */

#include <asm/bitsperlong.h>

#ifndef _UAPI___ERRMEM_H__
#define _UAPI___ERRMEM_H__

#define ERRMEM_VERSION          0x302 /* v3.02 */
#define ERRMEM_MAGIC_IOC        0xee

enum errmem_ioctls {
	_IOCTL_ERRMEM_VERSION = 0,
	_IOCTL_ERRMEM_GET_NUMBER_OF_SLOTS,
	_IOCTL_ERRMEM_GET_SLOTSIZE,
	_IOCTL_ERRMEM_GET_USED_SLOTS,
	_IOCTL_ERRMEM_ACKNOWLEDGE,
	_IOCTL_ERRMEM_FLUSH,
	_IOCTL_ERRMEM_GET_WATERMARK_LOW,
	_IOCTL_ERRMEM_SET_WATERMARK_LOW,
	_IOCTL_ERRMEM_GET_WATERMARK_HIGH,
	_IOCTL_ERRMEM_SET_WATERMARK_HIGH,
	_IOCTL_ERRMEM_GET_KERNEL_MSG_LEVEL,
	_IOCTL_ERRMEM_SET_KERNEL_MSG_LEVEL,
	_IOCTL_ERRMEM_WRITE,
	_IOCTL_ERRMEM_READ,
	_IOCTL_ERRMEM_SET_READ_SIZE,
	_IOCTL_ERRMEM_SET_EPOCH_MS,
	_IOCTL_ERRMEM_COUNT
};

#define IOCTL_ERRMEM_COUNT _IOCTL_ERRMEM_COUNT

#define IOCTL_ERRMEM_VERSION                                              \
	_IOR(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_VERSION,  unsigned long)
#define IOCTL_ERRMEM_GET_NUMBER_OF_SLOTS                                  \
	_IOR(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_GET_NUMBER_OF_SLOTS,  unsigned int)
#define IOCTL_ERRMEM_GET_SLOTSIZE                                         \
	_IOR(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_GET_SLOTSIZE,  unsigned int)
#define IOCTL_ERRMEM_GET_USED_SLOTS                                       \
	_IOR(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_GET_USED_SLOTS,  unsigned int)
#define IOCTL_ERRMEM_ACKNOWLEDGE                                          \
	_IOW(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_ACKNOWLEDGE,  unsigned int)
#define IOCTL_ERRMEM_FLUSH                                                \
	_IO(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_FLUSH)
#define IOCTL_ERRMEM_GET_WATERMARK_LOW                                    \
	_IOR(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_GET_WATERMARK_LOW,  unsigned int)
#define IOCTL_ERRMEM_SET_WATERMARK_LOW                                    \
	_IOW(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_SET_WATERMARK_LOW,  unsigned int)
#define IOCTL_ERRMEM_GET_WATERMARK_HIGH                                   \
	_IOR(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_GET_WATERMARK_HIGH,  unsigned int)
#define IOCTL_ERRMEM_SET_WATERMARK_HIGH                                   \
	_IOW(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_SET_WATERMARK_HIGH,  unsigned int)
#define IOCTL_ERRMEM_GET_KERNEL_MSG_LEVEL                                 \
	_IOR(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_GET_KERNEL_MSG_LEVEL, unsigned int)
#define IOCTL_ERRMEM_SET_KERNEL_MSG_LEVEL                                 \
	_IOW(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_SET_KERNEL_MSG_LEVEL, unsigned int)
#define IOCTL_ERRMEM_WRITE                                                \
	_IOW(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_WRITE, struct errmem_message)
#define IOCTL_ERRMEM_READ                                                 \
	_IOR(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_READ, struct errmem_message)
#define IOCTL_ERRMEM_SET_READ_SIZE                                        \
	_IOW(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_SET_READ_SIZE, unsigned int)
#define IOCTL_ERRMEM_SET_EPOCH_MS                                         \
	_IOW(ERRMEM_MAGIC_IOC, _IOCTL_ERRMEM_SET_EPOCH_MS, unsigned long long)

/* Message for type declaration, actual size can be different */
#define ERRMEM_MAX_ENTRY_LENGTH 256

/* Flag definitions */
#define ERRMEM_DROPPED_BIT		0
#define ERRMEM_BUSY_BIT			1
#define ERRMEM_INCOMPLETE_BIT		2
#define ERRMEM_OUTDATED_BIT		3
#define ERRMEM_CRC_ERROR_BIT		4
#define ERRMEM_RESTART_BIT		5
#define ERRMEM_INITIALIZED_BIT		6
#define ERRMEM_READ_BIT			7 /* Deprecated */
#define ERRMEM_ALLOC_FAIL_BIT		8 /* Deprecated */

/* Kernel reserved area for extensions and internals only */
#define ERRMEM_RESERVED_START		9
#define ERRMEM_RESERVED_END		15
#define ERRMEM_VM_START			16
#define ERRMEM_VM_END			23

/* USER SPACE reserved area ... For backward compatibility */
#define ERRMEM_USER_RESERVED_START	24
#define ERRMEM_USER_RESERVED_END	31

#define ERRMEM_KERNEL_RESERVED_MASK					\
	(((~0UL) << (ERRMEM_RESERVED_START)) &				\
	 (~0UL >> (__BITS_PER_LONG - 1 - (ERRMEM_RESERVED_END))))

#define ERRMEM_VM_MASK							\
	(((~0UL) << (ERRMEM_VM_START)) &				\
	 (~0UL >> (__BITS_PER_LONG - 1 - (ERRMEM_VM_END))))

#define ERRMEM_USER_RESERVED_MASK					\
	(((~0UL) << (ERRMEM_USER_RESERVED_START)) &			\
	 (~0UL >> (__BITS_PER_LONG - 1 - (ERRMEM_USER_RESERVED_END))))

/* Errmem status flags of messages read */
#define ERRMEM_FLAG_OK			0UL
#define ERRMEM_FLAG_DROPPED		(1UL << (ERRMEM_DROPPED_BIT))
#define ERRMEM_FLAG_BUSY		(1UL << (ERRMEM_BUSY_BIT))
#define ERRMEM_FLAG_INCOMPLETE		(1UL << (ERRMEM_INCOMPLETE_BIT))
#define ERRMEM_FLAG_OUTDATED		(1UL << (ERRMEM_OUTDATED_BIT))
#define ERRMEM_FLAG_CRC_ERROR		(1UL << (ERRMEM_CRC_ERROR_BIT))

#define ERRMEM_FLAG_ANY_ERROR						\
	(((~0UL) << (ERRMEM_DROPPED_BIT)) &				\
	 (~0UL >> (__BITS_PER_LONG - 1 - (ERRMEM_CRC_ERROR_BIT))))

#define ERRMEM_FLAG_RESTART		(1UL << (ERRMEM_RESTART_BIT))
#define ERRMEM_FLAG_INITIALIZED		(1UL << (ERRMEM_INITIALIZED_BIT))
#define ERRMEM_FLAG_ALLOC_FAIL		(1UL << (ERRMEM_ALLOC_FAIL_BIT))

#define ERRMEM_ALL_FLAGS						\
	(((~0UL) << (ERRMEM_DROPPED_BIT)) &				\
	 (~0UL >> (__BITS_PER_LONG - 1 - (ERRMEM_FLAG_ALLOC_FAIL_BIT))))

#define ERRMEM_MASK_VM(flags) ((flags) & ERRMEM_VM_MASK)
#define ERRMEM_GET_VM(flags) (ERRMEM_MASK_VM(flags) >> ERRMEM_VM_START)
#define ERRMEM_SET_VM(flags, vm) ((flags) |= ((vm) << ERRMEM_VM_START))

/*
 * Errmem message types
 */
enum errmem_entry_type {
	ERRMEM_TYPE_TRACE = 0,
	ERRMEM_TYPE_ASCII
};

/*
 * Errmem internal fields, used between driver and daemon
 */
struct errmem_internal {
	unsigned int seqnum;
	unsigned int local_clock[2];
	unsigned int offset;
	unsigned int flags;
	unsigned int crc;
	unsigned int storage;
};

/*
 * Messages read with IOCTL_ERRMEM_READ
 * message[] can be of arbitrary length
 */
struct errmem_message {
	enum errmem_entry_type type;
	struct errmem_internal internal;

	unsigned short length;
	unsigned char message[ERRMEM_MAX_ENTRY_LENGTH];
};

#endif /* _UAPI__ERRMEM_H__ */
