#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/highmem.h>
#include <linux/page_ext.h>
#include <linux/poison.h>
#include <linux/ratelimit.h>
#include <linux/module.h>

#include <linux/slab.h>

static bool __page_poisoning_enabled __read_mostly;
static bool want_page_poisoning __read_mostly;

static int early_page_poison_param(char *buf)
{
	if (!buf)
		return -EINVAL;
	return strtobool(buf, &want_page_poisoning);
}
early_param("page_poison", early_page_poison_param);

bool page_poisoning_enabled(void)
{
	return __page_poisoning_enabled;
}

static bool need_page_poisoning(void)
{
	return want_page_poisoning;
}

static void init_page_poisoning(void)
{
	/*
	 * page poisoning is debug page alloc for some arches. If either
	 * of those options are enabled, enable poisoning
	 */
	if (!IS_ENABLED(CONFIG_ARCH_SUPPORTS_DEBUG_PAGEALLOC)) {
		if (!want_page_poisoning && !debug_pagealloc_enabled())
			return;
	} else {
		if (!want_page_poisoning)
			return;
	}

	__page_poisoning_enabled = true;
}

struct page_ext_operations page_poisoning_ops = {
	.need = need_page_poisoning,
	.init = init_page_poisoning,
};

static inline void set_page_poison(struct page *page)
{
	struct page_ext *page_ext;

	page_ext = lookup_page_ext(page);
	if (unlikely(!page_ext))
		return;

	__set_bit(PAGE_EXT_DEBUG_POISON, &page_ext->flags);
}

static inline void clear_page_poison(struct page *page)
{
	struct page_ext *page_ext;

	page_ext = lookup_page_ext(page);
	if (unlikely(!page_ext))
		return;

	__clear_bit(PAGE_EXT_DEBUG_POISON, &page_ext->flags);
}

bool page_is_poisoned(struct page *page)
{
	struct page_ext *page_ext;

	page_ext = lookup_page_ext(page);
	if (unlikely(!page_ext))
		return false;

	return test_bit(PAGE_EXT_DEBUG_POISON, &page_ext->flags);
}

/*
 * poisoning stack trace
 * Copyright DENSO corporation
 * rev 1. 30-Aug-2018, initial coding
 * rev 2. 29-Sep-2018, increase stack num, until log size is power of 2
 * 	add information
 */
#define POISON_STACK_NUM	(4 * 2)
#define DEFINE_MODULE_PARAM(param, type, value)	\
	static type param = value;	\
	module_param(param, type, S_IRUGO | S_IWUSR | S_IWGRP);	\
	MODULE_PARM_DESC(param, #param)

DEFINE_MODULE_PARAM(poison_stack_available, int, 0);
DEFINE_MODULE_PARAM(poison_stack_skip, int, 5);

struct poison_stack {
	uint32_t	entries[POISON_STACK_NUM];
};
struct poison_log {
	struct list_head	list;
	void			*addr;
	char			n, comm[7];
	struct poison_stack	stack;
};
static LIST_HEAD(poison_log);
static LIST_HEAD(poison_log_pool);
DEFINE_SPINLOCK(poison_log_lock);

static void poison_page(struct page *page, unsigned long *trace_entries,
		int n)
{
	void *addr = kmap_atomic(page);

	set_page_poison(page);
	memset(addr, PAGE_POISON, PAGE_SIZE);
	if (unlikely(trace_entries)) {
		struct poison_stack	*dest;
		struct poison_log	*log, *alloc;
		unsigned long	flags;

		dest = addr + PAGE_SIZE;
		dest--;
		if (likely(sizeof(*trace_entries) ==
					sizeof(dest->entries))) {
			*dest = *(typeof(dest))trace_entries;
		} else {
			int	i;

			for (i = 0; i < POISON_STACK_NUM; i++) {
				dest->entries[i] = trace_entries[i];
			}
		}

		alloc = kmalloc(sizeof(*log), GFP_ATOMIC);

		spin_lock_irqsave(&poison_log_lock, flags);
		if (list_empty(&poison_log_pool)) {
			log = alloc;
			alloc = NULL;
		} else {
			log = list_first_entry(&poison_log_pool,
					typeof(*log), list);
			list_del(&log->list);
		}
		if (log) {
			log->addr = page_address(page);
			log->stack = *dest;
			log->n = (char)n;
			if (current) {
				memcpy(log->comm, current->comm,
						sizeof(log->comm));
			} else {
				memset(log->comm, ' ', sizeof(log->comm));
			}

			list_add(&log->list, &poison_log);
		}
		spin_unlock_irqrestore(&poison_log_lock, flags);
		if (alloc) {
#define DEBUG_POISON_STACK_REUSE
#undef  DEBUG_POISON_STACK_REUSE
#ifdef  DEBUG_POISON_STACK_REUSE
			static int	debug = 5;
			if (debug) {
				debug--;
				printk(KERN_INFO
					"%s reuse poison_log size=%ld\n",
						__func__,
						sizeof(*alloc));
			}
#endif
			kfree(alloc);
		}
	}
	kunmap_atomic(addr);
}

static void poison_pages(struct page *page, int n)
{
	int i;

	if (unlikely(poison_stack_available)) {
		unsigned long	trace_entries[POISON_STACK_NUM];
		struct stack_trace	trace = {
			.nr_entries = 0,
			.entries = trace_entries,
			.max_entries = ARRAY_SIZE(trace_entries),
			.skip = poison_stack_skip,
		};

		save_stack_trace(&trace);
		for (i = 0; i < n; i++) {
			poison_page(page + i, trace_entries, n);
		}
		return;
	}

	for (i = 0; i < n; i++)
		poison_page(page + i, NULL, n);
}

static bool single_bit_flip(unsigned char a, unsigned char b)
{
	unsigned char error = a ^ b;

	return error && !(error & (error - 1));
}

static int check_poison_mem(unsigned char *mem, size_t bytes)
{
	static DEFINE_RATELIMIT_STATE(ratelimit, 5 * HZ, 10);
	unsigned char *start;
	unsigned char *end;
	const char	*msg;
	unsigned long	length;

	if (IS_ENABLED(CONFIG_PAGE_POISONING_NO_SANITY))
		return 0;

	if (poison_stack_available) {
		bytes -= sizeof(struct poison_stack);
	}
	start = memchr_inv(mem, PAGE_POISON, bytes);
	if (!start)
		return 0;

	for (end = mem + bytes - 1; end > start; end--) {
		if (*end != PAGE_POISON)
			break;
	}

	if (!__ratelimit(&ratelimit))
		return 2;
	else if (start == end && single_bit_flip(*start, PAGE_POISON))
		msg = "pagealloc: single bit error\n";
	else
		msg = "pagealloc: memory corruption\n";

	pr_err("%s %s start=%p:%#lx len=%ld\n", __func__, msg,
			start, __pa(start), end - start);
	length = start - mem + 0x20;
	if (length > 256) {
		length = 256;
	}
	print_hex_dump(KERN_ERR, "", DUMP_PREFIX_ADDRESS, 16,
			sizeof(unsigned long), mem,
			length, 1);
	return 1;
}

static void unpoison_page(struct page *page)
{
	void *addr;
	int	ret;
	struct poison_log	*find, logimage;

	if (!page_is_poisoned(page))
		return;

	addr = kmap_atomic(page);

	find = NULL;
	if (unlikely(poison_stack_available)) {
		struct poison_log	*log;
		unsigned long		flags;

		spin_lock_irqsave(&poison_log_lock, flags);
		list_for_each_entry(log, &poison_log, list) {
			if (log->addr == addr) {
				find = log;

				list_del(&log->list);
				logimage = *log;
				list_add(&log->list, &poison_log_pool);
				break;
			}
		}
		spin_unlock_irqrestore(&poison_log_lock, flags);

		if (!find) {
			printk(KERN_ERR "BUG %s not found page addr=%pK:%#lx\n",
					__func__, addr, __pa(addr));
		}
	}

	if (unlikely(ret = check_poison_mem(addr, PAGE_SIZE) && find)) {
		int	i;

		pr_err("%s %dpages (%.*s) addr=%pK:%#lx\n", __func__,
				logimage.n,
				(int)sizeof(logimage.comm),
				logimage.comm,
				addr, __pa(addr) );
		for (i = 0; i < ARRAY_SIZE(logimage.stack.entries); i++) {
			unsigned long	caller;

			caller = (typeof(caller))logimage.stack.entries[i] |
				(ULONG_MAX ^ U32_MAX);
			pr_err("%s %#x %pS\n", __func__,
					logimage.stack.entries[i],
					(void*)caller);
		}
	};


	clear_page_poison(page);
	kunmap_atomic(addr);
}

static void unpoison_pages(struct page *page, int n)
{
	int i;

	for (i = 0; i < n; i++)
		unpoison_page(page + i);
}

void kernel_poison_pages(struct page *page, int numpages, int enable)
{
	if (!page_poisoning_enabled())
		return;

	if (enable)
		unpoison_pages(page, numpages);
	else
		poison_pages(page, numpages);
}

#ifndef CONFIG_ARCH_SUPPORTS_DEBUG_PAGEALLOC
void __kernel_map_pages(struct page *page, int numpages, int enable)
{
	/* This function does nothing, all work is done via poison pages */
}
#endif
