/*******************************************************************************
*    Copyright (c) 2013 Neusoft Co., Ltd.
*    FileName	:	bvd_if_dev.c
*    Module		:	bvd driver
*    Function	:	bvd driver module use for Neusoft Bluedragon
*					Bluetooth stack.
*-------------------------------------------------------------------------------
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc..
*******************************************************************************/

#include <linux/types.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <linux/errno.h>
#include <linux/poll.h>
#include <linux/mutex.h>

#include "bvd_if_dev.h"
#include "bvd_device.h"
#include "bvd_net_dev.h"
#include "bvd_tty_dev.h"
#ifdef __ALPS_FILE_ID__
#undef __ALPS_FILE_ID__
#endif	/* ifdef __ALPS_FILE_ID__ */
#define __ALPS_FILE_ID__    0x8118

/* bvd if dev */
#define BVD_IF_CDEV_CLASS_NAME		"bvd_if_cdev"
#define BVD_IF_VERSION				"1.8"
#define BVD_IF_CDEV_NUM				(1)

/*modify to 4096 for SPP 4K(tty device), 20150525*/
#define BVD_IF_DEV_CHUNK_DEFAULT	(4096)	/*2048*/
#define BVD_IF_DEV_CHUNK_MIN		(1024)

struct bvd_if_dev {
	struct mutex	mutex;
	unsigned char	*write_chunk;
	int				write_chunksize;
	unsigned char	*read_chunk;
	int				read_chunksize;
	int				type;
	void			*private_data;
};

static struct cdev	bvd_if_dev_driver;
static int			bvd_if_dev_driver_major;
static struct class *bvd_if_dev_driver_class;

/* bvd devcie */
static DEFINE_MUTEX(bvd_device_mutex);

static const
struct bvd_device_operations *bvd_device_ldiscs[BVD_DEVICE_TYPE_MAX];

static char version[] =
	BVD_IF_CDEV_CLASS_NAME ":v" BVD_IF_VERSION "\n";

/* bvd device ldiscs operation */

/* bvd devcie */
int bvd_device_register(int type, const struct bvd_device_operations *new_ops)
{
	if (type >= BVD_DEVICE_TYPE_MAX)
		return -EINVAL;

	if (!new_ops->create || !new_ops->destroy ||
		!new_ops->get_status || !new_ops->ioctl ||
		!new_ops->poll || !new_ops->read ||
		!new_ops->write) {
		return -EINVAL;
	}

	mutex_lock(&bvd_device_mutex);
	bvd_device_ldiscs[type] = new_ops;
	mutex_unlock(&bvd_device_mutex);

	return 0;
}

int bvd_device_unregister(int type)
{
	if (type >= BVD_DEVICE_TYPE_MAX)
		return -EINVAL;

	mutex_lock(&bvd_device_mutex);
	bvd_device_ldiscs[type] = NULL;
	mutex_unlock(&bvd_device_mutex);

	return 0;
}

static int bvd_device_read(struct bvd_if_dev *dev, char *buf, int count)
{
	int len = -EINVAL;

	if (!dev->private_data)
		return -ENODEV;

	mutex_lock(&bvd_device_mutex);
	if (bvd_device_ldiscs[dev->type]) {
		BUG_ON(bvd_device_ldiscs[dev->type]->read == NULL);
		len = bvd_device_ldiscs[dev->type]->read(dev->private_data, buf,
				count);
	}
	mutex_unlock(&bvd_device_mutex);

	return len;
}

static int bvd_device_write(struct bvd_if_dev *dev, const char *buf, int count)
{
	int len = -EINVAL;

	if (!dev->private_data)
		return -ENODEV;

	mutex_lock(&bvd_device_mutex);
	if (bvd_device_ldiscs[dev->type]) {
		BUG_ON(bvd_device_ldiscs[dev->type]->write == NULL);
		len = bvd_device_ldiscs[dev->type]->write(dev->private_data,
				buf, count);
	}
	mutex_unlock(&bvd_device_mutex);

	return len;
}

static unsigned int bvd_device_poll(struct bvd_if_dev *dev,
		struct file *file, struct poll_table_struct *wait)
{
	unsigned int mask = 0;

	if (!dev->private_data)
		return mask;

	mutex_lock(&bvd_device_mutex);
	if (bvd_device_ldiscs[dev->type]) {
		BUG_ON(bvd_device_ldiscs[dev->type]->poll == NULL);
		mask = bvd_device_ldiscs[dev->type]->poll(dev->private_data,
				file, wait);
	}
	mutex_unlock(&bvd_device_mutex);

	return mask;
}

static long bvd_device_ioctl(struct bvd_if_dev *dev, unsigned int cmd,
							 unsigned long arg)
{
	int ret = -EINVAL;

	if (!dev->private_data)
		return -ENODEV;

	mutex_lock(&bvd_device_mutex);
	if (bvd_device_ldiscs[dev->type]) {
		BUG_ON(bvd_device_ldiscs[dev->type]->ioctl == NULL);
		ret = bvd_device_ldiscs[dev->type]->ioctl(dev->private_data,
				cmd, arg);
	}
	mutex_unlock(&bvd_device_mutex);

	return ret;
}

static void *bvd_device_create(int type, void __user *arg)
{
	void *ret = NULL;

	mutex_lock(&bvd_device_mutex);
	if (bvd_device_ldiscs[type]) {
		BUG_ON(bvd_device_ldiscs[type]->create == NULL);
		ret = bvd_device_ldiscs[type]->create(arg);
	}
	mutex_unlock(&bvd_device_mutex);

	return ret;
}

static int bvd_device_destroy(struct bvd_if_dev *dev, unsigned long force)
{
	int ret = -EINVAL;

	if (!dev->private_data) {
		BVD_DBG("private_data is null, force %ld\n", force);
		return -ENODEV;
	}

	mutex_lock(&bvd_device_mutex);
	if (bvd_device_ldiscs[dev->type]) {
		BUG_ON(bvd_device_ldiscs[dev->type]->destroy == NULL);
		ret = bvd_device_ldiscs[dev->type]->destroy
				  (dev->private_data, force);
	}
	mutex_unlock(&bvd_device_mutex);
	BVD_DBG("ret:%d\n", ret);
	return ret;
}

static int bvd_device_get_status(struct bvd_if_dev *dev)
{
	int ret = -EINVAL;

	if (!dev->private_data)
		return -ENODEV;

	mutex_lock(&bvd_device_mutex);
	if (bvd_device_ldiscs[dev->type]) {
		ret = bvd_device_ldiscs[dev->type]->get_status(
			dev->private_data);
	}
	mutex_unlock(&bvd_device_mutex);

	return ret;
}

/* bvd if dev */
static int bvd_if_dev_chunkalloc(unsigned char **chunk, int count)
{
	unsigned char *buf_chunk;

	if (count < BVD_IF_DEV_CHUNK_MIN)
		count = BVD_IF_DEV_CHUNK_MIN;

	buf_chunk = kmalloc(count, GFP_KERNEL);

	if (!buf_chunk) {
		BVD_DBG("kmalloc failed, count %d\n", count);
		return -ENOMEM;
	}
	kfree(*chunk);
	*chunk = buf_chunk;

	return 0;
}

static ssize_t bvd_if_dev_read(struct file *file, char __user *buf,
		size_t count, loff_t *f_pos)
{
	struct bvd_if_dev *if_dev = (struct bvd_if_dev *)file->private_data;
	unsigned int chunksize = BVD_IF_DEV_CHUNK_DEFAULT;
	int len;

	if (!if_dev)
		return -ENODEV;

	mutex_lock(&if_dev->mutex);
	if (count < chunksize)
		chunksize = count;
	else
		count = chunksize;

	if (if_dev->read_chunksize < chunksize) {
		len = bvd_if_dev_chunkalloc(&if_dev->read_chunk, chunksize);
		if (len < 0)
			goto out;

		if_dev->read_chunksize = chunksize;
	}

	len = bvd_device_read(if_dev, if_dev->read_chunk, count);
	if (len > 0) {
		if (copy_to_user(buf, if_dev->read_chunk, len))
			len = -EFAULT;
	}

out:
	mutex_unlock(&if_dev->mutex);

	return len;
}

static ssize_t bvd_if_dev_write(struct file *file, const char __user *buf,
		size_t count, loff_t *f_pos)
{
	struct bvd_if_dev *if_dev = (struct bvd_if_dev *)file->private_data;
	unsigned int chunksize = BVD_IF_DEV_CHUNK_DEFAULT;
	int len;

	if (!if_dev)
		return -ENODEV;

	mutex_lock(&if_dev->mutex);
	if (count < chunksize)
		chunksize = count;
	else
		count = chunksize;

	if (if_dev->write_chunksize < chunksize) {
		len = bvd_if_dev_chunkalloc(&if_dev->write_chunk, chunksize);
		if (len < 0)
			goto out;

		if_dev->write_chunksize = chunksize;
	}

	if (copy_from_user(if_dev->write_chunk, buf, count)) {
		len = -EFAULT;
		goto out;
	}
	len = bvd_device_write(if_dev, if_dev->write_chunk, count);

out:
	mutex_unlock(&if_dev->mutex);

	return len;
}

static unsigned int bvd_if_dev_poll(struct file *file,
		struct poll_table_struct *wait)
{
	struct bvd_if_dev *if_dev = (struct bvd_if_dev *)file->private_data;
	unsigned int mask	= 0;

	if (!if_dev)
		return mask;

	mutex_lock(&if_dev->mutex);
	mask = bvd_device_poll(if_dev, file, wait);
	mutex_unlock(&if_dev->mutex);

	return mask;
}

static int bvd_if_dev_open(struct inode *inode, struct file *file)
{
	struct bvd_if_dev *if_dev;

	if_dev = kzalloc(sizeof(struct bvd_if_dev), GFP_KERNEL);
	if (!if_dev) {
		BVD_DBG("kzalloc failed, size %d\n", sizeof(struct bvd_if_dev));
		return -ENOMEM;
	}
	mutex_init(&if_dev->mutex);
	file->private_data = if_dev;

	return 0;
}

static int bvd_if_dev_release(struct inode *inode, struct file *file)
{
	struct bvd_if_dev *if_dev = (struct bvd_if_dev *)file->private_data;

	BVD_DBG("if dev release begin\n");
	mutex_lock(&if_dev->mutex);
	bvd_device_destroy(if_dev, 1);
	file->private_data = NULL;
	kfree(if_dev->read_chunk);
	kfree(if_dev->write_chunk);
	mutex_unlock(&if_dev->mutex);

	kfree(if_dev);
	BVD_DBG("if dev release end\n");
	return 0;
}

static int bvd_if_dev_create_device(struct bvd_if_dev *if_dev,
		struct bvd_device_create_req_header header, void __user *arg)
{
	void *device;

	if (header.type >= BVD_DEVICE_TYPE_MAX)
		return -EINVAL;

	device = bvd_device_create(header.type, arg);
	if (device) {
		if_dev->type			= header.type;
		if_dev->private_data	= device;
	} else {
		return -EINVAL;
	}

	return 0;
}

static long bvd_if_dev_ioctl(struct file *file, unsigned int cmd,
							 unsigned long arg)
{
	struct bvd_if_dev *if_dev = (struct bvd_if_dev *)file->private_data;
	int ret		= 0;
	int status;
	struct bvd_device_create_req_header header;

	if (!if_dev)
		return -ENODEV;

	mutex_lock(&if_dev->mutex);

	switch (cmd) {
	case BVD_IOCTL_CREATE_DEVICE:
	{
		ret = copy_from_user(&header, (void __user *)arg,
				sizeof(struct bvd_device_create_req_header));
		if (!ret) {
			BVD_DBG("create type %d\n", header.type);
			ret = bvd_if_dev_create_device(if_dev, header,
					(void __user *)(arg + sizeof(
					struct bvd_device_create_req_header)));
			BVD_DBG("bvd_if_dev_create_device ret: %d\n", ret);
		}
		break;
	}
	case BVD_IOCTL_RELEASE_DEVICE:
	{
		ret = bvd_device_destroy(if_dev, arg);
		if (!ret) {
			if_dev->type			= BVD_DEVICE_TYPE_MAX;
			if_dev->private_data	= NULL;
		}
		break;
	}
	case BVD_IOCTL_GET_DEVICE_STATUS:
	{
		status	= bvd_device_get_status(if_dev);
		ret		= put_user(status, (int __user *)arg);
		break;
	}
	default:
	{
		ret = bvd_device_ioctl(if_dev, cmd, arg);
		break;
	}
	}

	mutex_unlock(&if_dev->mutex);

	return ret;
}

static const struct file_operations bvd_if_dev_fops = {
	.owner			= THIS_MODULE,
	.read			= bvd_if_dev_read,
	.write			= bvd_if_dev_write,
	.poll			= bvd_if_dev_poll,
	.open			= bvd_if_dev_open,
	.release		= bvd_if_dev_release,
	.unlocked_ioctl = bvd_if_dev_ioctl,
};

static int __init bvd_if_init(void)
{
	int		ret;
	dev_t	dev;

	ret = alloc_chrdev_region(&dev, 0, BVD_IF_CDEV_NUM, "bvd_if_cdev");
	if (ret < 0)
		return ret;

	bvd_if_dev_driver_major = MAJOR(dev);
	bvd_if_dev_driver_class = class_create(THIS_MODULE,
			BVD_IF_CDEV_CLASS_NAME);

	cdev_init(&bvd_if_dev_driver, &bvd_if_dev_fops);

	bvd_if_dev_driver.owner = THIS_MODULE;
	bvd_if_dev_driver.ops	= &bvd_if_dev_fops;

	ret = cdev_add(&bvd_if_dev_driver, dev, 1);
	if (ret < 0)
		goto fail_cdev;

	device_create(bvd_if_dev_driver_class, NULL,
				  dev, NULL, "%s", "bvd_if_cdev");

	ret = bvd_net_dev_init();
	if (ret)
		goto fail_net;

	ret = bvd_tty_init();
	if (ret)
		goto fail_tty;

	pr_info("%s", version);

	return 0;

fail_tty:
	bvd_net_dev_exit();
fail_net:
	device_destroy(bvd_if_dev_driver_class,
				   MKDEV(bvd_if_dev_driver_major, 0));
	cdev_del(&bvd_if_dev_driver);
fail_cdev:
	unregister_chrdev_region(dev, BVD_IF_CDEV_NUM);
	class_destroy(bvd_if_dev_driver_class);
	return ret;
}

static void __exit bvd_if_exit(void)
{
	bvd_tty_exit();
	bvd_net_dev_exit();
	cdev_del(&bvd_if_dev_driver);
	device_destroy(bvd_if_dev_driver_class,
				   MKDEV(bvd_if_dev_driver_major, 0));
	unregister_chrdev_region(MKDEV(bvd_if_dev_driver_major, 0),
							 BVD_IF_CDEV_NUM);
	class_destroy(bvd_if_dev_driver_class);
}

module_init(bvd_if_init);

module_exit(bvd_if_exit);
MODULE_DESCRIPTION("Bluedragon Virtual Driver");
MODULE_AUTHOR("Neusoft");
MODULE_LICENSE("GPL");
