/*******************************************************************************
*    Copyright (c) 2013 Neusoft Co., Ltd.
*    FileName	:	bvd_net_dev.c
*    Module		:	bvd driver
*    Function	:	bvd driver module use for Neusoft Bluedragon
*					Bluetooth stack.
*-------------------------------------------------------------------------------
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc..
*******************************************************************************/

#include <linux/init.h>
#include <linux/etherdevice.h>
#include <linux/crc32.h>
#include <linux/types.h>
#include <linux/poll.h>
#include <linux/fs.h>
#include <linux/device.h>
#include <linux/slab.h>
#include <linux/wait.h>
#include <linux/uaccess.h>
#include <linux/sched.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/spinlock.h>
#include <linux/version.h>
#include "bvd_net_dev.h"
#include "bvd_device.h"
#ifdef __ALPS_FILE_ID__
#undef __ALPS_FILE_ID__
#endif	/* ifdef __ALPS_FILE_ID__ */
#define __ALPS_FILE_ID__    0x8119

static LIST_HEAD(bvd_net_dev_list);
static DEFINE_MUTEX(bvd_net_dev_list_lock);

#define BVD_NET_DEV_TX_QUEUE_LEN			20
#define BVD_NET_DEV_ADDR_MULTICAST_BIT		1
#define BVD_NET_DEV_ETHERNET_TYPE_START		0x0600
#define BVD_NET_DEV_ETHERNET_TYPE_END		0xffff
#define BVD_NET_DEV_MC_ADDR_MAX_INDEX		5
#define BVD_NET_DEV_MC_ADDR_START			0x01

/* broadcast addr */
static const bvd_net_dev_addr bc_addr = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

struct bvd_net_dev_mc_info_node {
	struct list_head			list;
	struct bvd_net_dev_mc_info	mc_info;
};

struct bvd_net_dev {
	struct list_head				list;
	unsigned long long				mc_filter;
	struct list_head				mc_list;
	struct bvd_net_dev_proto_filter proto_filter;
	struct sk_buff_head				sk_write_queue;
	unsigned int					flages;
	struct net_device				*dev;
	spinlock_t						lock;
	wait_queue_head_t				read_wq;
	wait_queue_head_t				status_wq;
};

static int bvd_net_dev_open(struct net_device *dev)
{
	BVD_DBG("\n");

	netif_start_queue(dev);

	return 0;
}

static int bvd_net_dev_close(struct net_device *dev)
{
	BVD_DBG("\n");

	netif_stop_queue(dev);

	return 0;
}

static inline int bvd_net_dev_mc_hash(__u8 *addr)
{
	return crc32_be(~0, addr, ETH_ALEN) >> 26;
}

#ifdef CONFIG_BVD_NET_DEV_MC_FILTER
static inline int bvd_net_dev_mc_filter(struct bvd_net_dev *netdev,
		struct sk_buff *skb)
{
	struct ethhdr *eh = (void *)skb->data;

	if ((eh->h_dest[0] & BVD_NET_DEV_ADDR_MULTICAST_BIT) &&
		 !test_bit(bvd_net_dev_mc_hash(eh->h_dest),
				   (ulong *)&netdev->mc_filter)) {
		return 1;
	}

	BVD_DBG("%u,%u,%u,%u,%u,%u\n",
			eh->h_dest[0], eh->h_dest[1], eh->h_dest[2],
			eh->h_dest[3], eh->h_dest[4], eh->h_dest[5]);
	return 0;
}

static void bvd_net_dev_set_default_mc_filter(struct bvd_net_dev *netdev)
{
	set_bit(bvd_net_dev_mc_hash(netdev->dev->broadcast),
			(ulong *)&netdev->mc_filter);
}
#endif	/* ifdef CONFIG_BVD_NET_DEV_MC_FILTER */

#ifdef CONFIG_BVD_NET_DEV_PROTO_FILTER
static inline u16 bvd_net_dev_eth_proto(struct sk_buff *skb)
{
	struct ethhdr	*eh		= (void *)skb->data;
	u16				proto	= ntohs(eh->h_proto);

	if (proto >= BVD_NET_DEV_ETHERNET_TYPE_START)
		return proto;

	if (get_unaligned((__be16 *)skb->data) ==
		htons(BVD_NET_DEV_ETHERNET_TYPE_END)) {
		return ETH_P_802_3;
	}

	return ETH_P_802_2;
}

static int bvd_net_dev_proto_filter(struct bvd_net_dev *netdev,
		struct sk_buff *skb)
{
	int index;
	u16 proto = bvd_net_dev_eth_proto(skb);
	struct bvd_net_dev_proto_range	*proto_list =
		netdev->proto_filter.proto_range_list;

	for (index = 0; (index < netdev->proto_filter.count) &&
		 proto_list[index].end; index++) {
		if ((proto >= proto_list[index].start) &&
			 (proto <= proto_list[index].end)) {
			BVD_DBG("proto: %u\n", proto);
			return 0;
		}
	}

	return 1;
}

static void bvd_net_dev_set_default_proto_filter(struct bvd_net_dev *netdev)
{
	struct bvd_net_dev_proto_range *proto_list =
		netdev->proto_filter.proto_range_list;

	netdev->proto_filter.count = 0;

	/* (IPv4, ARP) */
	proto_list[0].start = ETH_P_IP;
	proto_list[0].end	= ETH_P_ARP;
	netdev->proto_filter.count++;

	/* (RARP, AppleTalk) */
	proto_list[1].start = ETH_P_RARP;
	proto_list[1].end	= ETH_P_AARP;
	netdev->proto_filter.count++;

	/* (IPX, IPv6) */
	proto_list[2].start = ETH_P_IPX;
	proto_list[2].end	= ETH_P_IPV6;
	netdev->proto_filter.count++;
}
#endif	/* ifdef CONFIG_BVD_NET_DEV_PROTO_FILTER */

static netdev_tx_t bvd_net_dev_xmit(struct sk_buff *skb,
		struct net_device *dev)
{
	struct bvd_net_dev *netdev = netdev_priv(dev);
	unsigned long flags;
	struct netdev_queue *txq = netdev_get_tx_queue(dev, 0);

	BVD_DBG("\n");

	if (!spin_trylock_irqsave(&netdev->lock, flags))
		return NETDEV_TX_BUSY;

#ifdef CONFIG_BVD_NET_DEV_MC_FILTER
	if (bvd_net_dev_mc_filter(netdev, skb)) {
		dev->stats.tx_dropped++;
		spin_unlock_irqrestore(&netdev->lock, flags);
		kfree_skb(skb);
		return NETDEV_TX_OK;
	}
#endif	/* ifdef CONFIG_BVD_NET_DEV_MC_FILTER */

#ifdef CONFIG_BVD_NET_DEV_PROTO_FILTER
	if (bvd_net_dev_proto_filter(netdev, skb)) {
		dev->stats.tx_dropped++;
		spin_unlock_irqrestore(&netdev->lock, flags);
		kfree_skb(skb);
		return NETDEV_TX_OK;
	}
#endif	/* ifdef CONFIG_BVD_NET_DEV_PROTO_FILTER */

	/*update trans_start of netdev queue 0 instead of dev->trans_start*/
	/*for update kernel version to 4.6(Renesas R-Car BSP)*/
	if (txq && (txq->trans_start != jiffies))
		txq->trans_start = jiffies;
	skb_queue_tail(&netdev->sk_write_queue, skb);

	wake_up_interruptible(&netdev->read_wq);

	if (skb_queue_len(&netdev->sk_write_queue) >=
		BVD_NET_DEV_TX_QUEUE_LEN) {
		BVD_DBG("tx queue is full\n");

		netif_stop_queue(dev);
	}

	spin_unlock_irqrestore(&netdev->lock, flags);

	return NETDEV_TX_OK;
}

static void bvd_net_dev_set_mc_list(struct net_device *dev)
{
#ifdef CONFIG_BVD_NET_DEV_MC_FILTER
	int	mc_count = 0;
	struct bvd_net_dev_mc_info_node *mc;
	struct bvd_net_dev *netdev = netdev_priv(dev);
	struct bvd_net_dev_mc_range mc_range_list[BVD_NET_DEV_MAX_MC_FILTER+2];
	unsigned long flags;

	BVD_DBG("dev name %s, mc_count %d\n", dev->name, netdev_mc_count(dev));

	mc = kmalloc(sizeof(struct bvd_net_dev_mc_info_node), GFP_ATOMIC);
	if (!mc) {
		BVD_DBG("kmalloc failed, size %d\n",
				sizeof(struct bvd_net_dev_mc_info_node));
		return;
	}
	spin_lock_irqsave(&netdev->lock, flags);

	if (dev->flags & (IFF_PROMISC | IFF_ALLMULTI)) {
		u8 start[ETH_ALEN] = { BVD_NET_DEV_MC_ADDR_START };

		BVD_DBG("Request all addresses\n");

		/* Request all addresses */
		memcpy(mc_range_list[0].start,
				start, ETH_ALEN);
		memcpy(mc_range_list[0].end,
				dev->broadcast, ETH_ALEN);

		mc_count = 1;
	} else {
		struct netdev_hw_addr	*ha;
		int						index = 0;

		BVD_DBG("Request Multicast addresses\n");

		if (dev->flags & IFF_BROADCAST) {
			memcpy(mc_range_list[index].start,
				   dev->broadcast, ETH_ALEN);
			memcpy(mc_range_list[index].end,
				   dev->broadcast, ETH_ALEN);

			index++;
		}

		netdev_for_each_mc_addr(ha, dev) {
			if (index == BVD_NET_DEV_MAX_MC_FILTER)
				break;

			memcpy(mc_range_list[index].start,
				   ha->addr, ETH_ALEN);
			memcpy(mc_range_list[index].end,
				   ha->addr, ETH_ALEN);

			index++;
		}

		mc_count += index;
	}

	BVD_DBG("mc_count: %d\n", mc_count);

	mc->mc_info.mc_range_list = kmalloc(mc_count *
			sizeof(struct bvd_net_dev_mc_range), GFP_ATOMIC);
	if (!mc->mc_info.mc_range_list) {
		BVD_DBG("kmalloc failed, size %d\n",
				sizeof(struct bvd_net_dev_mc_range));
		kfree(mc);
		goto out;
	}

	memcpy(mc->mc_info.mc_range_list, mc_range_list,
		   mc_count * sizeof(struct bvd_net_dev_mc_range));

	mc->mc_info.count = mc_count;

	list_add_tail(&mc->list, &netdev->mc_list);

	netdev->flages |= 1 << BVD_NET_DEV_BIT_MC_CHANGED;

	wake_up_interruptible(&netdev->status_wq);

out:
	spin_unlock_irqrestore(&netdev->lock, flags);
#endif	/* ifdef CONFIG_BVD_NET_DEV_MC_FILTER */
}

static int bvd_net_dev_set_mc_addr(struct net_device *dev, void *arg)
{
	BVD_DBG("\n");

	return 0;
}

static void bvd_net_dev_timeout(struct net_device *dev)
{
	BVD_DBG("\n");

	netif_wake_queue(dev);
}

static const struct net_device_ops bvd_net_dev_ops = {
	.ndo_open				= bvd_net_dev_open,
	.ndo_stop				= bvd_net_dev_close,
	.ndo_start_xmit			= bvd_net_dev_xmit,
	.ndo_validate_addr		= eth_validate_addr,
	.ndo_set_rx_mode		= bvd_net_dev_set_mc_list,
	.ndo_set_mac_address	= bvd_net_dev_set_mc_addr,
	.ndo_tx_timeout			= bvd_net_dev_timeout,
	.ndo_change_mtu			= eth_change_mtu,
};

static void bvd_net_dev_setup(struct net_device *dev)
{
	BVD_DBG("\n");

	memcpy(dev->broadcast,
			bc_addr, ETH_ALEN);

	dev->addr_len = ETH_ALEN;

	ether_setup(dev);

	dev->priv_flags		&= ~IFF_TX_SKB_SHARING;
	dev->netdev_ops		= &bvd_net_dev_ops;
	dev->watchdog_timeo = HZ * 2;
}

static struct bvd_net_dev *bvd_net_dev_find(u8 *dst)
{
	struct bvd_net_dev *netdev;

	BVD_DBG("\n");

	list_for_each_entry(netdev, &bvd_net_dev_list, list) {
		unsigned long flags;

		spin_lock_irqsave(&netdev->lock, flags);
#if (LINUX_VERSION_LT(3, 5, 0))
		if (!compare_ether_addr(dst, netdev->dev->dev_addr)) {
#else	/* (LINUX_VERSION >= (3, 5, 0)) */
		if (ether_addr_equal(dst, netdev->dev->dev_addr)) {
#endif	/* (LINUX_VERSION_LT(3, 5, 0)) end*/
			spin_unlock_irqrestore(&netdev->lock, flags);
			return netdev;
		}
		spin_unlock_irqrestore(&netdev->lock, flags);
	}

	return NULL;
}

static void bvd_net_dev_add(struct bvd_net_dev *netdev)
{
	BVD_DBG("\n");

	list_add_tail(&netdev->list, &bvd_net_dev_list);
}

static void bvd_net_dev_del(struct bvd_net_dev *netdev)
{
	BVD_DBG("\n");

	list_del(&netdev->list);
}

static struct bvd_net_dev *__bvd_net_dev_create(
		struct bvd_net_dev_info *dev_info)
{
	int							err;
	struct net_device			*dev;
	struct bvd_net_dev			*netdev;
	static struct device_type	bvd_net_dev_type = {
			.name = "bluetooth", };

	BVD_DBG("addr:[%u,%u,%u,%u,%u,%u]\n",
			dev_info->dev_addr[0], dev_info->dev_addr[1],
			dev_info->dev_addr[2], dev_info->dev_addr[3],
			dev_info->dev_addr[4], dev_info->dev_addr[5]);

	mutex_lock(&bvd_net_dev_list_lock);

	netdev = bvd_net_dev_find(dev_info->dev_addr);
	if (netdev) {
		mutex_unlock(&bvd_net_dev_list_lock);
		return NULL;
	}
#if (LINUX_VERSION_LT(3, 17, 0))
	dev = alloc_netdev(sizeof(struct bvd_net_dev), dev_info->name,
					   bvd_net_dev_setup);
#else
	dev = alloc_netdev(sizeof(struct bvd_net_dev), dev_info->name,
			NET_NAME_UNKNOWN, bvd_net_dev_setup);
#endif
	if (!dev) {
		mutex_unlock(&bvd_net_dev_list_lock);
		return NULL;
	}

	netdev = netdev_priv(dev);

	memcpy(dev->dev_addr,
			dev_info->dev_addr, ETH_ALEN);
	netdev->dev					= dev;
	netdev->mc_filter			= 0;
	netdev->proto_filter.count	= 0;

	skb_queue_head_init(&netdev->sk_write_queue);
	init_waitqueue_head(&netdev->read_wq);
	init_waitqueue_head(&netdev->status_wq);
	INIT_LIST_HEAD(&netdev->mc_list);
	spin_lock_init(&netdev->lock);

#ifdef CONFIG_BVD_NET_DEV_MC_FILTER
	bvd_net_dev_set_default_mc_filter(netdev);
#endif	/* ifdef CONFIG_BVD_NET_DEV_MC_FILTER */

#ifdef CONFIG_BVD_NET_DEV_PROTO_FILTER
	bvd_net_dev_set_default_proto_filter(netdev);
#endif	/* ifdef CONFIG_BVD_NET_DEV_PROTO_FILTER */

	SET_NETDEV_DEVTYPE(dev, &bvd_net_dev_type);

	err = register_netdev(dev);
	if (err) {
		mutex_unlock(&bvd_net_dev_list_lock);
		free_netdev(dev);
		return NULL;
	}

	bvd_net_dev_add(netdev);

	mutex_unlock(&bvd_net_dev_list_lock);
	BVD_DBG("netdev create success\n");

	return netdev;
}

static int __bvd_net_dev_release(struct bvd_net_dev *netdev)
{
	struct bvd_net_dev_mc_info_node *mc, *tmp_mc;
	unsigned long flags;

	BVD_DBG("\n");

	mutex_lock(&bvd_net_dev_list_lock);

	unregister_netdev(netdev->dev);
	/* Avoid "BUG: scheduling while atmic" at 20160304 */
	spin_lock_irqsave(&netdev->lock, flags);
	skb_queue_purge(&netdev->sk_write_queue);

	list_for_each_entry_safe(mc, tmp_mc, &netdev->mc_list, list) {
		BVD_DBG("release mc\n");
		list_del(&mc->list);
		kfree(mc->mc_info.mc_range_list);
		kfree(mc);
	}

	bvd_net_dev_del(netdev);

	wake_up_interruptible(&netdev->status_wq);
	wake_up_interruptible(&netdev->read_wq);

	spin_unlock_irqrestore(&netdev->lock, flags);

	free_netdev(netdev->dev);

	mutex_unlock(&bvd_net_dev_list_lock);

	return 0;
}

static int __bvd_net_dev_pop_data(struct bvd_net_dev *netdev,
		char *buf, unsigned int count)
{
	struct net_device	*dev;
	struct sk_buff	*skb;
	unsigned int	read_count		= 0;
	unsigned int	rest_count		= count;
	unsigned int	current_count	= 0;
	char			*pos			= buf;
	unsigned long flags;

	BVD_DBG("\n");

	spin_lock_irqsave(&netdev->lock, flags);
	dev = netdev->dev;

	while (rest_count > 0) {
		skb = skb_peek(&netdev->sk_write_queue);
		if (!skb)
			break;

		current_count = min_t(unsigned int, rest_count, skb->len);

		memcpy(pos, skb->data, current_count);
		skb_pull(skb, current_count);

		read_count	+= current_count;
		rest_count	-= current_count;
		pos			+= current_count;
		dev->stats.tx_bytes += current_count;

		if (!skb->len) {
			dev->stats.tx_packets++;
			skb_dequeue(&netdev->sk_write_queue);
			kfree_skb(skb);
			if (netif_queue_stopped(dev))
				netif_wake_queue(dev);
			break;
		}
	}

	spin_unlock_irqrestore(&netdev->lock, flags);

	return read_count;
}

static int __bvd_net_dev_push_data(struct bvd_net_dev *netdev,
		const char *buf, unsigned int count)
{
	struct net_device	*dev;
	struct sk_buff		*skb;
	unsigned long flags;

	BVD_DBG("\n");

	skb = alloc_skb(count, GFP_ATOMIC);
	if (!skb)
		return -EAGAIN;

	spin_lock_irqsave(&netdev->lock, flags);
	dev = netdev->dev;

	memcpy(skb_put(skb, count), buf, count);

	dev->stats.rx_packets++;
	dev->stats.rx_bytes += count;
	skb->ip_summed	= CHECKSUM_NONE;
	skb->protocol	= eth_type_trans(skb, dev);

	spin_unlock_irqrestore(&netdev->lock, flags);

	netif_rx_ni(skb);

	return count;
}

static int __bvd_net_dev_get_status(struct bvd_net_dev *netdev)
{
	int status;
	unsigned long flags;

	BVD_DBG("\n");

	spin_lock_irqsave(&netdev->lock, flags);

	status = netdev->flages;

	spin_unlock_irqrestore(&netdev->lock, flags);

	return status;
}

static long bvd_net_dev_get_mc_list(struct bvd_net_dev *netdev,
		struct bvd_net_dev_mc_info *mc_info)
{
	struct bvd_net_dev_mc_info_node *mc;
	unsigned long flags;

	BVD_DBG("\n");

	spin_lock_irqsave(&netdev->lock, flags);

	if (list_empty(&netdev->mc_list)) {
		spin_unlock_irqrestore(&netdev->lock, flags);
		return -ENODATA;
	}

	mc = list_first_entry(&netdev->mc_list,
			struct bvd_net_dev_mc_info_node, list);

	list_del(&mc->list);

	mc_info->count = mc->mc_info.count;

	if (copy_to_user(mc_info->mc_range_list, mc->mc_info.mc_range_list,
			(mc_info->count *
					sizeof(struct bvd_net_dev_mc_range)))) {
		spin_unlock_irqrestore(&netdev->lock, flags);
		return -EFAULT;
	}

	if (list_empty(&netdev->mc_list))
		netdev->flages &= ~(1 << BVD_NET_DEV_BIT_MC_CHANGED);

	spin_unlock_irqrestore(&netdev->lock, flags);

	kfree(mc->mc_info.mc_range_list);
	kfree(mc);

	return 0;
}

static long bvd_net_dev_set_proto_filter(struct bvd_net_dev	*netdev,
		struct bvd_net_dev_proto_filter *req)
{
	int count;
	unsigned long flags;

	BVD_DBG("req->count: %d start: %d end: %d\n",
			req->count, req->proto_range_list[0].start,
			req->proto_range_list[0].end);

	count = req->count;

	if (count > BVD_NET_DEV_MAX_PROTO_FILTER)
		return -EINVAL;

	spin_lock_irqsave(&netdev->lock, flags);
	memcpy(&netdev->proto_filter, req,
		   sizeof(struct bvd_net_dev_proto_filter));

	if (count == 0)
		bvd_net_dev_set_default_proto_filter(netdev);

	spin_unlock_irqrestore(&netdev->lock, flags);

	return 0;
}

static void bvd_net_dev_mc_iterate_set_bit(struct bvd_net_dev *netdev,
		u8 *start, u8 *end)
{
	int index;

	set_bit(bvd_net_dev_mc_hash(start),
			(ulong *)&netdev->mc_filter);

	while ((memcmp(start, end, ETH_ALEN) < 0) &&
			(netdev->mc_filter != ~0LL)) {
		index = ETH_ALEN - 1;

		/* Increment start */
		while ((index >= 0) && (++start[index--] == 0))
			;

		set_bit(bvd_net_dev_mc_hash(start),
				(ulong *)&netdev->mc_filter);
	}
}

static long bvd_net_dev_set_mc_filter(struct bvd_net_dev *netdev,
		struct bvd_net_dev_mc_info *req)
{
	int							index = 0;
	struct bvd_net_dev_mc_range *mc_range_list;
	unsigned long flags;

	BVD_DBG("count %d\n", req->count);

	if (req->count <= 0)
		return -EINVAL;

	mc_range_list = kmalloc(req->count *
			sizeof(struct bvd_net_dev_mc_range), GFP_ATOMIC);
	if (!mc_range_list) {
		BVD_DBG("alloc failed, size %d\n",
			(req->count * sizeof(struct bvd_net_dev_mc_range)));
		return -ENOMEM;
	}

	if (copy_from_user(mc_range_list,
			req->mc_range_list,
			req->count * sizeof(struct bvd_net_dev_mc_range))) {
		kfree(mc_range_list);
		return -EFAULT;
	}

	spin_lock_irqsave(&netdev->lock, flags);

	netdev->mc_filter = 0;

	/* Always send broadcast */
	set_bit(bvd_net_dev_mc_hash(netdev->dev->broadcast),
			(ulong *)&netdev->mc_filter);

	/* Add address ranges to the multicast hash */
	while (index < req->count) {
		u8 *start, *end;

		start	= mc_range_list[index].start;
		end		= mc_range_list[index].end;

		bvd_net_dev_mc_iterate_set_bit(netdev, start, end);

		index++;
	}

	kfree(mc_range_list);

	spin_unlock_irqrestore(&netdev->lock, flags);

	return 0;
}

static long __bvd_net_dev_ioctl(struct bvd_net_dev *netdev,
		unsigned int cmd, unsigned long arg)
{
	int								ret = 0;
	struct bvd_net_dev_mc_info		mc_info;
	struct bvd_net_dev_proto_filter proto_info;
	unsigned long __user *argp = (unsigned long __user *)arg;

	BVD_DBG("cmd:%u\n", cmd);

	switch (cmd) {
#ifdef CONFIG_BVD_NET_DEV_MC_FILTER
	case BVD_NET_DEV_GET_MC_LIST:
	{
		if (copy_from_user(&mc_info, argp,
				sizeof(struct bvd_net_dev_mc_info))) {
			return -EFAULT;
		}

		ret = bvd_net_dev_get_mc_list(netdev, &mc_info);
		if (copy_to_user(argp, &mc_info,
				sizeof(struct bvd_net_dev_mc_info))) {
			return -EFAULT;
		}
		break;
	}
	case BVD_NET_DEV_SET_MC_FILTER:
	{
		if (copy_from_user(&mc_info, argp,
				sizeof(struct bvd_net_dev_mc_info))) {
			return -EFAULT;
		}

		ret = bvd_net_dev_set_mc_filter(netdev, &mc_info);
		break;
	}
#endif	/* ifdef CONFIG_BVD_NET_DEV_MC_FILTER */


	case BVD_NET_DEV_SET_PROTO_FILTER:
	{
#ifdef CONFIG_BVD_NET_DEV_PROTO_FILTER
		if (copy_from_user(&proto_info, argp,
				sizeof(struct bvd_net_dev_proto_filter))) {
			return -EFAULT;
		}

		ret = bvd_net_dev_set_proto_filter(netdev, &proto_info);
#endif	/* ifdef CONFIG_BVD_NET_DEV_PROTO_FILTER */
		break;
	}
	default:
		return -EINVAL;
	}

	return ret;
}

static unsigned int __bvd_net_dev_poll(struct bvd_net_dev *netdev,
		struct file *file, struct poll_table_struct *wait)
{
	unsigned int mask = 0;
	unsigned long flags;

	BVD_DBG("\n");
	spin_lock_irqsave(&netdev->lock, flags);

	poll_wait(file, &netdev->read_wq, wait);
	poll_wait(file, &netdev->status_wq, wait);

	if (!skb_queue_empty(&netdev->sk_write_queue))
		mask |= POLLIN | POLLRDNORM;

	if (netdev->flages)
		mask |= POLLPRI;

	spin_unlock_irqrestore(&netdev->lock, flags);

	return mask;
}

static void *bvd_net_dev_create(void __user *arg)
{
	struct bvd_net_dev		*netdev;
	struct bvd_net_dev_info dev_info;

	BVD_DBG("\n");

	if (copy_from_user(&dev_info, arg,
					   sizeof(struct bvd_net_dev_info))) {
		return NULL;
	}

	if ((strlen(dev_info.name) > BVD_NET_DEV_MAX_NAME_LEN) ||
		 !(*dev_info.name)) {
		return NULL;
	}

	netdev = __bvd_net_dev_create(&dev_info);

	return (void *)netdev;
}

static int bvd_net_dev_release(void *dev, unsigned long force)
{
	struct bvd_net_dev *netdev = (struct bvd_net_dev *)dev;

	BUG_ON(netdev == NULL);

	return __bvd_net_dev_release(netdev);
}

static int bvd_net_dev_pop_data(void *dev, char *buf, unsigned int count)
{
	struct bvd_net_dev *netdev = (struct bvd_net_dev *)dev;

	BUG_ON(buf == NULL);
	BUG_ON(netdev == NULL);

	return __bvd_net_dev_pop_data(netdev, buf, count);
}

static int bvd_net_dev_push_data(void *dev,
		const char *buf, unsigned int count)
{
	struct bvd_net_dev *netdev = (struct bvd_net_dev *)dev;

	BUG_ON(buf == NULL);
	BUG_ON(netdev == NULL);

	return __bvd_net_dev_push_data(netdev, buf, count);
}

static int bvd_net_dev_get_status(void *dev)
{
	struct bvd_net_dev *netdev = (struct bvd_net_dev *)dev;

	BUG_ON(netdev == NULL);

	return __bvd_net_dev_get_status(netdev);
}

static long bvd_net_dev_ioctl(void			*dev,
							  unsigned int	cmd,
							  unsigned long arg)
{
	struct bvd_net_dev *netdev = (struct bvd_net_dev *)dev;

	BUG_ON(netdev == NULL);

	return __bvd_net_dev_ioctl(netdev, cmd, arg);
}

static unsigned int bvd_net_dev_poll(void *dev,
		struct file *file,	 struct poll_table_struct *wait)
{
	struct bvd_net_dev *netdev = (struct bvd_net_dev *)dev;

	BUG_ON(netdev == NULL);

	return __bvd_net_dev_poll(netdev, file, wait);
}

static const struct bvd_device_operations bvd_net_dev_if_ops = {
	.create		= bvd_net_dev_create,
	.destroy	= bvd_net_dev_release,
	.read		= bvd_net_dev_pop_data,
	.write		= bvd_net_dev_push_data,
	.get_status = bvd_net_dev_get_status,
	.ioctl		= bvd_net_dev_ioctl,
	.poll		= bvd_net_dev_poll,
};

int __init bvd_net_dev_init(void)
{
	BVD_DBG("\n");

	return bvd_device_register(BVD_DEVICE_TYPE_NET, &bvd_net_dev_if_ops);
}

int bvd_net_dev_exit(void)
{
	BVD_DBG("\n");

	return bvd_device_unregister(BVD_DEVICE_TYPE_NET);
}
