/*******************************************************************************
*    Copyright (c) 2013 Neusoft Co., Ltd.
*    FileName	:	bvd_tty_dev.c
*    Module		:	bvd driver
*    Function	:	bvd driver module use for Neusoft Bluedragon
*					Bluetooth stack.
*-------------------------------------------------------------------------------
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc..
*******************************************************************************/

#include <linux/types.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/tty.h>
#include <linux/tty_driver.h>
#include <linux/tty_flip.h>
#include <linux/list.h>
#include <linux/skbuff.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/poll.h>
#include <linux/version.h>
#include "bvd_tty_dev.h"
#include "bvd_device.h"
#ifdef __ALPS_FILE_ID__
#undef __ALPS_FILE_ID__
#endif	/* ifdef __ALPS_FILE_ID__ */
#define __ALPS_FILE_ID__    0x811a

struct bvd_tty_dev {
	struct tty_port			tty_port;
	struct list_head		list;
	spinlock_t				lock;
	unsigned long			flags;
	int						id;
	wait_queue_head_t		read_wait;
	wait_queue_head_t		status_wait;
	wait_queue_head_t		clear_wait;
	int						in_alloc;
	int						in_limit;
	struct sk_buff_head		in_buff;
	int						modem_status;
	struct bvd_tty_tiocm	tiocm;
};

#define BVD_TTY_PORTS		(14)
#define BVD_TTY_NAME		"bvd_tty"

#define BVD_TTY_DEV_BUFFER_LIMIT    (7 * 661)

#define BVD_TTY_DEV_OPEN		1
#define BVD_TTY_DEV_HANGUP		2
#define BVD_TTY_DEV_THROTTLE	3
#define BVD_TTY_DEV_TIOCMSET	4

static struct tty_driver *bvd_tty_driver;

static LIST_HEAD(bvd_tty_dev_list);
static DEFINE_MUTEX(bvd_tty_dev_list_mutex);

static void bvd_tty_dev_delete(struct bvd_tty_dev *dev)
{
	BVD_DBG("list_del_init\n");

	mutex_lock(&bvd_tty_dev_list_mutex);
	list_del_init(&dev->list);
	mutex_unlock(&bvd_tty_dev_list_mutex);
	BVD_DBG("list_del_init end\n");
}

static int bvd_tty_port_activate(struct tty_port *port, struct tty_struct *tty)
{
	struct bvd_tty_dev *dev =
			container_of(port, struct bvd_tty_dev, tty_port);
	unsigned long flags;

	BVD_DBG("tty_port activate\n");
	if (!dev) {
		BVD_DBG("dev is null\n");
		return -ENODEV;
	}

	spin_lock_irqsave(&dev->lock, flags);
	tty->driver_data = dev;
	set_bit(BVD_TTY_DEV_OPEN, &dev->flags);
	spin_unlock_irqrestore(&dev->lock, flags);
	return 0;
}

static void bvd_tty_port_shutdown(struct tty_port *port)
{
	struct bvd_tty_dev *dev =
			container_of(port, struct bvd_tty_dev, tty_port);
	unsigned long flags;

	BVD_DBG("tty_port shutdown\n");

	if (!dev) {
		BVD_DBG("dev is null\n");
		return;
	}

	spin_lock_irqsave(&dev->lock, flags);
	clear_bit(BVD_TTY_DEV_OPEN, &dev->flags);
	wake_up_interruptible(&dev->read_wait);
	wake_up_interruptible(&dev->status_wait);
	spin_unlock_irqrestore(&dev->lock, flags);
	BVD_DBG("tty_port shutdown end\n");
}

static void bvd_tty_port_destruct(struct tty_port *port)
{
	struct bvd_tty_dev *dev =
		container_of(port, struct bvd_tty_dev, tty_port);
	unsigned long flags;

	BVD_DBG("tty_port destruct\n");
	if (!dev) {
		BVD_DBG("dev is null\n");
		return;
	}

	spin_lock_irqsave(&dev->lock, flags);
	skb_queue_purge(&dev->in_buff);
	spin_unlock_irqrestore(&dev->lock, flags);

	bvd_tty_dev_delete(dev);

	tty_unregister_device(bvd_tty_driver, dev->id);
	BVD_DBG("tty dev free\n");
	kfree(dev);
	BVD_DBG("tty_port destruct end\n");
}

static const struct tty_port_operations bvd_tty_dev_port_ops = {
	.shutdown = bvd_tty_port_shutdown,
	.activate = bvd_tty_port_activate,
	.destruct = bvd_tty_port_destruct,
};

static struct bvd_tty_dev *__bvd_tty_dev_get(int id)
{
	struct bvd_tty_dev *dev;

	list_for_each_entry(dev, &bvd_tty_dev_list, list)
		if (dev->id == id)
			return dev;

	return NULL;
}

static struct bvd_tty_dev *bvd_tty_dev_get(int id)
{
	struct bvd_tty_dev *dev;

	mutex_lock(&bvd_tty_dev_list_mutex);
	dev = __bvd_tty_dev_get(id);
	if (dev)
		tty_port_get(&dev->tty_port);

	mutex_unlock(&bvd_tty_dev_list_mutex);

	return dev;
}

static struct bvd_tty_dev *__bvd_tty_dev_device_create(int id)
{
	struct bvd_tty_dev	*dev;
	struct device		*tty_dev;

	/* alloc bvd_tty_dev and insert to bvd_tty_dev_list
	 *  should be atomic operation */
	mutex_lock(&bvd_tty_dev_list_mutex);
	dev = __bvd_tty_dev_get(id);
	if (dev) {
		BVD_DBG("tty %d has been exist\n", id);
		goto unlock_list;
	}

	dev = kzalloc(sizeof(struct bvd_tty_dev), GFP_KERNEL);
	if (!dev) {
		BVD_DBG("kzalloc failed, size %d\n",
				sizeof(struct bvd_tty_dev));
		goto unlock_list;
	}

	dev->id			= id;
	dev->in_limit	= BVD_TTY_DEV_BUFFER_LIMIT;
	list_add(&dev->list, &bvd_tty_dev_list);
	tty_port_init(&dev->tty_port);
	dev->tty_port.ops = &bvd_tty_dev_port_ops;
	spin_lock_init(&dev->lock);
	init_waitqueue_head(&dev->read_wait);
	init_waitqueue_head(&dev->status_wait);
	/*add clear_wait queue for read data when tty closing*/
	init_waitqueue_head(&dev->clear_wait);
	skb_queue_head_init(&dev->in_buff);
#if (LINUX_VERSION_LT(3, 8, 0))
	tty_dev = tty_register_device(bvd_tty_driver, dev->id, NULL);
#else	/* (LINUX_VERSION >= (3, 8, 0)) */
	tty_dev = tty_port_register_device(&dev->tty_port,
			bvd_tty_driver, dev->id, NULL);
#endif	/* (LINUX_VERSION_LT(3, 8, 0)) end*/
	if (IS_ERR(tty_dev)) {
		BVD_DBG("tty register device failed\n");
		list_del(&dev->list);
		kfree(dev);
		goto unlock_list;
	}
	mutex_unlock(&bvd_tty_dev_list_mutex);

	BVD_DBG("tty device create success\n");

	return dev;

unlock_list:
	BVD_DBG("tty device create failed\n");
	mutex_unlock(&bvd_tty_dev_list_mutex);
	return NULL;
}

static void *bvd_tty_dev_device_create(void __user *arg)
{
	int							ret;
	struct bvd_create_tty_req	req;

	ret = __copy_from_user(&req, (void __user *)arg,
			sizeof(struct bvd_create_tty_req));
	if ((ret) || (req.id < 0) || (req.id >= BVD_TTY_PORTS)) {
		BVD_DBG("copy_from_user ret = %d, req.id = %d\n", ret, req.id);
		return NULL;
	}

	return (void *)__bvd_tty_dev_device_create(req.id);
}

static int __bvd_tty_dev_device_destroy(struct bvd_tty_dev *dev,
		unsigned long force)
{
	unsigned long flags;
	struct tty_struct *tty;

	BVD_DBG("tty device destroy, force: %ld\n", force);

	if (!force) {
		spin_lock_irqsave(&dev->lock, flags);
		if (test_bit(BVD_TTY_DEV_OPEN, &dev->flags)) {
			BVD_DBG("dev using, can't destroy\n");
			spin_unlock_irqrestore(&dev->lock, flags);
			return -EBUSY;
		}
		spin_unlock_irqrestore(&dev->lock, flags);
	}

	tty = tty_port_tty_get(&dev->tty_port);
	if (tty) {
		tty_vhangup(tty);
		BVD_DBG("tty_vhangup end\n");
		tty_kref_put(tty);
		BVD_DBG("tty_kref_put end\n");
	}

	BVD_DBG("tty_port_put begin\n");
	tty_port_put(&dev->tty_port);

	BVD_DBG("device_destroy end\n");
	return 0;
}

static int bvd_tty_dev_device_destroy(void *dev, unsigned long force)
{
	struct bvd_tty_dev *device = (struct bvd_tty_dev *)dev;

	if (!device)
		return -ENODEV;

	return __bvd_tty_dev_device_destroy(device, force);
}

static int __bvd_tty_dev_device_read(struct bvd_tty_dev *dev,
		char *buf, unsigned int count)
{
	struct sk_buff	*skb;
	int				total = 0, offset = 0;

	if (!test_bit(BVD_TTY_DEV_OPEN, &dev->flags))
		return -EIO;

	if (test_bit(BVD_TTY_DEV_HANGUP, &dev->flags))
		return -ENODEV;

	BVD_DBG("need read count = %d\n", count);

	do {
		skb = skb_peek(&dev->in_buff);
		if (!skb) {
			BVD_DBG("in alloc queue is empty\n");
			break;
		}	/* in_buff is empty */

		if (skb->len > count) {
			memcpy(buf + offset, skb->data, count);
			BVD_DBG("buffer len > count, skb_pull\n");
			skb_pull(skb, count);
			total	+= count;
			count	= 0;
		} else {
			BVD_DBG("buffer len < count\n");
			skb = skb_dequeue(&dev->in_buff);
			memcpy(buf + offset, skb->data, skb->len);
			offset	+= skb->len;
			count	-= skb->len;
			total	+= skb->len;
			kfree_skb(skb);
		}
	} while (count > 0);

	BVD_DBG("if read count = %d\n", total);

	/*Add for lost data when close*/
	if ((dev->in_alloc <= 0) && waitqueue_active(&dev->clear_wait)) {
		BVD_DBG("wake_up clear_wait queue\n");
		wake_up_interruptible(&dev->clear_wait);
	}

	return total;
}

static int bvd_tty_dev_device_read(void *dev, char *buf,
		unsigned int count)
{
	struct bvd_tty_dev	*device = (struct bvd_tty_dev *)dev;
	int					ret;
	unsigned long flags;

	spin_lock_irqsave(&device->lock, flags);
	ret = __bvd_tty_dev_device_read(device, buf, count);
	spin_unlock_irqrestore(&device->lock, flags);

	return ret;
}

static int __bvd_tty_dev_device_write(struct bvd_tty_dev *dev,
		const char *buf, unsigned int count)
{
	int ret;
#if	(LINUX_VERSION_LT(3, 9, 0))
	struct tty_struct	*tty = dev->tty_port.tty;
#else	/* (LINUX_VERSION >= (3, 9, 0)) */
	struct tty_port *tty = &dev->tty_port;
#endif	/* (LINUX_VERSION_LT(3, 9, 0)) end*/
	BVD_DBG("write count %d\n", count);

	if (!test_bit(BVD_TTY_DEV_OPEN, &dev->flags)) {
		BVD_DBG("dev not open\n");
		return -EIO;
	}

	if (test_bit(BVD_TTY_DEV_THROTTLE, &dev->flags)) {
		BVD_DBG("dev throttle\n");
		return -EAGAIN;
	}

	if (test_bit(BVD_TTY_DEV_HANGUP, &dev->flags)) {
		BVD_DBG("dev hangup\n");
		return -ENODEV;
	}

	ret = tty_insert_flip_string(tty, buf, count);
	tty_flip_buffer_push(tty);

	return ret;
}

static int bvd_tty_dev_device_write(void *dev, const char *buf,
				unsigned int count)
{
	struct bvd_tty_dev	*device = (struct bvd_tty_dev *)dev;

	int					ret;
	unsigned long flags;

	spin_lock_irqsave(&device->lock, flags);
	ret = __bvd_tty_dev_device_write(device, buf, count);
	spin_unlock_irqrestore(&device->lock, flags);
	return ret;
}

static int __bvd_tty_dev_device_get_status(struct bvd_tty_dev *dev)
{
	int status = 0;

	if (test_bit(BVD_TTY_DEV_HANGUP, &dev->flags))
		status |= BVD_TTY_DEV_STATUS_HANGUP;

	if (test_bit(BVD_TTY_DEV_TIOCMSET, &dev->flags))
		status |= BVD_TTY_DEV_STATUS_TIOCMSET;

	return status;
}

static int bvd_tty_dev_device_get_status(void *dev)
{
	struct bvd_tty_dev	*device = (struct bvd_tty_dev *)dev;
	int					status;
	unsigned long flags;

	spin_lock_irqsave(&device->lock, flags);
	status = __bvd_tty_dev_device_get_status(device);
	spin_unlock_irqrestore(&device->lock, flags);

	return status;
}

static long __bvd_tty_dev_device_ioctl(struct bvd_tty_dev *dev,
		unsigned int cmd, unsigned long arg)
{
	int						ret, modem_status;
	struct bvd_tty_tiocm	tiocm;

	switch (cmd) {
	case BVD_IOCTL_TTY_GET_TIOCM:
	{
		clear_bit(BVD_TTY_DEV_TIOCMSET, &dev->flags);
		tiocm.set = dev->tiocm.set;
		tiocm.clear = dev->tiocm.clear;
		ret	= copy_to_user((void __user *)arg, &tiocm,
				sizeof(struct bvd_tty_tiocm));
		break;
	}
	case BVD_IOCTL_TTY_SET_MODEM_STATUS:
	{
		ret = __get_user(modem_status, (int __user *)arg);
		if (!ret)
			dev->modem_status = modem_status;
		break;
	}
	default:
	{
		ret = -ENOTTY;
		break;
	}
	}

	return ret;
}

static long bvd_tty_dev_device_ioctl(void *dev, unsigned int cmd,
		unsigned long arg)
{
	struct bvd_tty_dev	*device = (struct bvd_tty_dev *)dev;
	int					ret;
	unsigned long flags;

	spin_lock_irqsave(&device->lock, flags);
	ret = __bvd_tty_dev_device_ioctl(device, cmd, arg);
	spin_unlock_irqrestore(&device->lock, flags);

	return ret;
}

static unsigned int __bvd_tty_dev_device_poll(struct bvd_tty_dev *dev,
		struct file *file, struct poll_table_struct *wait)
{
	unsigned int mask = 0;

	poll_wait(file, &dev->read_wait, wait);
	poll_wait(file, &dev->status_wait, wait);

	if (!skb_queue_empty(&dev->in_buff))
		mask |= POLLIN | POLLRDNORM;

	if (test_bit(BVD_TTY_DEV_HANGUP, &dev->flags)
		|| test_bit(BVD_TTY_DEV_TIOCMSET, &dev->flags)) {
		mask |= POLLPRI;
	}

	BVD_DBG("poll mask %d\n", mask);

	return mask;
}

static unsigned int bvd_tty_dev_device_poll(void *dev, struct file *file,
					struct poll_table_struct *wait)
{
	struct bvd_tty_dev	*device = (struct bvd_tty_dev *)dev;
	unsigned int		mask	= 0;
	unsigned long flags;

	spin_lock_irqsave(&device->lock, flags);
	mask = __bvd_tty_dev_device_poll(device, file, wait);
	spin_unlock_irqrestore(&device->lock, flags);

	return mask;
}

static const struct bvd_device_operations bvd_tty_device_ops = {
	.create		= bvd_tty_dev_device_create,
	.destroy	= bvd_tty_dev_device_destroy,
	.read		= bvd_tty_dev_device_read,
	.write		= bvd_tty_dev_device_write,
	.get_status = bvd_tty_dev_device_get_status,
	.ioctl		= bvd_tty_dev_device_ioctl,
	.poll		= bvd_tty_dev_device_poll,
};

static int __bvd_tty_dev_write_room(struct bvd_tty_dev *dev)
{
	int room;

	room = dev->in_limit - dev->in_alloc;
	if (room < 0)
		room = 0;

	return room;
}

static void bvd_tty_dev_skbfree(struct sk_buff *skb)
{
	struct bvd_tty_dev	*dev	= (struct bvd_tty_dev *)skb->sk;
	struct tty_struct	*tty	= dev->tty_port.tty;

	BVD_DBG("free in buffer skb, alloc = %d, headroom = %d, len = %d\n",
			dev->in_alloc, skb_headroom(skb), skb->len);

	/* if part of data be pulled from skbuff,
	 * headroom will increase and len will reduce,
	 * thus "skb_headroom(skb) + skb->len" is the original size
	 * that we plus to in_alloc
	 * */
	dev->in_alloc -= (skb_headroom(skb) + skb->len);
	BVD_DBG("free in buffer, skb alloc = %d\n", dev->in_alloc);
	if (test_bit(BVD_TTY_DEV_OPEN, &dev->flags) && tty)
		tty_wakeup(tty);

	tty_port_put(&dev->tty_port);
}

static void bvd_tty_dev_skb_setowner_w(struct sk_buff *skb,
		struct bvd_tty_dev *dev, unsigned long size)
{
	tty_port_get(&dev->tty_port);
	dev->in_alloc	+= size;
	skb->sk			= (void *)dev;
	skb->destructor = bvd_tty_dev_skbfree;
}

static struct sk_buff *bvd_tty_dev_skballoc(struct bvd_tty_dev *dev,
			unsigned long size, gfp_t priority)
{
	BVD_DBG("skb alloc\n limit = %d, alloc = %d\n",
			dev->in_limit, dev->in_alloc);

	if (size <= __bvd_tty_dev_write_room(dev)) {
		struct sk_buff *skb = alloc_skb(size, priority);

		if (skb) {
			bvd_tty_dev_skb_setowner_w(skb, dev, size);
			return skb;
		}
	}

	return NULL;
}

static int bvd_tty_dev_open(struct tty_struct *tty, struct file *filp)
{
	int id;
	int ret;
	struct bvd_tty_dev *dev = (struct bvd_tty_dev *)tty->driver_data;

	if (!dev) {
		id	= tty->index;
		dev = bvd_tty_dev_get(id);
		BVD_DBG("tty open, id = %d\n", id);
		if (!dev) {
			BVD_DBG("bvd_tty_dev_get failed\n");
			return -ENODEV;
		}
	}
	ret = tty_port_open(&dev->tty_port, tty, filp);
	BVD_DBG("tty_port_open ret = %d\n", ret);

	return ret;
}

/*Add function to get count of chars in buffer, at 2015-07-07*/
static int bvd_tty_dev_count_in_buffer(struct bvd_tty_dev *dev)
{
	int count = 0;
	unsigned long flags;

	if (!dev)
		return count;

	spin_lock_irqsave(&dev->lock, flags);
	count = dev->in_alloc;
	spin_unlock_irqrestore(&dev->lock, flags);

	return count;
}

static void bvd_tty_dev_close(struct tty_struct *tty, struct file *filp)
{
	struct bvd_tty_dev	*dev = (struct bvd_tty_dev *)tty->driver_data;
	unsigned long		flags;
	long ret = 0;

	if (!dev) {
		BVD_DBG("bvd_tty_dev_close dev null\n");
		return;
	}

	BVD_DBG("tty device close\n");

	spin_lock_irqsave(&dev->lock, flags);
	/*Add handle for lost data when trigger closing, at 2015-07-07, begin*/
	if (dev->in_alloc > 0) {
		spin_unlock_irqrestore(&dev->lock, flags);
		BVD_DBG("will wait for reading data completely\n");
		ret = wait_event_interruptible_timeout(dev->clear_wait,
				(bvd_tty_dev_count_in_buffer(dev) <= 0), HZ);
		BVD_DBG("wait event return, ret: %ld\n", ret);
		spin_lock_irqsave(&dev->lock, flags);
	}
	/*Add handle for lost data when trigger closing, at 2015-07-07, end*/
	spin_unlock_irqrestore(&dev->lock, flags);

	tty_port_close(&dev->tty_port, tty, filp);
	BVD_DBG("tty device close end\n");
}

static int bvd_tty_dev_write(struct tty_struct *tty, const unsigned char *buf,
							 int count)
{
	struct bvd_tty_dev	*dev = (struct bvd_tty_dev *)tty->driver_data;
	struct sk_buff		*skb;
	unsigned long flags;

	BVD_DBG("tty write count %d\n", count);

	if (!dev) {
		BVD_DBG("bvd_tty_dev_write dev null\n");
		return -ENODEV;
	}

	spin_lock_irqsave(&dev->lock, flags);
	if (!test_bit(BVD_TTY_DEV_OPEN, &dev->flags)) {
		spin_unlock_irqrestore(&dev->lock, flags);
		return -EIO;
	}

	skb = bvd_tty_dev_skballoc(dev, count, GFP_ATOMIC);
	if (!skb) {
		BVD_DBG("alloc failed!\n");
		spin_unlock_irqrestore(&dev->lock, flags);
		return 0;
	}
	memcpy(skb_put(skb, count), buf, count);
	BVD_DBG("inster queue skb len = %d\n", skb->len);
	skb_queue_tail(&dev->in_buff, skb);
	wake_up_interruptible(&dev->read_wait);
	spin_unlock_irqrestore(&dev->lock, flags);

	return count;
}

static int bvd_tty_dev_write_room(struct tty_struct *tty)
{
	struct bvd_tty_dev	*dev = (struct bvd_tty_dev *)tty->driver_data;
	int					room;
	unsigned long flags;

	if (!dev)
		return 0;

	spin_lock_irqsave(&dev->lock, flags);
	room = __bvd_tty_dev_write_room(dev);
	spin_unlock_irqrestore(&dev->lock, flags);
	BVD_DBG("write room =%d\n", room);

	return room;
}

static int bvd_tty_dev_chars_in_buffer(struct tty_struct *tty)
{
	struct bvd_tty_dev	*dev = (struct bvd_tty_dev *)tty->driver_data;
	int					count;
	unsigned long flags;

	if (!dev)
		return 0;

	spin_lock_irqsave(&dev->lock, flags);
	count = dev->in_alloc;
	spin_unlock_irqrestore(&dev->lock, flags);

	return count;
}

static void bvd_tty_dev_flush_buffer(struct tty_struct *tty)
{
	struct bvd_tty_dev *dev = (struct bvd_tty_dev *)tty->driver_data;
	unsigned long flags;

	BVD_DBG("flush\n");

	if (!dev)
		return;

	spin_lock_irqsave(&dev->lock, flags);
	skb_queue_purge(&dev->in_buff);
	spin_unlock_irqrestore(&dev->lock, flags);
	tty_wakeup(tty);
}

static void bvd_tty_dev_throttle(struct tty_struct *tty)
{
	struct bvd_tty_dev *dev = (struct bvd_tty_dev *)tty->driver_data;
	unsigned long flags;

	BVD_DBG("throttle\n");

	if (!dev)
		return;

	spin_lock_irqsave(&dev->lock, flags);
	set_bit(BVD_TTY_DEV_THROTTLE, &dev->flags);
	spin_unlock_irqrestore(&dev->lock, flags);
}

static void bvd_tty_dev_unthrottle(struct tty_struct *tty)
{
	struct bvd_tty_dev *dev = (struct bvd_tty_dev *)tty->driver_data;
	unsigned long flags;

	BVD_DBG("unthrottle\n");

	if (!dev)
		return;

	spin_lock_irqsave(&dev->lock, flags);
	clear_bit(BVD_TTY_DEV_THROTTLE, &dev->flags);

	/* TODO: user just sleep - retry , do not need notify event */
	spin_unlock_irqrestore(&dev->lock, flags);
}

static void bvd_tty_dev_hangup(struct tty_struct *tty)
{
	struct bvd_tty_dev *dev = (struct bvd_tty_dev *)tty->driver_data;
	unsigned long flags;

	if (!dev)
		return;

	BVD_DBG("hang up\n");

	bvd_tty_dev_flush_buffer(tty);
	spin_lock_irqsave(&dev->lock, flags);
	set_bit(BVD_TTY_DEV_HANGUP, &dev->flags);
	wake_up_interruptible(&dev->status_wait);
	spin_unlock_irqrestore(&dev->lock, flags);

	tty_port_hangup(&dev->tty_port);
	BVD_DBG("tty_port_hangup end\n");
}

static int bvd_tty_dev_tiocmget(struct tty_struct *tty)
{
	struct bvd_tty_dev	*dev = (struct bvd_tty_dev *)tty->driver_data;
	int					set;
	unsigned long flags;

	if (!dev)
		return 0;

	spin_lock_irqsave(&dev->lock, flags);
	set = dev->modem_status;
	spin_unlock_irqrestore(&dev->lock, flags);

	BVD_DBG("tiocmget %x\n", set);

	return set;
}

static int bvd_tty_dev_tiocmset(struct tty_struct *tty, unsigned int set,
							unsigned int clear)
{
	struct bvd_tty_dev *dev = (struct bvd_tty_dev *)tty->driver_data;
	unsigned long flags;

	if (!dev)
		return -ENODEV;

	spin_lock_irqsave(&dev->lock, flags);
	dev->tiocm.set		= set;
	dev->tiocm.clear	= clear;
	set_bit(BVD_TTY_DEV_TIOCMSET, &dev->flags);
	wake_up_interruptible(&dev->status_wait);
	spin_unlock_irqrestore(&dev->lock, flags);

	BVD_DBG("tiocmset  set = %x, clear = %x\n", set, clear);

	return 0;
}

static void bvd_tty_dev_cleanup(struct tty_struct *tty)
{
	struct bvd_tty_dev *dev = (struct bvd_tty_dev *)tty->driver_data;
	unsigned long flags;

	BVD_DBG("cleanup\n");

	if (!dev) {
		BVD_DBG("dev is null\n");
		return;
	}

	spin_lock_irqsave(&dev->lock, flags);
	tty->driver_data = NULL;
	skb_queue_purge(&dev->in_buff);
	spin_unlock_irqrestore(&dev->lock, flags);

	tty_port_put(&dev->tty_port);
	BVD_DBG("cleanup end\n");
}

static const struct tty_operations bvd_tty_ops = {
	.open				= bvd_tty_dev_open,
	.close				= bvd_tty_dev_close,
	.write				= bvd_tty_dev_write,
	.write_room			= bvd_tty_dev_write_room,
	.chars_in_buffer	= bvd_tty_dev_chars_in_buffer,
	.flush_buffer		= bvd_tty_dev_flush_buffer,
	.throttle			= bvd_tty_dev_throttle,
	.unthrottle			= bvd_tty_dev_unthrottle,
	.hangup				= bvd_tty_dev_hangup,
	.tiocmget			= bvd_tty_dev_tiocmget,
	.tiocmset			= bvd_tty_dev_tiocmset,
	.cleanup			= bvd_tty_dev_cleanup,
};

int __init bvd_tty_init(void)
{
	int ret = 0;

	bvd_tty_driver = alloc_tty_driver(BVD_TTY_PORTS);
	if (!bvd_tty_driver)
		return -ENOMEM;

	bvd_tty_driver->driver_name = "bvd_tty_driver";
	bvd_tty_driver->name = BVD_TTY_NAME;
	bvd_tty_driver->major = 0;
	bvd_tty_driver->minor_start = 0;
	bvd_tty_driver->type = TTY_DRIVER_TYPE_SERIAL;
	bvd_tty_driver->subtype = SERIAL_TYPE_NORMAL;
	bvd_tty_driver->flags = TTY_DRIVER_REAL_RAW|TTY_DRIVER_DYNAMIC_DEV;
	bvd_tty_driver->init_termios = tty_std_termios;
	bvd_tty_driver->init_termios.c_cflag = B9600|CS8|CREAD|HUPCL|CLOCAL;
	bvd_tty_driver->init_termios.c_lflag &= ~ICANON;

	tty_set_operations(bvd_tty_driver, &bvd_tty_ops);

	ret = tty_register_driver(bvd_tty_driver);
	if (ret) {
		BVD_DBG("register driver failed, ret %d\n", ret);
		put_tty_driver(bvd_tty_driver);
		return ret;
	}

	ret = bvd_device_register(BVD_DEVICE_TYPE_TTY, &bvd_tty_device_ops);
	if (ret)
		bvd_tty_exit();

	BVD_DBG("tty init end, ret %d\n", ret);
	return ret;
}

void bvd_tty_exit(void)
{
	bvd_device_unregister(BVD_DEVICE_TYPE_TTY);
	tty_unregister_driver(bvd_tty_driver);
	put_tty_driver(bvd_tty_driver);
	BVD_DBG("tty exit\n");
}
