/*
 * skeleton.c  -- ADIT Skeleton ALSA SoC Codec Driver
 * Copyright (C) 2012 Mentor Graphics, Inc.
 * Copyright (C) 2012 Advanced Driver Information Technology GmbH
 * All Rights Reserved.
 */
/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>

#include "skeleton.h"

#define SKELETON_FORMATS (SNDRV_PCM_FMTBIT_S8 |\
			SNDRV_PCM_FMTBIT_S16_LE | SNDRV_PCM_FMTBIT_S20_3LE |\
			SNDRV_PCM_FMTBIT_S24_LE | SNDRV_PCM_FMTBIT_S32_LE)

#ifdef CONFIG_SND_SOC_SKELETON_SYSFS
#define SYSFS_ACCESS	0644
#else
#define SYSFS_ACCESS	0444
#endif

static const char * const stream_rates[] = {
	"RATE_5512",
	"RATE_8000",
	"RATE_11025",
	"RATE_16000",
	"RATE_22050",
	"RATE_32000",
	"RATE_44100",
	"RATE_48000",
	"RATE_64000",
	"RATE_88200",
	"RATE_96000",
	"RATE_176400",
	"RATE_192000"
};

/* Private data for the skeleton, can be extended beyond basic functionality */
struct skeleton_private {
	struct snd_soc_codec *codec;
	unsigned int mclk; /* Input frequency of the MCLK pin */
	unsigned int mode; /* The mode (I2S or left-justified) */
	int direction;
	int dai_format;
	struct snd_soc_dai_driver *skeleton_dai;
	unsigned int attribute_unlock:1;
	unsigned int attribute_mute:1;
	unsigned char skel_playback_stream_name[MAX_STREAM_NAME_SIZE];
	unsigned char skel_capture_stream_name[MAX_STREAM_NAME_SIZE];
};

/**
 * skeleton_set_dai_sysclk - determine the SKELETON samples rates.
 * @codec_dai: the codec DAI
 * @clk_id: the clock ID (ignored)
 * @freq: the MCLK input frequency
 * @dir: the clock direction (ignored)
 *
 * This function is used to tell the codec driver what the input MCLK
 * frequency is.
 *
 */
static int skeleton_set_dai_sysclk(struct snd_soc_dai *codec_dai,
				   int clk_id, unsigned int freq, int dir)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct skeleton_private *skeleton = snd_soc_codec_get_drvdata(codec);

	dev_dbg(codec_dai->dev, "set sys clock\n");

	skeleton->mclk = freq;
	skeleton->direction = dir;

	return 0;
}

/*
 * Skeleton Clock dividers
 * @codec_dai: the codec DAI
 * @div_id: the ID of the divider reg (ignored)
 * @div: the divider value
 */
static int skeleton_set_dai_clkdiv(struct snd_soc_dai *codec_dai,
				   int div_id, int div)
{
	dev_dbg(codec_dai->dev, "set dai clock divider id:%d div:%d\n",
		div_id, div);

	return 0;
}

/**
 * skeleton_set_dai_fmt - configure the codec for the selected audio format
 * @codec_dai: the codec DAI
 * @format: a SND_SOC_DAIFMT_x value indicating the data format
 *
 * This function takes a bitmask of SND_SOC_DAIFMT_x bits and programs the
 * codec accordingly.
 */
static int skeleton_set_dai_fmt(struct snd_soc_dai *codec_dai,
				unsigned int format)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct skeleton_private *skeleton =  snd_soc_codec_get_drvdata(codec);

	dev_dbg(codec_dai->dev, "set dai format\n");
	skeleton->dai_format = format;

	return 0;
}

/**
 * skeleton_hw_params - program the SKELETON with the given hardware parameters.
 * @substream: the audio stream
 * @params: the hardware parameters to set
 * @dai: the SOC DAI is not used
 *
 * This function programs the hardware with the values provided.
 * Specifically, the sample rate and the data format.
 */
static int skeleton_hw_params(struct snd_pcm_substream *substream,
			      struct snd_pcm_hw_params *params,
			    struct snd_soc_dai *dai)
{
	dev_dbg(dai->dev, "set hw params\n");

	return 0;
}

/**
 * skeleton_shutdown - skeleton enters into low power mode again.
 * @substream: the audio stream
 * @dai: the SOC DAI (ignored)
 */
static void skeleton_shutdown(struct snd_pcm_substream *substream,
			      struct snd_soc_dai *dai)
{
	dev_dbg(dai->dev, "enter shutdown\n");
}

static const struct snd_soc_dai_ops skeleton_dai_ops = {
	.set_fmt	= skeleton_set_dai_fmt,
	.set_sysclk	= skeleton_set_dai_sysclk,
	.set_clkdiv	= skeleton_set_dai_clkdiv,
	.hw_params	= skeleton_hw_params,
	.shutdown	= skeleton_shutdown,
};

static const struct snd_soc_dai_driver skeleton_dai = {
	.name = "skeleton-dai",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 32,
		.rates = SNDRV_PCM_RATE_CONTINUOUS,
		.rate_min = 8000,
		.rate_max = 192000,
		.formats = SKELETON_FORMATS,
		.sig_bits = 32,
	},
	.capture = {
		.stream_name = "Capture",
		.channels_min = 1,
		.channels_max = 32,
		.rate_min = 8000,
		.rate_max = 192000,
		.rates = SNDRV_PCM_RATE_CONTINUOUS,
		.formats = SKELETON_FORMATS,
		.sig_bits = 32,
	},
	.ops = &skeleton_dai_ops,
};

static ssize_t show_unlock(struct device *dev, struct device_attribute *attr,
			   char *buf)
{
	struct skeleton_private *p = NULL;
	char *sym_status;

	p = dev_get_drvdata(dev);
	sym_status = p->attribute_unlock ? "UNLOCKED" : "LOCKED";

	return sprintf(buf, "%s:%s\n", attr->attr.name, sym_status);
}

static ssize_t store_unlock(struct device *dev, struct device_attribute *attr,
			    const char *buf, size_t count)
{
	struct skeleton_private *p = NULL;
	char temp[10];

	snprintf(temp, ARRAY_SIZE(temp), "%s", buf);

	p = dev_get_drvdata(dev);

	if (strncmp(temp, "UNLOCK\n", ARRAY_SIZE(temp)) == 0) {
		dev_info(dev, "Unlocked attributes\n");
		p->attribute_unlock = true;
	} else if (strncmp(temp, "LOCK\n", ARRAY_SIZE(temp)) == 0) {
		dev_info(dev, "Locked attributes\n");
		p->attribute_unlock = false;
	} else {
		dev_err(dev, "Invalid value: #%s#\n", temp);
		return -EINVAL;
	}

	return count;
}

static ssize_t show_mute(struct device *dev, struct device_attribute *attr,
			 char *buf)
{
	struct skeleton_private *p = NULL;
	char *sym_status;

	p = dev_get_drvdata(dev);
	sym_status = p->attribute_mute ? "MUTED" : "UNMUTED";

	return sprintf(buf, "%s:%s\n", attr->attr.name, sym_status);
}

static ssize_t store_mute(struct device *dev, struct device_attribute *attr,
			  const char *buf, size_t count)
{
	dev_info(dev, "cannot mute from sysfs, use ALSA kcontrol\n");
	return count;
}

static ssize_t show_dai_name(struct device *dev, struct device_attribute *attr,
			     char *buf)
{
	struct skeleton_private *p = NULL;

	p = dev_get_drvdata(dev);
	return sprintf(buf, "%s:%s\n", attr->attr.name,
			p->skeleton_dai->name);
}

static ssize_t store_dai_name(struct device *dev, struct device_attribute *attr,
			      const char *buf, size_t count)
{
	dev_info(dev, "%s:cannot change\n", attr->attr.name);
	return count;
}

static ssize_t show_symmetric_rate(struct device *dev,
				   struct device_attribute *attr,
				   char *buf)
{
	struct skeleton_private *p = NULL;
	char *is_sym;

	p = dev_get_drvdata(dev);
	is_sym = (p->skeleton_dai->symmetric_rates == 1) ? "YES" : "NO";

	return sprintf(buf, "%s:%s\n", attr->attr.name, is_sym);
}

static ssize_t store_symmetric_rate(struct device *dev,
				    struct device_attribute *attr,
				    const char *buf, size_t count)
{
	struct skeleton_private *p = NULL;
	char temp[10];

	snprintf(temp, ARRAY_SIZE(temp), "%s", buf);

	p = dev_get_drvdata(dev);
	if (!p->attribute_unlock) {
		dev_info(dev, "unlock attributes first\n");
	} else {
		if (strncmp(temp, "YES\n", ARRAY_SIZE(temp)) == 0) {
			dev_info(dev, "Symmetric Rates Set\n");
			p->skeleton_dai->symmetric_rates = 1;
		} else if (strncmp(temp, "NO\n", ARRAY_SIZE(temp)) == 0) {
			dev_info(dev, "Symmetric Rates Unset\n");
			p->skeleton_dai->symmetric_rates = 0;
		} else {
			dev_err(dev, "Invalid value\n");
			return -EINVAL;
		}
	}

	return count;
}

/*
 * generates show/store functions for c_rates/p_rates attribute value
 */
#define gen_func_rates(d, stream)					\
static ssize_t show_##d##_rates(struct device *dev,			\
		struct device_attribute *attr, char *buf)		\
{									\
	struct skeleton_private *p = NULL;				\
	unsigned long rates;						\
	int i, count;							\
									\
	p = dev_get_drvdata(dev);					\
	rates = p->skeleton_dai->stream.rates;				\
									\
	count = sprintf(buf, "supported rates:\n");			\
									\
	for (i = 0; i < ARRAY_SIZE(stream_rates); i++) {		\
		if ((1 << i) & rates)					\
			count += sprintf(buf + count,			\
					 "%s\n", stream_rates[i]);	\
	}								\
									\
	if (SNDRV_PCM_RATE_CONTINUOUS & rates)				\
		count += sprintf(buf + count,				\
				 "%s\n", "RATE_CONTINUOUS");		\
									\
	return count;							\
}									\
									\
static ssize_t store_##d##_rates(struct device *dev,			\
		struct device_attribute *attr,				\
		const char *buf, size_t count)				\
{									\
	struct skeleton_private *p = NULL;				\
	unsigned long rates = 0;					\
	int i;								\
									\
	for (i = 0; i < ARRAY_SIZE(stream_rates); i++)			\
		if (strnstr(buf, stream_rates[i], count))		\
			rates |= (1 << i);				\
									\
	p = dev_get_drvdata(dev);					\
									\
	if (!p->attribute_unlock)					\
		dev_info(dev, "unlock attributes first\n");		\
	else								\
		p->skeleton_dai->stream.rates = rates;			\
									\
	return count;							\
}									\

/*
 * generates show/store functions for
 * c_chan_min/c_chan_max/p_chan_min/p_chan_max attribute value
 */
#define gen_func_chan(d, stream, min_max)				\
static ssize_t show_##d##_chan_##min_max(struct device *dev,		\
		struct device_attribute *attr, char *buf)		\
{									\
	unsigned int value;						\
	struct skeleton_private *p = NULL;				\
									\
	p = dev_get_drvdata(dev);					\
	value = p->skeleton_dai->stream.channels_##min_max;		\
									\
	return sprintf(buf, "%s:%u\n", attr->attr.name, value);		\
}									\
									\
static ssize_t store_##d##_chan_##min_max(struct device *dev,		\
		struct device_attribute *attr,				\
		const char *buf, size_t count)				\
{									\
	unsigned int value;						\
	struct skeleton_private *p = NULL;				\
									\
	if (kstrtouint(buf, 0, &value)) {				\
		dev_err(dev, "Invalid value\n");			\
		return -EINVAL;						\
	}								\
									\
	p = dev_get_drvdata(dev);					\
									\
	if (!p->attribute_unlock)					\
		dev_info(dev, "unlock attributes first\n");		\
	else								\
		p->skeleton_dai->stream.channels_##min_max = value;	\
									\
	return count;							\
}									\

/*
 * generates show/store functions for
 * c_rate_min/c_rate_max/p_rate_min/p_rate_max attribute value
 */
#define gen_func_rate(d, stream, min_max)				\
static ssize_t show_##d##_rate_##min_max(struct device *dev,		\
		struct device_attribute *attr, char *buf)		\
{									\
	unsigned int value;						\
	struct skeleton_private *p = NULL;				\
									\
	p = dev_get_drvdata(dev);					\
	value = p->skeleton_dai->stream.rate_##min_max;			\
									\
	return sprintf(buf, "%s:%u\n", attr->attr.name, value);		\
}									\
									\
static ssize_t store_##d##_rate_##min_max(struct device *dev,		\
		struct device_attribute *attr,				\
		 const char *buf, size_t count)				\
{									\
	unsigned int value;						\
	struct skeleton_private *p = NULL;				\
									\
	if (kstrtouint(buf, 0, &value)) {				\
		dev_err(dev, "Invalid value\n");			\
		return -EINVAL;						\
	}								\
									\
	p = dev_get_drvdata(dev);					\
									\
	if (!p->attribute_unlock)					\
		dev_info(dev, "unlock attributes first\n");		\
	else								\
		p->skeleton_dai->stream.rate_##min_max = value;		\
									\
	return count;							\
}									\

/*
 * generates show/store functions for c_sigbits/p_sigbits attribute value
 */
#define gen_func_sigbits(d, stream)					\
static ssize_t show_##d##_sigbits(struct device *dev,			\
		struct device_attribute *attr, char *buf)		\
{									\
	unsigned int value;						\
	struct skeleton_private *p = NULL;				\
									\
	p = dev_get_drvdata(dev);					\
	value = p->skeleton_dai->stream.sig_bits;			\
									\
	return sprintf(buf, "%s:0x%08x\n", attr->attr.name, value);	\
}									\
									\
static ssize_t store_##d##_sigbits(struct device *dev,			\
		struct device_attribute *attr, const char *buf,		\
		size_t count)						\
{									\
	unsigned int value;						\
	struct skeleton_private *p = NULL;				\
									\
	if (kstrtouint(buf, 0, &value)) {				\
		dev_err(dev, "Invalid value\n");			\
		return -EINVAL;						\
	}								\
									\
	p = dev_get_drvdata(dev);					\
									\
	if (!p->attribute_unlock)					\
		dev_info(dev, "unlock attributes first\n");		\
	else								\
		p->skeleton_dai->stream.sig_bits = value;		\
									\
	return count;							\
}									\

/*
 * generates show/store functions for c_formats/p_formats attribute value
 */
#define gen_func_formats(d, stream)					\
static ssize_t show_##d##_formats(struct device *dev,			\
		struct device_attribute *attr, char *buf)		\
{									\
	struct skeleton_private *p = NULL;				\
	u64 format;							\
									\
	p = dev_get_drvdata(dev);					\
	format = p->skeleton_dai->stream.formats;			\
									\
	return sprintf(buf, "%s:0x%016llX\n", attr->attr.name, format);	\
}									\
									\
static ssize_t store_##d##_formats(struct device *dev,			\
		struct device_attribute *attr,				\
		 const char *buf, size_t count)				\
{									\
	struct skeleton_private *p = NULL;				\
	u64 format;							\
									\
	if (kstrtoull(buf, 0, &format)) {				\
		dev_err(dev, "Invalid value\n");			\
		return -EINVAL;						\
	}								\
									\
	p = dev_get_drvdata(dev);					\
									\
	if (!p->attribute_unlock)					\
		dev_info(dev, "unlock attributes first\n");		\
	else								\
		p->skeleton_dai->stream.formats = format;		\
									\
	return count;							\
}									\

/* generate functions */
gen_func_rates(p, playback);
gen_func_rates(c, capture);

gen_func_chan(p, playback, min);
gen_func_chan(p, playback, max);
gen_func_chan(c, capture, min);
gen_func_chan(c, capture, max);

gen_func_rate(p, playback, min);
gen_func_rate(p, playback, max);
gen_func_rate(c, capture, min);
gen_func_rate(c, capture, max);

gen_func_sigbits(p, playback);
gen_func_sigbits(c, capture);

gen_func_formats(p, playback);
gen_func_formats(c, capture);

static DEVICE_ATTR(dai_name, SYSFS_ACCESS,
		show_dai_name, store_dai_name);
static DEVICE_ATTR(symmetric_rates, SYSFS_ACCESS,
		show_symmetric_rate,
		store_symmetric_rate);
static DEVICE_ATTR(unlock, SYSFS_ACCESS,
		show_unlock, store_unlock);
static DEVICE_ATTR(mute, SYSFS_ACCESS, show_mute, store_mute);
static DEVICE_ATTR(c_sigbits, SYSFS_ACCESS,
		show_c_sigbits, store_c_sigbits);
static DEVICE_ATTR(c_chan_max, SYSFS_ACCESS,
		show_c_chan_max, store_c_chan_max);
static DEVICE_ATTR(c_chan_min, SYSFS_ACCESS,
		show_c_chan_min, store_c_chan_min);
static DEVICE_ATTR(c_rate_max, SYSFS_ACCESS,
		show_c_rate_max, store_c_rate_max);
static DEVICE_ATTR(c_rate_min, SYSFS_ACCESS,
		show_c_rate_min, store_c_rate_min);
static DEVICE_ATTR(c_rates, SYSFS_ACCESS,
		show_c_rates, store_c_rates);
static DEVICE_ATTR(c_formats, SYSFS_ACCESS,
		show_c_formats, store_c_formats);
static DEVICE_ATTR(p_sigbits, SYSFS_ACCESS,
		show_p_sigbits, store_p_sigbits);
static DEVICE_ATTR(p_chan_max, SYSFS_ACCESS,
		show_p_chan_max, store_p_chan_max);
static DEVICE_ATTR(p_chan_min, SYSFS_ACCESS,
		show_p_chan_min, store_p_chan_min);
static DEVICE_ATTR(p_rate_max, SYSFS_ACCESS,
		show_p_rate_max, store_p_rate_max);
static DEVICE_ATTR(p_rate_min, SYSFS_ACCESS,
		show_p_rate_min, store_p_rate_min);
static DEVICE_ATTR(p_rates, SYSFS_ACCESS,
		show_p_rates, store_p_rates);
static DEVICE_ATTR(p_formats, SYSFS_ACCESS,
		show_p_formats, store_p_formats);

static struct attribute *skel_attrs[] = {
	&dev_attr_dai_name.attr,
	&dev_attr_symmetric_rates.attr,
	&dev_attr_unlock.attr,
	&dev_attr_mute.attr,
	&dev_attr_c_sigbits.attr,
	&dev_attr_c_chan_max.attr,
	&dev_attr_c_chan_min.attr,
	&dev_attr_c_rate_max.attr,
	&dev_attr_c_rate_min.attr,
	&dev_attr_c_rates.attr,
	&dev_attr_c_formats.attr,
	&dev_attr_p_sigbits.attr,
	&dev_attr_p_chan_max.attr,
	&dev_attr_p_chan_min.attr,
	&dev_attr_p_rate_max.attr,
	&dev_attr_p_rate_min.attr,
	&dev_attr_p_rates.attr,
	&dev_attr_p_formats.attr,
	NULL
};

static const struct attribute_group skel_attr_group = {
	.name = "skel_attribs",
	.attrs = skel_attrs
};

/**
 * skeleton_probe - ASoC probe function
 * @pdev: platform device
 *
 * This function is called when ASoC has all the pieces it needs to
 * instantiate a sound driver.
 */
static int skeleton_probe(struct snd_soc_codec *codec)
{
	dev_dbg(codec->dev, "probing snd soc codec\n");

	/* TODO - Add the non-DAPM controls */

	return 0;
}

/**
 * skeleton_remove - ASoC remove function
 * @pdev: platform device
 *
 * This function is the counterpart to skeleton_probe().
 */
static int skeleton_remove(struct snd_soc_codec *codec)
{
	dev_dbg(codec->dev, "removing snd soc codec\n");
	return 0;
};

/*
 * ASoC codec device structure
 *
 * Assign this variable to the codec_dev field of the machine driver's
 * snd_soc_device structure.
 */
static const struct snd_soc_codec_driver skeleton_driver = {
	.probe =	skeleton_probe,
	.remove =	skeleton_remove,
	.reg_cache_size = 0,
	.reg_word_size = sizeof(u8),
	.reg_cache_step = 1,
};

static const struct of_device_id skeleton_ids[] = {
	{ .compatible = "asoc,limco", },
	{}
};
MODULE_DEVICE_TABLE(of, skeleton_ids);

static int skeleton_fake_parse_fmt(struct platform_device *pdev,
				   struct device_node *from,
				   const char *name,
				   struct snd_soc_pcm_stream *stream)
{
	struct device_node *np;
	unsigned int ret, val;

	if (!from || !stream)
		return 1;

	np = of_get_child_by_name(from, name);
	if (!np)
		return 1;

	/* parse boundary channel count */
	ret = of_property_read_u32(np, "ch_mm", &val);
	if (!ret) {
		stream->channels_min = val;
		stream->channels_max = val;
	} else if (ret == -EINVAL) {
		ret = of_property_read_u32(np,
					   "ch_min", &stream->channels_min);
		if (!(!ret || ret == -EINVAL))
			dev_err(&pdev->dev,
				"DT reading ch_min (%i)\n", ret);
		ret = of_property_read_u32(np,
					   "ch_max", &stream->channels_max);
		if (!(!ret || ret == -EINVAL))
			dev_err(&pdev->dev,
				"DT reading ch_max (%i)\n", ret);
	} else {
		dev_err(&pdev->dev, "DT reading ch_mm (%i)\n", ret);
	}

	/* parse formats (INFO: codec fixup is done in soc-core.c) */
	of_property_read_u64(np, "fmts", &stream->formats);
	if (!(!ret || ret == -EINVAL))
		dev_err(&pdev->dev, "DT reading fmts (%i)\n", ret);

	/* parse supported rates */
	ret = of_property_match_string(np, "rates", "continuous");
	if (!ret) {
		stream->rates = SNDRV_PCM_RATE_CONTINUOUS;
	} else {
		ret = of_property_read_u32(np, "rates", &stream->rates);
		if (!(!ret || ret == -EINVAL))
			dev_err(&pdev->dev,
				"DT reading rates (%i)\n", ret);
	}

	/* parse boundary rates */
	ret = of_property_read_u32(np, "rate_mm", &val);
	if (!ret) {
		stream->rate_min = val;
		stream->rate_max = val;
	} else if (ret == -EINVAL) {
		ret = of_property_read_u32(np,
					   "rate_min", &stream->rate_min);
		if (!(!ret || ret == -EINVAL))
			dev_err(&pdev->dev,
				"DT reading rate_min (%i)\n", ret);
		ret = of_property_read_u32(np, "rate_max",
					   &stream->rate_max);
		if (!(!ret || ret == -EINVAL))
			dev_err(&pdev->dev,
				"DT reading rate_max (%i)\n", ret);
	} else {
		dev_err(&pdev->dev, "DT reading rate_mm (%i)\n", ret);
	}

	of_node_put(np);
	return 0;
}

static int skeleton_fake_probe(struct platform_device *pdev)
{
	struct skeleton_private *priv;
	struct device_node *np = pdev->dev.of_node;
	const char *sprop;
	int ret;

#ifdef CONFIG_SND_SOC_SKELETON_SYSFS
	dev_dbg(&pdev->dev, "ADIT skeleton ALSA SoC Codec Driver with sysfs\n");
#else
	dev_dbg(&pdev->dev, "ADIT skeleton ALSA SoC Codec Driver\n");
#endif
	priv = kzalloc(sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->skeleton_dai = kzalloc(sizeof(*priv->skeleton_dai),
			GFP_KERNEL);

	if (!priv->skeleton_dai) {
		ret = -ENOMEM;
		goto fail;
	}

	memcpy(priv->skeleton_dai, &skeleton_dai,
	       sizeof(struct snd_soc_dai_driver));

	sprop = (char *)of_get_property(np, "dai_name", NULL);
	if (!sprop) {
		dev_err(&pdev->dev, "dai_name not provided\n");
		return -ENODEV;
	}

	priv->skeleton_dai->name = sprop;

	/* Assigning stream names */
	snprintf(priv->skel_playback_stream_name,
		 MAX_STREAM_NAME_SIZE,
		 "%s_playback", sprop);
	snprintf(priv->skel_capture_stream_name,
		 MAX_STREAM_NAME_SIZE,
		 "%s_capture", sprop);
	priv->skeleton_dai->playback.stream_name =
			priv->skel_playback_stream_name;
	priv->skeleton_dai->capture.stream_name =
			priv->skel_capture_stream_name;

	/* find our symmetric rate */
	if (of_find_property(np, "symmetric_rates", NULL))
		priv->skeleton_dai->symmetric_rates = 1;
	else
		priv->skeleton_dai->symmetric_rates = 0;

	if (!skeleton_fake_parse_fmt(pdev, np, "pb_cap",
				     &priv->skeleton_dai->playback)) {
		priv->skeleton_dai->capture.channels_min =
				priv->skeleton_dai->playback.channels_min;
		priv->skeleton_dai->capture.channels_max =
				priv->skeleton_dai->playback.channels_max;
		priv->skeleton_dai->capture.formats =
				priv->skeleton_dai->playback.formats;
		priv->skeleton_dai->capture.rates =
				priv->skeleton_dai->playback.rates;
		priv->skeleton_dai->capture.rate_min =
				priv->skeleton_dai->playback.rate_min;
		priv->skeleton_dai->capture.rate_max =
				priv->skeleton_dai->playback.rate_max;
	} else {
		skeleton_fake_parse_fmt(pdev, np, "pb",
					&priv->skeleton_dai->playback);
		skeleton_fake_parse_fmt(pdev, np, "cap",
					&priv->skeleton_dai->capture);
	}

	ret = sysfs_create_group(&pdev->dev.kobj, &skel_attr_group);
	if (ret) {
		dev_err(&pdev->dev, "could not create system group\n");
		goto fail_kobj;
	}

	dev_set_drvdata(&pdev->dev, priv);

	ret = snd_soc_register_codec(&pdev->dev,
				     &skeleton_driver, priv->skeleton_dai, 1);
	if (ret) {
		dev_err(&pdev->dev, "failed to register snd soc codec\n");
		goto fail_group;
	}

	return 0;

fail_group:
	sysfs_remove_group(&pdev->dev.kobj, &skel_attr_group);
fail_kobj:
	dev_set_drvdata(&pdev->dev, NULL);
	kfree(priv->skeleton_dai);
fail:
	kfree(priv);
	return ret;
}

static int skeleton_fake_remove(struct platform_device *pdev)
{
	struct skeleton_private *priv = dev_get_drvdata(&pdev->dev);

	snd_soc_unregister_codec(&pdev->dev);
	sysfs_remove_group(&pdev->dev.kobj, &skel_attr_group);
	dev_set_drvdata(&pdev->dev, NULL);
	kfree(priv->skeleton_dai);
	kfree(priv);
	return 0;
}

static struct platform_driver skeleton_fake_driver = {
	.driver = {
		.name = "adit-skeleton",
		.owner = THIS_MODULE,
		.of_match_table = skeleton_ids,
	},
	.probe = skeleton_fake_probe,
	.remove = skeleton_fake_remove,
};

module_platform_driver(skeleton_fake_driver);

MODULE_AUTHOR("Joshua Frkuska <joshua_frkuska@mentor.com>");
MODULE_DESCRIPTION("ADIT skeleton ALSA SoC Codec Driver");
MODULE_LICENSE("GPL");
