/*
 * Copyright (C) 2012 Mentor Graphics Corp.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/clk.h>

#include <sound/soc.h>
#include <sound/soc-dpcm.h>
#include <sound/tlv.h>
#include <sound/pcm_params.h>

#define SND_KCTL_MAX_NAME_LEN			64

/* Stereo (2 slots) */
#define DAI_LINK_SLOTS_DEFAULT			2
/* 32-bit slot width */
#define DAI_LINK_SLOT_WIDTH_DEFAULT		32
/* I2S mode & NORMAL polarity */
#define DAI_LINK_FMT_DEFAULT			(SND_SOC_DAIFMT_I2S | \
						SND_SOC_DAIFMT_NB_NF)
/* NORMAL polarity */
#define DAI_LINK_CODEC_FMT_DEFAULT		SND_SOC_DAIFMT_NB_NF
/* CPU DAI to determine the mask */
#define DAI_LINK_TX_MASK_DEFAULT		0x0
#define DAI_LINK_RX_MASK_DEFAULT		0x0
/* sysclk framesink of CBM_CFM */
#define DAI_LINK_CLKFSY_DEFAULT			0

#define ASOC_GENERIC_PCM_DYNAMIC		100

/*  sysclk clkdiv table defines for keyvalue and header lengths
 *  Note: KEYVAL_LEN should not be 0
 */
#define SYS_HEADER_LEN		0
#define SYS_KEYVAL_LEN		3
#define RSYS_HEADER_LEN		2
#define DIV_HEADER_LEN		0
#define DIV_KEYVAL_LEN		2
#define RDIV_HEADER_LEN		2

#define SYSDIV_TABLE_CELL_MIN	1
#define SYSDIV_TABLE_CELL_MAX	64

#define ASOC_GENERIC_DT_CLK_CONFIG	"asoc,audio-gmd-clkcfg"
#define ASOC_GENERIC_DT_DAI_LINK	"asoc,audio-gmd-dai-link"

#define RSYS_CELL_TAG	"#rate-sysclk-cells"
#define SYS_CELL_TAG	"#sysclk-cells"
#define RDIV_CELL_TAG	"#rate-clkdiv-cells"
#define DIV_CELL_TAG	"#clkdiv-cells"
#define RSYS_TAG	"rate-sysclk"
#define SYS_TAG		"sysclk"
#define RDIV_TAG	"rate-clkdiv"
#define DIV_TAG		"clkdiv"
#define DAI_PLAYBACK	"playback"
#define DAI_CAPTURE	"capture"
#define DAI_CELL_TAG	"#sound-dai-cells"


#define ASOC_GENERIC_VALUE_INT(xname, xmax, xget, xput) \
{	.iface = SNDRV_CTL_ELEM_IFACE_MIXER, .name = xname, \
	.info = snd_soc_info_volsw, \
	.get = xget, \
	.put = xput, \
	.private_value = SOC_SINGLE_VALUE(0, 0, xmax, 0, 0) }

static int snd_soc_info_masks(struct snd_kcontrol *kcontrol,
			      struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;

	/* To handle 32 bit wide masks
	 * we need 2 integer values from 0 to 0xffff.
	 * Amixer currently can not parse -xyz values.
	 */
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 0xffff;
	return 0;
}

#define ASOC_GENERIC_VALUE_MASKS(xname, xget, xput) \
{	.iface = SNDRV_CTL_ELEM_IFACE_MIXER, .name = xname, \
	.info = snd_soc_info_masks, \
	.get = xget, \
	.put = xput}

struct sysdiv_table {
	unsigned int **table;
	unsigned int row_cell_cnt;
	unsigned int row_cnt;
};

/* DAI system clock and clock divider struct */
struct asoc_generic_dai_sysdiv_data {
	struct sysdiv_table rate_sys;
	struct sysdiv_table sys;
	struct sysdiv_table rate_div;
	struct sysdiv_table div;
};

struct asoc_generic_dai_link_data {
	int slot_width;
	unsigned int tx_mask;
	unsigned int rx_mask;
	unsigned int slots;
	struct asoc_generic_dai_sysdiv_data *cpu_sysdiv_conf;
	struct asoc_generic_dai_sysdiv_data *codec_sysdiv_conf;
	unsigned int format;
	unsigned int codec_format;
	unsigned int clkfsy;
	unsigned int be_format[2];
	unsigned int be_channels[2];
	int be_samplerate[2];
};

struct asoc_generic_data {
	struct snd_soc_card card;
	struct platform_device *pdev;
	struct snd_soc_dai_link *dai_link;
	struct asoc_generic_dai_link_data *dai_link_data;
};

static const char * const hw_clk_master_text[] = {
	"CBM_CFM",
	"CBS_CFM",
	"CBM_CFS",
	"CBS_CFS"

};

static unsigned int hw_clk_master_values[] = {
	SND_SOC_DAIFMT_CBM_CFM,
	SND_SOC_DAIFMT_CBS_CFM,
	SND_SOC_DAIFMT_CBM_CFS,
	SND_SOC_DAIFMT_CBS_CFS
};

static inline int of_get_compat_count(const struct device_node *np)
{
	struct device_node *child;
	int num = 0;

	for_each_child_of_node(np, child)
		num++;

	return num;
}

static int call_sysclk(struct platform_device *pdev,
		       struct snd_soc_dai *dai,
		       struct sysdiv_table *data,
		       int header_len,
		       int idx)
{
	int cells, j, ret = 0;

	cells = data->row_cell_cnt * SYS_KEYVAL_LEN;

	if (data->table) {
		for (j = 0; j < cells; j += SYS_KEYVAL_LEN) {
			ret = snd_soc_dai_set_sysclk(
				dai,
				data->table[idx][header_len + j],
				data->table[idx][header_len + j + 1],
				data->table[idx][header_len + j + 2]);
			if (ret && ret != -ENODEV) {
				dev_err(&pdev->dev,
					"unable to set dai sysclk %d %d %d",
					data->table[idx][header_len + j],
					data->table[idx][header_len + j + 1],
					data->table[idx][header_len + j + 2]);
				return ret;
			}

			dev_dbg(&pdev->dev,
				"set dai sysclk %d %d %d",
				data->table[idx][header_len + j],
				data->table[idx][header_len + j + 1],
				data->table[idx][header_len + j + 2]);
		}
	}

	return ret;
}

static int call_rate_sysclk(struct platform_device *pdev,
			    struct snd_soc_dai *dai,
			    struct sysdiv_table *data,
			    unsigned int bclk) {
	int i, ret = 0, found = 0;

	if (data->table) {
		for (i = 0 ; i < data->row_cnt; i++) {
			if ((bclk >= data->table[i][0]) &&
			    (bclk < data->table[i][1])) {
				dev_dbg(&pdev->dev,
					"set dai sysclk table rate: %d - %d",
					data->table[i][0],
					data->table[i][1]);
				ret = call_sysclk(pdev,
						  dai,
						  data,
						  RSYS_HEADER_LEN,
						  i);
				if (ret)
					return ret;
				found = 1;
			}
		}
		if (!found) {
			dev_err(&pdev->dev,
				"no sysclk table entry found for rate %d",
				bclk);
			ret = -EINVAL;
		}
	}
	return ret;
}

static int call_clkdiv(struct platform_device *pdev,
		       struct snd_soc_dai *dai,
		       struct sysdiv_table *data,
		       int header_len,
		       int idx) {
	int cells, j, ret = 0;

	cells = data->row_cell_cnt * DIV_KEYVAL_LEN;

	if (data->table)
		for (j = 0; j < cells; j += DIV_KEYVAL_LEN) {
			ret = snd_soc_dai_set_clkdiv(
				dai,
				data->table[idx][header_len + j],
				data->table[idx][header_len + j + 1]);
			if (ret && ret != -ENODEV) {
				dev_err(&pdev->dev,
					"unable to set dai clkdiv %d %d",
					data->table[idx][header_len + j],
					data->table[idx][header_len + j + 1]);
				return ret;
			}
			dev_dbg(&pdev->dev,
				"set dai clkdiv %d %d",
				data->table[idx][header_len + j],
				data->table[idx][header_len + j + 1]);
		}

	return ret;
}

static int call_rate_clkdiv(struct platform_device *pdev,
			    struct snd_soc_dai *dai,
			    struct sysdiv_table *data,
			    unsigned int bclk) {
	int i, ret = 0, found = 0;

	if (data->table) {
		for (i = 0 ; i < data->row_cnt; i++) {
			if ((bclk >= data->table[i][0]) &&
			    (bclk < data->table[i][1])) {
				dev_dbg(&pdev->dev,
					"set dai clkdiv table rate %d - %d",
					data->table[i][0],
					data->table[i][1]);
				ret = call_clkdiv(pdev,
						  dai,
						  data,
						  RDIV_HEADER_LEN,
						  i);
				if (ret)
					return ret;
				found = 1;
			}
		}
		if (!found) {
			dev_err(&pdev->dev,
				"no clkdiv table entry found for rate %d",
				bclk);
			ret = -EINVAL;
		}
	}
	return ret;
}

static int asoc_generic_find_rtd_idx(struct snd_soc_pcm_runtime *rtd)
{
	int i = 0;
	struct snd_soc_card *card = rtd->card;
	struct snd_soc_pcm_runtime *rtdl;

	list_for_each_entry(rtdl, &card->rtd_list, list) {
		if (rtdl == rtd)
			return i;
		i++;
	}

	return -EINVAL;
}

static int asoc_generic_get_dai_link_data_for_rtd(
		struct snd_soc_pcm_runtime *rtd,
		struct asoc_generic_dai_link_data **link_data,
		struct platform_device **pdev)
{
	struct asoc_generic_data *data = container_of(rtd->card,
					struct asoc_generic_data, card);
	int idx;

	idx = asoc_generic_find_rtd_idx(rtd);
	if (idx < 0)
		return idx;

	if (idx >= data->card.num_links) {
		dev_err(&data->pdev->dev, "rtd index out of range");
		return -EINVAL;
	}

	*link_data = &data->dai_link_data[idx];
	*pdev = data->pdev;
	return idx;
}

static int asoc_generic_call_sysdiv(
		struct platform_device *pdev,
		struct asoc_generic_dai_sysdiv_data *data,
		struct snd_soc_dai *dai)
{
	int ret = 0;

	if (data && dai) {
		/* dai-clk */
		ret = call_sysclk(pdev, dai,
				  &data->sys, SYS_HEADER_LEN, 0);
		if (ret)
			return ret;

		/* dai-div */
		ret = call_clkdiv(pdev, dai,
				  &data->div, DIV_HEADER_LEN, 0);
	}

	return ret;
}

static int asoc_generic_init(struct snd_soc_pcm_runtime *rtd)
{
	struct asoc_generic_dai_link_data *data;
	struct platform_device *pdev;
	int ret = 0;

	ret = asoc_generic_get_dai_link_data_for_rtd(rtd, &data, &pdev);
	if (ret < 0)
		return ret;

	/* call sysclk and clkdiv for cpu dai */
	ret = asoc_generic_call_sysdiv(pdev, data->cpu_sysdiv_conf,
				       rtd->cpu_dai);
	if (ret)
		return ret;
	/* call sysclk and clkdiv for codec dai */
	ret = asoc_generic_call_sysdiv(pdev, data->codec_sysdiv_conf,
				       rtd->codec_dai);

	return ret;
}

static int asoc_generic_hw_params(struct snd_pcm_substream *substream,
				  struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct asoc_generic_dai_link_data *data;
	struct platform_device *pdev;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	unsigned int rate = params_rate(params);
	unsigned int bclk = 0;
	int ret;

	ret = asoc_generic_get_dai_link_data_for_rtd(rtd, &data, &pdev);
	if (ret < 0)
		return ret;

	dev_dbg(&pdev->dev, "asoc_generic_hw_params rtd idx %i\n", ret);

	bclk = data->slots * rate * data->slot_width;

	if (data->cpu_sysdiv_conf) {
		/* call only on first open */
		if ((cpu_dai->rate == 0) || (cpu_dai->active < 2)) {
			/* cpu rate_sysclk */
			if (data->cpu_sysdiv_conf->rate_sys.table) {
				if (bclk == 0) {
					ret = -EINVAL;
					goto fail_bck;
				}

				ret = call_rate_sysclk(
					pdev,
					cpu_dai,
					&data->cpu_sysdiv_conf->rate_sys,
					bclk);
				if (ret)
					return ret;
			}

			/* cpu dai rate dividers */
			if (data->cpu_sysdiv_conf->rate_div.table) {
				if (bclk == 0) {
					ret = -EINVAL;
					goto fail_bck;
				}
				ret = call_rate_clkdiv(
					pdev,
					cpu_dai,
					&data->cpu_sysdiv_conf->rate_div,
					bclk);
				if (ret)
					return ret;
			}
		} else if (cpu_dai->rate != params_rate(params)) {
			dev_warn(&pdev->dev,
				 "warning, synchronous cpu-dai rate mismatch - requested: %u, current: %u",
				 cpu_dai->rate, params_rate(params));
		}
	}

	if (data->codec_sysdiv_conf) {
		/* call only on first open */
		if ((codec_dai->rate == 0) || (codec_dai->active < 2)) {
			/* codec rate_sysclk */
			if (data->codec_sysdiv_conf->rate_sys.table) {
				if (bclk == 0) {
					ret = -EINVAL;
					goto fail_bck;
				}
				ret = call_rate_sysclk(
					pdev,
					codec_dai,
					&data->codec_sysdiv_conf->rate_sys,
					bclk);
				if (ret)
					return ret;
			}

			/* codec dai dividers */
			if (data->codec_sysdiv_conf->rate_div.table) {
				if (bclk == 0) {
					ret = -EINVAL;
					goto fail_bck;
				}
				ret = call_rate_clkdiv(
					pdev,
					codec_dai,
					&data->codec_sysdiv_conf->rate_div,
					bclk);
				if (ret)
					return ret;
			}
		} else if (codec_dai->rate != params_rate(params)) {
			dev_warn(&pdev->dev,
				 "warning, synchronous codec-dai rate mismatch - requested: %u, current: %u",
				 codec_dai->rate, params_rate(params));
		}
	}

	return 0;
fail_bck:
	dev_err(&pdev->dev,
		"unable to calculate bck; slots or slot_width = 0");
	return ret;
}

static int asoc_generic_check_constraints(
				struct asoc_generic_dai_link_data *data,
				struct device *dev)
{
	int ret = 0;

	switch (data->format & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
	case SND_SOC_DAIFMT_RIGHT_J:
	case SND_SOC_DAIFMT_LEFT_J:
		if (data->slots != 2) {
			dev_err(dev,
				"formats I2S, RJF and LJF can only support slots =2\n");
			ret = -EINVAL;
		}
		break;
	}

	return ret;
}

static int asoc_generic_startup(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct asoc_generic_dai_link_data *data;
	struct platform_device *pdev;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	unsigned int format;
	int ret;

	ret = asoc_generic_get_dai_link_data_for_rtd(rtd, &data, &pdev);
	if (ret < 0)
		return ret;

	ret = asoc_generic_check_constraints(data, &pdev->dev);
	if (ret)
		return ret;

	dev_dbg(&pdev->dev, "asoc_generic_startup rtd idx %i\n", ret);

	/* call only on first open */
	if (cpu_dai->active == 0) {
		/* call sysclk and clkdiv for cpu dai */
		ret = asoc_generic_call_sysdiv(pdev, data->cpu_sysdiv_conf,
					       cpu_dai);
		if (ret)
			return ret;
	}

	/* call only on first open */
	if (codec_dai->active == 0) {
		/* call sysclk and clkdiv for codec dai */
		ret = asoc_generic_call_sysdiv(pdev, data->codec_sysdiv_conf,
					       codec_dai);
		if (ret)
			return ret;
	}

	/* call only on first open */
	if (cpu_dai->active == 0) {
		ret = snd_soc_dai_set_tdm_slot(cpu_dai,
					       data->tx_mask,
					       data->rx_mask,
					       data->slots,
					       data->slot_width);
		if (ret && ret != -ENOTSUPP) {
			dev_err(&pdev->dev,
				"unable to set cpu dai tdm params %x %x %d %d",
				data->tx_mask,
				data->rx_mask,
				data->slots,
				data->slot_width);
			return ret;
		}
	}

	/* call only on first open */
	if (codec_dai->active == 0) {
		ret = snd_soc_dai_set_tdm_slot(codec_dai,
					       data->tx_mask,
					       data->rx_mask,
					       data->slots,
					       data->slot_width);
		if (ret && ret != -ENOTSUPP) {
			dev_err(&pdev->dev,
				"unable to set codec dai tdm params %x %x %d %d",
				data->tx_mask,
				data->rx_mask,
				data->slots,
				data->slot_width);
			return ret;
		}
	}

	format = data->format | hw_clk_master_values[data->clkfsy];

	/* call only on first open */
	if (cpu_dai->active == 0) {
		ret = snd_soc_dai_set_fmt(cpu_dai, format);
		if (ret && ret != -ENOTSUPP) {
			dev_err(&pdev->dev, "unable to set cpu dai format %x",
				format);
			return ret;
		}
	}

	/* call only on first open */
	if (codec_dai->active == 0) {
		format &= ~SND_SOC_DAIFMT_INV_MASK;
		format |= data->codec_format;

		ret = snd_soc_dai_set_fmt(codec_dai, format);
		if (ret && ret != -ENOTSUPP) {
			dev_err(&pdev->dev, "unable to set codec dai format %x",
				format);
			return ret;
		}
	}
	return 0;
}

static struct snd_soc_ops asoc_generic_ops = {
	.hw_params = asoc_generic_hw_params,
	.startup = asoc_generic_startup,
};

static const char * const be_format_text[] = {
	"PCM_DYNAMIC",
	"SNDRV_PCM_FORMAT_S8",
	"SNDRV_PCM_FORMAT_S16_LE",
	"SNDRV_PCM_FORMAT_S24_3LE",
	"SNDRV_PCM_FORMAT_S24_LE",	/* low 3 bytes */
	"SNDRV_PCM_FORMAT_S32_LE",
};

static unsigned int be_format_values[] = {
	ASOC_GENERIC_PCM_DYNAMIC,
	SNDRV_PCM_FORMAT_S8,
	SNDRV_PCM_FORMAT_S16_LE,
	SNDRV_PCM_FORMAT_S24_3LE,
	SNDRV_PCM_FORMAT_S24_LE,
	SNDRV_PCM_FORMAT_S32_LE,
};

int asoc_generic_hw_params_fixup(struct snd_soc_pcm_runtime *rtd,
				 struct snd_pcm_hw_params *params)
{
	struct snd_soc_dpcm *dpcm, *be_dpcm;
	int dir, found = 0;
	unsigned int be_format;
	struct snd_interval *channels = hw_param_interval(params,
			SNDRV_PCM_HW_PARAM_CHANNELS);
	struct snd_interval *rate = hw_param_interval(params,
			SNDRV_PCM_HW_PARAM_RATE);
	struct asoc_generic_dai_link_data *fe_data;
	struct platform_device *pdev;
	int ret;

	dpcm = container_of(params, struct snd_soc_dpcm, hw_params);

	/* retrieve fe dai_link_data struct */
	ret = asoc_generic_get_dai_link_data_for_rtd(dpcm->fe, &fe_data, &pdev);
	if (ret < 0)
		return ret;

	/* determine stream direction */
	for (dir = SNDRV_PCM_STREAM_PLAYBACK;
			dir < SNDRV_PCM_STREAM_LAST; dir++) {
		list_for_each_entry(be_dpcm, &dpcm->fe->dpcm[dir].be_clients,
				    list_be) {
			if (&be_dpcm->hw_params == params) {
				found = 1;
				break;
			}
		}
		if (found == 1)
			break;
	}

	be_format = be_format_values[fe_data->be_format[dir]];

	dev_dbg(&pdev->dev, "%s dir:%s format:%u rate:%d chan:%d",
		__func__,
		(dir == SNDRV_PCM_STREAM_PLAYBACK) ?
			"Playback" : "Capture",
		be_format,
		fe_data->be_samplerate[dir],
		fe_data->be_channels[dir]);

	if (be_format != ASOC_GENERIC_PCM_DYNAMIC) {
		/* clear the BE format */
		snd_mask_none(&params->masks[SNDRV_PCM_HW_PARAM_FORMAT -
						SNDRV_PCM_HW_PARAM_FIRST_MASK]);
		/* then update the BE mask */
		snd_mask_set(&params->masks[SNDRV_PCM_HW_PARAM_FORMAT -
						SNDRV_PCM_HW_PARAM_FIRST_MASK],
						be_format);
	}

	if (fe_data->be_channels[dir] != 0) {
		channels->min = fe_data->be_channels[dir];
		channels->max = fe_data->be_channels[dir];
	}

	if (fe_data->be_samplerate[dir] != 0) {
		rate->min = fe_data->be_samplerate[dir];
		rate->max = fe_data->be_samplerate[dir];
	}

	return 0;
};

static int asoc_generic_get_clk_master(struct snd_kcontrol *kcontrol,
				       struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	ucontrol->value.integer.value[0] = data->clkfsy;
	return 0;
}

static int asoc_generic_set_clk_master(struct snd_kcontrol *kcontrol,
				       struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	if (data->clkfsy == ucontrol->value.integer.value[0])
		return 0;
	data->clkfsy = ucontrol->value.integer.value[0];
	return 1;
}

static const struct soc_enum hw_clk_master_enum =
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(hw_clk_master_text), hw_clk_master_text);

static const char * const polarities_text[] = {
	"Normal",
	"Inverted_frame",
	"Inverted_bit",
	"Inverted",
};

static const int polarities_value[] = {
	SND_SOC_DAIFMT_NB_NF,
	SND_SOC_DAIFMT_NB_IF,
	SND_SOC_DAIFMT_IB_NF,
	SND_SOC_DAIFMT_IB_IF,
};

static inline int asoc_generic_find_polarity(
		struct asoc_generic_dai_link_data *data)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(polarities_value); i++) {
		if ((data->format & SND_SOC_DAIFMT_INV_MASK) ==
				polarities_value[i])
			break;
	}

	return i;
}

static int asoc_generic_get_polarities(struct snd_kcontrol *kcontrol,
				       struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	ucontrol->value.integer.value[0] = asoc_generic_find_polarity(data);

	return 0;
}

static int asoc_generic_set_polarities(struct snd_kcontrol *kcontrol,
				       struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);
	int i = ucontrol->value.integer.value[0];

	if ((data->format & SND_SOC_DAIFMT_INV_MASK) ==
		polarities_value[i])
		return 0;

	data->format &= ~SND_SOC_DAIFMT_INV_MASK;
	data->format |= polarities_value[i];

	return 1;
}

static int asoc_generic_get_codec_polarities(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);
	int i;

	for (i = 0; i < ARRAY_SIZE(polarities_value); i++) {
		if ((data->codec_format & SND_SOC_DAIFMT_INV_MASK) ==
				polarities_value[i])
			break;
	}

	ucontrol->value.integer.value[0] = i;

	return 0;
}

static int asoc_generic_set_codec_polarities(
				struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);
	int i = ucontrol->value.integer.value[0];

	if ((data->codec_format & SND_SOC_DAIFMT_INV_MASK) ==
		polarities_value[i])
		return 0;

	data->codec_format &= ~SND_SOC_DAIFMT_INV_MASK;
	data->codec_format |= polarities_value[i];

	return 1;
}

static int asoc_generic_get_format(struct snd_kcontrol *kcontrol,
				   struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	ucontrol->value.integer.value[0] =
		(data->format & SND_SOC_DAIFMT_FORMAT_MASK) - 1;
	return 0;
}

static int asoc_generic_set_format(struct snd_kcontrol *kcontrol,
				   struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	if ((data->format & SND_SOC_DAIFMT_FORMAT_MASK) - 1
		 == ucontrol->value.integer.value[0])
		return 0;

	data->format &= ~SND_SOC_DAIFMT_FORMAT_MASK;
	data->format |= ucontrol->value.integer.value[0] + 1;

	return 1;
}

static const struct soc_enum polarities_enum =
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(polarities_text), polarities_text);

static const char * const format_text[] = {
	"I2S",
	"RIGHT_J", /* Right justified */
	"LEFT_J", /* Left justified */
	"DSP_A",
	"DSP_B",
};

static const struct soc_enum format_enum =
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(format_text), format_text);

static const struct soc_enum be_format_enum =
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(be_format_text), be_format_text);

static int asoc_generic_get_witdh(struct snd_kcontrol *kcontrol,
				  struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	ucontrol->value.integer.value[0] = data->slot_width;
	return 0;
}

static int asoc_generic_set_width(struct snd_kcontrol *kcontrol,
				  struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	if (data->slot_width == ucontrol->value.integer.value[0])
		return 0;

	data->slot_width = ucontrol->value.integer.value[0];
	return 1;
}

static int asoc_generic_get_slots(struct snd_kcontrol *kcontrol,
				  struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	ucontrol->value.integer.value[0] = data->slots;
	return 0;
}

static int asoc_generic_set_slots(struct snd_kcontrol *kcontrol,
				  struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	int value = ucontrol->value.integer.value[0];

	if (data->slots == ucontrol->value.integer.value[0])
		return 0;

	data->slots = value;
	return 1;
}

static int asoc_generic_get_tx_mask(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	/* value[0] stores low 16 bit of mask and value[1] - high 16 bit */
	ucontrol->value.integer.value[0] = data->tx_mask & 0xffff;
	ucontrol->value.integer.value[1] = (data->tx_mask >> 16) & 0xffff;
	return 0;
}

static int asoc_generic_set_tx_mask(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	/* value[0] stores low 16 bit of mask and value[1] - high 16 bit */
	unsigned int tx_mask = ucontrol->value.integer.value[0] |
			(ucontrol->value.integer.value[1] << 16);

	if (data->tx_mask == tx_mask)
		return 0;

	data->tx_mask = tx_mask;
	return 1;
}

static int asoc_generic_get_rx_mask(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	/* value[0] stores low 16 bit of mask and value[1] - high 16 bit */
	ucontrol->value.integer.value[0] = data->rx_mask & 0xffff;
	ucontrol->value.integer.value[1] = (data->rx_mask >> 16) & 0xffff;
	return 0;
}

static int asoc_generic_set_rx_mask(struct snd_kcontrol *kcontrol,
				    struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	/* value[0] stores low 16 bit of mask and value[1] - high 16 bit */
	unsigned int rx_mask = ucontrol->value.integer.value[0] |
			(ucontrol->value.integer.value[1] << 16);

	if (data->rx_mask == rx_mask)
		return 0;

	data->rx_mask = rx_mask;
	return 1;
}

static const struct snd_kcontrol_new  card_kcontrols_set_format[] = {
	SOC_DAPM_ENUM_EXT("clkfsy_mst_slv", hw_clk_master_enum,
			  asoc_generic_get_clk_master,
			  asoc_generic_set_clk_master),
	SOC_DAPM_ENUM_EXT("polarities", polarities_enum,
			  asoc_generic_get_polarities,
			  asoc_generic_set_polarities),
	SOC_DAPM_ENUM_EXT("codec_polarities", polarities_enum,
			  asoc_generic_get_codec_polarities,
			  asoc_generic_set_codec_polarities),
	SOC_DAPM_ENUM_EXT("format", format_enum, asoc_generic_get_format,
			  asoc_generic_set_format),
};

static const struct snd_kcontrol_new  card_kcontrols_set_tdm_slot[] = {
	ASOC_GENERIC_VALUE_INT("width", 32, asoc_generic_get_witdh,
			       asoc_generic_set_width),
	ASOC_GENERIC_VALUE_INT("slots", 32, asoc_generic_get_slots,
			       asoc_generic_set_slots),
	ASOC_GENERIC_VALUE_MASKS("tx_mask",
				 asoc_generic_get_tx_mask,
				 asoc_generic_set_tx_mask),
	ASOC_GENERIC_VALUE_MASKS("rx_mask",
				 asoc_generic_get_rx_mask,
				 asoc_generic_set_rx_mask),
};

/* BE snd_kcontrols */

static int asoc_generic_get_be_format(struct snd_kcontrol *kcontrol,
				      struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	int dir = kcontrol->id.subdevice;

	ucontrol->value.integer.value[0] = data->be_format[dir];
	return 0;
}

static int asoc_generic_set_be_format(struct snd_kcontrol *kcontrol,
				      struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	int dir = kcontrol->id.subdevice;

	data->be_format[dir] = ucontrol->value.integer.value[0];

	return 1;
}

static int asoc_generic_get_be_samplerate(struct snd_kcontrol *kcontrol,
					  struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	int dir = kcontrol->id.subdevice;

	ucontrol->value.integer.value[0] = data->be_samplerate[dir];
	return 0;
}

static int asoc_generic_set_be_samplerate(struct snd_kcontrol *kcontrol,
					  struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	int dir = kcontrol->id.subdevice;

	if (data->be_samplerate[dir] == ucontrol->value.integer.value[0])
		return 0;

	data->be_samplerate[dir] = ucontrol->value.integer.value[0];
	return 1;
}

static int asoc_generic_get_be_channels(struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	int dir = kcontrol->id.subdevice;

	ucontrol->value.integer.value[0] = data->be_channels[dir];
	return 0;
}

static int asoc_generic_set_be_channels(struct snd_kcontrol *kcontrol,
					struct snd_ctl_elem_value *ucontrol)
{
	struct asoc_generic_dai_link_data *data = snd_kcontrol_chip(kcontrol);

	int dir = kcontrol->id.subdevice;

	if (data->be_channels[dir] == ucontrol->value.integer.value[0])
		return 0;

	data->be_channels[dir] = ucontrol->value.integer.value[0];
	return 1;
}

static const struct snd_kcontrol_new be_card_kcontrols[] = {
	SOC_DAPM_ENUM_EXT("be_format", be_format_enum,
			  asoc_generic_get_be_format,
			  asoc_generic_set_be_format),
	ASOC_GENERIC_VALUE_INT("be_samplerate", 0x7fffffff,
			       asoc_generic_get_be_samplerate,
			       asoc_generic_set_be_samplerate),
	ASOC_GENERIC_VALUE_INT("be_channels", 32,
			       asoc_generic_get_be_channels,
			       asoc_generic_set_be_channels),
};

static const struct snd_soc_dapm_widget asoc_generic_dapm_widgets[] = {
	SND_SOC_DAPM_LINE("GMD Line Out", NULL),
	SND_SOC_DAPM_LINE("GMD Line In", NULL),
};

static int read_dt_table(
		struct platform_device *pdev,
		struct device_node *np,
		struct sysdiv_table *data,
		unsigned int header_len,
		unsigned int key_value_len,
		const char *cell_name,
		const char *table_name) {
	unsigned int tot_len, row_len, *single_row;
	int ret, i;

	if (np && of_find_property(np, table_name, &tot_len)) {
		ret = of_property_read_u32(np,
					   cell_name,
					   &data->row_cell_cnt);
		if (ret) {
			dev_dbg(&pdev->dev,
				"unable to read %s using cellcnt = %d",
				cell_name,
				SYSDIV_TABLE_CELL_MIN);
			data->row_cell_cnt = SYSDIV_TABLE_CELL_MIN;
		}

		/* check the cell count */
		if (data->row_cell_cnt < SYSDIV_TABLE_CELL_MIN ||
		    data->row_cell_cnt > SYSDIV_TABLE_CELL_MAX) {
			dev_err(&pdev->dev,
				"improper cell count in %s",
				cell_name);
			return -EINVAL;
		}

		tot_len = tot_len / sizeof(*single_row);

		row_len = ((data->row_cell_cnt * key_value_len) +
				header_len);
		data->row_cnt = tot_len / row_len;

		/* check the array length */
		if (tot_len % row_len) {
			dev_err(&pdev->dev,
				"improper array length in %s",
				table_name);
			return -EINVAL;
		}

		/* create dynamic array and populate */
		single_row = (unsigned int *)devm_kzalloc(&pdev->dev,
				sizeof(unsigned int) *
				tot_len, GFP_KERNEL);
		if (!single_row)
			return -ENOMEM;

		ret = of_property_read_u32_array(np,
						 table_name,
						 single_row,
						 tot_len);
		if (ret)
			return -EINVAL;

		/* create and fill out table with rows */
		data->table = (unsigned int **)devm_kzalloc(&pdev->dev,
				sizeof(unsigned int *) *
				data->row_cnt, GFP_KERNEL);
		if (!data->table)
			return -ENOMEM;

		for (i = 0; i < data->row_cnt; i++) {
			data->table[i] = &single_row[i * row_len];
			dev_dbg(&pdev->dev, "%s[%u]:<%u %u>",
				table_name,
				i,
				data->table[i][0],
				data->table[i][1]);
		}
	}
	return 0;
}

static int read_sysdiv_tables(struct platform_device *pdev,
			      struct device_node *np,
			      struct asoc_generic_dai_sysdiv_data *sysdiv_conf)
{
	int ret;

	/* handle rate sysclk */
	ret = read_dt_table(pdev,
			    np,
			    &sysdiv_conf->rate_sys,
			    RSYS_HEADER_LEN,
			    SYS_KEYVAL_LEN,
			    RSYS_CELL_TAG,
			    RSYS_TAG);
	if (ret)
		return ret;

	/* handle sysclk */
	ret = read_dt_table(pdev,
			    np,
			    &sysdiv_conf->sys,
			    SYS_HEADER_LEN,
			    SYS_KEYVAL_LEN,
			    SYS_CELL_TAG,
			    SYS_TAG);
	if (ret)
		return ret;

	/* handle rate clkdiv */
	ret = read_dt_table(pdev,
			    np,
			    &sysdiv_conf->rate_div,
			    RDIV_HEADER_LEN,
			    DIV_KEYVAL_LEN,
			    RDIV_CELL_TAG,
			    RDIV_TAG);
	if (ret)
		return ret;

	/* handle clkdiv */
	ret = read_dt_table(pdev,
			    np,
			    &sysdiv_conf->div,
			    DIV_HEADER_LEN,
			    DIV_KEYVAL_LEN,
			    DIV_CELL_TAG,
			    DIV_TAG);

	return ret;
}


/* Parse dai information for both single dai and multi dai */
static int asoc_generic_parse_dai(
				struct device_node *node,
				const char *list_name,
				const char **dai_name,
				struct device_node **dai_of_node)
{
	struct of_phandle_args args;
	int ret;
	const char *name;

	if (!node)
		return -ENODEV;

	ret = of_parse_phandle_with_args(node, list_name,
					 DAI_CELL_TAG, 0, &args);
	if (ret)
		return -ENODEV;

	ret = snd_soc_get_dai_name(&args, &name);
	if (ret)
		return ret;

	//only pass dai_name for multi dai
	if (args.args_count)
		*dai_name = name;

	*dai_of_node = args.np;

	return ret;
}

static int asoc_generic_config_dai_link(
			struct platform_device *pdev,
			struct device_node *node,
			struct snd_soc_dai_link *link,
			struct asoc_generic_dai_link_data *link_data)
{
	struct device_node *soc_np = NULL, *codec_np = NULL,
			   *cpu_clk_np, *codec_clk_np;
	int ret;
	const char *dai_dir;

	/* use dai-link defaults */
	link_data->clkfsy = DAI_LINK_CLKFSY_DEFAULT;
	link_data->rx_mask = DAI_LINK_TX_MASK_DEFAULT;
	link_data->tx_mask = DAI_LINK_RX_MASK_DEFAULT;
	link_data->slot_width = DAI_LINK_SLOT_WIDTH_DEFAULT;
	link_data->slots = DAI_LINK_SLOTS_DEFAULT;
	link_data->format = DAI_LINK_FMT_DEFAULT;
	link_data->codec_format = DAI_LINK_CODEC_FMT_DEFAULT;

	/* of nodes for linking */
	ret = asoc_generic_parse_dai(node, "codec-node",
				     &link->codec_dai_name, &codec_np);
	if (ret == -EPROBE_DEFER) {
		dev_dbg(&pdev->dev, "parse of codec dai node deferred\n");
		goto fail;
	}

	/* strings for linking */
	if (ret)
		ret = of_property_read_string_index(node, "codec-name",
						    0, &link->codec_name);

	if (!codec_np && ret) {
		dev_err(&pdev->dev,
			"phandle or name for codec should be provided");
		ret = -EINVAL;
		goto fail;
	}

	/* of node for linking */
	ret = asoc_generic_parse_dai(node, "cpu-dai-node",
				     &link->cpu_dai_name, &soc_np);
	if (ret == -EPROBE_DEFER) {
		dev_dbg(&pdev->dev, "parse of cpu dai node deferred\n");
		goto fail;
	}

	/* string for linking */
	if (ret)
		ret = of_property_read_string_index(node, "cpu-dai-name",
						    0, &link->cpu_dai_name);

	if (!soc_np && ret) {
		dev_err(&pdev->dev,
			"phandle or name for cpu dai should be provided");
		ret = -EINVAL;
		goto fail;
	}

	if (!link->codec_dai_name) {
		ret = of_property_read_string_index(node, "codec-dai-name",
						    0, &link->codec_dai_name);
		if (ret < 0 && ret != -EINVAL) {
			dev_err(&pdev->dev,
				"Property '%s' could not be read: %d\n",
				"codec_dai_name", ret);
			goto fail;
		}
	}

	dev_dbg(&pdev->dev, "codec-dai-name %s", link->codec_dai_name);

	ret = of_property_read_string_index(node, "platform-name",
					    0, &link->platform_name);
	if (ret) {
		dev_err(&pdev->dev,
			"Property '%s' could not be read: %d\n",
			"platform-name", ret);
		goto fail;
	}
	dev_dbg(&pdev->dev, "platform-name %s", link->platform_name);
	link->codec_of_node = codec_np;
	link->cpu_of_node = soc_np;
	link->name = node->name;
	link->stream_name = node->name;
	link->no_pcm = of_get_property(node, "no-pcm", NULL) ? 1 : 0;
	link->dynamic = of_get_property(node, "dynamic", NULL) ? 1 : 0;
	/* always enable playback and capture by default,
	 * user can change dai_link direction by playback_only
	 * capture_only in DTS node
	 */
	link->dpcm_playback = 1;
	link->dpcm_capture = 1;
	if (!of_property_read_string(node, "dai-unidirectional", &dai_dir)) {
		if (!strcmp(dai_dir, DAI_PLAYBACK))
			link->playback_only = 1;
		else if (!strcmp(dai_dir, DAI_CAPTURE))
			link->capture_only = 1;
	}

	if (link->no_pcm)
		link->be_hw_params_fixup = asoc_generic_hw_params_fixup;

	link->ops = &asoc_generic_ops;
	/* dai_fmt will be set in startup based on snd_kcontrol values */
	link->dai_fmt = 0;
	link->init = asoc_generic_init;

	cpu_clk_np = of_parse_phandle(node, "cpu-clk", 0);
	codec_clk_np = of_parse_phandle(node, "codec-clk", 0);

	if (cpu_clk_np) {
		link_data->cpu_sysdiv_conf = devm_kzalloc(&pdev->dev,
				sizeof(*link_data->cpu_sysdiv_conf),
				GFP_KERNEL);
		if (!link_data->cpu_sysdiv_conf) {
			ret = -ENOMEM;
			goto fail_clocks;
		}

		ret = read_sysdiv_tables(pdev,
					 cpu_clk_np,
					 link_data->cpu_sysdiv_conf);
		if (ret)
			goto fail_clocks;
	} else
		dev_dbg(&pdev->dev,
			"cpu clock phandle not provided, using DAI defaults");

	if (codec_clk_np) {
		link_data->codec_sysdiv_conf = devm_kzalloc(&pdev->dev,
				sizeof(*link_data->codec_sysdiv_conf),
				GFP_KERNEL);
		if (!link_data->codec_sysdiv_conf) {
			ret = -ENOMEM;
			goto fail_clocks;
		}

		ret = read_sysdiv_tables(pdev,
					 codec_clk_np,
					 link_data->codec_sysdiv_conf);
		if (ret)
			goto fail_clocks;
	} else
		dev_dbg(&pdev->dev,
			"codec clock phandle not provided, using DAI defaults");

fail_clocks:
	if (cpu_clk_np)
		of_node_put(cpu_clk_np);
	if (codec_clk_np)
		of_node_put(codec_clk_np);
fail:
	if (soc_np)
		of_node_put(soc_np);
	if (codec_np)
		of_node_put(codec_np);

	return ret;
}

/* have to do it here as snd_soc_add_controls is not exported */
static int asoc_generic_add_controls(struct snd_card *card,
				     struct device *dev,
				     const struct snd_kcontrol_new *controls,
				     int num_controls,
				     const char *prefix,
				     void *data)
{
	int err, i;

	for (i = 0; i < num_controls; i++) {
		const struct snd_kcontrol_new *control = &controls[i];

		err = snd_ctl_add(card, snd_soc_cnew(control, data,
						     control->name, prefix));
		if (err < 0) {
			dev_err(dev, "Failed to add %s: %d\n",
				control->name, err);
			return err;
		}
	}

	return 0;
}

int asoc_generic_add_play_capture_controls(
			struct snd_card *card,
			const struct snd_kcontrol_new *controls,
			int num_controls,
			struct asoc_generic_dai_link_data *dai_link_data,
			const char *prefix)
{
	int i;

	for (i = 0; i < num_controls; i++) {
		struct snd_kcontrol_new new;
		char name[SND_KCTL_MAX_NAME_LEN];
		int ret;

		new = controls[i];
		new.name = name;
		new.subdevice = SNDRV_PCM_STREAM_CAPTURE;
		snprintf(name, SND_KCTL_MAX_NAME_LEN, "%s.%s", "capture",
			 controls[i].name);
		ret = asoc_generic_add_controls(card,
						card->dev,
						&new,
						1,
						prefix,
						(void *)dai_link_data);
		if (ret)
			return ret;
		new.subdevice = SNDRV_PCM_STREAM_PLAYBACK;
		snprintf(name, SND_KCTL_MAX_NAME_LEN, "%s.%s", "playback",
			 controls[i].name);
		ret = asoc_generic_add_controls(card,
						card->dev,
						&new,
						1,
						prefix,
						(void *)dai_link_data);
		if (ret)
			return ret;
	}
	return 0;
}

static int asoc_generic_late_probe(struct snd_soc_card *card)
{
	struct snd_soc_pcm_runtime *rtd;
	int i = 0;
	struct asoc_generic_data *data = container_of(card,
					struct asoc_generic_data, card);
	const char *prefix = NULL;
	int ret;
	struct snd_card *snd_card = card->snd_card;

	list_for_each_entry(rtd, &card->rtd_list, list) {
		struct snd_soc_dai_link *link = &card->dai_link[i];

		prefix = link->stream_name;

		if ((rtd->cpu_dai->driver &&
		     rtd->cpu_dai->driver->ops->set_fmt) ||
		     (rtd->codec_dai->driver &&
				rtd->codec_dai->driver->ops->set_fmt)) {
			ret = asoc_generic_add_controls(
					snd_card,
					snd_card->dev,
					(const struct snd_kcontrol_new *)
						&card_kcontrols_set_format,
					ARRAY_SIZE(card_kcontrols_set_format),
					prefix,
					&data->dai_link_data[i]);
			if (ret < 0)
				return ret;
		}

		if ((rtd->cpu_dai->driver &&
		     rtd->cpu_dai->driver->ops->set_tdm_slot) ||
		    (rtd->codec_dai->driver &&
		     rtd->codec_dai->driver->ops->set_tdm_slot)) {
			ret = asoc_generic_add_controls(
					snd_card,
					snd_card->dev,
					(const struct snd_kcontrol_new *)
						&card_kcontrols_set_tdm_slot,
					ARRAY_SIZE(card_kcontrols_set_tdm_slot),
					prefix,
					&data->dai_link_data[i]);
			if (ret < 0)
				return ret;
		}

		if (rtd->dai_link->dynamic) {
			ret = asoc_generic_add_play_capture_controls(
					snd_card,
					be_card_kcontrols,
					ARRAY_SIZE(be_card_kcontrols),
					&data->dai_link_data[i],
					prefix);
			if (ret < 0)
				return ret;
		}
		i++;
	}
	return 0;
}

static int asoc_generic_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct asoc_generic_data *data;
	struct device_node *node = NULL;
	int ret;
	int i = 0, link_num = 0;
	int alloc_link_num;

	for_each_child_of_node(np, node) {
		if (!of_device_is_compatible(node, ASOC_GENERIC_DT_DAI_LINK))
			continue;
		link_num++;
	}

	/* for being able to parse simple DAI-Link */
	alloc_link_num = max(link_num, 1);

	data = devm_kzalloc(&pdev->dev, sizeof(*data) +
			alloc_link_num * (sizeof(struct snd_soc_dai_link) +
			sizeof(struct asoc_generic_dai_link_data)), GFP_KERNEL);
	if (!data) {
		ret = -ENOMEM;
		goto fail;
	}
	data->pdev = pdev;
	data->dai_link = (void *)data + sizeof(*data);
	data->dai_link_data = (void *)data->dai_link + alloc_link_num *
			sizeof(struct snd_soc_dai_link);

	if (link_num > 0) {
		for_each_child_of_node(np, node) {
			struct snd_soc_dai_link *dai_link;

			if (!of_device_is_compatible(node,
						     ASOC_GENERIC_DT_DAI_LINK))
				continue;

			dai_link = &data->dai_link[i];

			ret = asoc_generic_config_dai_link(
						pdev,
						node,
						dai_link,
						&data->dai_link_data[i]);
			if (ret) {
				if (node)
					of_node_put(node);
				goto fail;
			}
			i++;
		}
	} else {
		/* to keep compatibility to old settings - simple config */
		ret = asoc_generic_config_dai_link(
						pdev,
						pdev->dev.of_node,
						&data->dai_link[i],
						&data->dai_link_data[i]);
		if (ret)
			goto fail;
	}

	data->card.dev = &pdev->dev;
	ret = snd_soc_of_parse_card_name(&data->card, "model");
	if (ret)
		goto fail;

	/* Determine if audio routing table is given or not and create route */
	if (of_property_count_strings(data->card.dev->of_node,
				      "audio-routing") > 0) {
		ret = snd_soc_of_parse_audio_routing(&data->card,
						     "audio-routing");
		if (ret)
			goto fail;
	}

	/* to keep compatibility to old settings */
	data->card.num_links = alloc_link_num;
	data->card.owner = THIS_MODULE;
	data->card.dai_link = data->dai_link;
	data->card.dapm_widgets = asoc_generic_dapm_widgets;
	data->card.num_dapm_widgets = ARRAY_SIZE(asoc_generic_dapm_widgets);
	data->card.controls = NULL;
	data->card.num_controls = 0;
	data->card.late_probe = asoc_generic_late_probe;

	ret = snd_soc_register_card(&data->card);
	if (ret) {
		if (ret == -EPROBE_DEFER)
			dev_dbg(&pdev->dev, "snd_soc_register_card is deferred\n");
		else
			dev_err(&pdev->dev, "snd_soc_register_card failed (%d)\n",
				ret);
		goto fail;
	}

	platform_set_drvdata(pdev, data);

fail:
	return ret;
}

static int asoc_generic_remove(struct platform_device *pdev)
{
	struct asoc_generic_data *data = platform_get_drvdata(pdev);

	snd_soc_unregister_card(&data->card);
	return 0;
}

static const struct of_device_id asoc_generic_dt_ids[] = {
	{.compatible = "asoc,audio-generic",},
	{ /* sentinel */ }
};

MODULE_DEVICE_TABLE(of, asoc_generic_dt_ids);

static struct platform_driver asoc_generic_driver = {
	.driver = {
		   .name = "asoc-generic",
		   .owner = THIS_MODULE,
		   .of_match_table = asoc_generic_dt_ids,
		   },
	.probe = asoc_generic_probe,
	.remove = asoc_generic_remove,
};

module_platform_driver(asoc_generic_driver);

MODULE_AUTHOR("valentin_sitdikov@mentor.com");
MODULE_DESCRIPTION("ALSA SoC Generic Machine Layer Driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:asoc-generic");
