/*
 *  mrgfld_dpcm_florida.c - ASOC Machine driver for Intel Morganfield platform
 *
 *  Copyright (C) 2014 Wolfson Micro
 *  Copyright (C) 2014 Intel Corp
 *  Author: Samreen Nilofer <samreen.nilofer@intel.com>
 *
 * Based on
 *	moor_dpcm_florida.c - ASOC Machine driver for Intel Moorefield MID platform
 *  Copyright (C) 2014 Wolfson Micro
 *  Copyright (C) 2014 Intel Corp
 *  Author: Nikesh Oswal <Nikesh.Oswal@wolfsonmicro.com>
 *	    Praveen Diwakar <praveen.diwakar@intel.com>
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 */
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/module.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/async.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/acpi.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/jack.h>
#include <linux/input.h>

#include <linux/mfd/arizona/registers.h>
#include "../../codecs/wm5110.h"
#include "../../codecs/wm8998.h"


/* Codec PLL output clk rate */
#define CODEC_SYSCLK_RATE			49152000
/* Input clock to codec at MCLK1 PIN */
#define CODEC_IN_MCLK1_RATE			19200000
/* Input clock to codec at MCLK2 PIN */
#define CODEC_IN_MCLK2_RATE			32768
/* Input bit clock to codec */
#define CODEC_IN_BCLK_RATE			4800000

/*  define to select between MCLK1 and MCLK2 input to codec as its clock */
#define CODEC_IN_MCLK1				1
#define CODEC_IN_MCLK2				2
#define CODEC_IN_BCLK				3

#define SLOT_MASK(x) ((1 << x) - 1)
static bool is_codec8998;

struct mrgfld_mc_private {
	u8		pmic_id;
	void __iomem    *osc_clk0_reg;
	int bt_mode;
};
static const struct snd_soc_pcm_stream bxtn_florida_dai_params_codec = {
	.formats = SNDRV_PCM_FMTBIT_S24_LE,
	.rate_min = 48000,
	.rate_max = 48000,
	.channels_min = 2,
	.channels_max = 2,
};
static const struct snd_soc_pcm_stream bxtn_florida_dai_params_modem = {
	.stream_name = "modem-loop mono",
	.formats = SNDRV_PCM_FMTBIT_S16_LE,
	.rate_min = 48000,
	.rate_max = 48000,
	.channels_min = 1,
	.channels_max = 1,
};
static const struct snd_soc_pcm_stream bxtn_florida_dai_params_bt[] = {
	{
		.stream_name = "A2DP",
		.formats = SNDRV_PCM_FMTBIT_S16_LE,
		.rate_min = 48000,
		.rate_max = 48000,
		.channels_min = 2,
		.channels_max = 2,
	},
	{
		.stream_name = "Wide Band",
		.formats = SNDRV_PCM_FMTBIT_S16_LE,
		.rate_min = 16000,
		.rate_max = 16000,
		.channels_min = 1,
		.channels_max = 1,
	},
	{
		.stream_name = "Narrow Band",
		.formats = SNDRV_PCM_FMTBIT_S16_LE,
		.rate_min = 8000,
		.rate_max = 8000,
		.channels_min = 1,
		.channels_max = 1,
	},
};

static const struct snd_soc_pcm_stream bxtn_florida_dai_params_dmic01 = {
	.stream_name = "dmic01",
	.formats = SNDRV_PCM_FMTBIT_S16_LE,
	.rate_min = 48000,
	.rate_max = 48000,
	.channels_min = 2,
	.channels_max = 2,
};

static const struct snd_soc_pcm_stream bxtn_florida_dai_params_dmic23 = {
	.stream_name = "dmic23",
	.formats = SNDRV_PCM_FMTBIT_S16_LE,
	.rate_min = 48000,
	.rate_max = 48000,
	.channels_min = 2,
	.channels_max = 2,
};

/* set_osc_clk0-	enable/disables the osc clock0
 * addr:		address of the register to write to
 * enable:		bool to enable or disable the clock
 */
static inline void set_soc_osc_clk0(void __iomem *addr, bool enable)
{
	u32 osc_clk_ctrl;

	osc_clk_ctrl = readl(addr);
	if (enable)
		osc_clk_ctrl |= BIT(31);
	else
		osc_clk_ctrl &= ~(BIT(31));

	pr_debug("%s: enable:%d val 0x%x\n", __func__, enable, osc_clk_ctrl);

	writel(osc_clk_ctrl, addr);
}

static inline struct snd_soc_codec *mrgfld_florida_get_codec(struct snd_soc_card *card)
{
	bool found = false;
	struct snd_soc_codec *codec;
	char *codec_name;

	if (is_codec8998)
		codec_name = "wm8998-codec";
	else
		codec_name = "wm5110-codec";

	list_for_each_entry(codec, &card->codec_dev_list, card_list) {
		if (!strstr(codec->component.name, codec_name)) {
			pr_debug("codec was %s", codec->component.name);
			continue;
		} else {
			found = true;
			break;
		}
	}
	if (found == false) {
		pr_err("%s: cant find codec", __func__);
		return NULL;
	}
	return codec;
}

static struct snd_soc_dai *mrgfld_florida_get_codec_dai(struct snd_soc_card *card, const char *dai_name)
{
	struct snd_soc_pcm_runtime *rtd;

	list_for_each_entry(rtd, &card->rtd_list, list) {
		if (!strcmp(rtd->codec_dai->name, dai_name))
			return rtd->codec_dai;
	}
	pr_err("%s: unable to find codec dai\n", __func__);
	/* this should never occur */
	WARN_ON(1);
	return NULL;
}

static int mrgfld_bt_mode_get(struct snd_kcontrol *kcontrol,
			      struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_card *card = snd_kcontrol_chip(kcontrol);
	struct mrgfld_mc_private *drv = snd_soc_card_get_drvdata(card);

	ucontrol->value.enumerated.item[0] = drv->bt_mode;
	pr_debug("%s %d BT mode = %d\n", __func__, __LINE__, drv->bt_mode);
	return 0;
}

static int mrgfld_bt_mode_put(struct snd_kcontrol *kcontrol,
			      struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_card *card = snd_kcontrol_chip(kcontrol);
	struct mrgfld_mc_private *drv = snd_soc_card_get_drvdata(card);
	struct soc_enum *e = (void *)kcontrol->private_value;

	if (ucontrol->value.enumerated.item[0] > e->items - 1)
		return -EINVAL;

	drv->bt_mode = ucontrol->value.enumerated.item[0];
	pr_debug("%s %d mode = %d\n", __func__, __LINE__, drv->bt_mode);
	return 0;
}

/* Function to switch the input clock for codec,  When audio is in
 * progress input clock to codec will be through MCLK1 which is 19.2MHz
 * while in off state input clock to codec will be through 32KHz through
 * MCLK2
 * card	: Sound card structure
 * src	: Input clock source to codec
 */

static int mrgfld_florida_set_codec_clk(struct snd_soc_codec *florida_codec, int src)
{
	int ret;

	pr_debug("mrgfld_florida_set_codec_clk: source %d\n", src);

	/*reset FLL1*/
	snd_soc_codec_set_pll(florida_codec, WM5110_FLL1_REFCLK,
				ARIZONA_FLL_SRC_NONE, 0, 0);
	snd_soc_codec_set_pll(florida_codec, WM5110_FLL1,
				ARIZONA_FLL_SRC_NONE, 0, 0);

	switch (src) {
	case CODEC_IN_MCLK1:
		/* Turn ON the PLL to generate required sysclk rate
		 * from MCLK1 */
		ret = snd_soc_codec_set_pll(florida_codec, WM5110_FLL1,
				ARIZONA_CLK_SRC_MCLK1, CODEC_IN_MCLK1_RATE,
				CODEC_SYSCLK_RATE);
		if (ret != 0) {
			dev_err(florida_codec->dev, "Failed to enable FLL1 with Ref(MCLK) Clock Loop: %d\n", ret);
			return ret;
		}
		break;
	case CODEC_IN_BCLK:
		/* Turn ON the PLL to generate required sysclk rate
		 * from BCLK */
		ret = snd_soc_codec_set_pll(florida_codec, WM5110_FLL1,
				ARIZONA_CLK_SRC_AIF1BCLK, CODEC_IN_BCLK_RATE,
				CODEC_SYSCLK_RATE);
		if (ret != 0) {
			dev_err(florida_codec->dev, "Failed to enable FLL1 with Ref Clock Loop: %d\n", ret);
			return ret;
		}

		break;
	default:
		return -EINVAL;
	}

	/*Switch to PLL*/
	ret = snd_soc_codec_set_sysclk(florida_codec,
			ARIZONA_CLK_SYSCLK, ARIZONA_CLK_SRC_FLL1,
			CODEC_SYSCLK_RATE, SND_SOC_CLOCK_IN);
	if (ret != 0) {
		dev_err(florida_codec->dev, "Failed to set SYSCLK to FLL1: %d\n", ret);
		return ret;
	}

	return 0;
}


static int mrgfld_clock_control(struct snd_soc_dapm_widget *w,
		struct snd_kcontrol *k, int  event)
{

	struct snd_soc_dapm_context *dapm = w->dapm;
	struct snd_soc_card *card = dapm->card;
	struct snd_soc_codec *florida_codec = mrgfld_florida_get_codec(card);
	int ret = 0;

	if (!florida_codec) {
		pr_err("%s: florida codec not found\n", __func__);
		return -EINVAL;
	}
	if (SND_SOC_DAPM_EVENT_ON(event)) {
		pr_info("%s %d Event On\n", __func__, __LINE__);
		/* TODO: Ideally MCLK should be used to drive codec PLL
		 * currently we are using BCLK
		 */
		ret = mrgfld_florida_set_codec_clk(florida_codec, CODEC_IN_BCLK);
	} else {
		pr_info("%s %d Event Off\n", __func__, __LINE__);
		/* TODO: Switch to 32K clock for saving power. */
		pr_info("Currently we are not switching to 32K PMIC clock\n");
	}
	return ret;

}
static const struct snd_soc_dapm_widget mrgfld_widgets[] = {
	SND_SOC_DAPM_HP("Headphones", NULL),
	SND_SOC_DAPM_SPK("Ext Spk", NULL),
	SND_SOC_DAPM_SPK("EP", NULL),
	SND_SOC_DAPM_MIC("AMIC", NULL),
	SND_SOC_DAPM_MIC("DMIC", NULL),
	SND_SOC_DAPM_MIC("SoC DMIC", NULL),
	SND_SOC_DAPM_MIC("HDMI", NULL),
	SND_SOC_DAPM_SUPPLY("Platform Clock", SND_SOC_NOPM, 0, 0,
			mrgfld_clock_control, SND_SOC_DAPM_PRE_PMU|
			SND_SOC_DAPM_POST_PMD),

};

static int bxt_dmic_fixup(struct snd_soc_pcm_runtime *rtd,
				struct snd_pcm_hw_params *params)
{
	struct snd_interval *channels = hw_param_interval(params,
						SNDRV_PCM_HW_PARAM_CHANNELS);
	channels->min = channels->max = 2;

	return 0;
}

static const struct snd_soc_dapm_route mrgfld_wm5110_map[] = {
	/*Headphones*/
	{ "Headphones", NULL, "HPOUT1L" },
	{ "Headphones", NULL, "HPOUT1R" },

	/*Speakers*/
	{"Ext Spk", NULL, "SPKOUTLP"},
	{"Ext Spk", NULL, "SPKOUTLN"},
	{"Ext Spk", NULL, "SPKOUTRP"},
	{"Ext Spk", NULL, "SPKOUTRN"},

	{ "AMIC", NULL, "MICBIAS2" },
	{ "AMIC", NULL, "MICBIAS1" },
	{ "IN1L", NULL, "AMIC" },
	{ "IN1R", NULL, "AMIC" },

	/* SWM map link the SWM outs to codec AIF */
	{ "AIF1 Playback", NULL, "ssp0 Tx"},
	{ "ssp0 Tx", NULL, "codec1_out"},
	{ "ssp0 Tx", NULL, "codec0_out"},

	{ "ssp0 Rx", NULL, "AIF1 Capture" },
	{ "codec0_in", NULL, "ssp0 Rx" },
	{ "codec1_in", NULL, "ssp0 Rx" },

	{ "DMic", NULL, "SoC DMIC"},

	{ "DMIC01 Rx", NULL, "Capture" },
	{ "DMIC23 Rx", NULL, "Capture" },
	{ "dmic01_hifi", NULL, "DMIC01 Rx" },
	{ "dmic23_hifi", NULL, "DMIC23 Rx" },

	/* HDMI */
	{"HDMI", NULL, "hif1 Output"},
	{"hif1", NULL, "iDisp Tx"},
	{"iDisp Tx", NULL, "iDisp_out"},

	/* Modem0 Path */
	{ "Dummy Playback", NULL, "ssp3 Tx"},
	{ "ssp3 Tx", NULL, "modem0_out"},

	{ "modem0_in", NULL, "ssp3 Rx" },
	{ "ssp3 Rx", NULL, "Dummy Capture" },

	/* Bt Path */
	{ "Dummy Playback", NULL, "ssp1 Tx"},
	{ "ssp1 Tx", NULL, "bt_out"},

	{ "bt_in", NULL, "ssp1 Rx" },
	{ "ssp1 Rx", NULL, "Dummy Capture" },

	{"Headphones", NULL, "Platform Clock"},
	{"AMIC", NULL, "Platform Clock"},
	{"DMIC", NULL, "Platform Clock"},
	{"Ext Spk", NULL, "Platform Clock"},
	{"EP", NULL, "Platform Clock"},
	{"Tone Generator 1", NULL, "Platform Clock" },
	{"Tone Generator 2", NULL, "Platform Clock" },
};


static const struct snd_soc_dapm_route mrgfld_wm8998_map[] = {
	/*Headphones*/
	{ "Headphones", NULL, "HPOUTL" },
	{ "Headphones", NULL, "HPOUTR" },

	/*Speakers*/
	{"Ext Spk", NULL, "SPKOUTLP"},
	{"Ext Spk", NULL, "SPKOUTLN"},
	{"Ext Spk", NULL, "SPKOUTRP"},
	{"Ext Spk", NULL, "SPKOUTRN"},

	/*Earpiece*/
	{ "EP", NULL, "EPOUTN" },
	{ "EP", NULL, "EPOUTP" },

	{"IN3L", NULL, "DMIC"},
	{"IN3R", NULL, "DMIC"},
	{"DMIC", NULL, "MICVDD"},

	{ "AMIC", NULL, "MICBIAS2" },
	{ "AMIC", NULL, "MICBIAS1" },
	{ "IN2B", NULL, "AMIC" },
	{ "DMic", NULL, "SoC DMIC"},

	/* SWM map link the SWM outs to codec AIF */
	{ "AIF1 Playback", NULL, "ssp0 Tx"},
	{ "ssp0 Tx", NULL, "codec1_out"},
	{ "ssp0 Tx", NULL, "codec0_out"},

	{ "ssp0 Rx", NULL, "AIF1 Capture" },
	{ "codec0_in", NULL, "ssp0 Rx" },
	{ "codec1_in", NULL, "ssp0 Rx" },

	{ "DMIC01 Rx", NULL, "Capture" },
	{ "DMIC23 Rx", NULL, "Capture" },
	{ "dmic01_hifi", NULL, "DMIC01 Rx" },
	{ "dmic23_hifi", NULL, "DMIC23 Rx" },

	/* TODO: map for rest of the ports */

	/* HDMI */
	{"HDMI", NULL, "hif1 Output"},
	{"hif1", NULL, "iDisp Tx"},
	{"iDisp Tx", NULL, "iDisp_out"},

	/* Modem0 Path */
	{ "Dummy Playback", NULL, "ssp3 Tx"},
	{ "ssp3 Tx", NULL, "modem0_out"},

	{ "modem0_in", NULL, "ssp3 Rx" },
	{ "ssp3 Rx", NULL, "Dummy Capture" },

	/* Bt Path */
	{ "Dummy Playback", NULL, "ssp1 Tx"},
	{ "ssp1 Tx", NULL, "bt_out"},

	{ "bt_in", NULL, "ssp1 Rx" },
	{ "ssp1 Rx", NULL, "Dummy Capture" },

	{"Headphones", NULL, "Platform Clock"},
	{"AMIC", NULL, "Platform Clock"},
	{"DMIC", NULL, "Platform Clock"},
	{"Ext Spk", NULL, "Platform Clock"},
	{"EP", NULL, "Platform Clock"},
	{"Tone Generator 1", NULL, "Platform Clock" },
	{"Tone Generator 2", NULL, "Platform Clock" },
};


static const char * const mrgfld_a2dp_nb_wb_texts[] = {
	"A2DP profile", "WB Profile", "NB profile",
};

static const struct snd_kcontrol_new mrgfld_controls[] = {
	SOC_DAPM_PIN_SWITCH("Headphones"),
	SOC_DAPM_PIN_SWITCH("Ext Spk"),
	SOC_DAPM_PIN_SWITCH("EP"),
	SOC_DAPM_PIN_SWITCH("AMIC"),
	SOC_DAPM_PIN_SWITCH("DMIC"),
/*	MRGFLD_MUX_CTL("domain bt mode", 0,
			0, mrgfld_a2dp_nb_wb_texts,
			mrgfld_bt_mode_get,
			mrgfld_bt_mode_put),
*/
};

static int mrgfld_florida_init(struct snd_soc_pcm_runtime *runtime)
{
	int ret;
	unsigned int fmt;
	struct snd_soc_card *card = runtime->card;
	char *dai_name;
	struct snd_soc_dai *florida_dai;

	pr_info("Entry %s\n", __func__);

	if (is_codec8998)
		dai_name = "wm8998-aif1";
	else
		dai_name = "wm5110-aif1";

	florida_dai = mrgfld_florida_get_codec_dai(card, dai_name);

	if (!florida_dai) {
		pr_err("%s: florida dai not found\n", __func__);
		return -EINVAL;
	}

	ret = snd_soc_dai_set_tdm_slot(florida_dai, 0, 0, 4, 24);
	/* slot width is set as 25, SNDRV_PCM_FORMAT_S32_LE */
	if (ret < 0) {
		pr_err("can't set codec pcm format %d\n", ret);
		return ret;
	}

	/* bit clock inverse not required */
	fmt =   SND_SOC_DAIFMT_DSP_A | SND_SOC_DAIFMT_NB_NF
		| SND_SOC_DAIFMT_CBS_CFS;
	ret = snd_soc_dai_set_fmt(florida_dai, fmt);
	if (ret < 0) {
		pr_err("can't set codec DAI configuration %d\n", ret);
		return ret;
	}

	card->dapm.idle_bias_off = true;

	ret = snd_soc_add_card_controls(card, mrgfld_controls,
					ARRAY_SIZE(mrgfld_controls));
	if (ret) {
		pr_err("unable to add card controls\n");
		return ret;
	}
	return 0;
}

static unsigned int rates_48000[] = {
	48000,
	16000,
	8000,
};

static struct snd_pcm_hw_constraint_list constraints_48000 = {
	.count = ARRAY_SIZE(rates_48000),
	.list  = rates_48000,
};

static int mrgfld_florida_startup(struct snd_pcm_substream *substream)
{
	return snd_pcm_hw_constraint_list(substream->runtime, 0,
			SNDRV_PCM_HW_PARAM_RATE,
			&constraints_48000);
}

static struct snd_soc_ops mrgfld_florida_ops = {
	.startup = mrgfld_florida_startup,
};

static int mrgfld_florida_btfm_fixup(struct snd_soc_pcm_runtime *rtd,
			    struct snd_pcm_hw_params *params)
{
	struct snd_interval *rate = hw_param_interval(params,
			SNDRV_PCM_HW_PARAM_RATE);
	struct snd_interval *channels = hw_param_interval(params,
						SNDRV_PCM_HW_PARAM_CHANNELS);

	pr_debug("Invoked %s for dailink %s\n", __func__, rtd->dai_link->name);
	rate->min  = 8000;
	rate->max  = 48000;
	channels->min = 1;
	channels->max = 2;
	snd_mask_none(hw_param_mask(params, SNDRV_PCM_HW_PARAM_FORMAT));
	snd_mask_set(hw_param_mask(params, SNDRV_PCM_HW_PARAM_FORMAT),
		     (__force unsigned int)SNDRV_PCM_FORMAT_S16_LE);

	pr_debug("param width set to:%#x\n",
				snd_pcm_format_width(params_format(params)));

	return 0;
}
static int mrgfld_florida_codec_fixup(struct snd_soc_pcm_runtime *rtd,
			    struct snd_pcm_hw_params *params)
{
	struct snd_interval *rate = hw_param_interval(params,
			SNDRV_PCM_HW_PARAM_RATE);
	struct snd_interval *channels = hw_param_interval(params,
						SNDRV_PCM_HW_PARAM_CHANNELS);

	pr_debug("Invoked %s for dailink %s\n", __func__, rtd->dai_link->name);
	rate->min = rate->max = 48000;
	channels->min = channels->max = 2;
	snd_mask_none(hw_param_mask(params, SNDRV_PCM_HW_PARAM_FORMAT));
	snd_mask_set(hw_param_mask(params, SNDRV_PCM_HW_PARAM_FORMAT),
		     (__force unsigned int)SNDRV_PCM_FORMAT_S24_LE);

	pr_debug("param width set to:0x%x\n",
			snd_pcm_format_width(params_format(params)));

	return 0;
}

static const struct snd_soc_pcm_stream mrgfld_florida_dai_params = {
	.formats = SNDRV_PCM_FMTBIT_S24_LE,
	.rate_min = 48000,
	.rate_max = 48000,
	.channels_min = 4,
	.channels_max = 4,
};

static struct snd_soc_dai_link mrgfld_florida_msic_dailink[] = {
	{
		.name = "Bxtn Audio Port",
		.stream_name = "Audio",
		.cpu_dai_name = "System Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = mrgfld_florida_init,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ops = &mrgfld_florida_ops,
	},

	{
		.name = "Bxtn DB Audio Port",
		.stream_name = "Deep Buffer Audio",
		.cpu_dai_name = "Deepbuffer Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.ops = &mrgfld_florida_ops,
	},
	{
		.name = "Bxt Compress Probe playback",
		.stream_name = "Probe Playback",
		.cpu_dai_name = "Compress Probe0 Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = NULL,
		.nonatomic = 1,
	},
	{
		.name = "Bxt Compress Probe capture",
		.stream_name = "Probe Capture",
		.cpu_dai_name = "Compress Probe1 Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = NULL,
		.nonatomic = 1,
	},
	{
		.name = "Bxtn Audio Reference cap",
		.stream_name = "refcap",
		.cpu_dai_name = "Reference Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = NULL,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.dpcm_capture = 1,
	},
	{
		.name = "Bxtn LL Audio Port",
		.stream_name = "Low Latency Audio",
		.cpu_dai_name = "LowLatency Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.ops = &mrgfld_florida_ops,
	},
	{
		.name = "Bxtn Audio Reference cap",
		.stream_name = "refcap",
		.cpu_dai_name = "Reference Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = NULL,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.dpcm_capture = 1,
	},

		/* CODEC<->CODEC link */
	{
		.name = "Bxtn Codec-Loop Port",
		.stream_name = "Bxtn Codec-Loop",
		.cpu_dai_name = "SSP0 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_name = "wm5110-codec",
		.codec_dai_name = "wm5110-aif1",
		.params = &bxtn_florida_dai_params_codec,
		.dsp_loopback = true,
		.dai_fmt = SND_SOC_DAIFMT_DSP_A |
			SND_SOC_DAIFMT_NB_NF |
			SND_SOC_DAIFMT_CBS_CFS,
	},

	{
		.name = "Bxtn Modem-Loop Port",
		.stream_name = "Bxtn Modem-Loop",
		.cpu_dai_name = "SSP3 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.params = &bxtn_florida_dai_params_modem,
		.dsp_loopback = true,
		.ops = &mrgfld_florida_ops,
	},

	{
		.name = "Bxtn BTFM-Loop Port",
		.stream_name = "Bxtn BTFM-Loop",
		.cpu_dai_name = "SSP1 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.params = bxtn_florida_dai_params_bt,
		.num_params = ARRAY_SIZE(bxtn_florida_dai_params_bt),
		.dsp_loopback = true,
	},

	{
		.name = "Bxtn DMIC01-Loop Port",
		.stream_name = "Bxtn DMIC01-Loop",
		.cpu_dai_name = "DMIC01 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_dai_name = "dmic-hifi",
		.codec_name = "dmic-codec",
		.params = &bxtn_florida_dai_params_dmic01,
		.dsp_loopback = true,
	},

	{
		.name = "Bxtn DMIC23-Loop Port",
		.stream_name = "Bxtn DMIC23-Loop",
		.cpu_dai_name = "DMIC23 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_dai_name = "dmic-hifi",
		.codec_name = "dmic-codec",
		.params = &bxtn_florida_dai_params_dmic23,
		.dsp_loopback = true,
	},

	{
		.name = "Bxtn HDMI Port",
		.stream_name = "Hdmi",
		.cpu_dai_name = "HDMI Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.dpcm_playback = 1,
		.init = NULL,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
	},

	/* back ends */
	{
		.name = "SSP0-Codec",
		.cpu_dai_name = "SSP0 Pin",
		.codec_name = "wm5110-codec",
		.codec_dai_name = "wm5110-aif1",
		.platform_name = "0000:00:0e.0",
		.be_hw_params_fixup = mrgfld_florida_codec_fixup,
		.ignore_suspend = 1,
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
	},

	{
		.name = "dmic01",
		.cpu_dai_name = "DMIC01 Pin",
		.codec_name = "dmic-codec",
		.codec_dai_name = "dmic-hifi",
		.platform_name = "0000:00:0e.0",
		.ignore_suspend = 1,
		.dpcm_capture = 1,
		.no_pcm = 1,
		.be_hw_params_fixup = bxt_dmic_fixup,
	},

	{
		.name = "dmic23",
		.cpu_dai_name = "DMIC23 Pin",
		.codec_name = "dmic-codec",
		.codec_dai_name = "dmic-hifi",
		.platform_name = "0000:00:0e.0",
		.be_hw_params_fixup = bxt_dmic_fixup,
		.ignore_suspend = 1,
		.dpcm_capture = 1,
		.no_pcm = 1,
	},

	{
		.name = "iDisp",
		.cpu_dai_name = "iDisp Pin",
		.codec_name = "ehdaudio0D1",
		.codec_dai_name = "intel-hdmi-hif1",
		.platform_name = "0000:00:0e.0",
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.no_pcm = 1,
	},
	{
		.name = "SSP1-BTFM",
		.cpu_dai_name = "SSP1 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.ignore_suspend = 1,
		.be_hw_params_fixup = mrgfld_florida_btfm_fixup,
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
	},

};

static struct snd_soc_dai_link mrgfld_wm8998_msic_dailink[] = {
	{
		.name = "Bxtn Audio Port",
		.stream_name = "Audio",
		.cpu_dai_name = "System Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = mrgfld_florida_init,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
		.ops = &mrgfld_florida_ops,
	},

	{
		.name = "Bxtn DB Audio Port",
		.stream_name = "Deep Buffer Audio",
		.cpu_dai_name = "Deepbuffer Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.ops = &mrgfld_florida_ops,
	},
	{
		.name = "Bxt Compress Probe playback",
		.stream_name = "Probe Playback",
		.cpu_dai_name = "Compress Probe0 Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = NULL,
		.nonatomic = 1,
	},
	{
		.name = "Bxt Compress Probe capture",
		.stream_name = "Probe Capture",
		.cpu_dai_name = "Compress Probe1 Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = NULL,
		.nonatomic = 1,
	},

	{
		.name = "Bxtn LL Audio Port",
		.stream_name = "Low Latency Audio",
		.cpu_dai_name = "LowLatency Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.ops = &mrgfld_florida_ops,
	},
	{
		.name = "Bxtn Audio Reference cap",
		.stream_name = "refcap",
		.cpu_dai_name = "Reference Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.init = NULL,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
		.dpcm_capture = 1,
	},

		/* CODEC<->CODEC link */
	{
		.name = "Bxtn Codec-Loop Port",
		.stream_name = "Bxtn Codec-Loop",
		.cpu_dai_name = "SSP0 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_name = "wm8998-codec",
		.codec_dai_name = "wm8998-aif1",
		.params = &bxtn_florida_dai_params_codec,
		.dsp_loopback = true,
		.dai_fmt = SND_SOC_DAIFMT_DSP_A |
			SND_SOC_DAIFMT_NB_NF |
			SND_SOC_DAIFMT_CBS_CFS,
	},

	{
		.name = "Bxtn Modem-Loop Port",
		.stream_name = "Bxtn Modem-Loop",
		.cpu_dai_name = "SSP3 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.params = &bxtn_florida_dai_params_modem,
		.dsp_loopback = true,
		.ops = &mrgfld_florida_ops,
	},

	{
		.name = "Bxtn BTFM-Loop Port",
		.stream_name = "Bxtn BTFM-Loop",
		.cpu_dai_name = "SSP1 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_dai_name = "snd-soc-dummy-dai",
		.codec_name = "snd-soc-dummy",
		.params = bxtn_florida_dai_params_bt,
		.num_params = ARRAY_SIZE(bxtn_florida_dai_params_bt),
		.dsp_loopback = true,
	},

	{
		.name = "Bxtn DMIC01-Loop Port",
		.stream_name = "Bxtn DMIC01-Loop",
		.cpu_dai_name = "DMIC01 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_dai_name = "dmic-hifi",
		.codec_name = "dmic-codec",
		.params = &bxtn_florida_dai_params_dmic01,
		.dsp_loopback = true,
	},

	{
		.name = "Bxtn DMIC23-Loop Port",
		.stream_name = "Bxtn DMIC23-Loop",
		.cpu_dai_name = "DMIC23 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_dai_name = "dmic-hifi",
		.codec_name = "dmic-codec",
		.params = &bxtn_florida_dai_params_dmic23,
		.dsp_loopback = true,
	},

	{
		.name = "Bxtn HDMI Port",
		.stream_name = "Hdmi",
		.cpu_dai_name = "HDMI Pin",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.platform_name = "0000:00:0e.0",
		.dpcm_playback = 1,
		.init = NULL,
		.ignore_suspend = 1,
		.nonatomic = 1,
		.dynamic = 1,
	},

	/* back ends */
	{
		.name = "SSP0-Codec",
		.cpu_dai_name = "SSP0 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_name = "wm8998-codec",
		.codec_dai_name = "wm8998-aif1",
		.be_hw_params_fixup = mrgfld_florida_codec_fixup,
		.ignore_suspend = 1,
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
	},

	{
		.name = "dmic01",
		.cpu_dai_name = "DMIC01 Pin",
		.codec_name = "dmic-codec",
		.codec_dai_name = "dmic-hifi",
		.platform_name = "0000:00:0e.0",
		.ignore_suspend = 1,
		.dpcm_capture = 1,
		.no_pcm = 1,
		.be_hw_params_fixup = bxt_dmic_fixup,
	},

	{
		.name = "dmic23",
		.cpu_dai_name = "DMIC23 Pin",
		.codec_name = "dmic-codec",
		.codec_dai_name = "dmic-hifi",
		.platform_name = "0000:00:0e.0",
		.ignore_suspend = 1,
		.dpcm_capture = 1,
		.no_pcm = 1,
		.be_hw_params_fixup = bxt_dmic_fixup,
	},

	{
		.name = "iDisp",
		.cpu_dai_name = "iDisp Pin",
		.codec_name = "ehdaudio0D1",
		.codec_dai_name = "intel-hdmi-hif1",
		.platform_name = "0000:00:0e.0",
		.dpcm_playback = 1,
		.ignore_suspend = 1,
		.no_pcm = 1,
	},
	{
		.name = "SSP1-BTFM",
		.cpu_dai_name = "SSP1 Pin",
		.platform_name = "0000:00:0e.0",
		.codec_name = "snd-soc-dummy",
		.codec_dai_name = "snd-soc-dummy-dai",
		.ignore_suspend = 1,
		.be_hw_params_fixup = mrgfld_florida_btfm_fixup,
		.no_pcm = 1,
		.dpcm_playback = 1,
		.dpcm_capture = 1,
	},
};

#ifdef CONFIG_PM_SLEEP
static int snd_mrgfld_florida_prepare(struct device *dev)
{
	pr_debug("In %s\n", __func__);
	return snd_soc_suspend(dev);
}

static void snd_mrgfld_florida_complete(struct device *dev)
{
	pr_debug("In %s\n", __func__);
	snd_soc_resume(dev);
}

static int snd_mrgfld_florida_poweroff(struct device *dev)
{
	pr_debug("In %s\n", __func__);
	return snd_soc_poweroff(dev);
}
#else
#define snd_mrgfld_florida_prepare NULL
#define snd_mrgfld_florida_complete NULL
#define snd_mrgfld_florida_poweroff NULL
#endif

/* SoC card */
static struct snd_soc_card snd_soc_card_mrgfld = {
	.name = "florida-audio",
	.dai_link = mrgfld_florida_msic_dailink,
	.num_links = ARRAY_SIZE(mrgfld_florida_msic_dailink),
	.dapm_widgets = mrgfld_widgets,
	.num_dapm_widgets = ARRAY_SIZE(mrgfld_widgets),
	.dapm_routes =  mrgfld_wm5110_map,
	.num_dapm_routes = ARRAY_SIZE(mrgfld_wm5110_map),
	.fully_routed = false,
};

static struct snd_soc_card snd_soc_card_wm8998_mrgfld = {
	.name = "wm8998-audio",
	.dai_link =  mrgfld_wm8998_msic_dailink,
	.num_links = ARRAY_SIZE(mrgfld_wm8998_msic_dailink),
	.dapm_widgets = mrgfld_widgets,
	.num_dapm_widgets = ARRAY_SIZE(mrgfld_widgets),
	.dapm_routes = mrgfld_wm8998_map,
	.num_dapm_routes = ARRAY_SIZE(mrgfld_wm8998_map),
	.fully_routed = false,
};

static char *bxtn_machines[] = {"INT34C1", "INT34E0"};

static acpi_status sst_acpi_mach_match(void *handle, u32 level,
						void *context, void **ret)
{
	*(bool *)context = true;
	return AE_OK;
}

static int  sst_acpi_find_machine(char **machine)
{
	char **mach;
	bool found = false;

	for (mach = bxtn_machines; *mach; mach++)
		if (!(acpi_get_devices(*mach, sst_acpi_mach_match,
					&found, NULL)) && found) {
			*machine = *mach;
			pr_debug("Found machine ID = %s", *machine);
			return 0;
		}
	return -ENODEV;
}

static int snd_mrgfld_florida_mc_probe(struct platform_device *pdev)
{
	int ret_val = 0;
	struct mrgfld_mc_private *drv;
	char *codec_id = NULL;
	struct snd_soc_card *card_mrgfld = NULL;
	int codec_found;

	pr_debug("Entry %s\n", __func__);

	codec_found = sst_acpi_find_machine(&codec_id);
	if (codec_id == NULL) {
		pr_err("Codec not found\n");
		return codec_found;
	}

	if (strncmp(codec_id, "INT34C1", strlen("INT34C1")) == 0) {
		pr_info("Audio: Detected HW codec WM8281\n");
		is_codec8998 = false;
		card_mrgfld = &snd_soc_card_mrgfld;
	} else if (strncmp(codec_id, "INT34E0", strlen("INT34E0")) == 0) {
		pr_info("Audio: Detected HW codec WM8998\n");
		is_codec8998 = true;
		card_mrgfld = &snd_soc_card_wm8998_mrgfld;
	} else
		return -ENODEV;

	drv = devm_kzalloc(&pdev->dev, sizeof(*drv), GFP_ATOMIC);
	if (!drv)
		return -ENOMEM;

	card_mrgfld->dev = &pdev->dev;
	snd_soc_card_set_drvdata(card_mrgfld, drv);
	/* Register the card */
	ret_val = snd_soc_register_card(card_mrgfld);
	if (ret_val) {
		pr_err("snd_soc_register_card failed %d\n", ret_val);
		goto unalloc;
	}
	platform_set_drvdata(pdev, card_mrgfld);
	pr_info("%s successful\n", __func__);
	return ret_val;

unalloc:
	devm_kfree(&pdev->dev, drv);
	return ret_val;
}

static int snd_mrgfld_florida_mc_remove(struct platform_device *pdev)
{
	struct snd_soc_card *soc_card = platform_get_drvdata(pdev);
	struct mrgfld_mc_private *drv = snd_soc_card_get_drvdata(soc_card);

	pr_debug("In %s\n", __func__);

	devm_kfree(&pdev->dev, drv);
	snd_soc_card_set_drvdata(soc_card, NULL);
	snd_soc_unregister_card(soc_card);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static const struct dev_pm_ops snd_mrgfld_florida_mc_pm_ops = {
	.prepare = snd_mrgfld_florida_prepare,
	.complete = snd_mrgfld_florida_complete,
	.poweroff = snd_mrgfld_florida_poweroff,
};

static struct platform_driver snd_mrgfld_florida_mc_driver = {
	.driver = {
		.name = "mrgfld_florida",
		.pm = &snd_mrgfld_florida_mc_pm_ops,
	},
	.probe = snd_mrgfld_florida_mc_probe,
	.remove = snd_mrgfld_florida_mc_remove,
};

static int snd_mrgfld_florida_driver_init(void)
{
	pr_info("Morganfield Machine Driver mrgfld_florida registered\n");
	return platform_driver_register(&snd_mrgfld_florida_mc_driver);
}
module_init(snd_mrgfld_florida_driver_init);

static void snd_mrgfld_florida_driver_exit(void)
{
	pr_debug("In %s\n", __func__);
	platform_driver_unregister(&snd_mrgfld_florida_mc_driver);
}
module_exit(snd_mrgfld_florida_driver_exit)

MODULE_DESCRIPTION("ASoC Morganfield Machine driver");
MODULE_AUTHOR("Samreen Nilofer <samreen.nilofer@intel.com>");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:mrgfld_florida");
