/*
 *  Virtio based driver for the QNX audio device
 *  Copyright (c)2017 by ADIT GmbH
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/virtio.h>
#include <linux/virtio_ids.h>
#include <linux/virtio_config.h>
#include <sound/core.h>
#include <sound/initval.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include "virtio_snd_qnx.h"

#ifdef DEBUG
#define DBG(...) pr_info(__VA_ARGS__)
#else
#define DBG(...) do {} while (0)
#endif
#define VDEV_ERR(v, ...) dev_err(&(v)->dev, ##__VA_ARGS__)
#define VCARD_ERR(v, ...) dev_err(&(v)->vdev->dev, ##__VA_ARGS__)
#define VCARD_INFO(v, ...) dev_info(&(v)->vdev->dev, ##__VA_ARGS__)

//#define VPCM_ERR(v, ...) pcm_err((v)->pcm, ##__VA_ARGS__)
#define VPCM_ERR(v, ...) dev_err((v)->pcm->card->dev, ##__VA_ARGS__)
#define VPCM_LOG(v, ...) do {	\
	if (!log) break;	\
	dev_info((v)->pcm->card->dev, ##__VA_ARGS__);	\
} while (0)

static int log = 0;
module_param(log, int, S_IRUGO | S_IWUSR | S_IWGRP);
MODULE_PARM_DESC(log, "activate virtio-snd log outputs");

static int sgmode;
static int sg = 1;
#ifdef CONFIG_SND_DMA_SGBUF
module_param(sg, int, S_IRUGO | S_IWUSR | S_IWGRP);
MODULE_PARM_DESC(log, "scatter and gather buffer control");
#endif

/*
 * QNX host side DMA buffer num
 */
#define MAX_HOSTQUESZ	4
#define MIN_HOSTQUESZ	2

/*some reasonable defaults*/
#define VIRTIO_SND_FORMATS_DEFAULT SNDRV_PCM_FMTBIT_S16_LE
#define VIRTIO_SND_RATES_DEFAULT SNDRV_PCM_RATE_8000_48000
#define VIRTIO_SND_RATE_MIN_DEFAULT 8000
#define VIRTIO_SND_RATE_MAX_DEFAULT 48000
#define VIRTIO_SND_CHANNELS_MIN_DEFAULT 1
#define VIRTIO_SND_CHANNELS_MAX_DEFAULT 32
#define VIRTIO_SND_BUFFER_BYTES_MAX_DEFAULT (1024 * 128)
#define VIRTIO_SND_PERIOD_BYTES_MIN_DEFAULT 32
#define VIRTIO_SND_PERIOD_BYTES_MAX_DEFAULT 32768
#define VIRTIO_SND_PERIODS_MIN_DEFAULT 2

#define VIRTIO_SND_PERIODS_MAX_DEFAULT		\
	(VIRTIO_SND_BUFFER_BYTES_MAX_DEFAULT /	\
	VIRTIO_SND_PERIOD_BYTES_MIN_DEFAULT)

#define VIRTIO_SND_PERIODS_HOSTGAP	3
	/*
	 * periods_min is calculated as
	 *   MIN_HOSTQUESZ + periods_hostgap module param
	 *
	 * consider about flow control, queuing amount must be
	 * more 3 than QNX DMA buffer num.
	 *
	 * detecting end of playback stream, needs 1 callback
	 * after last sending data. at do_force_flush()
	 * inflight
	 *
	 * MIN_HOSTQUESZ + 3 -> MIN_HOSTQUESZ + 2
	 *   save app_pos on app_pos_previous
	 * MIN_HOSTQUESZ + 2 -> MIN_HOSTQUESZ + 1
	 *   AP didn't send more data at returning Ack of +3
	 *   compare app_pos and app_pos_previous, then
	 *   detect stopping data.
	 *   QNX must receive FLUSH/DRAIN before MIN_HOSTQUESZ+1
	 */
static int	periods_hostgap = VIRTIO_SND_PERIODS_HOSTGAP;
module_param(periods_hostgap, int, S_IRUGO | S_IWUSR | S_IWGRP);
MODULE_PARM_DESC(periods_hostgap, "periods_min gap with host");

#ifdef FOR_SOFT_FLUSH
static int sfdevice = 1;
module_param(sfdevice, int, S_IRUGO | S_IWUSR | S_IWGRP);
MODULE_PARM_DESC(sfdevice, "device to activate softflush on.");
#endif

static const char shortname[] = "virtio";
static const char longname[] = "Virtio sound card";

static const struct snd_pcm_hardware virtio_snd_default_caps = {
	.info =			(SNDRV_PCM_INFO_INTERLEAVED |
				 SNDRV_PCM_INFO_BLOCK_TRANSFER |
				 SNDRV_PCM_INFO_BATCH |
				 SNDRV_PCM_INFO_DRAIN_TRIGGER |
				 SNDRV_PCM_INFO_PAUSE),
	.formats =		VIRTIO_SND_FORMATS_DEFAULT,
	.rates =		VIRTIO_SND_RATES_DEFAULT,
	.rate_min =		VIRTIO_SND_RATE_MIN_DEFAULT,
	.rate_max =		VIRTIO_SND_RATE_MAX_DEFAULT,
	.channels_min =		VIRTIO_SND_CHANNELS_MIN_DEFAULT,
	.channels_max =		VIRTIO_SND_CHANNELS_MAX_DEFAULT,
	.buffer_bytes_max =	VIRTIO_SND_BUFFER_BYTES_MAX_DEFAULT,
	.period_bytes_min =	VIRTIO_SND_PERIOD_BYTES_MIN_DEFAULT,
	.period_bytes_max =	VIRTIO_SND_PERIOD_BYTES_MAX_DEFAULT,
	.periods_min =		VIRTIO_SND_PERIODS_MIN_DEFAULT,
	.periods_max =		VIRTIO_SND_PERIODS_MAX_DEFAULT,
	.fifo_size =		0,
};

/*need to handle non-standard rates*/
struct virtio_snd_rate_caps {
	unsigned int rates_list[VIRTIO_AUDIO_RATE_LAST + 1];
	unsigned int rates_cnt;
	unsigned int min_rate;
	unsigned int max_rate;
};

enum virtio_snd_state {
	VIRTIO_AUDIO_STATE_CLOSED = 0,
	VIRTIO_AUDIO_STATE_ACQUIRED,  /*acquire was sent*/
	VIRTIO_AUDIO_STATE_PREPARED, /*prepare was successed*/
	VIRTIO_AUDIO_STATE_FLUSHING, /*ALSA draining*/
};

struct virtio_snd_pcm {
	bool configuration_parsed; /*prevent 0 divide*/
	unsigned int stream;	/*direction*/
	unsigned int devno;	/*device number as exported by host*/
	struct snd_pcm_hardware hwcaps;	/*caps to be exported to ALSA*/
	struct virtio_snd_rate_caps rate_caps;/*rate caps given by host*/
	struct snd_pcm_hw_constraint_list rate_constraints;/*rate constraints*/
	snd_pcm_uframes_t period_align; /*opt. alignment requirement by host*/
	struct virtio_snd_card *vcard;
	struct snd_pcm *pcm;
	struct virtqueue *vq_stream;	/*the stream queue*/
	struct virtqueue *vq_cmd;	/*command queue*/
	unsigned int max_s;		/*max stream pakets*/
	unsigned int max_c;		/*max command pakets*/
	wait_queue_head_t wqueue;	/*to wait for inflight buffers*/
	struct stream_job *sjobs;	/*jobs*/
	struct stream_job **sjobs_list; /*jobs list*/

	/*runtime*/
	struct snd_pcm_substream *substream;
	struct virtio_audio_acquire_params acquire_params;
	snd_pcm_uframes_t hw_pos;	/*acked by hardware*/
	snd_pcm_uframes_t app_pos;	/*signalled to hardware*/
	enum virtio_snd_state state;	/*device state*/
	unsigned int num_inflight;	/*buffers on the road*/

	/*
	 * flow control
	 */
	int			max_hostquesz;
	int			force_flushing;
	snd_pcm_uframes_t	app_pos_previous, app_pos_previous_wq;
	ktime_t			last_update_time, force_flush_time;
	int			no_update_count;
	struct delayed_work	force_flush_work;

	int			draining;
};

struct virtio_snd_pair {
	struct virtio_snd_pcm *dir[2];
};

struct virtio_snd_card {
	struct virtio_device *vdev;
	struct snd_card  *card;
	unsigned int num_devices;
	unsigned int max_devno;
	struct virtio_snd_pcm *devices;
	struct virtio_snd_pair *pairs;
	vq_callback_t **callbacks;
	struct virtqueue **vqs;
	const char **names;
};

struct cmd_job {
	struct virtio_audio_cmd cmd;
	struct virtio_audio_status status;
	struct completion done;
	void (*action)(struct cmd_job *);
	struct virtio_snd_pcm *vpcm;
	atomic_t ref_cnt;
};

struct stream_job {
	struct virtio_audio_status status;
	snd_pcm_uframes_t	actual_frames;
};

#ifdef FOR_SOFT_FLUSH
static bool need_softflush(struct virtio_snd_pcm* vpcm) {
	/* devno = 1 is for playback device for interrupt sound from asound.conf*/
	return ((vpcm->stream == SNDRV_PCM_STREAM_PLAYBACK) &&
		    (sfdevice != -1) &&(vpcm->devno ==  sfdevice)) ? true : false;
}
#endif

static void virtio_snd_action_flush(struct cmd_job *job);
static void virtio_snd_action_prepare(struct cmd_job *job);
static void virtio_snd_action_sync(struct cmd_job *job);
static void virtio_snd_action_ignore(struct cmd_job *job);

static u64 fmt_map_vdev2alsa[VIRTIO_AUDIO_FMT_LAST + 1] = {
	[VIRTIO_AUDIO_FMT_U8] = SNDRV_PCM_FMTBIT_U8,
	[VIRTIO_AUDIO_FMT_S8] = SNDRV_PCM_FMTBIT_S8,
	[VIRTIO_AUDIO_FMT_U16_LE] = SNDRV_PCM_FMTBIT_U16_LE,
	[VIRTIO_AUDIO_FMT_U16_BE] = SNDRV_PCM_FMTBIT_U16_BE,
	[VIRTIO_AUDIO_FMT_S16_LE] = SNDRV_PCM_FMTBIT_S16_LE,
	[VIRTIO_AUDIO_FMT_S16_BE] = SNDRV_PCM_FMTBIT_S16_BE,
	[VIRTIO_AUDIO_FMT_U24_LE] = SNDRV_PCM_FMTBIT_U24_LE,
	[VIRTIO_AUDIO_FMT_U24_BE] = SNDRV_PCM_FMTBIT_U24_BE,
	[VIRTIO_AUDIO_FMT_S24_LE] = SNDRV_PCM_FMTBIT_S24_LE,
	[VIRTIO_AUDIO_FMT_S24_BE] = SNDRV_PCM_FMTBIT_S24_BE,
	[VIRTIO_AUDIO_FMT_U32_LE] = SNDRV_PCM_FMTBIT_U32_LE,
	[VIRTIO_AUDIO_FMT_U32_BE] = SNDRV_PCM_FMTBIT_U32_BE,
	[VIRTIO_AUDIO_FMT_S32_LE] = SNDRV_PCM_FMTBIT_S32_LE,
	[VIRTIO_AUDIO_FMT_S32_BE] = SNDRV_PCM_FMTBIT_S32_BE,
	[VIRTIO_AUDIO_FMT_A_LAW] = SNDRV_PCM_FMTBIT_A_LAW,
	[VIRTIO_AUDIO_FMT_MU_LAW] = SNDRV_PCM_FMTBIT_MU_LAW,
	[VIRTIO_AUDIO_FMT_IEC958_SUBFRAME_LE] =
					SNDRV_PCM_FMTBIT_IEC958_SUBFRAME_LE,
	[VIRTIO_AUDIO_FMT_IEC958_SUBFRAME_BE] =
					SNDRV_PCM_FMTBIT_IEC958_SUBFRAME_BE,
	[VIRTIO_AUDIO_FMT_AC3] = 0,
	[VIRTIO_AUDIO_FMT_FLOAT_LE] = SNDRV_PCM_FMTBIT_FLOAT_LE,
	[VIRTIO_AUDIO_FMT_FLOAT_BE] = SNDRV_PCM_FMTBIT_FLOAT_BE,
	[VIRTIO_AUDIO_FMT_FLOAT64_LE] = SNDRV_PCM_FMTBIT_FLOAT64_LE,
	[VIRTIO_AUDIO_FMT_FLOAT64_BE] = SNDRV_PCM_FMTBIT_FLOAT64_BE,
	[VIRTIO_AUDIO_FMT_IMA_ADPCM] = SNDRV_PCM_FMTBIT_IMA_ADPCM,
	[VIRTIO_AUDIO_FMT_GSM] = SNDRV_PCM_FMTBIT_GSM,
	[VIRTIO_AUDIO_FMT_MPEG] =	SNDRV_PCM_FMTBIT_MPEG,
	[VIRTIO_AUDIO_FMT_SPECIAL] = SNDRV_PCM_FMTBIT_SPECIAL,
	[VIRTIO_AUDIO_FMT_U24_3LE] = SNDRV_PCM_FMTBIT_U24_3LE,
	[VIRTIO_AUDIO_FMT_U24_3BE] = SNDRV_PCM_FMTBIT_U24_3BE,
	[VIRTIO_AUDIO_FMT_S24_3LE] = SNDRV_PCM_FMTBIT_S24_3LE,
	[VIRTIO_AUDIO_FMT_S24_3BE] = SNDRV_PCM_FMTBIT_S24_3BE
};

#define VDEV_AUDIO_FMT(fmt) (VIRTIO_AUDIO_FMT_##fmt)

static u32 fmt_map_alsa2vdev[((__force int)SNDRV_PCM_FORMAT_LAST) + 1] = {
	[SNDRV_PCM_FORMAT_U8]	= VDEV_AUDIO_FMT(U8),
	[SNDRV_PCM_FORMAT_S8]	= VDEV_AUDIO_FMT(S8),
	[SNDRV_PCM_FORMAT_U16_LE] = VDEV_AUDIO_FMT(U16_LE),
	[SNDRV_PCM_FORMAT_U16_BE] = VDEV_AUDIO_FMT(U16_BE),
	[SNDRV_PCM_FORMAT_S16_LE] = VDEV_AUDIO_FMT(S16_LE),
	[SNDRV_PCM_FORMAT_S16_BE] = VDEV_AUDIO_FMT(S16_BE),
	[SNDRV_PCM_FORMAT_U24_LE] = VDEV_AUDIO_FMT(U24_LE),
	[SNDRV_PCM_FORMAT_U24_BE] = VDEV_AUDIO_FMT(U24_BE),
	[SNDRV_PCM_FORMAT_S24_LE] = VDEV_AUDIO_FMT(S24_LE),
	[SNDRV_PCM_FORMAT_S24_BE] = VDEV_AUDIO_FMT(S24_BE),
	[SNDRV_PCM_FORMAT_U32_LE] = VDEV_AUDIO_FMT(U32_LE),
	[SNDRV_PCM_FORMAT_U32_BE] = VDEV_AUDIO_FMT(U32_BE),
	[SNDRV_PCM_FORMAT_S32_LE] = VDEV_AUDIO_FMT(S32_LE),
	[SNDRV_PCM_FORMAT_S32_BE] = VDEV_AUDIO_FMT(S32_BE),
	[SNDRV_PCM_FORMAT_A_LAW] = VDEV_AUDIO_FMT(A_LAW),
	[SNDRV_PCM_FORMAT_MU_LAW] = VDEV_AUDIO_FMT(MU_LAW),
	[SNDRV_PCM_FORMAT_IEC958_SUBFRAME_LE] =
					VDEV_AUDIO_FMT(IEC958_SUBFRAME_LE),
	[SNDRV_PCM_FORMAT_IEC958_SUBFRAME_BE] =
					VDEV_AUDIO_FMT(IEC958_SUBFRAME_BE),
	[SNDRV_PCM_FORMAT_FLOAT_LE]	= VDEV_AUDIO_FMT(FLOAT_LE),
	[SNDRV_PCM_FORMAT_FLOAT_BE]	= VDEV_AUDIO_FMT(FLOAT_BE),
	[SNDRV_PCM_FORMAT_FLOAT64_LE]	= VDEV_AUDIO_FMT(FLOAT64_LE),
	[SNDRV_PCM_FORMAT_FLOAT64_BE]	= VDEV_AUDIO_FMT(FLOAT64_BE),
	[SNDRV_PCM_FORMAT_IMA_ADPCM]	= VDEV_AUDIO_FMT(IMA_ADPCM),
	[SNDRV_PCM_FORMAT_GSM]		= VDEV_AUDIO_FMT(GSM),
	[SNDRV_PCM_FORMAT_MPEG]		= VDEV_AUDIO_FMT(MPEG),
	[SNDRV_PCM_FORMAT_SPECIAL]	= VDEV_AUDIO_FMT(SPECIAL),
};

#define VDEV_CMD_TOUT_DEFAULT 1000

static const char *vdev_cmd_name(u32 vdev_cmd)
{
	switch (vdev_cmd) {
	case VIRTIO_AUDIO_CMD_ACQUIRE:
		return "ACQUIRE";
	case VIRTIO_AUDIO_CMD_PREPARE:
		return "PREPARE";
	case VIRTIO_AUDIO_CMD_FLUSH:
		return "FLUSH";
	case VIRTIO_AUDIO_CMD_DRAIN:
		return "DRAIN";
	case VIRTIO_AUDIO_CMD_PAUSE:
		return "PAUSE";
	case VIRTIO_AUDIO_CMD_RESUME:
		return "RESUME";
	case VIRTIO_AUDIO_CMD_RELEASE:
		return "RELEASE";
	default:
		return "<UNKNOWN>";
	}
}

static inline unsigned int vdev_cmd_tout(u32 vdev_cmd)
{
/* FIXME: timeout should depend on command.
 * review after command sequence clarification
 */
	switch (vdev_cmd) {
	case VIRTIO_AUDIO_CMD_ACQUIRE:
	case VIRTIO_AUDIO_CMD_PREPARE:
	case VIRTIO_AUDIO_CMD_FLUSH:
	case VIRTIO_AUDIO_CMD_DRAIN:
	case VIRTIO_AUDIO_CMD_PAUSE:
	case VIRTIO_AUDIO_CMD_RESUME:
	case VIRTIO_AUDIO_CMD_RELEASE:
	default:
		return VDEV_CMD_TOUT_DEFAULT;
	}
}

static void cmd_job_put(struct cmd_job *job)
{
	if (atomic_dec_and_test(&job->ref_cnt)) {
		DBG("free job for cmd %s\n", vdev_cmd_name(job->cmd.id));
		kfree(job);
		return;
	}
}

static void cmd_job_get(struct cmd_job *job)
{
	atomic_inc(&job->ref_cnt);
}

static struct cmd_job *cmd_job_create(struct virtio_snd_pcm *vpcm, gfp_t gfp,
				      u32 cmd)
{
	/* allocate for now... */
	struct cmd_job *job = kmalloc(sizeof(*job), gfp);

	if (!job)
		return NULL;

	job->cmd.id = cmd;
	job->vpcm = vpcm;
	init_completion(&job->done);

	switch (cmd) {
	case VIRTIO_AUDIO_CMD_ACQUIRE:
	case VIRTIO_AUDIO_CMD_RELEASE:
		job->action = virtio_snd_action_sync;
		break;

	case VIRTIO_AUDIO_CMD_PREPARE:
		job->action = virtio_snd_action_prepare;
		break;

	case VIRTIO_AUDIO_CMD_FLUSH:
		job->action = virtio_snd_action_flush;
		break;

	case VIRTIO_AUDIO_CMD_DRAIN:
	case VIRTIO_AUDIO_CMD_PAUSE:
	case VIRTIO_AUDIO_CMD_RESUME:
	default:
		job->action = virtio_snd_action_ignore;
		break;
	}
	atomic_set(&job->ref_cnt, 1);
	return job;
}

static void cmd_job_complete(struct cmd_job *job)
{
	DBG("cb cmd->complete status %d\n", job->status.status);
	if (atomic_read(&job->ref_cnt) > 1) {
		/*
		 * virtio_snd_cmd_wait hasn't yet processing
		 * now job->action() is availale
		 */
		job->action(job);
	} else {
		struct virtio_snd_pcm *vpcm = job->vpcm;

		printk(KERN_ERR
			"%s(%d) command=%s return after timeout,"
			"status=%d refcnt=%d\n",
			__func__, vpcm ? vpcm->devno : -1,
			vdev_cmd_name(job->cmd.id),
			job->status.status,
			atomic_read(&job->ref_cnt));
	}
	cmd_job_put(job);
}

static int cmd_job_wait(struct cmd_job *job, unsigned int tout)
{
	int ret = wait_for_completion_timeout(&job->done,
					      msecs_to_jiffies(tout));

	return ret ? 0 : -ETIMEDOUT;
}

static struct stream_job *stream_job_get(struct virtio_snd_pcm *vpcm)
{
	struct stream_job *job;

	job = vpcm->sjobs_list[vpcm->num_inflight];
	vpcm->sjobs_list[vpcm->num_inflight] = NULL;
	vpcm->num_inflight++;
	return job;
}

static void stream_job_put(struct virtio_snd_pcm *vpcm, struct stream_job *job)
{
	vpcm->num_inflight--;
	vpcm->sjobs_list[vpcm->num_inflight] = job;
	if (!vpcm->num_inflight) {
		DBG("WAKE_UP\n");
		wake_up(&vpcm->wqueue);
	}
}

/*convert fmt mask*/
static u64 fmt_vdev2alsa(u32 fmt_mask)
{
	unsigned int bit;
	u64 alsa_mask = 0;

	for (bit = 0; bit < VIRTIO_AUDIO_FMT_LAST; bit++) {
		if (fmt_mask & (1u << bit))
			alsa_mask |= fmt_map_vdev2alsa[bit];
	}
	return alsa_mask;
}

static unsigned int rate_map_vdev2alsa[VIRTIO_AUDIO_RATE_LAST + 1] = {
	[VIRTIO_AUDIO_RATE_BIT_8000] = SNDRV_PCM_RATE_8000,
	[VIRTIO_AUDIO_RATE_BIT_11025] = SNDRV_PCM_RATE_11025,
	[VIRTIO_AUDIO_RATE_BIT_16000] = SNDRV_PCM_RATE_16000,
	[VIRTIO_AUDIO_RATE_BIT_22050] = SNDRV_PCM_RATE_22050,
	[VIRTIO_AUDIO_RATE_BIT_32000] = SNDRV_PCM_RATE_32000,
	[VIRTIO_AUDIO_RATE_BIT_44100] = SNDRV_PCM_RATE_44100,
	[VIRTIO_AUDIO_RATE_BIT_48000] = SNDRV_PCM_RATE_48000,
	[VIRTIO_AUDIO_RATE_BIT_88200] = SNDRV_PCM_RATE_88200,
	[VIRTIO_AUDIO_RATE_BIT_96000] = SNDRV_PCM_RATE_96000,
	[VIRTIO_AUDIO_RATE_BIT_176400] = SNDRV_PCM_RATE_176400,
	[VIRTIO_AUDIO_RATE_BIT_192000] = SNDRV_PCM_RATE_192000,
	[VIRTIO_AUDIO_RATE_BIT_24000] = SNDRV_PCM_RATE_KNOT,
	[VIRTIO_AUDIO_RATE_BIT_18900] = SNDRV_PCM_RATE_KNOT,
	[VIRTIO_AUDIO_RATE_BIT_12000] = SNDRV_PCM_RATE_KNOT,
	[VIRTIO_AUDIO_RATE_BIT_64000] = SNDRV_PCM_RATE_64000,
};

static unsigned int rate_map_vdev2abs[VIRTIO_AUDIO_RATE_LAST + 1] = {
	[VIRTIO_AUDIO_RATE_BIT_8000] = 8000,
	[VIRTIO_AUDIO_RATE_BIT_11025] = 11025,
	[VIRTIO_AUDIO_RATE_BIT_16000] = 16000,
	[VIRTIO_AUDIO_RATE_BIT_22050] = 22050,
	[VIRTIO_AUDIO_RATE_BIT_32000] = 32000,
	[VIRTIO_AUDIO_RATE_BIT_44100] = 44100,
	[VIRTIO_AUDIO_RATE_BIT_48000] = 48000,
	[VIRTIO_AUDIO_RATE_BIT_88200] = 88200,
	[VIRTIO_AUDIO_RATE_BIT_96000] = 96000,
	[VIRTIO_AUDIO_RATE_BIT_176400] = 176400,
	[VIRTIO_AUDIO_RATE_BIT_192000] = 192000,
	[VIRTIO_AUDIO_RATE_BIT_24000] = 24000,
	[VIRTIO_AUDIO_RATE_BIT_18900] = 18900,
	[VIRTIO_AUDIO_RATE_BIT_12000] = 12000,
	[VIRTIO_AUDIO_RATE_BIT_64000] = 64000,
};

/*convert fmt mask*/
static u32 rate_vdev2alsa(u32 rate_mask, struct virtio_snd_rate_caps *caps)
{
	unsigned int bit;
	u32 alsa_mask = 0;

	if (caps)
		memset(caps, 0, sizeof(*caps));

	for (bit = 0; bit <= VIRTIO_AUDIO_RATE_LAST; bit++) {
		if (rate_mask & (1u << bit)) {
			alsa_mask |= rate_map_vdev2alsa[bit];
			if (caps) {
				unsigned int curr_rate;

				curr_rate = rate_map_vdev2abs[bit];
				caps->rates_list[caps->rates_cnt] = curr_rate;
				if (curr_rate > caps->max_rate)
					caps->max_rate = curr_rate;
				if (!caps->min_rate ||
				    (curr_rate < caps->min_rate))
					caps->min_rate = curr_rate;
				caps->rates_cnt++;
			}
		}
	}
	return alsa_mask;
}

static int virtio_snd_cmd_wait(struct virtio_snd_pcm *vpcm,
			       struct cmd_job *buf, u32 cmd,
			       struct virtio_audio_status *status)
{
	int err = cmd_job_wait(buf, vdev_cmd_tout(cmd));

	if (err) {
		VPCM_ERR(vpcm, "timeout on cmd %s err=%d\n",
				vdev_cmd_name(cmd), err);
		cmd_job_put(buf);
		return err;
	}

	if (status)
		memcpy(status, &buf->status, sizeof(*status));

	if ((cmd != VIRTIO_AUDIO_CMD_PREPARE)&&(buf->status.status != VIRTIO_AUDIO_S_OK)) {
		VPCM_ERR(vpcm, "error %d on cmd %s\n", buf->status.status,
			 vdev_cmd_name(cmd));
		err = -EINVAL;
	}
	cmd_job_put(buf);

	VPCM_LOG(vpcm, "%s(%d) command=%s\n", __func__, vpcm->devno,
			vdev_cmd_name(cmd));
	return err;
}

static int virtio_snd_cmd_send(struct virtio_snd_pcm *vpcm,
			       u32 cmd, struct cmd_job *buf)
{
	struct scatterlist sg[2];
	int err;

	cmd_job_get(buf);
	buf->cmd.id = cmd;

	if (cmd == VIRTIO_AUDIO_CMD_ACQUIRE)
		buf->cmd.acquire_params = vpcm->acquire_params;

	buf->status.status = VIRTIO_AUDIO_S_OK;
	sg_init_table(sg, 2);
	sg_set_buf(&sg[0], &buf->cmd, sizeof(buf->cmd));
	sg_set_buf(&sg[1], &buf->status, sizeof(buf->status));

	err = virtqueue_add_outbuf(vpcm->vq_cmd, sg, 2, buf, GFP_ATOMIC);
	if (err) {
		VPCM_ERR(vpcm, "add to queue failed: %d\n", err);
		cmd_job_put(buf);
		return err;
	}
	virtqueue_kick(vpcm->vq_cmd);

	VPCM_LOG(vpcm, "%s(%d) command=%s\n", __func__, vpcm->devno,
			vdev_cmd_name(cmd));
	return err;
}

/*send command an wait for completion*/
static int virtio_snd_cmd_send_wait(struct virtio_snd_pcm *vpcm, u32 cmd,
				    struct virtio_audio_status *status)
{
	struct cmd_job *job;
	unsigned long flags;
	int err;

	if( cmd == VIRTIO_AUDIO_CMD_NONE)
		return 0;

	job = cmd_job_create(vpcm, GFP_KERNEL, cmd);
	if (!job) {
		VPCM_ERR(vpcm, "failed getting job for cmd %s\n",
			 vdev_cmd_name(cmd));
		return -ENOMEM;
	}

	snd_pcm_stream_lock_irqsave(vpcm->substream, flags);
	err = virtio_snd_cmd_send(vpcm, cmd, job);
	snd_pcm_stream_unlock_irqrestore(vpcm->substream, flags);

	if (err) {
		VPCM_ERR(vpcm, "send command failed");
		cmd_job_put(job);
		return err;
	}

	return virtio_snd_cmd_wait(vpcm, job, cmd, status);
}

/*send command. do not wait for completion. called with stream lock held*/
static int virtio_snd_cmd_send_nowait(struct virtio_snd_pcm *vpcm, u32 cmd)
{
	struct cmd_job *job;
	int err;

	if( cmd == VIRTIO_AUDIO_CMD_NONE)
		return 0;

	job = cmd_job_create(vpcm, GFP_ATOMIC, cmd);
	if (!job) {
		VPCM_ERR(vpcm, "failed getting job for cmd %s\n",
			 vdev_cmd_name(cmd));
		return -ENOMEM;
	}
	err = virtio_snd_cmd_send(vpcm, cmd, job);

	if (err) {
		VPCM_ERR(vpcm, "send command failed");
		cmd_job_put(job);
		return err;
	}
	cmd_job_put(job);
	return 0;
}

static void virtio_snd_cb_cmd(struct virtqueue *vq)
{
	struct cmd_job *buf;
	u32 len;

	while ((buf = virtqueue_get_buf(vq, &len)) != NULL)
		cmd_job_complete(buf);
}

static int virtio_snd_samples_send_sg(struct virtio_snd_pcm *vpcm,
		struct snd_pcm_substream *substream,
		unsigned int		offset,
		unsigned int		len,
		snd_pcm_uframes_t	actual_frames)
{
	struct snd_pcm_runtime	*rt = substream->runtime;
	struct stream_job *buf;
	struct scatterlist sg[2];
	int	err;

	while (len > 0) {
		void		*start;
		unsigned int	chunk;
		snd_pcm_uframes_t	chunkFrames;

		buf = stream_job_get(vpcm);
		if (!buf) {
			VPCM_ERR(vpcm, "%s(%d) stream_job empty\n",
					__func__, vpcm->devno);
			return -ENOMEM;
		}

		buf->status.status = VIRTIO_AUDIO_S_OK;
		sg_init_table(sg, ARRAY_SIZE(sg));

		start = snd_pcm_sgbuf_get_ptr(substream, offset);
		chunk = snd_pcm_sgbuf_get_chunk_size(substream,
				offset, len);
		sg_set_buf(sg, start, chunk);
		sg_set_buf(sg + 1, &buf->status, sizeof(buf->status));

		chunkFrames = bytes_to_frames(rt, chunk);
		if (actual_frames > chunkFrames) {
			buf->actual_frames = chunkFrames;
			actual_frames -= chunkFrames;
		} else {
			buf->actual_frames = actual_frames;
			actual_frames = 0;
		}

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			err = virtqueue_add_outbuf(vpcm->vq_stream,
					sg, 2, buf, GFP_ATOMIC);
		else
			err = virtqueue_add_inbuf(vpcm->vq_stream,
					sg, 2, buf, GFP_ATOMIC);

		if (err) {
			VPCM_ERR(vpcm, "%s(%d) virtqueue add failed: "
					"offset=%d size=%d err=%d\n",
					__func__, vpcm->devno,
					offset, chunk, err);
			stream_job_put(vpcm, buf);
			return err;
		}

		len -= chunk;
		offset += chunk;
	}
	return 0;
}

static int virtio_snd_samples_send(struct virtio_snd_pcm *vpcm,
		struct snd_pcm_substream *substream,
		unsigned int	offset,
		unsigned int	len,
		snd_pcm_uframes_t	actual_frames)
{
	struct stream_job *buf = stream_job_get(vpcm);
	struct scatterlist sg[2];
	int	err;
	if (!buf) {
		VPCM_ERR(vpcm, "%s(%d) stream_job empty\n",
				__func__, vpcm->devno);
		return -ENOMEM;
	}
	buf->actual_frames = actual_frames;

	buf->status.status = VIRTIO_AUDIO_S_OK;
	sg_init_table(sg, ARRAY_SIZE(sg));
	sg_set_buf(sg, substream->runtime->dma_area + offset, len);
	sg_set_buf(sg + 1, &buf->status, sizeof(buf->status));

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		err = virtqueue_add_outbuf(vpcm->vq_stream, sg, 2,
					   buf, GFP_ATOMIC);
	else
		err = virtqueue_add_inbuf(vpcm->vq_stream, sg, 2,
					  buf, GFP_ATOMIC);

	if (err) {
		VPCM_ERR(vpcm, "virtqueue add failed: %d\n", err);
		stream_job_put(vpcm, buf);
		return err;
	}

	return 0;
}

static inline void silence_padding(
		struct virtio_snd_pcm *vpcm,
		struct snd_pcm_substream *substream,
		struct snd_pcm_runtime *rt,
		unsigned int offset,
		snd_pcm_uframes_t avail,
		snd_pcm_uframes_t xfer)
{
	void		*start;
	unsigned int	len;

	if (avail >= xfer) {
		return;
	}
	len = frames_to_bytes(rt, xfer - avail);
	offset += frames_to_bytes(rt, avail);

	VPCM_LOG(vpcm, "%s(%d) avail=%lu xfer=%lu\n",
			__func__, vpcm->devno, avail, xfer);

	while (sgmode && len > 0) {
		unsigned int	chunk;

		start = snd_pcm_sgbuf_get_ptr(substream, offset);
		chunk = snd_pcm_sgbuf_get_chunk_size(substream,
				offset, len);
		snd_pcm_format_set_silence(rt->format,
				start,
				bytes_to_frames(rt, chunk)
					* rt->channels);
		VPCM_LOG(vpcm, "%s(%d) offset=%d len=%d "
				"=%lu frames x %d channels\n",
				__func__, vpcm->devno,
				offset, chunk,
				bytes_to_frames(rt, chunk),
				rt->channels);

		len -= chunk;
		offset += chunk;
	}
	if (!sgmode) {
		start = rt->dma_area + offset;
		snd_pcm_format_set_silence(rt->format,
				start,
				(xfer - avail) * rt->channels);
	}
}

#define buff_avail(v) ((v)->num_inflight < ((v)->max_s))
#define substr2vpcm(s) (((struct virtio_snd_pair *)\
		       ((s)->pcm->private_data))->dir[(s)->stream])

enum is_callback {
	async_work = 0,
	in_callback,
	in_tryxfer,
};
static void do_force_flush(struct virtio_snd_pcm *vpcm,
		enum is_callback in_callback);

/*must be called with substream lock held*/
static int virtio_snd_try_xfer(struct snd_pcm_substream *substream)
{
	struct virtio_snd_pcm *vpcm = substr2vpcm(substream);
	struct snd_pcm_runtime *rt = substream->runtime;
	int err = 0;
	unsigned int bufs_added = 0;
	bool drain = 0;

	snd_pcm_uframes_t avail;
	snd_pcm_uframes_t xfer = rt->period_size;
	snd_pcm_uframes_t avail_min = xfer;

	if (vpcm->state != VIRTIO_AUDIO_STATE_PREPARED) {
		VPCM_LOG(vpcm,"%s(%d) avoid bad state: stataus is %d",
			__func__, vpcm->devno, vpcm->state);
		return 0;
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		if (rt->control->appl_ptr < vpcm->app_pos)
			avail = rt->boundary - (vpcm->app_pos -
						rt->control->appl_ptr);
		else
			avail = rt->control->appl_ptr - vpcm->app_pos;
		if (vpcm->draining){
			drain = 1;
			avail_min = 1;
		}
	} else {
		if (rt->control->appl_ptr > vpcm->app_pos)
			avail = rt->buffer_size - (rt->boundary -
						   (rt->control->appl_ptr -
						    vpcm->app_pos));
		else
			avail = rt->buffer_size - (vpcm->app_pos -
						   rt->control->appl_ptr);
	}
	DBG("avail %ld\n", avail);

	while ((avail >= avail_min) && buff_avail(vpcm)) {
		unsigned int	offset, len;

		offset = frames_to_bytes(rt,
				vpcm->app_pos % rt->buffer_size);
		len = frames_to_bytes(rt, xfer);

		if (drain && (avail < xfer)) {
			silence_padding(vpcm, substream, rt,
					offset, avail, xfer);
			xfer = avail;
		}

		if (sgmode) {
			err = virtio_snd_samples_send_sg(vpcm, substream,
					offset, len, xfer);
		} else {
			err = virtio_snd_samples_send(vpcm, substream,
					offset, len, xfer);
		}
		if (err) {
			VPCM_ERR(vpcm, "send samples failed: %d\n", err);
			break;
		}
		bufs_added++;
		vpcm->app_pos += xfer;
		if (vpcm->app_pos >= substream->runtime->boundary)
			vpcm->app_pos -= substream->runtime->boundary;
		avail -= xfer;
	}

	/* kick once if at least one buffer has been added */
	if (bufs_added) {
		virtqueue_kick(vpcm->vq_stream);
		if (vpcm->num_inflight <= vpcm->max_hostquesz) {
			do_force_flush(vpcm, in_tryxfer);
		}
	}
	DBG("finished. added %d bufs\n", bufs_added);
	return err;
}

static int force_flush = 1;
module_param(force_flush, int, S_IRUGO | S_IWUSR | S_IWGRP);
MODULE_PARM_DESC(force_flush, "force flush, for flow control at vdev");

static void do_force_flush(struct virtio_snd_pcm *vpcm,
		enum is_callback in_callback)
{
	struct snd_pcm_substream	*substream = vpcm->substream;
	struct snd_pcm_runtime *rt = substream->runtime;
	unsigned long	flags;
	unsigned long	is_force_flush;
	snd_pcm_uframes_t	leftsz;

	flags = 0;	/* to prevent compiler warning */

#define FOR_DBG_FORCEFLUSH
#undef  FOR_DBG_FORCEFLUSH
#ifdef  FOR_DBG_FORCEFLUSH
	VPCM_LOG(vpcm, 
			"%s(%d) in_callback=%d\n",
			__func__, vpcm->devno, in_callback);
#endif

	if (in_callback) {
		is_force_flush = (vpcm->force_flushing <= 1);
	} else {
		/*
		 * force_flush work
		 */
		ktime_t now = ktime_get();
		unsigned long delay;

		snd_pcm_stream_lock_irqsave(substream, flags);
		is_force_flush = (vpcm->force_flushing == 1) &&
			vpcm->force_flush_time.tv64;

		delay = nsecs_to_jiffies(ktime_to_ns(
			ktime_sub(vpcm->force_flush_time, now)));

		if (is_force_flush 
			&& ktime_compare(vpcm->force_flush_time, now) > 0
			&& delay > 0) {
			/*
			 * wait more time
			 */ 
			schedule_delayed_work(&vpcm->force_flush_work, delay );

			VPCM_LOG(vpcm,
					"%s(%d) resched delayed work "
					"delay %ld tick\n",
					__func__, vpcm->devno, delay );
			snd_pcm_stream_unlock_irqrestore(substream, flags);
			return;
		}
	}

	is_force_flush &= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK);
			/*
			 * playback
			 */
	if (rt->control->appl_ptr >= vpcm->app_pos) {
		leftsz = rt->control->appl_ptr - vpcm->app_pos;
	} else {
		leftsz = rt->boundary -
			(vpcm->app_pos - rt->control->appl_ptr);
	}
	is_force_flush &= (leftsz < rt->period_size);
	if (is_force_flush) {
		/*
		 * avail not exist
		 * no progress application data
		 */
		if (vpcm->app_pos == (in_callback ?
				vpcm->app_pos_previous :
				vpcm->app_pos_previous_wq) ) {
			/*
			 * not xfer any additional data
			 */
			;
		} else if (in_callback == in_tryxfer) {
			/*
			 * just sent under vpcm->hostquesz
			 * -> not return ack now
			 */
			;
		} else {
			is_force_flush = false;
		}
	}
#ifdef  FOR_DBG_FORCEFLUSH
	VPCM_LOG(vpcm,
			"%s(%d) is_force_flush = %ld leftsz %ld "
			"rt->period_size %ld app_pos %ld "
			"app_pos_previous %ld\n",
			__func__, vpcm->devno,
			is_force_flush, leftsz,
			rt->period_size, vpcm->app_pos,
			vpcm->app_pos_previous);
#endif

	if (in_callback == async_work) {
		;
	} else if (vpcm->app_pos_previous != vpcm->app_pos) {
		/*
		 * in_callback, pregress app_pos
		 */
		vpcm->app_pos_previous = vpcm->app_pos;
		vpcm->last_update_time = ktime_get();
		vpcm->no_update_count = 0;

		if (vpcm->force_flushing == 1) {
			/*
			 * cancel work
			 */
			vpcm->force_flush_time.tv64 = 0LL;
		}
	} else {
		/*
		 * in_callback, no update app_pos
		 */
		vpcm->no_update_count++;
	}

	/*
	 * no more application data, as of now
	 */
	if (is_force_flush && in_callback &&
		(rt->control->appl_ptr == vpcm->app_pos) &&
		(vpcm->num_inflight <= vpcm->max_hostquesz + 3)) {
		/*
		 * debug trace start
		 */
		VPCM_LOG(vpcm,
			"## %s(%d) appr-ptr=%ld app-pos=%ld "
			"hw-ptr=%ld hw-pos=%ld "
			"inflight=%d hostquesz=%d force_flushing=%d "
			"caller=%pS\n",
			__func__, vpcm->devno,
			rt->control->appl_ptr, vpcm->app_pos,
			rt->status->hw_ptr, vpcm->hw_pos,
			vpcm->num_inflight,
			vpcm->max_hostquesz,
			vpcm->force_flushing,
			__builtin_return_address(0));
	}
	is_force_flush &= !!force_flush &&
		(vpcm->num_inflight > 0) &&
		(vpcm->num_inflight <= vpcm->max_hostquesz);
		/*
		 * data in virtque, but possiblely host might receive
		 * all data.
		 * -> callback may not be called under flow-control
		 *    - max_hostquesz in host DMA buffer
		 *    - 1 at writing snd_pcm_plugin_write, not return
		 * host will finish FLUSH at end of virtqueue.
		 * So FLUSH must be issue before max_hostquesz + 1
		 */

	if (in_callback == async_work) {
		;
	} else if (is_force_flush) {
		unsigned long	delay;
		ktime_t         now = ktime_get();

		vpcm->force_flush_time = ktime_add(
			vpcm->last_update_time,
			ns_to_ktime(100 * NSEC_PER_MSEC));

		if (ktime_compare(vpcm->force_flush_time,
			now) <= 0) {
			vpcm->force_flush_time = now;
			delay = 0;
		} else {
			ktime_t t;

			t = ktime_sub(vpcm->force_flush_time, now);

			delay = nsecs_to_jiffies(ktime_to_ns(t));
		}

		vpcm->app_pos_previous_wq = vpcm->app_pos_previous;

		if (vpcm->force_flushing == 0) {
			VPCM_LOG(vpcm, "## %s(%d) check after "
				"delay=%ld tick, noupd=%d\n",
				__func__, vpcm->devno, delay,
				vpcm->no_update_count);

			vpcm->force_flushing = 1;
			schedule_delayed_work(&vpcm->force_flush_work, delay);
		} else {
#ifdef  FOR_DBG_FORCEFLUSH
			VPCM_LOG(vpcm, "## %s(%d) update "
				"delay=%ld tick, noupd=%d\n",
				__func__, vpcm->devno, delay,
				vpcm->no_update_count);
#endif
		}
		return;
	} else {
		/*
		 * in_callback, !is_force_flush
		 */
		return;
	}

	if (is_force_flush) {
		rt->stop_threshold = rt->buffer_size;
			/*
			 * start detecting xrun
			 */
		vpcm->force_flushing = 2;
			/*
			 * no more watch app_pos and num_inflight
			 */
	} else {
		vpcm->force_flushing = 0;
			/*
			 * end force flushing in workqueue thread
			 */
	}
	snd_pcm_stream_unlock_irqrestore(substream, flags);
	if (!is_force_flush) {
		wake_up(&vpcm->wqueue);
		return;
	}

	/*
	 * possible not to return callback, by vdev flow control
	 * force flush to get all callback
	 */
	VPCM_LOG(vpcm,
			"%s(%d) appr-ptr=%ld app-pos=%ld "
			"hw-ptr=%ld hw-pos=%ld "
			"inflight=%d hostquesz=%d\n",
			__func__, vpcm->devno,
			rt->control->appl_ptr, vpcm->app_pos,
			rt->status->hw_ptr, vpcm->hw_pos,
			vpcm->num_inflight,
			vpcm->max_hostquesz);

	vpcm->state = VIRTIO_AUDIO_STATE_FLUSHING;
	virtio_snd_cmd_send_wait(vpcm, VIRTIO_AUDIO_CMD_FLUSH, NULL);
	virtio_snd_cmd_send_wait(vpcm, VIRTIO_AUDIO_CMD_PREPARE, NULL);

	snd_pcm_stream_lock_irqsave(substream, flags);
	vpcm->force_flushing = 0;

	if (snd_pcm_running(substream))
		virtio_snd_try_xfer(substream);
		/*
		 * stream may restarted again
		 */

	snd_pcm_stream_unlock_irqrestore(substream, flags);

	VPCM_LOG(vpcm, "%s(%d) finished\n", __func__, vpcm->devno);
	wake_up(&vpcm->wqueue);
}

static void force_flush_work_fn(struct work_struct *work)
{
	struct virtio_snd_pcm	*vpcm;

	vpcm = container_of(work, typeof(*vpcm), force_flush_work.work);
	do_force_flush(vpcm, false);
}

/*called with IRQ's disabled by virtio framework*/
static void virtio_snd_cb_stream(struct virtqueue *vq)
{
	struct virtio_snd_card *vcard = vq->vdev->priv;
	struct virtio_snd_pcm *vpcm = &vcard->devices[vq->index / 2];
	struct stream_job *buf;
	unsigned long flags;
	u32 len;
	unsigned int cnt = 0;
	unsigned int err_cnt = 0;
	u32	err_status;
	struct snd_pcm_substream *substream;
	snd_pcm_uframes_t	sent_frames, app_pos;

	substream = vpcm->substream;
	DBG("CB STREAM\n");
	if (!substream) {
		/*
		 * callback very delayed at FLUSH timeout at hw_free
		 * -- This is for safe to prevent panic
		 */
		printk(KERN_ERR "%s(%d) already closed\n",
				__func__, vpcm->devno);
		return;
	}

	snd_pcm_stream_lock_irqsave(substream, flags);

	/*block any update in flushing state*/
	if (vpcm->state == VIRTIO_AUDIO_STATE_FLUSHING) {
		DBG("draining: ignore buffer cb...\n");
		VPCM_LOG(vpcm, "%s(%d) draining: ignore buffer cb "
				"inflight=%d\n",
				__func__, vpcm->devno, vpcm->num_inflight);
		snd_pcm_stream_unlock_irqrestore(substream, flags);
		return;
	}

	err_status = VIRTIO_AUDIO_S_OK;
	sent_frames = 0;
	while ((buf = virtqueue_get_buf(vq, &len)) != NULL) {
		cnt++;
		if (buf->status.status == VIRTIO_AUDIO_S_OK) {
			sent_frames += buf->actual_frames;
		} else {
			err_status = buf->status.status;
			err_cnt++;
		}
		stream_job_put(vpcm, buf);
	}

	/*notify each elapsed period as ALSA won't detect a full buffer
	 *consumed at once which could happen on a sw driven device
	 */
	if (sent_frames) {
		vpcm->hw_pos += sent_frames;
		if (vpcm->hw_pos >= substream->runtime->boundary)
			vpcm->hw_pos -= substream->runtime->boundary;

		snd_pcm_stream_unlock_irqrestore(substream, flags);
		snd_pcm_period_elapsed(substream);
		snd_pcm_stream_lock_irqsave(substream, flags);
	}

	/* treat any error as XRUN and let user try to recover stream */
	if (err_cnt) {
		DBG("CB STREAM ERR\n");
		VPCM_ERR(vpcm, "%s(%d) err=%d(%d) cause xrun\n",
				__func__, vpcm->devno, err_cnt, err_status);

		if (cnt) {
			do_force_flush(vpcm, true);
		}
		snd_pcm_stop(substream, SNDRV_PCM_STATE_XRUN);
		snd_pcm_stream_unlock_irqrestore(substream, flags);
		return;
	}

	app_pos = vpcm->app_pos;
	if (snd_pcm_running(substream))
		virtio_snd_try_xfer(substream);
	if (cnt) {
		do_force_flush(vpcm, true);
	}

	snd_pcm_stream_unlock_irqrestore(substream, flags);
}

static void virtio_snd_action_sync(struct cmd_job *job)
{
	complete(&job->done);
}

static void virtio_snd_action_ignore(struct cmd_job *job)
{
	/*do nothing*/
}

/* flush is tricky. device already returns buffer as 'finished' once forwarded
 * to real hardware. If we notify this to ALSA drain will return too early
 * while the hardware is still producing sound.
 * This can result in unwanted side effects. To overcome this we block any
 * position update while flushing.
 * Once flush command is finished we unblock the update.
 */
static void virtio_snd_action_flush(struct cmd_job *job)
{
	struct virtio_snd_pcm *vpcm = job->vpcm;
	unsigned long flags;

	snd_pcm_stream_lock_irqsave(vpcm->substream, flags);
	if (vpcm->state == VIRTIO_AUDIO_STATE_FLUSHING) {
		DBG("drain finished\n");
		VPCM_LOG(vpcm, "%s(%d) drain finished\n",
				__func__, vpcm->devno);
		vpcm->state = VIRTIO_AUDIO_STATE_ACQUIRED;
	}
	snd_pcm_stream_unlock_irqrestore(vpcm->substream, flags);
	virtio_snd_cb_stream(vpcm->vq_stream);
	complete(&job->done);
}

static void virtio_snd_action_prepare(struct cmd_job *job)
{
	struct virtio_snd_pcm *vpcm = job->vpcm;
	unsigned long flags;

	snd_pcm_stream_lock_irqsave(vpcm->substream, flags);
	vpcm->state = VIRTIO_AUDIO_STATE_PREPARED;
	snd_pcm_stream_unlock_irqrestore(vpcm->substream, flags);
	complete(&job->done);
}

static int virtio_snd_open(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *rt = substream->runtime;
	struct snd_pcm_hardware *hw = &substream->runtime->hw;
	struct virtio_snd_pcm *vpcm = substr2vpcm(substream);
	int err;

	DBG("open\n");
	*hw = vpcm->hwcaps;
	if (vpcm->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		/*
		 * flow control: adjust periods_min
		 */
		hw->periods_min = MIN_HOSTQUESZ + periods_hostgap;
		if (hw->periods_min < VIRTIO_SND_PERIODS_MIN_DEFAULT) {
			hw->periods_min = VIRTIO_SND_PERIODS_MIN_DEFAULT;
		}
	}

	if (vpcm->rate_caps.rates_cnt) {
		vpcm->rate_constraints.count = vpcm->rate_caps.rates_cnt;
		vpcm->rate_constraints.list = vpcm->rate_caps.rates_list;
		vpcm->rate_constraints.mask = 0;
		snd_pcm_hw_constraint_list(rt, 0, SNDRV_PCM_HW_PARAM_RATE,
					   &vpcm->rate_constraints);
	}

	err = snd_pcm_hw_constraint_integer(rt, SNDRV_PCM_HW_PARAM_PERIODS);
	if (err < 0) {
		VPCM_ERR(vpcm, "periods constraint err: %d\n", err);
		return err;
	}

	if (vpcm->period_align != 0) {
		err = snd_pcm_hw_constraint_step(rt, 0,
					   SNDRV_PCM_HW_PARAM_PERIOD_BYTES,
					   vpcm->period_align);
		if (err < 0) {
			VPCM_ERR(vpcm, "p_bytes constraint err: %d\n", err);
			return err;
		}
	}

	vpcm->substream = substream;
	return 0;
}

static int virtio_snd_close(struct snd_pcm_substream *substream)
{
	struct virtio_snd_pcm *vpcm = substr2vpcm(substream);

	DBG("close\n");

	vpcm->substream = NULL;

	return 0;
}

static int virtio_snd_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params)
{
	struct virtio_snd_pcm *vpcm = substr2vpcm(substream);
	int err;
	struct virtio_audio_status status;
	int	periods;

	DBG("hw_params\n");

	if (vpcm->state != VIRTIO_AUDIO_STATE_CLOSED) {
		VPCM_ERR(vpcm, "device in use\n");
		return -EBUSY;
	}

	err = snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(params));
	if (err < 0) {
		VPCM_ERR(vpcm, "%s(%d) alloc sample pages failed: %d\n",
				__func__, vpcm->devno, err);
		return err;
	}
	/*setup audio format to send to host*/
	vpcm->acquire_params.rate = params_rate(params);
	vpcm->acquire_params.voices = params_channels(params);
	vpcm->acquire_params.format =
		fmt_map_alsa2vdev[(__force int)params_format(params)];
	vpcm->acquire_params.frag_size = params_period_bytes(params);

	periods = params_periods(params);
	if (periods > vpcm->max_s) {
		periods = vpcm->max_s;
	}
	periods /= 2;
	vpcm->max_hostquesz = periods;
		/*
		 * when AP filled with virtqueue,
		 * half of then are sent in host queue (DMA buffers)
		 * and leave half of then in virtqueue.
		 * It will prevent underrun at flow-control=off
		 */
	if (params_periods(params) > periods_hostgap) {
		periods = params_periods(params) - periods_hostgap;
	}
	if (vpcm->max_hostquesz > periods) {
		vpcm->max_hostquesz = periods;
	}
		/*
		 * keep gap of between periods and max_hostquesz
		 * for detecting force flush
		 */

	if (vpcm->max_hostquesz > MAX_HOSTQUESZ) {
		vpcm->max_hostquesz = MAX_HOSTQUESZ;
	}
	if (vpcm->max_hostquesz < MIN_HOSTQUESZ) {
		vpcm->max_hostquesz = MIN_HOSTQUESZ;
	}
	if (!force_flush || vpcm->stream == SNDRV_PCM_STREAM_CAPTURE) {
		/*
		 * compatible for past day's implementation
		 */
		vpcm->max_hostquesz = params_periods(params);
			/* FIXME ?? */
	}
	VPCM_LOG(vpcm, "%s(%d) "
			"params_periods=%d "
			"vpcm->max_s=%d "
			"max_hostquesz=%d\n",
			__func__, vpcm->devno,
			params_periods(params),
			vpcm->max_s, vpcm->max_hostquesz);

	vpcm->acquire_params.frags_max = vpcm->max_hostquesz;

	err = virtio_snd_cmd_send_wait(vpcm, VIRTIO_AUDIO_CMD_ACQUIRE, &status);

	if (err)
		return err;

	/* Resulting frag_size is not necessarily aligned to our request.. */
	if (!status.acquire_status.frag_size) {
		virtio_snd_cmd_send_wait(vpcm, VIRTIO_AUDIO_CMD_RELEASE, NULL);
		VPCM_ERR(vpcm, "host did not specify fragment size\n");
		return -EINVAL;
	}
	vpcm->state = VIRTIO_AUDIO_STATE_ACQUIRED;

	return 0;
}

static int virtio_snd_hw_free(struct snd_pcm_substream *substream)
{
	struct virtio_snd_pcm *vpcm = substr2vpcm(substream);
	int ret;
	unsigned long	flags;

	VPCM_LOG(vpcm, "%s(%d) inflight=%d\n",
			__func__, vpcm->devno, vpcm->num_inflight);

	if (!wait_event_timeout(vpcm->wqueue,
			!vpcm->force_flushing, msecs_to_jiffies(1000))) {
		VPCM_ERR(vpcm, "%s(%d) timeout inflight=%d %s\n",
				__func__, vpcm->devno,
				vpcm->num_inflight,
				vpcm->force_flushing ? " force_flushing" :"");
	}

	snd_pcm_stream_lock_irqsave(substream, flags);
	/* hw_free gets called after successful or failing hw_params
	 * or even w/o hw_params being called before.
	 * Prevent sending release to host if not acquired.
	 */
	if (vpcm->state == VIRTIO_AUDIO_STATE_CLOSED) {
		snd_pcm_stream_unlock_irqrestore(substream, flags);
		return 0;
	} 

	vpcm->state = VIRTIO_AUDIO_STATE_FLUSHING;
	snd_pcm_stream_unlock_irqrestore(substream, flags);

	ret = virtio_snd_cmd_send_wait(vpcm, VIRTIO_AUDIO_CMD_FLUSH, NULL);

	if (!wait_event_timeout(vpcm->wqueue,
			!vpcm->num_inflight && !vpcm->force_flushing,
			msecs_to_jiffies(5000))) {

		VPCM_ERR(vpcm, "%s(%d) timeout before RELEASE cmd inflight=%d%s\n",
				__func__, vpcm->devno,
				vpcm->num_inflight,
				vpcm->force_flushing ? " force_flushing" :"");
	}

	ret = virtio_snd_cmd_send_wait(vpcm, VIRTIO_AUDIO_CMD_RELEASE, NULL);

	snd_pcm_lib_free_pages(substream);

	snd_pcm_stream_lock_irqsave(substream, flags);
	vpcm->state = VIRTIO_AUDIO_STATE_CLOSED;
	snd_pcm_stream_unlock_irqrestore(substream, flags);

	return ret;
}

static int virtio_snd_prepare(struct snd_pcm_substream *substream)
{
	struct virtio_snd_pcm *vpcm = substr2vpcm(substream);

	VPCM_LOG(vpcm, "%s(%d) inflight=%d\n",
			__func__, vpcm->devno, vpcm->num_inflight);

	if (!wait_event_timeout(vpcm->wqueue,
			!vpcm->num_inflight && !vpcm->force_flushing,
			msecs_to_jiffies(1000))) {
		VPCM_ERR(vpcm, "%s(%d) timeout inflight=%d%s\n",
				__func__, vpcm->devno,
				vpcm->num_inflight,
				vpcm->force_flushing ? " force_flushing" :"");
	}
	vpcm->hw_pos = 0;
	vpcm->app_pos = 0;
	vpcm->app_pos_previous = 0;
	vpcm->force_flushing = 0;
	vpcm->draining = 0;

	return virtio_snd_cmd_send_wait(vpcm, VIRTIO_AUDIO_CMD_PREPARE, NULL);
}

static int virtio_snd_trigger(struct snd_pcm_substream *substream, int cmd)
{
	int ret = 0;
	struct virtio_snd_pcm *vpcm = substr2vpcm(substream);
#define TRIGGER_CMD_DEF(s)	[SNDRV_PCM_TRIGGER_ ##s] = #s
	const static char	*trigger_cmd[] = {
		TRIGGER_CMD_DEF(STOP),
		TRIGGER_CMD_DEF(START),
		TRIGGER_CMD_DEF(PAUSE_PUSH),
		TRIGGER_CMD_DEF(PAUSE_RELEASE),
		TRIGGER_CMD_DEF(SUSPEND),
		TRIGGER_CMD_DEF(RESUME),
		TRIGGER_CMD_DEF(DRAIN),
	};

	VPCM_LOG(vpcm, "%s(%d) cmd=%s%s\n",
		__func__, vpcm->devno,
		cmd >= 0 && sizeof(trigger_cmd)/sizeof(trigger_cmd[0]) ?
		trigger_cmd[cmd] : "?",
		vpcm->force_flushing ? " force-flushing" : "");

		/*
		 * wait for force_flushing, if processing
		 */

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		DBG("UNPAUSE\n");
		ret = virtio_snd_cmd_send_nowait(vpcm, VIRTIO_AUDIO_CMD_RESUME);
		virtio_snd_try_xfer(substream);
		break;
	case SNDRV_PCM_TRIGGER_START:
		DBG("START\n");

		/* disabe xrun detection as it makes no sense on sw device.
		 * xrun on hardware is signalled via status in stream queue.
		 */
		substream->runtime->stop_threshold =
			substream->runtime->boundary;
		virtio_snd_try_xfer(substream);
		break;
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		DBG("PAUSE\n");
		ret = virtio_snd_cmd_send_nowait(vpcm, VIRTIO_AUDIO_CMD_PAUSE);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		DBG("STOP\n");
		if (substream->stream == SNDRV_PCM_STREAM_CAPTURE)
			break;
		if (vpcm->num_inflight) {
			/*
			 * STOP means AP stopped send more data
			 * so try to force flush
			 */
			vpcm->app_pos_previous = vpcm->app_pos;
			do_force_flush(vpcm, true);
		}
		break;

	case SNDRV_PCM_TRIGGER_DRAIN:
		DBG("DRAIN\n");
		if (substream->stream == SNDRV_PCM_STREAM_CAPTURE)
			break;
		vpcm->draining = 1;
		virtio_snd_try_xfer(substream);
		/** detect xrun from now */
		substream->runtime->stop_threshold =
			substream->runtime->buffer_size;

		if (!vpcm->num_inflight) {
			snd_pcm_stop(substream, SNDRV_PCM_STATE_XRUN);
		} else {
			/*
			 * DRAIN means AP stopped send more data
			 * so try to force flush
			 *
			 * force_flush=0:
			 *   Nothing to do else log in do_force_flush()
			 *   All Ack will be return soon.
			 * force_flush=1
			 *   Detecting to stop AP sending may be
			 *   impossible if prevoius sending count
			 *   is less than hostquesz.
			 *   so set app_pos_previous as AP progress
			 *   has been stopped, then do_force_flush()
			 *   will make vdev to return all Ack.
			 *
			 * Both case returned Ack derive XRUN, then
			 * DRAIN will finished.
			 */
			vpcm->app_pos_previous = vpcm->app_pos;
			do_force_flush(vpcm, true);
		}

		break;
	default:
		break;
	}

	return ret;
}

/*called with IRQ disabled and substream lock held*/
static snd_pcm_uframes_t virtio_snd_pointer(struct snd_pcm_substream *substream)
{
	struct virtio_snd_pcm *vpcm = substr2vpcm(substream);
	snd_pcm_uframes_t hw_ptr;

	hw_ptr = vpcm->hw_pos % substream->runtime->buffer_size;
	DBG("PTR: %ld\n", hw_ptr);
	return hw_ptr;
}

/*called with IRQ disabled and substream lock held*/
static int virtio_snd_ack(struct snd_pcm_substream *substream)
{
	DBG("ACK\n");
	if (snd_pcm_running(substream))
		virtio_snd_try_xfer(substream);
	return 0;
}

static const struct snd_pcm_ops virtio_snd_ops = {
	.open		= virtio_snd_open,
	.close		= virtio_snd_close,
	.ioctl		= snd_pcm_lib_ioctl,
	.hw_params	= virtio_snd_hw_params,
	.hw_free	= virtio_snd_hw_free,
	.prepare	= virtio_snd_prepare,
	.trigger	= virtio_snd_trigger,
	.pointer	= virtio_snd_pointer,
	.ack		= virtio_snd_ack,
};

static int virtio_snd_empty_queues(struct virtio_snd_card *vcard)
{
	unsigned int i;

	for (i = 0; i < vcard->num_devices; i++) {
		void *buf;
		struct virtqueue *vq_stream = vcard->devices[i].vq_stream;
		struct virtqueue *vq_cmd = vcard->devices[i].vq_cmd;
		struct virtio_snd_pcm *vpcm = &vcard->devices[i / 2];

		if (!(vcard->devices[i].configuration_parsed))
			continue;

		while ((buf = virtqueue_detach_unused_buf(vq_stream)) != NULL)
			stream_job_put(vpcm, buf);
		while ((buf = virtqueue_detach_unused_buf(vq_cmd)) != NULL)
			cmd_job_put(buf);
	}

	return 0;
}

static int virtio_snd_destroy_queues(struct virtio_snd_card *vcard)
{
	struct virtio_device *vdev = vcard->vdev;

	vdev->config->del_vqs(vdev);
	kfree(vcard->callbacks);
	kfree(vcard->vqs);
	kfree(vcard->names);

	return 0;
}

static int virtio_snd_create_queues(struct virtio_snd_card *vcard)
{
	unsigned int num_queues = vcard->num_devices * 2;
	unsigned int i;
	int err;

	vcard->callbacks = kmalloc_array(num_queues, sizeof(*vcard->callbacks),
					 GFP_KERNEL);
	vcard->vqs = kcalloc(num_queues, sizeof(*vcard->vqs), GFP_KERNEL);
	vcard->names = kmalloc_array(num_queues, sizeof(*vcard->names),
				     GFP_KERNEL);

	if (!vcard->callbacks || !vcard->vqs || !vcard->names) {
		VCARD_ERR(vcard, "alloc queue structures failed");
		err = -ENOMEM;
		goto err_alloc;
	}
	/* interleaved queue layout: streamQ/cmdQ */
	for (i = 0; i < vcard->num_devices; i++) {

		if (vcard->devices[i].configuration_parsed) {

			vcard->callbacks[i * 2] = virtio_snd_cb_stream;
			vcard->names[i * 2] = "stream";

			vcard->callbacks[(i * 2) + 1] = virtio_snd_cb_cmd;
			vcard->names[(i * 2) + 1] = "cmd";
		} else {

			vcard->callbacks[i * 2] = NULL;
			vcard->names[i * 2] = NULL;
			vcard->callbacks[(i * 2) + 1] = NULL;
			vcard->names[(i * 2) + 1] = NULL;
		}
	}

	err = vcard->vdev->config->find_vqs(vcard->vdev, num_queues,
					    vcard->vqs, vcard->callbacks,
					    vcard->names);

	if (err) {
		VCARD_ERR(vcard, "find vqueues failed: %d", err);
		goto err_find_queues;
	}

	DBG("created %d queues\n", num_queues);
	return 0;

err_find_queues:
err_alloc:
	kfree(vcard->callbacks);
	kfree(vcard->vqs);
	kfree(vcard->names);

	return err;
}

static void virtio_snd_destroy_vpcm(struct virtio_snd_pcm *vpcm)
{
	kfree(vpcm->sjobs);
	kfree(vpcm->sjobs_list);
}

static int virtio_snd_setup_vpcm(struct virtio_snd_pcm *vpcm, unsigned int dev)
{
	unsigned int i;
	struct virtio_snd_card *vcard = vpcm->vcard;

	vpcm->vq_stream = vcard->vqs[dev * 2];
	vpcm->max_s = virtqueue_get_vring_size(vpcm->vq_stream) / 2;
	vpcm->sjobs = kcalloc(vpcm->max_s, sizeof(struct stream_job),
			      GFP_KERNEL);
	if (!vpcm->sjobs) {
		VPCM_ERR(vpcm, "alloc jobs failed\n");
		return -ENOMEM;
	}
	vpcm->sjobs_list = kcalloc(vpcm->max_s, sizeof(struct stream_job *),
				   GFP_KERNEL);
	if (!vpcm->sjobs_list) {
		VPCM_ERR(vpcm, "alloc jobs list failed\n");
		kfree(vpcm->sjobs);
		return -ENOMEM;
	}

	for (i = 0 ; i < vpcm->max_s; i++)
		vpcm->sjobs_list[i] = &vpcm->sjobs[i];

	DBG("queue %s size %d devno %d\n", vcard->names[dev * 2], vpcm->max_s, vpcm->devno);

	vpcm->vq_cmd = vcard->vqs[(dev * 2) + 1];
	vpcm->max_c = virtqueue_get_vring_size(vpcm->vq_cmd);
	DBG("queue %s size %d devno %d\n", vcard->names[(dev * 2) + 1], vpcm->max_c, vpcm->devno);
	return 0;
}

#define MAX_PREALLOC_CHANNELS	20
static int prealloc[MAX_PREALLOC_CHANNELS];

module_param_array(prealloc, int, NULL, S_IRUGO | S_IWUSR | S_IWGRP);
MODULE_PARM_DESC(prealloc, "preallocate buffer size in KB");

static int virtio_snd_setup_stream(struct snd_pcm *pcm,
				   struct virtio_snd_pcm *vpcm, int stream)
{
	size_t	size;
	int	type;
	struct device	*data;

	if (!vpcm)
		return 0;

	vpcm->pcm = pcm;
	snd_pcm_set_ops(pcm, stream, &virtio_snd_ops);
	if (vpcm->devno < ARRAY_SIZE(prealloc)) {
		size = prealloc[vpcm->devno] * 1024;
	} else {
		size = 0;
	}
	if (size > 0) {
		VPCM_LOG(vpcm, "%s(%d) prealloc size=%ld, "
				"take as buffer_bytes_max\n",
				__func__, vpcm->devno, size);
		vpcm->hwcaps.buffer_bytes_max = size;
	} else {
		size = 0;
	}
	sgmode = sg;	/* module param:sg is available at setup */
	if (sgmode) {
		type = SNDRV_DMA_TYPE_DEV_SG;
		data = NULL;
	} else {
		type = SNDRV_DMA_TYPE_CONTINUOUS;
		data = snd_dma_continuous_data(GFP_KERNEL);
	}

	return snd_pcm_lib_preallocate_pages(pcm->streams[stream].substream,
					     type, data,
					     size,
					     vpcm->hwcaps.buffer_bytes_max);
}

static int virtio_snd_setup_pcm(struct virtio_snd_card *vcard,
				struct virtio_snd_pair *vpair, unsigned int dev)
{
	int err;
	struct snd_pcm *pcm;
	struct virtio_snd_pcm *p = vpair->dir[SNDRV_PCM_STREAM_PLAYBACK];
	struct virtio_snd_pcm *c = vpair->dir[SNDRV_PCM_STREAM_CAPTURE];

	if (!p && !c)
		return 0;

	DBG("snd_pcm_new dev=%d", dev);
	err = snd_pcm_new(vcard->card, "VIRTIO PCM", dev,
			  p ? 1 : 0,
			  c ? 1 : 0,
			  &pcm);
	if (err) {
		VCARD_ERR(vcard, "creating pcm failed: %d", err);
		return err;
	}

	sprintf(pcm->name, "VIRTIO PCM %d", (int)dev);
	pcm->private_data = vpair;

	err = virtio_snd_setup_stream(pcm, p, SNDRV_PCM_STREAM_PLAYBACK);
	if (err) {
		VCARD_ERR(vcard, "alloc playback sample area failed: %d", err);
		return err;
	}

	err = virtio_snd_setup_stream(pcm, c, SNDRV_PCM_STREAM_CAPTURE);
	if (err) {
		VCARD_ERR(vcard, "alloc capture sample area failed: %d", err);
		return err;
	}

	return 0;
}

static int virtio_snd_destroy_devices(struct virtio_snd_card *vcard)
{
	unsigned int i;

	/*just care for vpcm. the pcm is destroyed by ALSA*/
	for (i = 0; i < vcard->num_devices; i++)
		virtio_snd_destroy_vpcm(&vcard->devices[i]);
	kfree(vcard->pairs);

	return 0;
}

/*create pcm devices according to config*/
static int virtio_snd_create_devices(struct virtio_snd_card *vcard)
{
	unsigned int i;

	vcard->pairs = kcalloc(vcard->max_devno + 1,
			       sizeof(struct virtio_snd_pair), GFP_KERNEL);
	if (!vcard->pairs)
		return -ENOMEM;

	for (i = 0; i < vcard->num_devices; i++) {
		int err;
		struct virtio_snd_pcm *vpcm = &vcard->devices[i];

		if (!vpcm->configuration_parsed)
			continue;

		err = virtio_snd_setup_vpcm(vpcm, i);
		if (err) {
			VCARD_ERR(vcard, "setup vpcm %d failed", i);
			kfree(vcard->pairs);
			return err;
		}
		vcard->pairs[vpcm->devno].dir[vpcm->stream] = vpcm;
	}

	for (i = 0; i < vcard->num_devices; i++) {
		int err;

		if (!vcard->devices[i].configuration_parsed)
			continue;
		err = virtio_snd_setup_pcm(vcard, &vcard->pairs[vcard->devices[i].devno],
					   vcard->devices[i].devno);
		if (err) {
			VCARD_ERR(vcard, "setup pcm %d failed", i);
			kfree(vcard->pairs);
			return err;
		}
	}
	return 0;
}

#define devcfg_rd32(_vdev, _dev, _which)				\
	({								\
		u32 entry = virtio_cread32(_vdev,			\
		(sizeof(u32)						\
		+ (sizeof(struct virtio_audio_chan_config) * (_dev))    \
		+ offsetof(struct virtio_audio_chan_config, _which)));	\
		entry;							\
	})

static void virtio_snd_put_config(struct virtio_snd_card *vcard)
{
	kfree(vcard->devices);
}

/*this should be the only part knowing about config layout*/
static int virtio_snd_parse_config(struct virtio_snd_card *vcard)
{
	u32 num_devices = 0;
	unsigned int i;
	struct virtio_device *vdev = vcard->vdev;

	/*how many devices?*/
	virtio_cread(vdev, struct virtio_audio_config,
		     channel_num, &num_devices);

	/*no devices */
	if (num_devices == 0) {
		VCARD_ERR(vcard, "no devices");
		return -EINVAL;
	}

	vcard->devices = kcalloc(num_devices, sizeof(struct virtio_snd_pcm),
				GFP_KERNEL);

	if (!vcard->devices) {
		VCARD_ERR(vcard, "no memory for devices");
		return -ENOMEM;
	}
	vcard->num_devices = num_devices;

	for (i = 0; i < vcard->num_devices; i++) {
		struct snd_pcm_hardware *caps;
		struct virtio_snd_pcm *vpcm;

		vpcm = &vcard->devices[i];
		vpcm->configuration_parsed = true;
		vpcm->stream = devcfg_rd32(vdev, i, channel);
		vpcm->devno = devcfg_rd32(vdev, i, devno);
		if (vpcm->devno > vcard->max_devno)
			vcard->max_devno = vpcm->devno;
		vpcm->vcard = vcard;
		init_waitqueue_head(&vpcm->wqueue);
		INIT_DELAYED_WORK(&vpcm->force_flush_work,
				force_flush_work_fn);
		vpcm->force_flushing = 0;

		caps = &vpcm->hwcaps;
		/* load default caps and overwrite with restrictions
		 * given by host
		 */
		*caps = virtio_snd_default_caps;
		/* pause not supported on host side - leads to blocking */
		if (vpcm->stream == SNDRV_PCM_STREAM_CAPTURE)
			caps->info &= ~SNDRV_PCM_INFO_PAUSE;

		DBG("dev %d dir %d rates %04x formats %04x frag %d-%d ch %d-%d\n",
		    devcfg_rd32(vdev, i, devno), devcfg_rd32(vdev, i, channel),
		    devcfg_rd32(vdev, i, rates), devcfg_rd32(vdev, i, formats),
		    devcfg_rd32(vdev, i, min_fragsize),
		    devcfg_rd32(vdev, i, max_fragsize),
		    devcfg_rd32(vdev, i, min_voices),
		    devcfg_rd32(vdev, i, max_voices));

		caps->formats = fmt_vdev2alsa(devcfg_rd32(vdev, i, formats));
		caps->rates =  rate_vdev2alsa(devcfg_rd32(vdev, i, rates),
					      &vpcm->rate_caps);
		caps->rate_min = vpcm->rate_caps.min_rate;
		caps->rate_max = vpcm->rate_caps.max_rate;
		caps->channels_min = devcfg_rd32(vdev, i, min_voices);
		caps->channels_max = devcfg_rd32(vdev, i, max_voices);
		caps->period_bytes_min = devcfg_rd32(vdev, i, min_fragsize);
		caps->period_bytes_max = devcfg_rd32(vdev, i, max_fragsize);

		if (vpcm->hwcaps.period_bytes_min != 0) {

			caps->periods_max = vpcm->hwcaps.buffer_bytes_max /
						vpcm->hwcaps.period_bytes_min;

		} else {
			vpcm->configuration_parsed = false;
			VCARD_INFO(vcard, "dev:%d can not be available", i);
			continue;
		}
		vpcm->period_align = devcfg_rd32(vdev, i, fragment_align);
	}
	return 0;
}

static int virtio_snd_probe(struct virtio_device *vdev)
{
	struct virtio_snd_card *vcard;
	int ret;

	/* having accessible configuration is mandatory*/
	if (!vdev->config->get) {
		VDEV_ERR(vdev, "%s failure: config access disabled\n",
			 __func__);
		return -EINVAL;
	}

	/*create ALSA and virtio card instance */
	vcard = kzalloc(sizeof(*vcard), GFP_KERNEL);
	if (!vcard) {
		VDEV_ERR(vdev, "no mem for vcard");
		return -ENOMEM;
	}
	vdev->priv = vcard;
	vcard->vdev = vdev;
	ret = snd_card_new(&vdev->dev, SNDRV_DEFAULT_IDX1, SNDRV_DEFAULT_STR1,
			   THIS_MODULE, 0, &vcard->card);
	if (ret) {
		VCARD_ERR(vcard, "create card failed: %d", ret);
		goto err_no_card;
	}
	vcard->card->private_data = vcard;
	strncpy(vcard->card->driver, shortname,	sizeof(vcard->card->driver));
	strncpy(vcard->card->shortname, shortname,
		sizeof(vcard->card->shortname));
	strncpy(vcard->card->longname, longname,
		sizeof(vcard->card->longname));

	/*create pcm devices*/
	ret = virtio_snd_parse_config(vcard);
	if (ret) {
		VCARD_ERR(vcard, "parsing config failed: %d", ret);
		goto err_parse_fail;
	}

	ret = virtio_snd_create_queues(vcard);
	if (ret) {
		VCARD_ERR(vcard, "create queues failed: %d", ret);
		goto err_queues_fail;
	}

	ret = virtio_snd_create_devices(vcard);
	if (ret) {
		VCARD_ERR(vcard, "create pcms failed: %d", ret);
		goto err_pcm_fail;
	}

	ret = snd_card_register(vcard->card);
	if (ret) {
		VCARD_ERR(vcard, "card register failed: %d", ret);
		goto err_register_fail;
	}

	VCARD_INFO(vcard, "virtio sound card with %d devices registered",
		   vcard->num_devices);

	return 0;

err_register_fail:
	virtio_snd_destroy_devices(vcard);
err_pcm_fail:
	virtio_snd_destroy_queues(vcard);
err_queues_fail:
	virtio_snd_put_config(vcard);
err_parse_fail:
	snd_card_free(vcard->card);
err_no_card:
	kfree(vcard);
	return ret;
}

static void virtio_snd_remove(struct virtio_device *vdev)
{
	struct virtio_snd_card *vcard = (struct virtio_snd_card *)vdev->priv;

	VCARD_INFO(vcard, "remove virtio sound card\n");
	/* this will close all devices and free attached pcm's */
	snd_card_free(vcard->card);

	/*stop queues and discard all pending requests*/
	vdev->config->reset(vdev);
	virtio_snd_empty_queues(vcard);
	virtio_snd_destroy_queues(vcard);
	virtio_snd_destroy_devices(vcard);
	kfree(vcard->devices);
	kfree(vcard);
}

/*virtio sound is not standardized, we support only on the QNX implementation*/
static struct virtio_device_id id_table[] = {
	{ VIRTIO_ID_SOUND, VIRTIO_DEV_ANY_ID },
	{ 0 },
};

static struct virtio_driver virtio_snd_driver = {
	.feature_table_size = 0,
	.driver.name = KBUILD_MODNAME,
	.driver.owner = THIS_MODULE,
	.id_table = id_table,
	.probe = virtio_snd_probe,
	.remove = virtio_snd_remove,
};

module_virtio_driver(virtio_snd_driver);

MODULE_DEVICE_TABLE(virtio, id_table);
MODULE_AUTHOR("Andreas Pape <apape@de.adit-jv.com>");
MODULE_DESCRIPTION("virtio based ALSA driver");
MODULE_LICENSE("GPL");
