/*
 * $QNXLicenseC:
 * Copyright 2015, QNX Software Systems.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"). You
 * may not reproduce, modify or distribute this software except in
 * compliance with the License. You may obtain a copy of the License
 * at: http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTIES OF ANY KIND, either express or implied.
 *
 * This file may contain contributions from others, either as
 * contributors under the License or as licensors under other terms.
 * Please review this entire file for other proprietary rights or license
 * notices, as well as the QNX Development Suite License Guide at
 * http://licensing.qnx.com/license-guide/ for other information.
 * $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <devctl.h>
#include "asoundlib.h"


#define SND_FILE_AFM_SERVICE "/dev/snd/afmC%iD%i"

struct snd_afm
{
	int     fd;
};

/* Open an Acoustic Function Module (AFM) by name */
int
snd_afm_open_name (snd_afm_t **handle, const char *filename)
{
	int fd, ver;
	snd_afm_t *afm;

	*handle = NULL;

	if (!filename || !filename[0])
		return -EINVAL;
	if ((fd = open (filename, O_RDWR | O_CLOEXEC)) < 0)
		return -errno;

	if (ioctl (fd, SND_AFM_IOCTL_PVERSION, &ver) < 0)
	{
		close (fd);
		return -errno;
	}
	if (SND_PROTOCOL_INCOMPATIBLE (ver, SND_AFM_VERSION))
	{
		close (fd);
		return -SND_ERROR_INCOMPATIBLE_VERSION;
	}
	afm = (snd_afm_t *) calloc (1, sizeof (snd_afm_t));
	if (afm == NULL)
	{
		close (fd);
		return -ENOMEM;
	}
	afm->fd = fd;
	*handle = afm;
	return EOK;
}

/* Open an Acoustic Function Module (AFM) by card and device number */
int
snd_afm_open (snd_afm_t **handle, int card, int device)
{
	char    filename[32];

	*handle = NULL;

	if (card < 0 || card >= INT_MAX)
		return -EINVAL;
	if (device < 0 || device >= INT_MAX)
		return -EINVAL;

	sprintf (filename, SND_FILE_AFM_SERVICE, card, device);
	return (snd_afm_open_name (handle, filename));
}

/* Close handle to an Acoustic Function Module (AFM) */
int
snd_afm_close (snd_afm_t *handle)
{
	int status = EOK;

	if (!handle)
		return -EINVAL;

	status = close (handle->fd) < 0 ? -errno : EOK;
	free (handle);
	return status;
}

/* Get the file descriptor associated with the Acoustic Function Module (AFM) handle */
int
snd_afm_file_descriptor (snd_afm_t *handle)
{
	if (!handle)
		return -EINVAL;
	return handle->fd;
}

/* Start Acoustic Function Module (AFM) processing */
int
snd_afm_service_start (snd_afm_t *handle)
{
	if (!handle)
		return -EINVAL;

	if (ioctl (handle->fd, SND_AFM_IOCTL_START) < 0)
		return -errno;

	return EOK;
}

/* Stop Acoustic Function Module (AFM) processing */
int
snd_afm_service_stop (snd_afm_t *handle)
{
	if (!handle)
		return -EINVAL;

	if (ioctl (handle->fd, SND_AFM_IOCTL_STOP) < 0)
		return -errno;

	return EOK;
}

/* Set Acoustic Function Module (AFM) file path for the defined type
 * See afm_path_t defintion for supporetd types
 * Note: This call is not support by all AFMs
 */
int
snd_afm_service_set_path (snd_afm_t *handle, int type, const char *path)
{
	int status = EOK;
	iov_t iov[2];

	if (!handle || !path)
		return -EINVAL;

	SETIOV(&iov[0], &type, sizeof(int));
	SETIOV(&iov[1], path, strlen(path)+1);

	if ((status = devctlv(handle->fd, SND_AFM_DCMD_SET_PATH, 2, 0, iov, NULL, NULL)) != EOK)
		return -status;

	return EOK;
}

/* Set the audio mode used to tune acoustic processing parameters of the AFM */
int
snd_afm_service_set_audio_mode (snd_afm_t *handle, const char *mode)
{
	int status = EOK;

	if (!handle || !mode)
		return -EINVAL;

	if ((status = devctl(handle->fd, SND_AFM_DCMD_SET_AUDIO_MODE, (char*)mode, strlen(mode)+1, NULL)) != EOK)
		return -status;

	return EOK;
}

/* Get the currently applied audio mode of the AFM */
int
snd_afm_service_get_audio_mode (snd_afm_t *handle, char *mode, int size)
{
	int status = EOK;

	if (!handle || !mode)
		return -EINVAL;

	if ((status = devctl(handle->fd, SND_AFM_DCMD_GET_AUDIO_MODE, mode, size, NULL)) != EOK)
		return -status;

	return EOK;
}

/* Get the size of an AFM's vehicle info list */
int
snd_afm_service_get_vin_list_count(snd_afm_t *handle, int *num)
{
    int status = EOK;

    if (!handle || !num)
        return -EINVAL;

    if ((status = devctl(handle->fd, SND_AFM_DCMD_GET_VIN_LIST_COUNT, num, sizeof(int), NULL)) != EOK)
        return -status;

    return EOK;
}

/* Get an AFM's vehicle info list, this can contain parameters such as RPM and Speed.  */
int
snd_afm_service_get_vin_list(snd_afm_t *handle, afm_vin_list_item_t *list, int num)
{
    int status = EOK;

    if (!handle || !list)
        return -EINVAL;

    if ((status = devctl(handle->fd, SND_AFM_DCMD_GET_VIN_LIST, list, num*sizeof(afm_vin_list_item_t), NULL)) != EOK)
        return -status;

    return EOK;
}

/* Set an AFM's vehicle info stream  */
int
snd_afm_service_set_vin_stream(snd_afm_t *handle, afm_vin_pair_t *stream, int num)
{
    int status = EOK;

    if (!handle || !stream)
        return -EINVAL;

    if ((status = devctl(handle->fd, SND_AFM_DCMD_SET_VIN_STREAM, stream, num*sizeof(afm_vin_pair_t), NULL)) != EOK)
        return -status;

    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/afm_service/afm_service.c $ $Rev: 805218 $")
#endif
