/*
 * $QNXLicenseC:
 * Copyright 2007, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable license fees to QNX
 * Software Systems before you may reproduce, modify or distribute this software,
 * or any work that includes all or part of this software.   Free development
 * licenses are available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review this entire
 * file for other proprietary rights or license notices, as well as the QNX
 * Development Suite License Guide at http://licensing.qnx.com/license-guide/
 * for other information.
 * $
 */

#include <sys/asound_common.h>
#include <alsa/asoundlib.h>
#include "pcm_internal.h"
#include <sys/poll.h>
#include <pthread.h>
#include <sys/slog2.h>
#ifdef __QNX__
#include <sys/neutrino.h>
#include <sys/iomgr.h>
#endif

#ifdef __QNX__
enum PulseType {
    SHUTDOWN = _PULSE_CODE_MINAVAIL,
    DATA
};
#ifndef SIGEV_PULSE_PTR_INIT
#define SIGEV_PULSE_PTR_INIT SIGEV_PULSE_INIT
#define MsgSendPulsePtr( coid, prio, code, ptr ) MsgSendPulse( coid, prio, code, (int)(ptr) )
#endif
#endif

struct _snd_async_handler
{
    void *private_data;
    snd_async_callback_t callback;
    snd_pcm_t *pcm;
#ifdef __QNX__
    struct sigevent event;
#endif
    struct _snd_async_handler *next;
};

static struct _snd_async_handler *handlers = NULL;
static pthread_t thread;
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
#ifdef __QNX__
static int chid;
static int coid;
#endif

static void *async_handler(void *dummy)
{
#ifdef __QNX__
    struct _pulse pulse;
    while( 1 ) {
        if( MsgReceivePulse(chid, &pulse, sizeof(pulse), NULL) == -1 ) {
            slog2fa(NULL, 0, SLOG2_WARNING, "Error receiving pulse: %d\n", SLOG2_FA_SIGNED(errno), SLOG2_FA_END);
        }
        switch( pulse.code ) {
            case SHUTDOWN:
                return NULL;
            case DATA:
                {
                    int count;
                    struct _snd_async_handler *iter;
                    pthread_mutex_lock( &mutex );
                    struct pollfd fds;

                    // Get the associated handler
                    struct _snd_async_handler *handler;
                    handler = (struct _snd_async_handler *)pulse.value.sival_ptr;

                    // Now that we have the lock, make sure the pulse is still valid
                    for( iter = handlers; iter != NULL && iter != handler; iter = iter->next );
                    if( iter == NULL ) break;

                    // pulse is valid, run the callback
                    snd_pcm_avail( handler->pcm );
                    handler->callback( handler );

                    // TODO - This won't work with count > 1 because we only
                    // wish to re-arm the triggered fd, not all fd's, but
                    // fortunately count is always 1 currently
                    count = snd_pcm_poll_descriptors_count( handler->pcm );
                    if( count > 1 ) {
                        slog2fa(NULL, 0, SLOG2_WARNING, "fd count too large for async callback: %d\n", SLOG2_FA_SIGNED(count), SLOG2_FA_END);
                        count = 1;
                    }

                    snd_pcm_poll_descriptors( handler->pcm, &fds, count );

                    if( ionotify( fds.fd, _NOTIFY_ACTION_POLLARM, snd_pcm_stream( handler->pcm ) == SND_PCM_STREAM_CAPTURE ? _NOTIFY_COND_INPUT : _NOTIFY_COND_OUTPUT, &handler->event ) > 0 ) {
                        MsgSendPulsePtr( coid, -1, handler->event.sigev_code, handler );
                    }
                    pthread_mutex_unlock( &mutex );
                }
                break;
        }
    }
#else
    while( 1 ) {
        struct _snd_async_handler *iter;
        struct pollfd *fds;
        int i, count = 0, nextfds;
        for( iter = handlers; iter != NULL; iter = iter->next ) {
            count += snd_pcm_poll_descriptors_count( iter->pcm );
        }
        do {
            fds = malloc(sizeof(struct pollfd) * count);
            if( fds == NULL ) {
                usleep(1000);
            }
        } while( fds == NULL );
        poll(fds, count, -1);
        for( i = 0; i < count; i ++ ) {
            if( fds[i].revents & (POLLIN | POLLOUT) ) {
                for( count = 0, iter = handlers; iter != NULL && count + (nextfds = snd_pcm_poll_descriptors_count( iter->pcm )) <= i; iter = iter->next ) {
                    count += snd_pcm_poll_descriptors_count( iter->pcm );
                    iter->callback( iter );
                    break;
                }
            }
        }
        free( fds );
    }
#endif

    return NULL;
}

// Not implemented, as we do not generate EIO, and I don't believe signals are a good method to support this
// anyway. Might want to add an extension to allow for qnx pulses, though.
int snd_async_add_handler(snd_async_handler_t **handler, int fd, snd_async_callback_t callback, void *private_data)
{
    return -ENOTSUP;
}

int snd_async_del_handler(snd_async_handler_t *handler)
{
    void *dummy;
    int ret = -EINVAL;
    struct _snd_async_handler *iter, *next;

    pthread_mutex_lock( &mutex );
    if( handlers != NULL ) {
        if( handlers == handler ) {
            next = handlers->next;
            free( handlers );
            handlers = next;
            if( next == NULL ) {
                // Last async handler, stop the thread
#ifdef __QNX__
                MsgSendPulsePtr(coid, -1, SHUTDOWN, NULL);
#else
                // Not really safe, but good enough for testing
                pthread_cancel(thread);
#endif
                pthread_join(thread, &dummy);
#ifdef __QNX__
                ConnectDetach( coid );
                ChannelDestroy( chid );
#endif
            }
            ret = EOK;
        } else {
            for( iter = handlers; iter->next != NULL; iter = iter->next ) {
                if( iter->next == handler ) {
                    next = iter->next->next;
                    free( iter->next );
                    iter->next = next;
                    ret = EOK;
                    break;
                }
            }
        }
    }
    pthread_mutex_unlock( &mutex );
    return ret;
}

// Not implemented, because snd_async_add_handler is not implemented
int snd_async_handler_get_fd(snd_async_handler_t *handler)
{
    return -ENOTSUP;
}

// Not implemented, because snd_async_add_handler is not implemented
int snd_async_handler_get_signo(snd_async_handler_t *handler)
{
    return ENOTSUP;
}

void *snd_async_handler_get_callback_private(snd_async_handler_t *handler)
{
    return handler->private_data;
}

int snd_async_add_pcm_handler(snd_async_handler_t **handler, snd_pcm_t *pcm, snd_async_callback_t callback, void *private_data)
{
    int count;
    int ret = EOK;
    struct pollfd fds;

    *handler = malloc(sizeof(struct _snd_async_handler));

    if( *handler == NULL ) {
        return ENOMEM;
    }

    (*handler)->pcm = pcm;
    (*handler)->callback = callback;
    (*handler)->private_data = private_data;
    (*handler)->next = handlers;

    // First async handler, start the thread
    pthread_mutex_lock( &mutex );
    if( handlers == NULL ) {
#ifdef __QNX__
        if( (chid = ChannelCreate(_NTO_CHF_DISCONNECT | _NTO_CHF_UNBLOCK)) == -1 ) {
            ret = errno;
            goto err0;
        }

        if( (coid = ConnectAttach(0, 0, chid, _NTO_SIDE_CHANNEL, 0)) == -1) {
            ret = errno;
            goto err1;
        }
#endif

        if( (ret = pthread_create(&thread, NULL, async_handler, NULL)) != EOK ) {
            goto err2;
        }
    }

    // TODO - This won't work with count > 1 because we only
    // wish to re-arm the triggered fd, not all fd's, but
    // fortunately count is always 1 currently
    count = snd_pcm_poll_descriptors_count( (*handler)->pcm );
    if( count > 1 ) {
        slog2fa(NULL, 0, SLOG2_WARNING, "fd count too large for async callback: %d\n", SLOG2_FA_SIGNED(count), SLOG2_FA_END);
        count = 1;
    }

    snd_pcm_poll_descriptors( (*handler)->pcm, &fds, count );

#ifdef __QNX__
    SIGEV_PULSE_PTR_INIT(&(*handler)->event, coid, -1, DATA, *handler);
    if( ionotify( fds.fd, _NOTIFY_ACTION_POLLARM, snd_pcm_stream( (*handler)->pcm ) == SND_PCM_STREAM_CAPTURE ? _NOTIFY_COND_INPUT : _NOTIFY_COND_OUTPUT, &(*handler)->event ) > 0 ) {
        MsgSendPulsePtr( coid, -1, (*handler)->event.sigev_code, *handler );
    }
#endif

    handlers = *handler;
    pthread_mutex_unlock( &mutex );

    return ret;
err2:
#ifdef __QNX__
    ConnectDetach( coid );
err1:
    ChannelDestroy( chid );
err0:
#endif
    free( *handler );
    *handler = NULL;
    return ret;
}

snd_pcm_t *snd_async_handler_get_pcm(snd_async_handler_t *handler)
{
    return handler->pcm;
}

int snd_async_add_timer_handler(snd_async_handler_t **handler, snd_timer_t *timer, snd_async_callback_t callback, void *private_data)
{
    UNIMPLEMENTED;
    return 0;
}

snd_timer_t *snd_async_handler_get_timer(snd_async_handler_t *handler)
{
    UNIMPLEMENTED;
    return NULL;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/async.c $ $Rev: 781520 $")
#endif
