/*
 * $QNXLicenseC:
 * Copyright 2007, QNX Software Systems. All Rights Reserved.
 * 
 * You must obtain a written license from and pay applicable license fees to QNX 
 * Software Systems before you may reproduce, modify or distribute this software, 
 * or any work that includes all or part of this software.   Free development 
 * licenses are available for evaluation and non-commercial purposes.  For more 
 * information visit http://licensing.qnx.com or email licensing@qnx.com.
 *  
 * This file may contain contributions from others.  Please review this entire 
 * file for other proprietary rights or license notices, as well as the QNX 
 * Development Suite License Guide at http://licensing.qnx.com/license-guide/ 
 * for other information.
 * $
 */

#include <sys/time.h>
#include <sys/asound_common.h>
#include <alsa/asoundlib.h>
#include "pcm_internal.h"

static const char *map_long_names[] = {
    "Unknown",
    "N/A, silent",
    "mono stream",
    "front left",
    "front right",
    "rear left",
    "rear right",
    "front center",
    "LFE",
    "side left",
    "side right",
    "rear center",
    "front left center",
    "front right center",
    "rear left center",
    "rear right center",
    "front left wide",
    "front right wide",
    "front left high",
    "front center high",
    "front right high",
    "top center",
    "top front left",
    "top front right",
    "top front center",
    "top rear left",
    "top rear right",
    "top rear center",
    "top front left center",
    "top front right center",
    "top side left",
    "top side right",
    "left LFE",
    "right LFE",
    "bottom center",
    "bottom left center",
    "bottom right center"
};

static const char *map_long_names_bb[] = {
    "side left upper",
    "side left lower",
    "side right upper",
    "side right lower",
    "bottom front center",
    "rear center bottom",
    "rear center top",
    "rear right upper center",
    "rear left upper center"
};

static const char *map_names[] = {
	"SND_CHMAP_UNKNOWN",
	"SND_CHMAP_NA",
	"SND_CHMAP_MONO",
	"SND_CHMAP_FL",
	"SND_CHMAP_FR",
	"SND_CHMAP_RL",
	"SND_CHMAP_RR",
	"SND_CHMAP_FC",
	"SND_CHMAP_LFE",
	"SND_CHMAP_SL",
	"SND_CHMAP_SR",
	"SND_CHMAP_RC",
	"SND_CHMAP_FLC",
	"SND_CHMAP_FRC",
	"SND_CHMAP_RLC",
	"SND_CHMAP_RRC",
	"SND_CHMAP_FLW",
	"SND_CHMAP_FRW",
	"SND_CHMAP_FLH",
	"SND_CHMAP_FCH",
	"SND_CHMAP_FRH",
	"SND_CHMAP_TC",
	"SND_CHMAP_TFL",
	"SND_CHMAP_TFR",
	"SND_CHMAP_TFC",
	"SND_CHMAP_TRL",
	"SND_CHMAP_TRR",
	"SND_CHMAP_TRC",
	"SND_CHMAP_TFLC",
	"SND_CHMAP_TFRC",
	"SND_CHMAP_TSL",
	"SND_CHMAP_TSR",
	"SND_CHMAP_LLFE",
	"SND_CHMAP_RLFE",
	"SND_CHMAP_BC",
	"SND_CHMAP_BLC",
	"SND_CHMAP_BRC"
};

static const char *map_names_bb[] = {
    "SND_CHMAP_BB_SLU",
    "SND_CHMAP_BB_SLL",
    "SND_CHMAP_BB_SRU",
    "SND_CHMAP_BB_SRL",
    "SND_CHMAP_BB_BFC",
    "SND_CHMAP_BB_RCB",
    "SND_CHMAP_BB_RCT",
    "SND_CHMAP_BB_RRUC",
    "SND_CHMAP_BB_RLUC"
};

snd_pcm_chmap_query_t **snd_pcm_query_chmaps(snd_pcm_t *pcm)
{
    return pcm->callbacks->query_chmaps(pcm);
}

snd_pcm_chmap_query_t **snd_pcm_query_chmaps_from_hw(int card, int dev,
                             int subdev,
                             snd_pcm_stream_t stream)
{
    UNIMPLEMENTED;
    return NULL;
}

void snd_pcm_free_chmaps(snd_pcm_chmap_query_t **maps)
{
    int i;
    for( i = 0; maps[i]; i ++ ) {
        free( maps[ i ] );
    }
    free( maps );
}

snd_pcm_chmap_t *snd_pcm_get_chmap(snd_pcm_t *pcm)
{
    return pcm->callbacks->get_chmap(pcm);
}

int snd_pcm_set_chmap(snd_pcm_t *pcm, const snd_pcm_chmap_t *map)
{
    return pcm->callbacks->set_chmap(pcm, map);
}

const char *snd_pcm_chmap_type_name(enum snd_pcm_chmap_type val)
{
    switch( val ) {
        default:
        case SND_CHMAP_TYPE_NONE:
            return "Unspecified channel position";
        case SND_CHMAP_TYPE_FIXED:
            return "Fixed channel position";
        case SND_CHMAP_TYPE_VAR:
            return "Freely swappable channel position";
        case SND_CHMAP_TYPE_PAIRED:
            return "Pair-wise swappable channel position";
    }
}

const char *snd_pcm_chmap_name(enum snd_pcm_chmap_position val)
{
    if( val < sizeof(map_names)/sizeof(map_names[0]) ) {
        return map_names[val];
    } else if( val >= SND_CHMAP_BB_SLU && val <= SND_CHMAP_BB_LAST ) {
        return map_names_bb[val-SND_CHMAP_BB_SLU];
    } else {
        return NULL;
    }
}

const char *snd_pcm_chmap_long_name(enum snd_pcm_chmap_position val)
{
    if( val < sizeof(map_long_names)/sizeof(map_long_names[0]) ) {
        return map_long_names[val];
    } else if( val >= SND_CHMAP_BB_SLU && val <= SND_CHMAP_BB_LAST ) {
        return map_long_names_bb[val-SND_CHMAP_BB_SLU];
    } else {
        return NULL;
    }
}

int snd_pcm_chmap_print(const snd_pcm_chmap_t *map, size_t maxlen, char *buf)
{
    int i;
    int count;
    int length;
    const char *name;

    count = snprintf(buf, maxlen, "%d: ", map->channels);
    if( count < 0 ) return count;

    for( i = 0; i < map->channels; i ++ ) {
        if( map->pos[i] < SND_CHMAP_LAST ) {
            name = map_long_names[map->pos[i]];
        } else if ( map->pos[i] >= SND_CHMAP_BB_SLU && map->pos[i] <= SND_CHMAP_BB_LAST ) {
            name = map_long_names_bb[map->pos[i] - SND_CHMAP_BB_SLU];
        } else {
            name = map_long_names[0];
        }
        length = strlen(name);
        if( maxlen - count >= length + 1 + ((i > 0) ? 2 : 0)) {
            if( i > 0 ) {
                strcpy(buf+count, ", ");
                count += 2;
            }
            strcpy(buf+count, name);
            count += length;
        } else {
            return -ENOMEM;
        }
    }

    return count;
}

unsigned int snd_pcm_chmap_from_string(const char *str)
{
    int i;
    for( i = 0; i < sizeof(map_names)/sizeof(map_names[0]); i ++ ) {
        if( !strcmp(str, map_names[i]) ) {
            return i;
        }
    }
    for( i = 0; i < sizeof(map_names_bb)/sizeof(map_names_bb[0]); i ++ ) {
        if( !strcmp(str, map_names_bb[i]) ) {
            return i + SND_CHMAP_BB_SLU;
        }
    }
    for( i = 0; i < sizeof(map_long_names)/sizeof(map_long_names[0]); i ++ ) {
        if( !strcmp(str, map_long_names[i]) ) {
            return i;
        }
    }
    for( i = 0; i < sizeof(map_long_names_bb)/sizeof(map_long_names_bb[0]); i ++ ) {
        if( !strcmp(str, map_long_names_bb[i]) ) {
            return i + SND_CHMAP_BB_SLU;
        }
    }
    return SND_CHMAP_UNKNOWN;
}

snd_pcm_chmap_t *snd_pcm_chmap_parse_string(const char *str)
{
    UNIMPLEMENTED;
    return 0;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/chmap.c $ $Rev: 781520 $")
#endif
