#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <inttypes.h>
#include <alsa/asoundlib.h>
#include "pcm_internal.h"

typedef struct _snd_config_iterator {
    snd_config_t *data;
} _snd_config_iterator;

typedef struct _snd_config_compound_data {
    int child_count;
    _snd_config_iterator *child_iterators;
} _snd_config_compound_data;

typedef struct _snd_config {
    snd_config_type_t type;
    char *id;
    union {
        char *string;
        const void *pointer;
        struct _snd_config_compound_data compound;
    } data;
} _snd_config;

int snd_config_top(snd_config_t **config)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_load(snd_config_t *config, snd_input_t *in)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_load_override(snd_config_t *config, snd_input_t *in)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_save(snd_config_t *config, snd_output_t *out)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_update(void)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_update_r(snd_config_t **top, snd_config_update_t **update, const char *path)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_update_free(snd_config_update_t *update)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_update_free_global(void)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_search(snd_config_t *config, const char *key,
		      snd_config_t **result)
{
    int i;

    if( config->type != SND_CONFIG_TYPE_COMPOUND ) {
        return EINVAL;
    }

    for( i = 0; i < config->data.compound.child_count; i ++ ) {
        if( !strcmp(config->data.compound.child_iterators[i].data->id, key) ) {
            *result = config->data.compound.child_iterators[i].data;
            return EOK;
        }
    }
    return -ENOENT;
}

int snd_config_searchv(snd_config_t *config,
		       snd_config_t **result, ...)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_search_definition(snd_config_t *config,
				 const char *base, const char *key,
				 snd_config_t **result)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_expand(snd_config_t *config, snd_config_t *root,
		      const char *args, snd_config_t *private_data,
		      snd_config_t **result)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_evaluate(snd_config_t *config, snd_config_t *root,
			snd_config_t *private_data, snd_config_t **result)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_add(snd_config_t *config, snd_config_t *leaf)
{
    _snd_config_iterator *new_iterators;
    if( config->type != SND_CONFIG_TYPE_COMPOUND ) {
        return -EINVAL;
    }

    new_iterators = realloc(config->data.compound.child_iterators, sizeof(_snd_config_iterator) * (config->data.compound.child_count + 1));
    if( new_iterators == NULL ) {
        free( new_iterators );
        return -ENOMEM;
    }

    config->data.compound.child_iterators = new_iterators;
    config->data.compound.child_iterators[ config->data.compound.child_count ].data
        = leaf;
    config->data.compound.child_count ++;

    return EOK;
}

int snd_config_delete(snd_config_t *config)
{
    int i;

    switch( config->type ) {
        case SND_CONFIG_TYPE_COMPOUND:
            for( i = 0; i < config->data.compound.child_count; i ++ ) {
                snd_config_delete( config->data.compound.child_iterators[ i ].data );
            }
            free( config->data.compound.child_iterators );
            break;
        case SND_CONFIG_TYPE_STRING:
        case SND_CONFIG_TYPE_INTEGER:
        case SND_CONFIG_TYPE_INTEGER64:
        case SND_CONFIG_TYPE_REAL:
            free( config->data.string );
            break;
        case SND_CONFIG_TYPE_POINTER:
            break;
    }


    // Free the config itself
    free( config->id );
    free( config );

    return EOK;
}

int snd_config_delete_compound_members(const snd_config_t *config)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_copy(snd_config_t **dst, snd_config_t *src)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_make(snd_config_t **config, const char *key,
		    snd_config_type_t type)
{
    *config = calloc(1, sizeof(snd_config_t));
    if( !*config ) {
        return -ENOMEM;
    }

    (*config)->type = type;
    (*config)->id = strdup( key );

    if( (*config)->id == NULL ) {
        free( *config );
        *config = NULL;
        return -ENOMEM;
    }

    return EOK;
}

int snd_config_make_integer(snd_config_t **config, const char *key)
{
    int ret;
    ret = snd_config_make( config, key, SND_CONFIG_TYPE_INTEGER );
    if( ret == EOK ) {
        (*config)->data.string = NULL;
    }
    return ret;
}

int snd_config_make_integer64(snd_config_t **config, const char *key)
{
    int ret;

    ret = snd_config_make( config, key, SND_CONFIG_TYPE_INTEGER64 );
    if( ret == EOK ) {
        (*config)->data.string = NULL;
    }
    return ret;
}

int snd_config_make_real(snd_config_t **config, const char *key)
{
    int ret;

    ret = snd_config_make( config, key, SND_CONFIG_TYPE_REAL );
    if( ret == EOK ) {
        (*config)->data.string = NULL;
    }
    return ret;
}

int snd_config_make_string(snd_config_t **config, const char *key)
{
    int ret;

    ret = snd_config_make( config, key, SND_CONFIG_TYPE_STRING );
    if( ret == EOK ) {
        (*config)->data.string = NULL;
    }
    return ret;
}

int snd_config_make_pointer(snd_config_t **config, const char *key)
{
    int ret;

    ret = snd_config_make( config, key, SND_CONFIG_TYPE_POINTER );
    if( ret == EOK ) {
        (*config)->data.pointer = NULL;
    }
    return ret;
}

int snd_config_make_compound(snd_config_t **config, const char *key, int join)
{
    *config = calloc(1, sizeof(snd_config_t));
    if( !*config ) {
        return -ENOMEM;
    }

    (*config)->type = SND_CONFIG_TYPE_COMPOUND;
    (*config)->data.compound.child_count = 0;
    (*config)->data.compound.child_iterators = NULL;

    return EOK;
}

int snd_config_imake_integer(snd_config_t **config, const char *key, const long value)
{
    int ret;
    int len;
    ret = snd_config_make( config, key, SND_CONFIG_TYPE_INTEGER );
    if( ret == EOK ) {
        len = snprintf(NULL, 0, "%ld", value);
        (*config)->data.string = malloc(len);
        if( (*config)->data.string == NULL ) {
            snd_config_delete(*config);
            return -ENOMEM;
        }
        sprintf((*config)->data.string, "%ld", value);
    }
    return ret;
}

int snd_config_imake_integer64(snd_config_t **config, const char *key, const long long value)
{
    int ret;
    int len;
    ret = snd_config_make( config, key, SND_CONFIG_TYPE_INTEGER64 );
    if( ret == EOK ) {
        len = snprintf(NULL, 0, "%lld", value);
        (*config)->data.string = malloc(len);
        if( (*config)->data.string == NULL ) {
            snd_config_delete(*config);
            return -ENOMEM;
        }
        sprintf((*config)->data.string, "%lld", value);
    }
    return ret;
}

int snd_config_imake_real(snd_config_t **config, const char *key, const double value)
{
    int ret;
    int len;
    ret = snd_config_make( config, key, SND_CONFIG_TYPE_REAL );
    if( ret == EOK ) {
        len = snprintf(NULL, 0, "%g", value);
        (*config)->data.string = malloc(len);
        if( (*config)->data.string == NULL ) {
            snd_config_delete(*config);
            return -ENOMEM;
        }
        sprintf((*config)->data.string, "%g", value);
    }
    return ret;
}

int snd_config_imake_string(snd_config_t **config, const char *key, const char *ascii)
{
    int ret;

    ret = snd_config_make( config, key, SND_CONFIG_TYPE_STRING );
    if( ret == EOK ) {
        (*config)->data.string = strdup( ascii );

        if( (*config)->data.string == NULL ) {
            snd_config_delete( *config );
            return -ENOMEM;
        }
    }
    return ret;
}

int snd_config_imake_pointer(snd_config_t **config, const char *key, const void *ptr)
{
    int ret;

    ret = snd_config_make( config, key, SND_CONFIG_TYPE_POINTER );
    if( ret == EOK ) {
        (*config)->data.pointer = ptr;
    }

    return ret;
}

snd_config_type_t snd_config_get_type(const snd_config_t *config)
{
    return config->type;
}

int snd_config_set_id(snd_config_t *config, const char *id)
{
    char *new_id = strdup( id );
    if( new_id == NULL ) {
        return -ENOMEM;
    }

    free( config->id );
    config->id = new_id;

    return EOK;
}

int snd_config_set_integer(snd_config_t *config, long value)
{
    int len;

    if( config->type != SND_CONFIG_TYPE_INTEGER ) {
        return -EINVAL;
    }

    len = snprintf(NULL, 0, "%ld", value);
    config->data.string = malloc(len);
    if( config->data.string == NULL ) {
        return -ENOMEM;
    }
    sprintf(config->data.string, "%ld", value);
    return EOK;
}

int snd_config_set_integer64(snd_config_t *config, long long value)
{
    int len;

    if( config->type != SND_CONFIG_TYPE_INTEGER64 ) {
        return -EINVAL;
    }

    len = snprintf(NULL, 0, "%lld", value);
    config->data.string = malloc(len);
    if( config->data.string == NULL ) {
        return -ENOMEM;
    }
    sprintf(config->data.string, "%lld", value);
    return EOK;
}

int snd_config_set_real(snd_config_t *config, double value)
{
    int len;

    if( config->type != SND_CONFIG_TYPE_REAL ) {
        return -EINVAL;
    }

    len = snprintf(NULL, 0, "%g", value);
    config->data.string = malloc(len);
    if( config->data.string == NULL ) {
        return -ENOMEM;
    }
    sprintf(config->data.string, "%g", value);
    return EOK;
}

int snd_config_set_string(snd_config_t *config, const char *value)
{
    char *newstr;
    if( config->type != SND_CONFIG_TYPE_STRING ) {
        return -EINVAL;
    }

    newstr = strdup( value );
    if( newstr == NULL ) {
        return -ENOMEM;
    }

    config->data.string = newstr;
    return EOK;
}

int snd_config_set_ascii(snd_config_t *config, const char *ascii)
{
    char *newstr;
    if( config->type != SND_CONFIG_TYPE_STRING ) {
        return -EINVAL;
    }

    newstr = strdup( ascii );
    if( newstr == NULL ) {
        return -ENOMEM;
    }

    config->data.string = newstr;
    return EOK;
}

int snd_config_set_pointer(snd_config_t *config, const void *ptr)
{
    if( config->type != SND_CONFIG_TYPE_POINTER ) {
        return -EINVAL;
    }

    config->data.pointer = ptr;
    return EOK;
}

int snd_config_get_id(const snd_config_t *config, const char **value)
{
    *value = config->id;

    if( config->id == NULL ) {
        return -EINVAL;
    }

    return EOK;
}

int snd_config_get_integer(const snd_config_t *config, long *value)
{
    if( config->type == SND_CONFIG_TYPE_POINTER ) {
        return -EINVAL;
    }

    *value = atoi(config->data.string);
    return EOK;
}

int snd_config_get_integer64(const snd_config_t *config, long long *value)
{
    if( config->type == SND_CONFIG_TYPE_POINTER ) {
        return -EINVAL;
    }

    *value = strtoll(config->data.string, NULL, 10);
    return EOK;
}

int snd_config_get_real(const snd_config_t *config, double *value)
{
    if( config->type == SND_CONFIG_TYPE_POINTER ) {
        return -EINVAL;
    }

    *value = strtod(config->data.string, NULL);
    return EOK;
}

int snd_config_get_ireal(const snd_config_t *config, double *value)
{
    if( config->type == SND_CONFIG_TYPE_POINTER ) {
        return -EINVAL;
    }

    *value = strtod(config->data.string, NULL);

    return EOK;
}

int snd_config_get_string(const snd_config_t *config, const char **value)
{
    if( config->type == SND_CONFIG_TYPE_STRING ) {
        *value = config->data.string;
        return EOK;
    }

    return -EINVAL;
}

int snd_config_get_ascii(const snd_config_t *config, char **value)
{
    if( config->type == SND_CONFIG_TYPE_STRING ) {
        *value = config->data.string;
        return EOK;
    }

    return -EINVAL;
}

int snd_config_get_pointer(const snd_config_t *config, const void **value)
{
    if( config->type == SND_CONFIG_TYPE_POINTER ) {
        *value = config->data.pointer;
        return EOK;
    }

    return -EINVAL;
}

int snd_config_test_id(const snd_config_t *config, const char *id)
{
    return strcmp( config->id, id );
}

snd_config_iterator_t snd_config_iterator_first(const snd_config_t *node)
{
    if( node->type != SND_CONFIG_TYPE_COMPOUND ) {
        return NULL;
    }
    return &node->data.compound.child_iterators[0];
}

snd_config_iterator_t snd_config_iterator_next(const snd_config_iterator_t iterator)
{
    return iterator + 1;
}

snd_config_iterator_t snd_config_iterator_end(const snd_config_t *node)
{
    if( node->type != SND_CONFIG_TYPE_COMPOUND ) {
        return NULL;
    }
    return &node->data.compound.child_iterators[node->data.compound.child_count];
}

snd_config_t *snd_config_iterator_entry(const snd_config_iterator_t iterator)
{
    return iterator->data;
}

int snd_config_get_bool_ascii(const char *ascii)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_get_bool(const snd_config_t *conf)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_get_ctl_iface_ascii(const char *ascii)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_config_get_ctl_iface(const snd_config_t *conf)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_names_list(const char *iface, snd_devname_t **list)
{
    UNIMPLEMENTED;
    return 0;
}

void snd_names_list_free(snd_devname_t *list)
{
    UNIMPLEMENTED;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/config.c $ $Rev: 781520 $")
#endif
