#include <errno.h>
#include <stdlib.h>
#include <stdbool.h>
#include "pcm_internal.h"
#include <alsa/pcm_extplug.h>

#define IS_POWER_OF_TWO(x) (!((x) & ((x) - 1)))

typedef struct extplug_data
{
    int format_mask;
    int slave_format_mask;
    int channel_mask;
    int slave_channel_mask;
    snd_pcm_t *slave;
    snd_pcm_hw_params_t slaveParams;
    snd_pcm_channel_area_t dst;
    uint32_t params_set;
} extplug_data_t;

static int ext_close(snd_pcm_t *pcm)
{
    return pcm->callback_param.extplug->callback->close( pcm->callback_param.extplug );
}

static int ext_poll_descriptors_count(snd_pcm_t *pcm)
{
    extplug_data_t *data = pcm->private_data;

    return snd_pcm_poll_descriptors_count( data->slave );
}

static int ext_poll_descriptors(snd_pcm_t *pcm, struct pollfd *pfds, unsigned int space)
{
    extplug_data_t *data = pcm->private_data;

    return snd_pcm_poll_descriptors( data->slave, pfds, space );
}

static int ext_hw_params(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    int i;
    int ret;
    bool found_match = false;
    extplug_data_t *data = pcm->private_data;
    snd_pcm_extplug_t *ext = pcm->callback_param.extplug;
    const snd_pcm_hw_params_t *parent;
    snd_pcm_access_t access;
    snd_pcm_uframes_t frames;
    unsigned int rate;

    parent = find_params_setting( &data->slaveParams, 1 << PARAM_VOICE_SET );
    if( !parent ) return -EINVAL;
    // See if we can just push the configuration through
    if( !(params->channels & parent->channels) ) {
        // Can't, so find the number of channels in the output that is closest
        // to the number of channels in the input
        for( i = params->voices; i < sizeof(params->channels)*_BITS_BYTE; i ++ ) {
            if( data->slave_channel_mask & (1 << (i - 1)) & parent->channels ) {
                data->slaveParams.channels = 1 << (i - 1);
                data->slaveParams.voices = i;
                found_match = true;
                break;
            }
        }
        if( !found_match ) {
            for( i = params->voices - 1; i >= 0; i -- ) {
                if( data->slave_channel_mask & (1 << (i - 1)) & parent->channels ) {
                    data->slaveParams.channels = 1 << (i - 1);
                    data->slaveParams.voices = i;
                    found_match = true;
                    break;
                }
            }
        }
        if( !found_match ) {
            // Could not match number of channels
            return -EINVAL;
        }
        ext->slave_channels = i;
    } else {
        ext->slave_channels = params->voices;
    }
    if( (ret = snd_pcm_hw_params_set_channels(data->slave, &data->slaveParams, ext->slave_channels)) ) {
        return ret;
    }

    parent = find_params_setting( &data->slaveParams, 1 << PARAM_FORMAT_SET );
    if( !parent ) return -EINVAL;
    // See if we can just push the configuration through
    if( !(params->formats & parent->formats) ) {
        // Can't, so find a format for the output that the slave supports
        for( i = params->format; i < SND_PCM_FORMAT_LAST; i ++ ) {
            if( data->slave_format_mask & (1 << i) & parent->formats ) {
                snd_pcm_hw_params_set_format( data->slave, &data->slaveParams, i );
                found_match = true;
                break;
            }
        }
        if( !found_match ) {
            for( i = params->format - 1; i >= 0; i -- ) {
                if( data->slave_format_mask & (1 << i) & parent->formats ) {
                    snd_pcm_hw_params_set_format( data->slave, &data->slaveParams, i );
                    found_match = true;
                    break;
                }
            }
        }
        if( !found_match ) {
            // Could not match format
            return -EINVAL;
        }
        ext->slave_format = i;
        ext->slave_subformat = SND_PCM_SUBFORMAT_STD;
    } else {
        ext->slave_format = params->format;
        ext->slave_subformat = SND_PCM_SUBFORMAT_STD;
    }
    if( (ret = snd_pcm_hw_params_set_format(data->slave, &data->slaveParams, ext->slave_format)) ) {
        return ret;
    }

    // Copy the remaining parameters
    if( (ret = snd_pcm_hw_params_get_rate( params, &rate, NULL)) ) {
        return ret;
    }
    if( (ret = snd_pcm_hw_params_set_rate(data->slave, &data->slaveParams, rate, 0)) ) {
        return ret;
    }

    if( (ret = snd_pcm_hw_params_get_access( params, &access)) ) {
        return ret;
    }
    if( (ret = snd_pcm_hw_params_set_access(data->slave, &data->slaveParams, access)) ) {
        return ret;
    }

    if( (ret = snd_pcm_hw_params_get_period_size_max( params, &frames, NULL)) ) {
        return ret;
    }

    // Convert the number of frames according to conversions that the explug provides
    frames /= (params->voices * snd_pcm_format_width(params->format)/8);
    if( (ret = snd_pcm_hw_params_set_period_size( data->slave, &data->slaveParams, frames, 0 )) ) {
        return ret;
    }


    if( (ret = snd_pcm_hw_params_get_buffer_size_max( params, &frames)) ) {
        return ret;
    }

    // Convert the number of frames according to conversions that the explug provides
    frames /= (params->voices * snd_pcm_format_width(params->format)/8);
    if( (ret = snd_pcm_hw_params_set_buffer_size(data->slave, &data->slaveParams, frames)) ) {
        return ret;
    }

    ext->format = params->format;
    ext->subformat = SND_PCM_SUBFORMAT_STD;
    ext->channels = params->voices;
    ext->rate = params->rate;

    if( (ret = pcm->callback_param.extplug->callback->hw_params(pcm->callback_param.extplug, &data->slaveParams)) != EOK ) {
        return ret;
    }

    return snd_pcm_hw_params( data->slave, &data->slaveParams );
}

static int ext_sw_params(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    extplug_data_t *data = pcm->private_data;

    return snd_pcm_sw_params(data->slave, params);
}


static int ext_hw_params_any(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    extplug_data_t *data = pcm->private_data;

    params->parent = &data->slaveParams;
    params->params_set = data->params_set;
    params->formats = data->format_mask;
    params->channels = data->channel_mask;
    if( params->params_set & (1 << PARAM_FORMAT_SET) ) {
        update_formats( params );
    }
    if( params->params_set & (1 << PARAM_VOICE_SET) ) {
        update_channels( params );
    }

    return EOK;
}

static int ext_nonblock(snd_pcm_t *pcm, int nonblock)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_nonblock(info->slave, nonblock);
}

static int ext_drain(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_drain( info->slave );
}

static int ext_start(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_start( info->slave );
}

static int ext_prepare(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_prepare( info->slave );
}

static snd_pcm_sframes_t ext_avail(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_avail( info->slave );
}

static snd_pcm_sframes_t ext_avail_update(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_avail_update( info->slave );
}

static int ext_drop(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_drop( info->slave );
}

static int ext_reset(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_reset( info->slave );
}

static int ext_status(snd_pcm_t *pcm, snd_pcm_status_t *status)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_status( info->slave, status );
}

static snd_pcm_chmap_query_t **ext_query_chmaps(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_query_chmaps( info->slave );
}

static snd_pcm_chmap_t *ext_get_chmap(snd_pcm_t *pcm)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_get_chmap( info->slave );
}

static int ext_set_chmap(snd_pcm_t *pcm, const snd_pcm_chmap_t *map)
{
    extplug_data_t *info = pcm->private_data;

    return snd_pcm_set_chmap( info->slave, map );
}

static snd_pcm_sframes_t ext_readi(snd_pcm_t *pcm, void *buffer, snd_pcm_uframes_t size)
{
    int ret;
    snd_pcm_channel_area_t dst;

    extplug_data_t *data = pcm->private_data;

    // Size input buffer to receive the input
    dst.addr = (void *)buffer;
    dst.first = 0;
    dst.step = snd_pcm_format_width( pcm->callback_param.extplug->format );

    if( (ret = snd_pcm_area_size( &data->dst, size, pcm->callback_param.extplug->slave_channels, pcm->callback_param.extplug->slave_format )) ) {
        return ret;
    }

    if( (ret = snd_pcm_readi(data->slave, data->dst.addr, size)) != size ) {
        return ret;
    }

    if( (ret = pcm->callback_param.extplug->callback->transfer(pcm->callback_param.extplug, &dst, 0, &data->dst, 0, size)) != EOK ) {
        return ret;
    }

    return size;
}

static snd_pcm_sframes_t ext_writei(snd_pcm_t *pcm, const void *buffer, snd_pcm_uframes_t size)
{
    int ret;
    snd_pcm_channel_area_t src;

    extplug_data_t *data = pcm->private_data;

    // Size output buffer to receive the output
    src.addr = (void *)buffer;
    src.first = 0;
    src.step = snd_pcm_format_width( pcm->callback_param.extplug->format );

    if( (ret = snd_pcm_area_size( &data->dst, size, pcm->callback_param.extplug->slave_channels, pcm->callback_param.extplug->slave_format )) ) {
        return ret;
    }

    if( (ret = pcm->callback_param.extplug->callback->transfer(pcm->callback_param.extplug, &data->dst, 0, &src, 0, size)) != size ) {
        return ret;
    }

    return snd_pcm_writei(data->slave, data->dst.addr, size);
}

static int ext_set_rate_resample(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val)
{
    extplug_data_t *data = pcm->private_data;

    return snd_pcm_hw_params_set_rate_resample( data->slave, params, val );
}

snd_pcm_plugin_callbacks_t ext_callbacks = {
    .hw_params = ext_hw_params,
    .sw_params = ext_sw_params,
    .hw_params_any = ext_hw_params_any,
    .nonblock = ext_nonblock,
    .drain = ext_drain,
    .drop = ext_drop,
    .reset = ext_reset,
    .start = ext_start,
    .prepare = ext_prepare,
    .avail = ext_avail,
    .avail_update = ext_avail_update,
    .status = ext_status,
    .query_chmaps = ext_query_chmaps,
    .get_chmap = ext_get_chmap,
    .set_chmap = ext_set_chmap,
    .set_rate_resample = ext_set_rate_resample,
    .readi = ext_readi,
    .writei = ext_writei,
    .close = ext_close,
    .poll_descriptors_count = ext_poll_descriptors_count,
    .poll_descriptors = ext_poll_descriptors,
};

int snd_pcm_extplug_create(snd_pcm_extplug_t *ext, const char *name,
			   snd_config_t *root, snd_config_t *slave_conf,
			   snd_pcm_stream_t stream, int mode)
{
    int ret = snd_pcm_allocate_pcm(&ext->pcm, &ext_callbacks);
    if( ret == EOK ) {
        extplug_data_t *data = calloc(1, sizeof(extplug_data_t));
        if( data != NULL ) {
            ret = snd_pcm_open_config(&data->slave, root, slave_conf, stream, mode);
            if( ret == EOK ) {
                ext->stream = stream;
                ext->pcm->private_data = data;
                ext->pcm->callback_param.extplug = ext;
                snd_pcm_hw_params_any( data->slave, &data->slaveParams );
            } else {
                free( data );
            }
        } else {
            snd_pcm_destroy_pcm(ext->pcm);
            ret = -ENOMEM;
        }
    }
    return ret;
}

int snd_pcm_extplug_delete(snd_pcm_extplug_t *ext)
{
    int ret;
    extplug_data_t *data = (extplug_data_t *)ext->pcm->private_data;

    // Close the slave. If it fails, retain the error code, but close the
    // plugin anyway
    ret = snd_pcm_close( data->slave );

    free( data->dst.addr );
    free( data );

    return ret;
}

int snd_pcm_extplug_set_param_list(snd_pcm_extplug_t *extplug, int type, unsigned int num_list, const unsigned int *list)
{
    extplug_data_t *data = (extplug_data_t *)extplug->pcm->private_data;
    int i;

    switch( type ) {
        case SND_PCM_EXTPLUG_HW_FORMAT:
            data->params_set |= (1 << PARAM_FORMAT_SET);
            for( i = 0; i < num_list; i ++ ) {
                if( list[ i ] <= SND_PCM_FORMAT_GSM
                        || (list[ i ] >= SND_PCM_FORMAT_SPECIAL && list[ i ] <= SND_PCM_FORMAT_LAST) ) {
                    data->format_mask |= 1 << list[ i ];
                }
            }
            break;
        case SND_PCM_EXTPLUG_HW_CHANNELS:
            data->params_set |= (1 << PARAM_VOICE_SET);
            for( i = 0; i < num_list; i ++ ) {
                if( list[ i ] <= 8 ) {
                    data->channel_mask |= 1 << (list[ i ] - 1);
                }
            }
            break;
        default:
            return -EINVAL;
    }
    return EOK;
}

int snd_pcm_extplug_set_param_minmax(snd_pcm_extplug_t *extplug, int type, unsigned int min, unsigned int max)
{
    extplug_data_t *data = (extplug_data_t *)extplug->pcm->private_data;
    int i;

    switch( type ) {
        case SND_PCM_EXTPLUG_HW_FORMAT:
            data->params_set |= (1 << PARAM_FORMAT_SET);
            for( i = min; i <= max; i ++ ) {
                data->format_mask |= 1 << i;
            }
            break;
        case SND_PCM_EXTPLUG_HW_CHANNELS:
            data->params_set |= (1 << PARAM_VOICE_SET);
            for( i = min; i <= max; i ++ ) {
                data->channel_mask |= 1 << (i - 1);
            }
            break;
        default:
            return -EINVAL;
    }

    return EOK;
}

int snd_pcm_extplug_set_slave_param_list(snd_pcm_extplug_t *extplug, int type, unsigned int num_list, const unsigned int *list)
{
    extplug_data_t *data = (extplug_data_t *)extplug->pcm->private_data;
    int i;

    switch( type ) {
        case SND_PCM_EXTPLUG_HW_FORMAT:
            for( i = 0; i < num_list; i ++ ) {
                if( list[ i ] <= SND_PCM_FORMAT_GSM
                 || (list[ i ] >= SND_PCM_FORMAT_SPECIAL
                  && list[ i ] <= SND_PCM_FORMAT_LAST) ) {
                    data->slave_format_mask |= 1 << list[ i ];
                }
            }
            break;
        case SND_PCM_EXTPLUG_HW_CHANNELS:
            for( i = 0; i < num_list; i ++ ) {
                if( list[ i ] <= 8 ) {
                    data->slave_channel_mask |= 1 << list[ i ];
                }
            }
            break;
        default:
            return -EINVAL;
    }
    return EOK;
}

int snd_pcm_extplug_set_slave_param_minmax(snd_pcm_extplug_t *extplug, int type, unsigned int min, unsigned int max)
{
    extplug_data_t *data = (extplug_data_t *)extplug->pcm->private_data;
    int i;

    switch( type ) {
        case SND_PCM_EXTPLUG_HW_FORMAT:
            for( i = min; i <= max; i ++ ) {
                data->slave_format_mask |= 1 << i;
            }
            break;
        case SND_PCM_EXTPLUG_HW_CHANNELS:
            for( i = min; i <= max; i ++ ) {
                data->slave_channel_mask |= 1 << i;
            }
            break;
        default:
            return -EINVAL;
    }

    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/extplug.c $ $Rev: 759464 $")
#endif
