#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <stdint.h>
#include <alsa/asoundlib.h>
#include "pcm_internal.h"
#include <alsa/pcm_ioplug.h>

typedef struct ioplug_data
{
    snd_pcm_hw_params_t params;
    snd_pcm_stream_t stream;
    int periodsize;
    int buffersize;
    int rd_frag;
    int wr_frag;
    int frags;
} ioplug_data_t;

static int io_close(snd_pcm_t *pcm)
{
    if( pcm->callback_param.ioplug->callback->close ) {
        return pcm->callback_param.ioplug->callback->close(pcm->callback_param.ioplug);
    } else {
        return EOK;
    }
}

static snd_pcm_sframes_t io_readi(snd_pcm_t *pcm, void *buffer, snd_pcm_uframes_t size)
{
    ioplug_data_t *data = pcm->private_data;

    snd_pcm_channel_area_t areas;
    areas.addr = (void *)buffer;
    areas.first = 0;
    areas.step = 0;

    data->wr_frag += size * snd_pcm_format_width( data->params.format ) * data->params.voices / data->params.period_bytes_max / 8;
    while( data->wr_frag >= data->frags ) {
        data->wr_frag -= data->frags;
    }
    if( pcm->callback_param.ioplug->callback->transfer ) {
        return pcm->callback_param.ioplug->callback->transfer(pcm->callback_param.ioplug, &areas, 0, size);
    } else {
        return 0;
    }
}

static snd_pcm_sframes_t io_writei(snd_pcm_t *pcm, const void *buffer, snd_pcm_uframes_t size)
{
    ioplug_data_t *data = pcm->private_data;

    snd_pcm_channel_area_t areas;
    areas.addr = (void *)buffer;
    areas.first = 0;
    areas.step = 0;

    data->wr_frag += size * snd_pcm_format_width( data->params.format ) * data->params.voices / data->params.period_bytes_max / 8;
    while( data->wr_frag >= data->frags ) {
        data->wr_frag -= data->frags;
    }
    return pcm->callback_param.ioplug->callback->transfer(pcm->callback_param.ioplug, &areas, 0, size);
}

static int io_recover(snd_pcm_t *pcm, int err, int silent)
{
    return pcm->callback_param.ioplug->callback->prepare(pcm->callback_param.ioplug);
}

static int io_drain(snd_pcm_t *pcm)
{
    if( pcm->callback_param.ioplug->callback->drain ) {
        return pcm->callback_param.ioplug->callback->drain(pcm->callback_param.ioplug);
    } else {
        return EOK;
    }
}

static int io_drop(snd_pcm_t *pcm)
{
    return pcm->callback_param.ioplug->callback->stop(pcm->callback_param.ioplug);
}

static int io_start(snd_pcm_t *pcm)
{
    return pcm->callback_param.ioplug->callback->start(pcm->callback_param.ioplug);
}

static int io_reset(snd_pcm_t *pcm)
{
    UNIMPLEMENTED;
    return EOK;
}

static int io_status(snd_pcm_t *pcm, snd_pcm_status_t *status)
{
    UNIMPLEMENTED;
    return EOK;
}

static snd_pcm_chmap_query_t **io_query_chmaps(snd_pcm_t *pcm)
{
    if( pcm->callback_param.ioplug->callback->query_chmaps ) {
        return pcm->callback_param.ioplug->callback->query_chmaps(pcm->callback_param.ioplug);
    } else {
        return NULL;
    }
}

static snd_pcm_chmap_t *io_get_chmap(snd_pcm_t *pcm)
{
    if( pcm->callback_param.ioplug->callback->get_chmap ) {
        return pcm->callback_param.ioplug->callback->get_chmap(pcm->callback_param.ioplug);
    } else {
        return NULL;
    }
}

static int io_set_chmap(snd_pcm_t *pcm, const snd_pcm_chmap_t *map)
{
    if( pcm->callback_param.ioplug->callback->set_chmap ) {
        return pcm->callback_param.ioplug->callback->set_chmap(pcm->callback_param.ioplug, map);
    } else {
        return -EINVAL;
    }
}

static snd_pcm_sframes_t io_avail_update(snd_pcm_t *pcm)
{
    int avail;
    ioplug_data_t *data = pcm->private_data;

    data->rd_frag = pcm->callback_param.ioplug->callback->pointer(pcm->callback_param.ioplug);
    if( data->stream == SND_PCM_STREAM_PLAYBACK ) {
        avail = data->wr_frag >= data->rd_frag
              ? data->frags - (data->wr_frag - data->rd_frag) - 1
              : data->rd_frag - data->wr_frag - 1;
    } else {
        avail = data->rd_frag >= data->wr_frag
              ? data->rd_frag - data->wr_frag
              : data->frags - data->wr_frag + data->rd_frag;
    }
    return avail * data->params.period_bytes_max / (data->params.channels * snd_pcm_format_width( data->params.format ) / 8);
}

static snd_pcm_sframes_t io_avail(snd_pcm_t *pcm)
{
    return io_avail_update(pcm);
}

static int io_prepare(snd_pcm_t *pcm)
{
    if( pcm->callback_param.ioplug->callback->prepare ) {
        return pcm->callback_param.ioplug->callback->prepare(pcm->callback_param.ioplug);
    } else {
        return EOK;
    }
}

static int io_nonblock(snd_pcm_t *pcm, int nonblock)
{
    pcm->callback_param.ioplug->nonblock = nonblock;
    return 0;
}

static int io_resume(snd_pcm_t *pcm)
{
    if( pcm->callback_param.ioplug->callback->resume ) {
        return pcm->callback_param.ioplug->callback->resume(pcm->callback_param.ioplug);
    } else {
        return EOK;
    }
}

static int io_hw_params(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    int ret;
    ioplug_data_t *data = pcm->private_data;

    if( pcm->callback_param.ioplug->callback->hw_params ) {
        ret = pcm->callback_param.ioplug->callback->hw_params(pcm->callback_param.ioplug, params);
    } else {
        ret = EOK;
    }

    data->frags = params->periods_max;

    return ret;
}

static int io_sw_params(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    if( pcm->callback_param.ioplug->callback->sw_params ) {
        return pcm->callback_param.ioplug->callback->sw_params(pcm->callback_param.ioplug, params);
    } else {
        return EOK;
    }
}

static int io_hw_params_any(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    ioplug_data_t *data = pcm->private_data;

    snd_pcm_hw_params_copy( params, &data->params );

    return EOK;
}

static int io_sw_params_any(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    // Nothing to do, hardware doesn't use software parameters
    return EOK;
}

static int io_set_rate_resample(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val)
{
    // Nothing to do, an io-plugin always returns hardware rates
    return EOK;
}

static int io_poll_descriptors_count(snd_pcm_t *pcm)
{
    if( pcm->callback_param.ioplug->callback->poll_descriptors_count ) {
        return pcm->callback_param.ioplug->callback->poll_descriptors_count(pcm->callback_param.ioplug);
    } else {
        return -EINVAL;
    }
}

static int io_poll_descriptors(snd_pcm_t *pcm, struct pollfd *pfds, unsigned int space)
{
    if( pcm->callback_param.ioplug->callback->poll_descriptors ) {
        return pcm->callback_param.ioplug->callback->poll_descriptors(pcm->callback_param.ioplug, pfds, space);
    } else {
        return -EINVAL;
    }
}

snd_pcm_plugin_callbacks_t io_callbacks = {
    .close = io_close,
    .readi = io_readi,
    .writei = io_writei,
    .recover = io_recover,
    .drain = io_drain,
    .drop = io_drop,
    .start = io_start,
    .reset = io_reset,
    .status = io_status,
    .query_chmaps = io_query_chmaps,
    .get_chmap = io_get_chmap,
    .set_chmap = io_set_chmap,
    .avail = io_avail,
    .avail_update = io_avail_update,
    .prepare = io_prepare,
    .nonblock = io_nonblock,
    .resume = io_resume,
    .hw_params = io_hw_params,
    .sw_params = io_sw_params,
    .hw_params_any = io_hw_params_any,
    .sw_params_any = io_sw_params_any,
    .set_rate_resample = io_set_rate_resample,
    .poll_descriptors_count = io_poll_descriptors_count,
    .poll_descriptors = io_poll_descriptors,
};

int snd_pcm_ioplug_create(snd_pcm_ioplug_t *io, const char *name,
              snd_pcm_stream_t stream, int mode)
{
    int ret = snd_pcm_allocate_pcm(&io->pcm, &io_callbacks);
    if( ret == EOK ) {
        ioplug_data_t *data = calloc(1, sizeof(ioplug_data_t));
        if( data != NULL ) {
            data->stream = stream;
            data->params.rate_max = UINT_MAX;
            data->params.period_bytes_max = UINT_MAX;
            data->params.buffer_bytes_max = UINT_MAX;
            data->params.periods_max = UINT_MAX;
            data->params.format = SND_PCM_FORMAT_UNKNOWN;
            io->pcm->private_data = data;
            io->pcm->callback_param.ioplug = io;
        } else {
            snd_pcm_destroy_pcm(io->pcm);
            ret = -ENOMEM;
        }
    }
    return ret;
}

int snd_pcm_ioplug_delete(snd_pcm_ioplug_t *io)
{
    ioplug_data_t *data = io->pcm->private_data;

    free( data );

    return EOK;
}

int snd_pcm_ioplug_reinit_status(snd_pcm_ioplug_t *ioplug)
{
    UNIMPLEMENTED;
    return 0;
}

const snd_pcm_channel_area_t *snd_pcm_ioplug_mmap_areas(snd_pcm_ioplug_t *ioplug)
{
    UNIMPLEMENTED;
    return NULL;
}

void snd_pcm_ioplug_params_reset(snd_pcm_ioplug_t *io)
{
    UNIMPLEMENTED;
}

/* hw_parameter setting */
int snd_pcm_ioplug_set_param_minmax(snd_pcm_ioplug_t *io, int type, unsigned int min, unsigned int max)
{
    ioplug_data_t *data = (ioplug_data_t *)io->pcm->private_data;
    int i;

    switch( type ) {
        case SND_PCM_IOPLUG_HW_CHANNELS:
            data->params.params_set |= (1 << PARAM_VOICE_SET);
            for( i = min; i <= max; i ++ ) {
                data->params.channels |= 1 << (i - 1);
            }
            update_channels( &data->params );
            break;
        case SND_PCM_IOPLUG_HW_RATE:
            data->params.params_set |= (1 << PARAM_RATE_SET);
            data->params.rate_min = min;
            data->params.rate_max = max;
            break;
        case SND_PCM_IOPLUG_HW_PERIOD_BYTES:
            data->params.params_set |= (1 << PARAM_PERIOD_SET);
            data->params.period_bytes_min = min;
            data->params.period_bytes_max = max;
            break;
        case SND_PCM_IOPLUG_HW_BUFFER_BYTES:
            data->params.params_set |= (1 << PARAM_BUFFER_SET);
            data->params.buffer_bytes_min = min;
            data->params.buffer_bytes_max = max;
            break;
        case SND_PCM_IOPLUG_HW_PERIODS:
            data->params.params_set |= (1 << PARAM_PERIOD_COUNT_SET);
            data->params.periods_min = min;
            data->params.periods_max = max;
            break;
        default:
            return -EINVAL;
    }

    return EOK;
}

int snd_pcm_ioplug_set_param_list(snd_pcm_ioplug_t *io, int type, unsigned int num_list, const unsigned int *list)
{
    ioplug_data_t *data = (ioplug_data_t *)io->pcm->private_data;
    int i;

    switch( type ) {
        case SND_PCM_IOPLUG_HW_ACCESS:
            data->params.params_set |= (1 << PARAM_ACCESS_SET);
            for( i = 0; i < num_list; i ++ ) {
                if( list[ i ] < SND_PCM_ACCESS_LAST ) {
                    data->params.access_modes |= 1 << list[ i ];
                }
            }
            break;
        case SND_PCM_IOPLUG_HW_FORMAT:
            data->params.params_set |= (1 << PARAM_FORMAT_SET);
            for( i = 0; i < num_list; i ++ ) {
                if( list[ i ] <= SND_PCM_FORMAT_GSM
                        || (list[ i ] >= SND_PCM_FORMAT_SPECIAL && list[ i ] <= SND_PCM_FORMAT_LAST) ) {
                    data->params.formats |= 1 << list[ i ];
                }
            }
            update_formats( &data->params );
            break;
        case SND_PCM_IOPLUG_HW_CHANNELS:
            data->params.params_set |= (1 << PARAM_VOICE_SET);
            for( i = 0; i < num_list; i ++ ) {
                if( list[ i ] <= 8 ) {
                    data->params.channels |= 1 << list[ i ];
                }
            }
            update_channels( &data->params );
            break;
        case SND_PCM_IOPLUG_HW_RATE:
            data->params.params_set |= (1 << PARAM_RATE_SET);
            data->params.rates = malloc( sizeof(int) * num_list );
            if( data->params.rates == NULL ) {
                return -ENOMEM;
            }
            memcpy( data->params.rates, list, sizeof(int) * num_list );
            data->params.rate_count = num_list;
            data->params.rate_min = INT_MAX;
            data->params.rate_max = 0;
            for( i = 0; i < num_list; i ++ ) {
                data->params.rate_min = MIN(data->params.rate_min, list[ i ]);
                data->params.rate_max = MAX(data->params.rate_max, list[ i ]);
            }
            break;
        case SND_PCM_IOPLUG_HW_PERIOD_BYTES:
            data->params.params_set |= (1 << PARAM_PERIOD_SET);
            if( num_list != 1 ) {
                return -EINVAL;
            }
            data->params.period_bytes_min = list[0];
            data->params.period_bytes_max = list[0];
            break;
        case SND_PCM_IOPLUG_HW_BUFFER_BYTES:
            data->params.params_set |= (1 << PARAM_BUFFER_SET);
            if( num_list != 1 ) {
                return -EINVAL;
            }
            data->params.buffer_bytes_min = list[0];
            data->params.buffer_bytes_max = list[0];
            break;
        case SND_PCM_IOPLUG_HW_PERIODS:
            data->params.params_set |= (1 << PARAM_PERIOD_COUNT_SET);
            if( num_list != 1 ) {
                return -EINVAL;
            }
            data->params.periods_min = list[0];
            data->params.periods_max = list[0];
            break;
        default:
            return -EINVAL;
    }

    return EOK;
}

/* change PCM status */
int snd_pcm_ioplug_set_state(snd_pcm_ioplug_t *ioplug, snd_pcm_state_t state)
{
    UNIMPLEMENTED;
    return 0;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/ioplug.c $ $Rev: 759464 $")
#endif
