/*
 * $QNXLicenseC:
 * Copyright 2007, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable license fees to QNX
 * Software Systems before you may reproduce, modify or distribute this software,
 * or any work that includes all or part of this software.   Free development
 * licenses are available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review this entire
 * file for other proprietary rights or license notices, as well as the QNX
 * Development Suite License Guide at http://licensing.qnx.com/license-guide/
 * for other information.
 * $
 */



/*
 *    alsa.c
 *      Implementation of alsa functions that are incompatible with legacy qnx
 *      audio
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <limits.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/asound_common.h>
#include <alsa/asoundlib.h>
#include <alsa/pcm_plugin.h>
#include "pcm_internal.h"
#include <stdbool.h>
#include <sys/slog2.h>

#define INTERNAL_SLOG_BUFFER

#define SND_FILE_PCM_PLAYBACK			"/dev/snd/pcmC%iD%ip"
#define SND_FILE_PCM_CAPTURE			"/dev/snd/pcmC%iD%ic"

#define IS_POWER_OF_TWO(x) (!((x) & ((x) - 1)))
#define TEST_SET(params, flags) ((params->params_set & (flags)) == (flags))

#ifdef INTERNAL_SLOG_BUFFER
static bool slog_initialized = false;
static slog2_buffer_t slog_handle;
static slog2_buffer_set_config_t slog_config;
#define TRY_START_SLOG                                          \
{                                                               \
    if(!slog_initialized) {                                     \
        slog_initialized = true;                                \
        memset(&slog_config, 0, sizeof(slog_config));           \
        slog_config.num_buffers = 1;                            \
        slog_config.verbosity_level = SLOG2_INFO;               \
        slog_config.buffer_set_name = "libasound";              \
        slog_config.buffer_config[0].buffer_name="libasound";   \
        slog_config.buffer_config[0].num_pages = 5;             \
        slog2_register(&slog_config, &slog_handle, 0);          \
        slog2_set_verbosity(slog_handle, SLOG2_INFO);           \
        slog2_set_default_buffer(slog_handle);                  \
    }                                                           \
}
#else
#define TRY_START_SLOG
#endif

snd_lib_error_handler_t snd_lib_error;

struct  snd_pcm_link_group {
	snd_pcm_t			*handle;
	struct snd_pcm_link_group	*next;
};

snd_config_t *snd_config = NULL;

const snd_pcm_hw_params_t *find_params_setting( const snd_pcm_hw_params_t *params, params_set_t param )
{
    const snd_pcm_hw_params_t *i;
    for( i = params; i != NULL; i = i->parent ) {
        if( i->params_set & (1 << param) ) {
            return i;
        }
    }

    return NULL;
}

unsigned int transform_buffer_size( const snd_pcm_hw_params_t *start, const snd_pcm_hw_params_t *end, unsigned int size )
{
    uint64_t newsize = size;
    if( start->rate_min == start->rate_max ) {
        // Have a fixed rate. Check whether it will be modified to hit the target
        if( start->rate_min < end->rate_min ) {
            newsize *= start->rate_min;
            newsize /= end->rate_min;
        } else if( start->rate_max > end->rate_max ) {
            newsize *= start->rate_max;
            newsize /= end->rate_max;
        }
    }

    // TODO - If 'end' supports a variable number of voices, pick the closest
    // one and still do a transform
    if( start->voices && end->voices ) {
        newsize *= start->voices;
        newsize /= end->voices;
    }

    // TODO - If 'end' supports a variable output format, pick the closest
    // one and still do a transform. It might be desirable to let the plugin
    // indicate which one it will choose for more precision, but it may not
    // know yet at this point
    if( (start->params_set & (1 << PARAM_FORMAT_SET)) && (end->params_set & (1 << PARAM_FORMAT_SET))) {
        newsize *= snd_pcm_format_width( start->format );
        newsize /= snd_pcm_format_width( end->format );
    }

    return (unsigned int)newsize;
}

void update_formats(snd_pcm_hw_params_t *params)
{
    int i;

    // Set formats
    if( IS_POWER_OF_TWO( params->formats ) ) {
        for( i = 0; i < sizeof(params->formats) * _BITS_BYTE; i ++ ) {
            if( params->formats & (1 << i) ) {
                params->format = i;
                break;
            }
        }
    }

}

void update_channels(snd_pcm_hw_params_t *params)
{
    int i;

    if( IS_POWER_OF_TWO( params->channels ) ) {
        for( i = 0; i < sizeof(params->channels) * _BITS_BYTE; i ++ ) {
            if( params->channels & (1 << i) ) {
                params->voices = i + 1;
                break;
            }
        }
    }
}


int snd_pcm_open_config(snd_pcm_t **pcmp, snd_config_t *root, snd_config_t *slave_conf, snd_pcm_stream_t stream, int mode)
{
    int ret = 0;

    TRY_START_SLOG
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(stream), SLOG2_FA_UNSIGNED(mode), SLOG2_FA_END);

	if (stream != SND_PCM_STREAM_PLAYBACK && stream != SND_PCM_STREAM_CAPTURE) {
		return -EINVAL;
    }

    ret = snd_pcm_load_plugin_config( pcmp, root, slave_conf, stream, mode );
    if( ret ) {
        return ret;
    }

    (*pcmp)->stream = stream;

    return ret;
}

int snd_pcm_open(snd_pcm_t **pcmp, const char *name, snd_pcm_stream_t stream, int mode)
{
    int ret = 0;

    TRY_START_SLOG
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %s %d %d", SLOG2_FA_STRING(__func__), SLOG2_FA_STRING(name), SLOG2_FA_UNSIGNED(stream), SLOG2_FA_UNSIGNED(mode), SLOG2_FA_END);
	if (stream != SND_PCM_STREAM_PLAYBACK && stream != SND_PCM_STREAM_CAPTURE) {
		return -EINVAL;
    }

    ret = snd_pcm_load_plugin( pcmp, name, stream, mode );
    if( ret ) {
        return ret;
    }

    (*pcmp)->stream = stream;

    return ret;
}

snd_pcm_stream_t snd_pcm_stream( snd_pcm_t *pcm )
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %"PRIxPTR, SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((intptr_t)pcm), SLOG2_FA_END);
    return pcm->stream;
}

int snd_pcm_close(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %"PRIxPTR, SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((intptr_t)pcm), SLOG2_FA_END);
    int ret;
    if( (ret = pcm->callbacks->close(pcm)) == EOK ) {
        free(pcm->name);
        free(pcm);
    }
    return ret;
}

snd_pcm_sframes_t snd_pcm_readi(snd_pcm_t *pcm, void *buffer, snd_pcm_uframes_t size)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %"PRIxPTR" %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((uintptr_t)pcm), SLOG2_FA_UNSIGNED((unsigned int)size), SLOG2_FA_END);
    if( size == 0 ) return 0;

    return pcm->callbacks->readi(pcm, buffer, size);
}

snd_pcm_sframes_t snd_pcm_writei(snd_pcm_t *pcm, const void *buffer, snd_pcm_uframes_t size)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %"PRIxPTR" %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((uintptr_t)pcm), SLOG2_FA_UNSIGNED((unsigned int)size), SLOG2_FA_END);
    if( size == 0 ) return 0;

    return pcm->callbacks->writei(pcm, buffer, size);
}

snd_pcm_sframes_t snd_pcm_writen(snd_pcm_t *pcm, void **buffs, snd_pcm_uframes_t size)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %"PRIxPTR" %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((uintptr_t)pcm), SLOG2_FA_UNSIGNED((unsigned int)size), SLOG2_FA_END);
    if( size == 0 ) return 0;

    return pcm->callbacks->writen(pcm, buffs, size);
}

snd_pcm_sframes_t snd_pcm_readn(snd_pcm_t *pcm, void **buffs, snd_pcm_uframes_t size)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %"PRIxPTR" %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((uintptr_t)pcm), SLOG2_FA_UNSIGNED((unsigned int)size), SLOG2_FA_END);
    if( size == 0 ) return 0;

    return pcm->callbacks->readn(pcm, buffs, size);
}

int snd_pcm_recover(snd_pcm_t *pcm, int err, int silent)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %"PRIxPTR" %d %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((uintptr_t)pcm), SLOG2_FA_UNSIGNED(err), SLOG2_FA_UNSIGNED(silent), SLOG2_FA_END);
    return pcm->callbacks->recover(pcm, err, silent);
}

const char *snd_strerror(int errnum)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(errnum), SLOG2_FA_END);
    return (const char *) strerror (-errnum);
}

int snd_pcm_drain(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %"PRIxPTR, SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((uintptr_t)pcm), SLOG2_FA_END);
    return pcm->callbacks->drain(pcm);
}

int snd_pcm_hw_params_malloc(snd_pcm_hw_params_t **params)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    *params = malloc(sizeof(snd_pcm_hw_params_t));
    return *params == NULL ? -ENOMEM : EOK;
}

void snd_pcm_hw_params_free(snd_pcm_hw_params_t *obj)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    free( obj );
}

int snd_pcm_hw_params(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    int ret;
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    ret = pcm->callbacks->hw_params(pcm, params);
    memcpy(&pcm->hw_params, params, sizeof(snd_pcm_hw_params_t));
    return ret;
}

int snd_pcm_hw_params_get_buffer_size(const snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *buffer = find_params_setting(params, PARAM_BUFFER_SET);

    if( buffer == NULL || buffer->buffer_bytes_min != buffer->buffer_bytes_max ) {
        return -EINVAL;
    }
    *val = transform_buffer_size(params, buffer, buffer->buffer_bytes_min);
    *val /= params->voices * snd_pcm_format_width( params->format ) / 8;
    return EOK;
}

int snd_pcm_hw_params_get_buffer_size_min(const snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *buffer = find_params_setting(params, PARAM_BUFFER_SET);

    if( buffer ) {
        *val = transform_buffer_size(params, buffer, buffer->buffer_bytes_min);
        *val /= params->voices * snd_pcm_format_width( params->format ) / 8;
        return EOK;
    } else {
        return -EINVAL;
    }
}

int snd_pcm_hw_params_get_buffer_size_max(const snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *buffer = find_params_setting(params, PARAM_BUFFER_SET);

    if( buffer ) {
        *val = transform_buffer_size(params, buffer, buffer->buffer_bytes_max);
        *val /= params->voices * snd_pcm_format_width( params->format ) / 8;
        return EOK;
    } else {
        return -EINVAL;
    }
}

int snd_pcm_hw_params_get_buffer_time(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    unsigned int min_time;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_BUFFER_SET);

    if( i != NULL ) {
        if( i->buffer_bytes_min != i->buffer_bytes_max ) {
            return -EINVAL;
        }
        min_time = transform_buffer_size( params, i, i->buffer_bytes_min ) * 1000000 / params->voices / snd_pcm_format_width( params->format ) / _BITS_BYTE / params->rate;

        if( dir ) {
            if( min_time < *val ) {
                *dir = -1;
            } else if( min_time == *val ) {
                *dir = 0;
            } else {
                *dir = 1;
            }
        }
        *val = min_time;
    } else {
        // Not set, value can be anything
        return -EINVAL;
    }

    return EOK;
}

int snd_pcm_hw_params_get_buffer_time_min(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    unsigned int min_time;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_BUFFER_SET);

    if( i != NULL ) {
        min_time = transform_buffer_size( params, i, i->buffer_bytes_min ) * 1000000 / params->voices / snd_pcm_format_width( params->format ) / _BITS_BYTE / params->rate;

        if( dir ) {
            if( min_time < *val ) {
                *dir = -1;
            } else if( min_time == *val ) {
                *dir = 0;
            } else {
                *dir = 1;
            }
        }
        *val = min_time;
    } else {
        // Not set, value can be anything
        *val = 0;
        if( dir ) *dir = ((0 == *val) ? 0 : -1);
    }

    return EOK;
}

int snd_pcm_hw_params_get_buffer_time_max(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    unsigned int max_time;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_BUFFER_SET);

    if( i != NULL ) {
        max_time = transform_buffer_size( params, i, i->buffer_bytes_max ) * 1000000 / params->voices / snd_pcm_format_width( params->format ) / _BITS_BYTE / params->rate;

        if( dir ) {
            if( max_time < *val ) {
                *dir = -1;
            } else if( max_time == *val ) {
                *dir = 0;
            } else {
                *dir = 1;
            }
        }
        *val = max_time;
    } else {
        // Not set, value can be anything
        if( dir ) *dir = ((*val == UINT_MAX) ? 0 : 1);
        *val = UINT_MAX;
    }

    return EOK;
}

int snd_pcm_hw_params_get_period_size(const snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    snd_pcm_uframes_t frames;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_PERIOD_SET);

    if( i != NULL ) {
        if( i->period_bytes_min != i->period_bytes_max ) {
            return -EINVAL;
        }
        frames = transform_buffer_size( params, i, i->period_bytes_min ) / params->voices / (snd_pcm_format_width( params->format ) / _BITS_BYTE);
        if( dir ) {
            if( frames < *val ) {
                *dir = -1;
            } else if( frames == *val ) {
                *dir = 0;
            } else {
                *dir = 1;
            }
        }
        *val = frames;
        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_period_size_min(const snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    snd_pcm_uframes_t frames;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_PERIOD_SET);

    if( i != NULL ) {
        frames = transform_buffer_size( params, i, i->period_bytes_min ) / params->voices / (snd_pcm_format_width( params->format ) / _BITS_BYTE);
        if( dir ) {
            if( frames < *val ) {
                *dir = -1;
            } else if( frames == *val ) {
                *dir = 0;
            } else {
                *dir = 1;
            }
        }
        *val = frames;
        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_period_size_max(const snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    snd_pcm_uframes_t frames;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_PERIOD_SET);

    if( i != NULL ) {
        frames = transform_buffer_size( params, i, i->period_bytes_max ) / params->voices / (snd_pcm_format_width( params->format ) / _BITS_BYTE);
        if( dir ) {
            if( frames < *val ) {
                *dir = -1;
            } else if( frames == *val ) {
                *dir = 0;
            } else {
                *dir = 1;
            }
        }
        *val = frames;
        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_period_time_min(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    unsigned int time;
    const snd_pcm_hw_params_t *i;

    if( !TEST_SET(params, (1 << PARAM_FORMAT_SET)|( 1 << PARAM_RATE_SET)|(1 << PARAM_VOICE_SET)) ) {
        return -EINVAL;
     }

    i = find_params_setting(params, PARAM_PERIOD_SET);
    if( i ) {
        time = transform_buffer_size( params, i, i->period_bytes_min) * 1000000 / (params->rate_min * params->voices * snd_pcm_format_width( params->format ) / 8);
    } else {
        time = 0;
    }
    if( dir ) {
        if( time < *val ) {
            *dir = -1;
        } else if( time == *val ) {
            *dir = 0;
        } else {
            *dir = 1;
        }
    }
    *val = time;
     return EOK;
}

int snd_pcm_hw_params_get_period_time_max(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    unsigned int time;
    const snd_pcm_hw_params_t *i;

    if( !TEST_SET(params, (1 << PARAM_FORMAT_SET)|( 1 << PARAM_RATE_SET)|(1 << PARAM_VOICE_SET)) ) {
        return -EINVAL;
     }

    i = find_params_setting(params, PARAM_PERIOD_SET);
    if( i ) {
        time = transform_buffer_size( params, i, i->period_bytes_max) * 1000000 / (params->rate_min * params->voices * snd_pcm_format_width( params->format ) / 8);
    } else {
        time = UINT_MAX;
    }
    if( dir ) {
        if( time < *val ) {
            *dir = -1;
        } else if( time == *val ) {
            *dir = 0;
        } else {
            *dir = 1;
        }
    }
    *val = time;
     return EOK;
}

int snd_pcm_hw_params_get_period_time(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    unsigned int time;

    if( params->period_bytes_max != params->period_bytes_min
     || params->rate_max != params->rate_min
     || params->voices == 0 ) {
        return -EINVAL;
     }

    time = params->period_bytes_max * 1000000 / params->rate_min / params->voices;
    if( dir ) {
        if( time < *val ) {
            *dir = -1;
        } else if( time == *val ) {
            *dir = 0;
        } else {
            *dir = 1;
        }
    }
    *val = time;
     return EOK;
}

int snd_pcm_hw_params_set_period_size(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val, int dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    uint64_t min, max;
    unsigned int requested_period_bytes;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_PERIOD_SET);

    // For now, don't allow setting the period unless voices/rate/format are known
    if( TEST_SET(params, (1 << PARAM_FORMAT_SET)|( 1 << PARAM_RATE_SET)|(1 << PARAM_VOICE_SET)) ) {

        requested_period_bytes = ((uint64_t)(val)) * params->voices * snd_pcm_format_width( params->format ) / 8;
        if( i ) {
            min = transform_buffer_size(params, i, i->period_bytes_min);
            max = transform_buffer_size(params, i, i->period_bytes_max);

            // Convert val to bytes
            if( requested_period_bytes < min && requested_period_bytes > max ) {
                return -EINVAL;
            }
        }

        params->period_bytes_min = params->period_bytes_max = requested_period_bytes;
        params->params_set |= (1 << PARAM_PERIOD_SET);

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_periods(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_PERIOD_COUNT_SET);

    if( i != NULL ) {
        if( i->periods_min != i->periods_max ) {
            return -EINVAL;
        }
        if( dir ) {
            if( i->periods_min < *val ) {
                *dir = -1;
            } else if( i->periods_min == *val ) {
                *dir = 0;
            } else {
                *dir = 1;
            }
        }
        *val = i->periods_min;
        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_access(const snd_pcm_hw_params_t *params, snd_pcm_access_t *_access)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    int j;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_ACCESS_SET);

    if( i != NULL ) {
        if( IS_POWER_OF_TWO (i->access_modes) ) {
            for( j = 0; j <= SND_PCM_ACCESS_LAST; j ++ ) {
                if( i->access_modes & (1 << j) ) {
                    *_access = j;
                    return EOK;
                }
            }
        }
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_rate( const snd_pcm_hw_params_t *params, unsigned int *rate, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_RATE_SET);

    if( i != NULL ) {
        if( i->rate_min != i->rate_max ) {
            return -EINVAL;
        }

        if( dir ) {
            if( *rate < i->rate_min ) {
                *dir = 1;
            } else if( *rate > i->rate_min ) {
                *dir = -1;
            } else {
                *dir = 0;
            }
        }

        *rate = i->rate_min;

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_rate_min(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_RATE_SET);

    if( i != NULL ) {
        if( *val < i->rate_min ) {
            *dir = 1;
        } else if( *val > i->rate_min ) {
            *dir = -1;
        } else {
            *dir = 0;
        }
        *val = params->rate_min;

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_rate_max(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_RATE_SET);

    if( i != NULL ) {
        if( *val < i->rate_max ) {
            *dir = 1;
        } else if( *val > i->rate_max ) {
            *dir = -1;
        } else {
            *dir = 0;
        }
        *val = i->rate_max;

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_rate_numden(const snd_pcm_hw_params_t *params,
				      unsigned int *rate_num,
				      unsigned int *rate_den)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_RATE_SET);

    if( i != NULL ) {
        if( i->rate_min != i->rate_max ) {
            return -EINVAL;
        }

        *rate_num = i->rate_min;
        *rate_den = 1;

        return EOK;
    }

    return -EINVAL;
}

void snd_pcm_hw_params_copy(snd_pcm_hw_params_t *dst, const snd_pcm_hw_params_t *src)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    memcpy(dst, src, sizeof(snd_pcm_hw_params_t));
}

int snd_pcm_hw_params_set_period_size_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    uint64_t min, max;
    unsigned int requested_period_bytes;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_PERIOD_SET);

    // For now, don't allow setting the period size unless voices/rate/format are known
    if( TEST_SET(params, (1 << PARAM_FORMAT_SET)|( 1 << PARAM_RATE_SET)|(1 << PARAM_VOICE_SET)) ) {

        if( i != NULL ) {
            min = transform_buffer_size(params, i, i->period_bytes_min);
            max = transform_buffer_size(params, i, i->period_bytes_max);
        } else {
            min = 0;
            max = UINT_MAX;
        }

        requested_period_bytes = ((uint64_t)(*val)) * params->voices * snd_pcm_format_width( params->format ) / _BITS_BYTE;
        if( requested_period_bytes < min ) {
            requested_period_bytes = params->period_bytes_max = params->period_bytes_min;
            if( dir ) *dir = 1;
        } else if( (*val) > max ) {
            requested_period_bytes = params->period_bytes_min = params->period_bytes_max;
            if( dir ) *dir = -1;
        } else {
            params->period_bytes_min = params->period_bytes_max = requested_period_bytes;
            if( dir ) *dir = 0;
        }
        (*val) = ((uint64_t)requested_period_bytes) / params->voices * _BITS_BYTE / snd_pcm_format_width( params->format );
        params->params_set |= (1 << PARAM_PERIOD_SET);

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_set_periods_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *period = find_params_setting(params, PARAM_PERIOD_SET);
    const snd_pcm_hw_params_t *buffer = find_params_setting(params, PARAM_BUFFER_SET);
    const snd_pcm_hw_params_t *count = find_params_setting(params, PARAM_PERIOD_COUNT_SET);

    if( (period == NULL || buffer == NULL) && count == NULL ) {
        // Can be anything, so accept the value
        params->periods_min = params->periods_max = *val;
        params->params_set |= (1 << PARAM_PERIOD_COUNT_SET);
        *dir = 0;
        return EOK;
    }
    if( period != NULL && buffer != NULL ) {
        // If the period count limits aren't set yet, set them now
        if( count == NULL ) {
            count = params;
            params->params_set |= (1 << PARAM_PERIOD_COUNT_SET);
            params->periods_min = buffer->buffer_bytes_min / period->period_bytes_max;
            params->periods_max = buffer->buffer_bytes_max / period->period_bytes_min;
        } else {
            // Update the period count limits
            params->periods_min = MAX(count->periods_min, buffer->buffer_bytes_min / period->period_bytes_max);
            params->periods_max = MIN(count->periods_max, buffer->buffer_bytes_max / period->period_bytes_min);
        }
    } else {
        // Just take the count limits
        params->periods_min = count->periods_min;
        params->periods_max = count->periods_max;
    }

    if( *val < count->periods_min ) {
        *val = params->periods_max = params->periods_min;
        *dir = -1;
    } else if ( *val > count->periods_max ) {
        *val = params->periods_min = params->periods_max;
        *dir = 1;
    } else {
        params->periods_min = params->periods_max = *val;
        *dir = 0;
    }

    // If one of periods and buffer size is set, and the other isn't, it can be set now
    if( period == NULL && buffer != NULL ) {
        params->period_bytes_min = params->period_bytes_max = buffer->buffer_bytes_min / (*val);
        params->params_set |= (1 << PARAM_PERIOD_SET);
    }
    if( period != NULL && buffer == NULL ) {
        params->buffer_bytes_min = params->buffer_bytes_max = period->period_bytes_min * (*val);
        params->params_set |= (1 << PARAM_BUFFER_SET);
    }

    params->periods_min = params->periods_max = *val;
    params->params_set |= (1 << PARAM_PERIOD_COUNT_SET);

    return EOK;
}

int snd_pcm_hw_params_set_buffer_size_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    uint64_t min, max;
    const snd_pcm_hw_params_t *period = find_params_setting(params, PARAM_PERIOD_SET);
    const snd_pcm_hw_params_t *buffer = find_params_setting(params, PARAM_BUFFER_SET);
    const snd_pcm_hw_params_t *count = find_params_setting(params, PARAM_PERIOD_COUNT_SET);

    // For now, don't allow setting the buffer size unless voices/rate/format are known
    if( TEST_SET(params, (1 << PARAM_FORMAT_SET)|( 1 << PARAM_RATE_SET)|(1 << PARAM_VOICE_SET)) ) {

        if( buffer != NULL ) {
            min = transform_buffer_size(params, buffer, buffer->buffer_bytes_min);
            max = transform_buffer_size(params, buffer, buffer->buffer_bytes_max);
        } else {
            min = 0;
            max = UINT_MAX;
        }

        if( (*val) < min ) {
            (*val) = params->buffer_bytes_max = params->buffer_bytes_min;
        } else if( (*val) > max ) {
            (*val) = params->buffer_bytes_min = params->buffer_bytes_max;
        } else {
            params->buffer_bytes_min = params->buffer_bytes_max = (*val);
        }
        params->params_set |= (1 << PARAM_BUFFER_SET);

        // If one of periods and num periods is set, and the other isn't, it can be set now
        if( period == NULL && count != NULL ) {
            params->period_bytes_min = params->period_bytes_max = (*val) / count->periods_min;
            params->params_set |= (1 << PARAM_PERIOD_SET);
        }
        if( period != NULL && count == NULL ) {
            params->periods_min = params->periods_max = (*val) / period->period_bytes_min;
            params->params_set |= (1 << PARAM_PERIOD_COUNT_SET);
        }

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_set_buffer_size(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    uint64_t min, max;
    const snd_pcm_hw_params_t *period = find_params_setting(params, PARAM_PERIOD_SET);
    const snd_pcm_hw_params_t *buffer = find_params_setting(params, PARAM_BUFFER_SET);
    const snd_pcm_hw_params_t *count = find_params_setting(params, PARAM_PERIOD_COUNT_SET);

    // For now, don't allow setting the buffer size unless voices/rate/format are known
    if( TEST_SET(params, (1 << PARAM_FORMAT_SET)|( 1 << PARAM_RATE_SET)|(1 << PARAM_VOICE_SET)) ) {

        // Convert val from frames to bytes
        val *= params->voices * snd_pcm_format_width( params->format ) / 8;

        if( buffer != NULL ) {
            min = transform_buffer_size(params, buffer, buffer->buffer_bytes_min);
            max = transform_buffer_size(params, buffer, buffer->buffer_bytes_max);
        } else {
            min = 0;
            max = UINT_MAX;
        }

        if( val < min || val > max ) {
            return -EINVAL;
        }
        params->buffer_bytes_min = params->buffer_bytes_max = val;
        params->params_set |= (1 << PARAM_BUFFER_SET);

        // If one of periods and num periods is set, and the other isn't, it can be set now
        if( period == NULL && count != NULL ) {
            params->period_bytes_min = params->period_bytes_max = val / count->periods_min;
            params->params_set |= (1 << PARAM_PERIOD_SET);
        }
        if( period != NULL && count == NULL ) {
            params->periods_min = params->periods_max = val / transform_buffer_size(params, period, period->period_bytes_min);
            params->params_set |= (1 << PARAM_PERIOD_COUNT_SET);
        }

        return EOK;
    }

    return -EINVAL;
}


int snd_pcm_hw_params_any(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->hw_params_any(pcm, params);
}

int snd_pcm_hw_params_set_access(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_access_t access)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_ACCESS_SET);

    if( i != NULL ) {
        if( i->access_modes & (1 << access) ) {
            params->access_modes = 1 << access;
            params->params_set |= (1 << PARAM_ACCESS_SET);
            return EOK;
        }
    }

    return -EINVAL;
}

int snd_pcm_hw_params_set_format(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_format_t format)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(format), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_FORMAT_SET);

    if( i != NULL ) {
        if( i->formats & (1 << format) ) {
            params->formats = 1 << format;
            params->format = format;
            params->params_set |= (1 << PARAM_FORMAT_SET);
            return EOK;
        }
    }

    return -EINVAL;
}

int snd_pcm_hw_params_set_channels(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(val), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i;

    if( val == 0 || val > sizeof(params->channels) * _BITS_BYTE ) {
        return -EINVAL;
    }

    i = find_params_setting(params, PARAM_VOICE_SET);
    if( i != NULL ) {
        if( (i->channels & (1 << (val - 1))) != 0 ) {
            params->channels = 1 << (val - 1);
            params->voices = val;
            params->params_set |= (1 << PARAM_VOICE_SET);

            return EOK;
        }
    }

    return -EINVAL;
}

int snd_pcm_hw_params_test_channels(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(val), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_VOICE_SET);

    if( i != NULL ) {
        if( i->channels & (1 << val) ) {
            return EOK;
        }
    }
    return -EINVAL;
}

int snd_pcm_hw_params_get_channels(const snd_pcm_hw_params_t *params, unsigned int *val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_VOICE_SET);

    if( i != NULL ) {
        if( i->voices ) {
            *val = params->voices;
            return EOK;
        }
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_format(const snd_pcm_hw_params_t *params, snd_pcm_format_t *val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_FORMAT_SET);

    if( i != NULL ) {
        if( i->format != SND_PCM_FORMAT_UNKNOWN ) {
            *val = params->format;
            return EOK;
        }
    }

    return -EINVAL;
}

int snd_pcm_hw_params_set_rate_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    int j;
    unsigned int difference, min_difference, best_rate = 0;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_RATE_SET);

    if( i != NULL ) {
        if( i->rates == NULL ) {
            params->rate = *val;
            if( params->rate < i->rate_min ) {
                params->rate = i->rate_min;
                if( dir != NULL ) {
                    *dir = -1;
                }
            } else if ( params->rate > i->rate_max ) {
                params->rate = i->rate_max;
                if( dir != NULL ) {
                    *dir = 1;
                }
            } else if( dir != NULL ) {
                *dir = 0;
            }
        } else {
            min_difference = UINT_MAX;
            for(j=0; j < i->rate_count; j ++  ) {
                difference = i->rates[ j ] > *val ? i->rates[ j ] - *val : *val - i->rates[ j ];
                if( difference < min_difference ) {
                    min_difference = difference;
                    best_rate = i->rates[ j ];
                }
            }

            if( min_difference == UINT_MAX ) {
                return -EINVAL;
            }

            if( dir != NULL ) {
                if( best_rate < *val ) {
                    *dir = -1;
                } else if( best_rate > *val ) {
                    *dir = 1;
                } else {
                    *dir = 0;
                }
            }
            params->rate = best_rate;
        }

        *val = params->rate_min = params->rate_max = params->rate;
        params->rate_count = 0;
        params->rates = NULL;
        params->params_set |= (1 << PARAM_RATE_SET);

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_sw_params_malloc(snd_pcm_sw_params_t **params)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    *params = malloc(sizeof(snd_pcm_sw_params_t));
    return *params == NULL ? -ENOMEM : 0;
}

void snd_pcm_sw_params_free(snd_pcm_sw_params_t *obj)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    free( obj );
}

int snd_pcm_sw_params_current(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    memcpy(params, &pcm->sw_params, sizeof(snd_pcm_sw_params_t));
    return EOK;
}

void snd_pcm_sw_params_copy(snd_pcm_sw_params_t *dst, const snd_pcm_sw_params_t *src)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    memcpy(dst, src, sizeof(snd_pcm_sw_params_t));
}


int snd_pcm_sw_params_set_start_threshold(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((unsigned int)val), SLOG2_FA_END);
    params->start_threshold  = val;

    return EOK;
}

int snd_pcm_sw_params(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    memcpy(&pcm->sw_params, params, sizeof(snd_pcm_sw_params_t));
    return pcm->callbacks->sw_params(pcm, params);
}

int snd_pcm_nonblock(snd_pcm_t *pcm, int nonblock)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->nonblock(pcm, nonblock);
}

int snd_pcm_prepare(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->prepare(pcm);
}

int snd_pcm_status(snd_pcm_t *pcm, snd_pcm_status_t *status)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->status(pcm, status);
}

int snd_pcm_resume(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->resume(pcm);
}

snd_pcm_state_t snd_pcm_state(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    UNIMPLEMENTED;
    return SND_PCM_STATE_OPEN;
}

int snd_pcm_format_width(snd_pcm_format_t format)
{
    switch(format)
    {
        case SND_PCM_FORMAT_S8:
        case SND_PCM_FORMAT_U8:
            return 8;
        case SND_PCM_FORMAT_S16_LE:
        case SND_PCM_FORMAT_S16_BE:
        case SND_PCM_FORMAT_U16_LE:
        case SND_PCM_FORMAT_U16_BE:
            return 16;
        case SND_PCM_FORMAT_S24_LE:
        case SND_PCM_FORMAT_S24_BE:
        case SND_PCM_FORMAT_U24_LE:
        case SND_PCM_FORMAT_U24_BE:
            return 24;
        case SND_PCM_FORMAT_S32_LE:
        case SND_PCM_FORMAT_S32_BE:
        case SND_PCM_FORMAT_U32_LE:
        case SND_PCM_FORMAT_U32_BE:
        case SND_PCM_FORMAT_FLOAT_LE:
        case SND_PCM_FORMAT_FLOAT_BE:
            return 32;
        case SND_PCM_FORMAT_FLOAT64_LE:
        case SND_PCM_FORMAT_FLOAT64_BE:
            return 64;
        case SND_PCM_FORMAT_IEC958_SUBFRAME_LE:
        case SND_PCM_FORMAT_IEC958_SUBFRAME_BE:
            return 24;
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
            return 8;
        case SND_PCM_FORMAT_IMA_ADPCM:
            return 4;
        case SND_PCM_FORMAT_S24_3LE:
        case SND_PCM_FORMAT_S24_3BE:
        case SND_PCM_FORMAT_U24_3LE:
        case SND_PCM_FORMAT_U24_3BE:
        case SND_PCM_FORMAT_S20_3LE:
        case SND_PCM_FORMAT_S20_3BE:
        case SND_PCM_FORMAT_U20_3LE:
        case SND_PCM_FORMAT_U20_3BE:
        case SND_PCM_FORMAT_S18_3LE:
        case SND_PCM_FORMAT_S18_3BE:
        case SND_PCM_FORMAT_U18_3LE:
        case SND_PCM_FORMAT_U18_3BE:
            return 24;
        default:
            return -EINVAL;
    }
}

int snd_pcm_format_physical_width(snd_pcm_format_t format)
{
    switch(format)
    {
        case SND_PCM_FORMAT_S8:
        case SND_PCM_FORMAT_U8:
            return 8;
        case SND_PCM_FORMAT_S16_LE:
        case SND_PCM_FORMAT_S16_BE:
        case SND_PCM_FORMAT_U16_LE:
        case SND_PCM_FORMAT_U16_BE:
            return 16;
        case SND_PCM_FORMAT_S24_LE:
        case SND_PCM_FORMAT_S24_BE:
        case SND_PCM_FORMAT_U24_LE:
        case SND_PCM_FORMAT_U24_BE:
        case SND_PCM_FORMAT_S32_LE:
        case SND_PCM_FORMAT_S32_BE:
        case SND_PCM_FORMAT_U32_LE:
        case SND_PCM_FORMAT_U32_BE:
        case SND_PCM_FORMAT_FLOAT_LE:
        case SND_PCM_FORMAT_FLOAT_BE:
            return 32;
        case SND_PCM_FORMAT_FLOAT64_LE:
        case SND_PCM_FORMAT_FLOAT64_BE:
            return 64;
        case SND_PCM_FORMAT_IEC958_SUBFRAME_LE:
        case SND_PCM_FORMAT_IEC958_SUBFRAME_BE:
            return 24;
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
            return 8;
        case SND_PCM_FORMAT_IMA_ADPCM:
            return 8;
        case SND_PCM_FORMAT_S24_3LE:
        case SND_PCM_FORMAT_S24_3BE:
        case SND_PCM_FORMAT_U24_3LE:
        case SND_PCM_FORMAT_U24_3BE:
        case SND_PCM_FORMAT_S20_3LE:
        case SND_PCM_FORMAT_S20_3BE:
        case SND_PCM_FORMAT_U20_3LE:
        case SND_PCM_FORMAT_U20_3BE:
        case SND_PCM_FORMAT_S18_3LE:
        case SND_PCM_FORMAT_S18_3BE:
        case SND_PCM_FORMAT_U18_3LE:
        case SND_PCM_FORMAT_U18_3BE:
            return 24;
        default:
            return -EINVAL;
    }
}

ssize_t snd_pcm_format_size(snd_pcm_format_t format, size_t samples)
{
    return (snd_pcm_format_width(format) + 7) * samples >> 3;
}

uint8_t snd_pcm_format_silence(snd_pcm_format_t format)
{
    switch(format) {
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
        case SND_PCM_FORMAT_IMA_ADPCM:
            UNIMPLEMENTED;
            return 0;
        default:
            return 0;
    }
}

uint16_t snd_pcm_format_silence_16(snd_pcm_format_t format)
{
    switch(format) {
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
        case SND_PCM_FORMAT_IMA_ADPCM:
            UNIMPLEMENTED;
            return 0;
        default:
            return 0;
    }
}

uint32_t snd_pcm_format_silence_32(snd_pcm_format_t format)
{
    switch(format) {
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
        case SND_PCM_FORMAT_IMA_ADPCM:
            UNIMPLEMENTED;
            return 0;
        default:
            return 0;
    }
}

uint64_t snd_pcm_format_silence_64(snd_pcm_format_t format)
{
    switch(format) {
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
        case SND_PCM_FORMAT_IMA_ADPCM:
            UNIMPLEMENTED;
            return 0;
        default:
            return 0;
    }
}

int snd_pcm_format_set_silence(snd_pcm_format_t format, void *buf, unsigned int samples)
{
    switch(format) {
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
        case SND_PCM_FORMAT_IMA_ADPCM:
            UNIMPLEMENTED;
            return 0;
        default:
            memset( buf, 0, samples * snd_pcm_format_width(format) / 8 );
            return EOK;
    }
}

snd_pcm_sframes_t snd_pcm_mmap_writei(snd_pcm_t *pcm, const void *buffer, snd_pcm_uframes_t size)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((unsigned int)size), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

snd_pcm_sframes_t snd_pcm_mmap_readi(snd_pcm_t *pcm, void *buffer, snd_pcm_uframes_t size)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((unsigned int)size), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

snd_pcm_sframes_t snd_pcm_mmap_writen(snd_pcm_t *pcm, void **bufs, snd_pcm_uframes_t size)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((unsigned int)size), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

snd_pcm_sframes_t snd_pcm_mmap_readn(snd_pcm_t *pcm, void **bufs, snd_pcm_uframes_t size)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((unsigned int)size), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

const char *snd_pcm_name(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->name;
}

snd_pcm_sframes_t snd_pcm_avail(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    snd_pcm_sframes_t ret;
    ret = pcm->callbacks->avail(pcm);
    return ret;
}

snd_pcm_sframes_t snd_pcm_avail_update(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    snd_pcm_sframes_t ret;
    ret = pcm->callbacks->avail_update(pcm);
    return ret;
}

int snd_pcm_start(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->start(pcm);
}

int snd_pcm_pause(snd_pcm_t *pcm, int enable)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(enable), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_wait(snd_pcm_t *pcm, int timeout)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(timeout), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_mmap_begin(snd_pcm_t *pcm, const snd_pcm_channel_area_t **areas, snd_pcm_uframes_t *offset, snd_pcm_uframes_t *frames)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

snd_pcm_sframes_t snd_pcm_mmap_commit(snd_pcm_t *pcm, snd_pcm_uframes_t offset, snd_pcm_uframes_t frames)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_poll_descriptors_count(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->poll_descriptors_count(pcm);
}

int snd_pcm_poll_descriptors(snd_pcm_t *pcm, struct pollfd *pfds, unsigned int space)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->poll_descriptors(pcm, pfds, space);
}

int snd_pcm_poll_descriptors_revents(snd_pcm_t *pcm, struct pollfd *pfds, unsigned int nfds, unsigned short *revents)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

size_t snd_pcm_hw_params_sizeof(void)
{
    return sizeof(snd_pcm_hw_params_t);
}

size_t snd_pcm_sw_params_sizeof(void)
{
    return sizeof(snd_pcm_sw_params_t);
}

const char *snd_pcm_format_name(const snd_pcm_format_t format)
{
    switch(format)
    {
        case SND_PCM_FORMAT_S8:
            return "SND_PCM_FORMAT_S8";
        case SND_PCM_FORMAT_U8:
            return "SND_PCM_FORMAT_U8";
        case SND_PCM_FORMAT_S16_LE:
            return "SND_PCM_FORMAT_S16_LE";
        case SND_PCM_FORMAT_S16_BE:
            return "SND_PCM_FORMAT_S16_BE";
        case SND_PCM_FORMAT_U16_LE:
            return "SND_PCM_FORMAT_U16_LE";
        case SND_PCM_FORMAT_U16_BE:
            return "SND_PCM_FORMAT_U16_BE";
        case SND_PCM_FORMAT_S24_LE:
            return "SND_PCM_FORMAT_S24_LE";
        case SND_PCM_FORMAT_S24_BE:
            return "SND_PCM_FORMAT_S24_BE";
        case SND_PCM_FORMAT_U24_LE:
            return "SND_PCM_FORMAT_U24_LE";
        case SND_PCM_FORMAT_U24_BE:
            return "SND_PCM_FORMAT_U24_BE";
        case SND_PCM_FORMAT_S32_LE:
            return "SND_PCM_FORMAT_S32_LE";
        case SND_PCM_FORMAT_S32_BE:
            return "SND_PCM_FORMAT_S32_BE";
        case SND_PCM_FORMAT_U32_LE:
            return "SND_PCM_FORMAT_U32_LE";
        case SND_PCM_FORMAT_U32_BE:
            return "SND_PCM_FORMAT_U32_BE";
        case SND_PCM_FORMAT_FLOAT_LE:
            return "SND_PCM_FORMAT_FLOAT_LE";
        case SND_PCM_FORMAT_FLOAT_BE:
            return "SND_PCM_FORMAT_FLOAT_BE";
        case SND_PCM_FORMAT_FLOAT64_LE:
            return "SND_PCM_FORMAT_FLOAT64_LE";
        case SND_PCM_FORMAT_FLOAT64_BE:
            return "SND_PCM_FORMAT_FLOAT64_BE";
        case SND_PCM_FORMAT_IEC958_SUBFRAME_LE:
            return "SND_PCM_FORMAT_IEC958_SUBFRAME_LE";
        case SND_PCM_FORMAT_IEC958_SUBFRAME_BE:
            return "SND_PCM_FORMAT_IEC958_SUBFRAME_BE";
        case SND_PCM_FORMAT_MU_LAW:
            return "SND_PCM_FORMAT_MU_LAW";
        case SND_PCM_FORMAT_A_LAW:
            return "SND_PCM_FORMAT_A_LAW";
        case SND_PCM_FORMAT_IMA_ADPCM:
            return "SND_PCM_FORMAT_IMA_ADPCM";
        case SND_PCM_FORMAT_MPEG:
            return "SND_PCM_FORMAT_MPEG";
        case SND_PCM_FORMAT_GSM:
            return "SND_PCM_FORMAT_GSM";
        case SND_PCM_FORMAT_SPECIAL:
            return "SND_PCM_FORMAT_SPECIAL";
        case SND_PCM_FORMAT_S24_3LE:
            return "SND_PCM_FORMAT_S24_3LE";
        case SND_PCM_FORMAT_S24_3BE:
            return "SND_PCM_FORMAT_S24_3BE";
        case SND_PCM_FORMAT_U24_3LE:
            return "SND_PCM_FORMAT_U24_3LE";
        case SND_PCM_FORMAT_U24_3BE:
            return "SND_PCM_FORMAT_U24_3BE";
        case SND_PCM_FORMAT_S20_3LE:
            return "SND_PCM_FORMAT_S20_3LE";
        case SND_PCM_FORMAT_S20_3BE:
            return "SND_PCM_FORMAT_S20_3BE";
        case SND_PCM_FORMAT_U20_3LE:
            return "SND_PCM_FORMAT_U20_3LE";
        case SND_PCM_FORMAT_U20_3BE:
            return "SND_PCM_FORMAT_U20_3BE";
        case SND_PCM_FORMAT_S18_3LE:
            return "SND_PCM_FORMAT_S18_3LE";
        case SND_PCM_FORMAT_S18_3BE:
            return "SND_PCM_FORMAT_S18_3BE";
        case SND_PCM_FORMAT_U18_3LE:
            return "SND_PCM_FORMAT_U18_3LE";
        case SND_PCM_FORMAT_U18_3BE:
            return "SND_PCM_FORMAT_U18_3BE";
        default:
           return "Unknown";
    }
}

int snd_pcm_format_big_endian(snd_pcm_format_t format)
{
    switch( format ) {
        case SND_PCM_FORMAT_S8:
        case SND_PCM_FORMAT_U8:
        case SND_PCM_FORMAT_S16_BE:
        case SND_PCM_FORMAT_U16_BE:
        case SND_PCM_FORMAT_S24_BE:
        case SND_PCM_FORMAT_U24_BE:
        case SND_PCM_FORMAT_S32_BE:
        case SND_PCM_FORMAT_U32_BE:
        case SND_PCM_FORMAT_FLOAT_BE:
        case SND_PCM_FORMAT_FLOAT64_BE:
        case SND_PCM_FORMAT_IEC958_SUBFRAME_BE:
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
        case SND_PCM_FORMAT_IMA_ADPCM:
        case SND_PCM_FORMAT_MPEG:
        case SND_PCM_FORMAT_GSM:
        case SND_PCM_FORMAT_S24_3BE:
        case SND_PCM_FORMAT_U24_3BE:
        case SND_PCM_FORMAT_S20_3BE:
        case SND_PCM_FORMAT_U20_3BE:
        case SND_PCM_FORMAT_S18_3BE:
        case SND_PCM_FORMAT_U18_3BE:
            return 1;
        default:
            return 0;
    }
}

int snd_pcm_format_little_endian(snd_pcm_format_t format)
{
    switch( format ) {
        case SND_PCM_FORMAT_S8:
        case SND_PCM_FORMAT_U8:
        case SND_PCM_FORMAT_S16_LE:
        case SND_PCM_FORMAT_U16_LE:
        case SND_PCM_FORMAT_S24_LE:
        case SND_PCM_FORMAT_U24_LE:
        case SND_PCM_FORMAT_S32_LE:
        case SND_PCM_FORMAT_U32_LE:
        case SND_PCM_FORMAT_FLOAT_LE:
        case SND_PCM_FORMAT_FLOAT64_LE:
        case SND_PCM_FORMAT_IEC958_SUBFRAME_LE:
        case SND_PCM_FORMAT_MU_LAW:
        case SND_PCM_FORMAT_A_LAW:
        case SND_PCM_FORMAT_IMA_ADPCM:
        case SND_PCM_FORMAT_MPEG:
        case SND_PCM_FORMAT_GSM:
        case SND_PCM_FORMAT_S24_3LE:
        case SND_PCM_FORMAT_U24_3LE:
        case SND_PCM_FORMAT_S20_3LE:
        case SND_PCM_FORMAT_U20_3LE:
        case SND_PCM_FORMAT_S18_3LE:
        case SND_PCM_FORMAT_U18_3LE:
            return 1;
        default:
            return 0;
    }
}

int snd_pcm_format_unsigned(snd_pcm_format_t format)
{
    switch( format ) {
        case SND_PCM_FORMAT_U8:
        case SND_PCM_FORMAT_U16_LE:
        case SND_PCM_FORMAT_U16_BE:
        case SND_PCM_FORMAT_U24_LE:
        case SND_PCM_FORMAT_U24_BE:
        case SND_PCM_FORMAT_U32_LE:
        case SND_PCM_FORMAT_U32_BE:
        case SND_PCM_FORMAT_U24_3LE:
        case SND_PCM_FORMAT_U24_3BE:
        case SND_PCM_FORMAT_U20_3LE:
        case SND_PCM_FORMAT_U20_3BE:
        case SND_PCM_FORMAT_U18_3LE:
        case SND_PCM_FORMAT_U18_3BE:
            return 1;
        default:
            return 0;
    }
}

int snd_pcm_format_signed(snd_pcm_format_t format)
{
    switch( format ) {
        case SND_PCM_FORMAT_S8:
        case SND_PCM_FORMAT_S16_LE:
        case SND_PCM_FORMAT_S16_BE:
        case SND_PCM_FORMAT_S24_LE:
        case SND_PCM_FORMAT_S24_BE:
        case SND_PCM_FORMAT_S32_LE:
        case SND_PCM_FORMAT_S32_BE:
        case SND_PCM_FORMAT_FLOAT_LE:
        case SND_PCM_FORMAT_FLOAT_BE:
        case SND_PCM_FORMAT_FLOAT64_LE:
        case SND_PCM_FORMAT_FLOAT64_BE:
        case SND_PCM_FORMAT_S24_3LE:
        case SND_PCM_FORMAT_S24_3BE:
        case SND_PCM_FORMAT_S20_3LE:
        case SND_PCM_FORMAT_S20_3BE:
        case SND_PCM_FORMAT_S18_3LE:
        case SND_PCM_FORMAT_S18_3BE:
            return 1;
        default:
            return 0;
    }
}

int snd_pcm_format_linear(snd_pcm_format_t format)
{
    switch( format ) {
        case SND_PCM_FORMAT_S8:
        case SND_PCM_FORMAT_U8:
        case SND_PCM_FORMAT_S16_LE:
        case SND_PCM_FORMAT_S16_BE:
        case SND_PCM_FORMAT_U16_LE:
        case SND_PCM_FORMAT_U16_BE:
        case SND_PCM_FORMAT_S24_LE:
        case SND_PCM_FORMAT_S24_BE:
        case SND_PCM_FORMAT_U24_LE:
        case SND_PCM_FORMAT_U24_BE:
        case SND_PCM_FORMAT_S32_LE:
        case SND_PCM_FORMAT_S32_BE:
        case SND_PCM_FORMAT_U32_LE:
        case SND_PCM_FORMAT_U32_BE:
        case SND_PCM_FORMAT_S24_3LE:
        case SND_PCM_FORMAT_S24_3BE:
        case SND_PCM_FORMAT_U24_3LE:
        case SND_PCM_FORMAT_U24_3BE:
        case SND_PCM_FORMAT_S20_3LE:
        case SND_PCM_FORMAT_S20_3BE:
        case SND_PCM_FORMAT_U20_3LE:
        case SND_PCM_FORMAT_U20_3BE:
        case SND_PCM_FORMAT_S18_3LE:
        case SND_PCM_FORMAT_S18_3BE:
        case SND_PCM_FORMAT_U18_3LE:
        case SND_PCM_FORMAT_U18_3BE:
            return 1;
        default:
            return 0;
    }
}

int snd_pcm_hw_params_set_rate_resample(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(val), SLOG2_FA_END);
    return pcm->callbacks->set_rate_resample(pcm, params, val);
}

int snd_pcm_hw_params_set_buffer_time_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(*val), SLOG2_FA_END);
    int ret;
    uint64_t min, max;
    int bps;
    unsigned int best_distance = UINT_MAX;
    unsigned int best_period, best_periods, best_buffer;
    unsigned int requested_buffer_bytes;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_BUFFER_SET);
    const snd_pcm_hw_params_t *period = find_params_setting(params, PARAM_PERIOD_SET);
    const snd_pcm_hw_params_t *periods = find_params_setting(params, PARAM_PERIOD_COUNT_SET);

    // For now, don't allow setting the period unless voices/rate/format are known
    if( TEST_SET(params, (1 << PARAM_FORMAT_SET)|( 1 << PARAM_RATE_SET)|(1 << PARAM_VOICE_SET)) ) {
        bps = snd_pcm_format_width( params->format ) / 8;

        if( i ) {
            min = transform_buffer_size(params, i, i->buffer_bytes_min);
            max = transform_buffer_size(params, i, i->buffer_bytes_max);
        } else {
            min = 0;
            max = UINT_MAX;
        }

        // If period is set, it must be an integer multiple of some value in the period range
        if( period ) {
            int i, j, period_size, distance, period_min, period_max;
            int min_factor, max_factor;

            period_min = transform_buffer_size(params, period, period->period_bytes_min);
            period_max = transform_buffer_size(params, period, period->period_bytes_max);
            // Figure out what range of factors can be used on the period range and still leave a value
            // in the buffer range
            min_factor = min / period->period_bytes_max;
            max_factor = max / period->period_bytes_min;

            // Obey periods constraints
            if( periods && min_factor < periods->periods_min ) min_factor = period_min;
            if( periods && max_factor > periods->periods_max ) max_factor = period_max;
            // Apply sensible restrictions. Can't have min_factor < 1 and can't have max_factor > requested_buffer_bytes
            // Convert val to bytes in buffer space
            requested_buffer_bytes = ((unsigned long long)*val)*params->rate_min*params->voices*snd_pcm_format_width(params->format)/8000000;

            if( min_factor < 2 ) min_factor = 2;
            if( max_factor > requested_buffer_bytes ) max_factor = requested_buffer_bytes;
            for( i = min_factor; i < max_factor; i ++ ) {
                // Use this factor to get down into the period range. Round to an integer on both sides,
                // and see if the value is within the buffer range. If it is, see if it's the closest
                // value yet
                period_size = *val / i;
                // We're looking for values where i*period_size is close to *val where j is in the range of period
                // then validate that the product is in the buffer range. If nothing is in the buffer range,
                // then choose the nearest point in the buffer range
                for( j = 0; j < 2; j ++ ) {
                    if( (period_size + j) >= period_min && period_size <= period_max)
                    {
                        distance = abs( (period_size + j) * i - requested_buffer_bytes );
                        if( distance < best_distance ) {
                            best_buffer = (period_size + j) * i;
                            best_distance = distance;
                            best_periods = i;
                            best_period = period_size + j;
                        }
                    }
                }

                // The values we tried are the closest, but these may be outside the range, so also try the min and max period size in case they happen to be closest
                distance = abs( period_min * i - requested_buffer_bytes );
                if( distance < best_distance ) {
                    best_buffer = period_min * i;
                    best_distance = distance;
                    best_periods = i;
                    best_period = period_min;
                }
                distance = abs( period_max * i - requested_buffer_bytes );
                if( distance < best_distance ) {
                    best_buffer = period_max * i;
                    best_distance = distance;
                    best_periods = i;
                    best_period = period_max;
                }
            }

            if( best_distance != UINT_MAX ) {
                params->buffer_bytes_min = params->buffer_bytes_max = best_buffer;
                params->params_set |= (1 << PARAM_BUFFER_SET);
                if( (ret = snd_pcm_hw_params_set_periods(pcm, params, best_periods, 0)) != EOK ) {
                    return ret;
                }
                if( (ret = snd_pcm_hw_params_set_period_size(pcm, params, best_period / params->voices / snd_pcm_format_width( params->format ) * 8, 0)) != EOK ) {
                    return ret;
                }

                return EOK;
            } else {
                return -EINVAL;
            }
        }

        // Convert val to bytes
        requested_buffer_bytes = ((unsigned long long)*val)*params->rate_min*params->voices*snd_pcm_format_width(params->format)/8000000;

        if( requested_buffer_bytes < min ) {
            if( dir ) *dir = 1;
            *val = min * 1000000 / params->rate_min / params->channels / bps;
            params->buffer_bytes_min = params->buffer_bytes_max = min;
        } else if( requested_buffer_bytes > max ) {
            if( dir ) *dir = -1;
            *val = max * 1000000 / params->rate_min / params->channels / bps;
            params->buffer_bytes_min = params->buffer_bytes_max = max;
        } else {
            if( dir ) *dir = 0;
            params->buffer_bytes_min = params->buffer_bytes_max = requested_buffer_bytes;
        }
        params->params_set |= (1 << PARAM_BUFFER_SET);

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_set_period_time_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(*val), SLOG2_FA_END);
    uint64_t min=0, max=UINT64_MAX;
    int bps;
    unsigned int requested_period_bytes;
    const snd_pcm_hw_params_t *i = find_params_setting(params, PARAM_PERIOD_SET);

    // For now, don't allow setting the period unless voices/rate/format are known
    if( TEST_SET(params, (1 << PARAM_FORMAT_SET)|( 1 << PARAM_RATE_SET)|(1 << PARAM_VOICE_SET)) ) {
        bps = snd_pcm_format_width( params->format ) / 8;

        if( i != NULL ) {
            min = transform_buffer_size(params, i, i->period_bytes_min);
            max = transform_buffer_size(params, i, i->period_bytes_max);
        }

        // Convert val to bytes
        requested_period_bytes = ((uint64_t)(*val)) * params->rate_min * params->voices * bps / 1000000;
        if( requested_period_bytes < min ) {
            if( dir ) *dir = 1;
            *val = min * 1000000 / params->rate_min / params->channels / bps;
            params->period_bytes_min = params->period_bytes_max = min;
        } else if( requested_period_bytes > max ) {
            if( dir ) *dir = -1;
            *val = max * 1000000 / params->rate_min / params->channels / bps;
            params->period_bytes_min = params->period_bytes_max = max;
        } else {
            if( dir ) *dir = 0;
            params->period_bytes_min = params->period_bytes_max = requested_period_bytes;
        }
        params->params_set |= (1 << PARAM_PERIOD_SET);

        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_sw_params_set_avail_min(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((unsigned int)val), SLOG2_FA_END);
    // Not supported
    return EOK;
}

int snd_pcm_sw_params_set_period_event(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, int val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(val), SLOG2_FA_END);
    // We only support having a period event enabled
    if( val != 1 ) {
        return -EINVAL;
    }

    return EOK;
}

int snd_pcm_dump(snd_pcm_t *pcm, snd_output_t *out)
{
    // TODO - This is possible to implement, but will require io-audio changes
    return -EINVAL;
}

int snd_pcm_dump_hw_setup(snd_pcm_t *pcm, snd_output_t *out)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_dump_sw_setup(snd_pcm_t *pcm, snd_output_t *out)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_dump_setup(snd_pcm_t *pcm, snd_output_t *out)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_hw_params_dump(snd_pcm_hw_params_t *params, snd_output_t *out)
{
    snd_output_printf(out, "Rate: [%d,%d], channels %d formats %d (%d) startmode %d voices %d rate %d buffer_time %d period [%d,%d], buffer [%d,%d], periods [%d,%d]\n", params->rate_min, params->rate_max, params->channels, params->formats, params->format, params->startmode, params->voices, params->rate, params->buffer_time, params->period_bytes_min, params->period_bytes_max, params->buffer_bytes_min, params->buffer_bytes_max, params->periods_min, params->periods_max);
    return EOK;
}

int snd_pcm_sw_params_dump(snd_pcm_sw_params_t *params, snd_output_t *out)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_status_dump(snd_pcm_status_t *status, snd_output_t *out)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_hw_params_test_rate(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir)
{
    int i;

    //TODO: support dir parameter
    if( params->rates == NULL ) {
        return val >= params->rate_min && val <= params->rate_max;
    }

    for( i = 0; i < params->rate_count; i ++ ) {
        if( params->rates[ i ] == val ) {
            return 1;
        }
    }

    return EOK;
}

int snd_pcm_allocate_pcm( snd_pcm_t **pcmp, snd_pcm_plugin_callbacks_t *callbacks )
{
    *pcmp = malloc(sizeof(snd_pcm_t));

    if( *pcmp == NULL ) {
        return -ENOMEM;
    }

    (*pcmp)->callbacks = callbacks;
    (*pcmp)->name = NULL;

    return EOK;
}

int snd_pcm_destroy_pcm( snd_pcm_t *pcmp )
{
    free( pcmp );

    return EOK;
}

size_t snd_pcm_access_mask_sizeof(void)
{
    return sizeof(snd_pcm_access_mask_t);
}

int snd_pcm_access_mask_malloc(snd_pcm_access_mask_t **ptr)
{
    *ptr = malloc(sizeof(snd_pcm_access_mask_t));

    if( *ptr  ) {
        return EOK;
    } else {
        return -ENOMEM;
    }
}

void snd_pcm_access_mask_none(snd_pcm_access_mask_t *mask)
{
    memset( mask, 0, sizeof(snd_pcm_access_mask_t) );
}

void snd_pcm_access_mask_set(snd_pcm_access_mask_t *mask, snd_pcm_access_t val)
{
    if( val <= SND_PCM_ACCESS_LAST ) {
        mask->data |= 1 << val;
    }
}

int snd_pcm_access_mask_test(const snd_pcm_access_mask_t *mask, snd_pcm_access_t val)
{
    if( val <= SND_PCM_ACCESS_LAST ) {
        if( mask->data & (1 << val) ) {
            return EOK;
        } else {
            return -EINVAL;
        }
    } else {
        return -EINVAL;
    }
}

int snd_pcm_hw_params_test_access(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_access_t _access)
{
    if( params->access_modes & (1 << _access) ) {
        return EOK;
    }

    return -EINVAL;
}

int snd_pcm_hw_params_set_access_mask(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_access_mask_t *mask)
{
    params->access_modes = mask->data;
    return params->access_modes == 0 ? -EINVAL : 0;
}

int snd_pcm_hw_params_get_access_mask(snd_pcm_hw_params_t *params, snd_pcm_access_mask_t *mask)
{
    mask->data = params->access_modes;
    return params->access_modes == 0 ? -EINVAL : 0;
}

void snd_pcm_access_mask_free(snd_pcm_access_mask_t *obj)
{
    free( obj );
}

void snd_pcm_hw_params_get_format_mask(snd_pcm_hw_params_t *params, snd_pcm_format_mask_t *mask)
{
    mask->data = params->formats;
}

int snd_pcm_format_mask_test(const snd_pcm_format_mask_t *mask, snd_pcm_format_t val)
{
    if( val <= SND_PCM_FORMAT_LAST ) {
        if( mask->data & (1 << val) ) {
            return EOK;
        } else {
            return -EINVAL;
        }
    } else {
        return -EINVAL;
    }
}

int snd_pcm_hw_params_test_format(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_format_t val)
{
    if( params->formats & (1 << val) ) {
        return EOK;
    } else {
        return -EINVAL;
    }
}

int snd_pcm_hw_params_is_monotonic(const snd_pcm_hw_params_t *params)
{
    // io-audio doesn't return monotonic timestamps at present
    return 0;
}

int snd_pcm_hw_params_can_pause(const snd_pcm_hw_params_t *params)
{
    // TODO - The common ports all support pause, and there is no way to query
    // which ones don't. Querying should be added
    return 1;
}

int snd_pcm_sw_params_set_stop_threshold(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED((int)val), SLOG2_FA_END);
    // TODO io-audio does not support this
    return EOK;
}

int snd_pcm_hw_params_get_channels_min(const snd_pcm_hw_params_t *params, unsigned int *val)
{
    int i;
    for( i = 0; i < sizeof(params->channels) * _BITS_BYTE; i ++ ) {
        if( params->channels & (1 << i) ) {
            *val = i;
            return EOK;
        }
    }

    return -EINVAL;
}

int snd_pcm_hw_params_get_channels_max(const snd_pcm_hw_params_t *params, unsigned int *val)
{
    int i;
    for( i = sizeof(params->channels) * _BITS_BYTE; i >= 0; i -- ) {
        if( params->channels & (1 << i) ) {
            *val = i;
            return EOK;
        }
    }

    return -EINVAL;
}


int snd_pcm_area_copy(const snd_pcm_channel_area_t *dst_channel, snd_pcm_uframes_t dst_offset,
		      const snd_pcm_channel_area_t *src_channel, snd_pcm_uframes_t src_offset,
		      unsigned int samples, snd_pcm_format_t format)
{
    int width = snd_pcm_format_width( format ) / 8;

    memcpy( dst_channel->addr + dst_offset * width, src_channel->addr + src_offset * width, samples * width );

    return EOK;
}

int snd_pcm_areas_copy(const snd_pcm_channel_area_t *dst_channels, snd_pcm_uframes_t dst_offset,
		       const snd_pcm_channel_area_t *src_channels, snd_pcm_uframes_t src_offset,
		       unsigned int channels, snd_pcm_uframes_t frames, snd_pcm_format_t format)
{
    int width = snd_pcm_format_width( format ) / 8;

    memcpy( dst_channels->addr + dst_offset * width, src_channels->addr + src_offset * width, frames * width );

    return EOK;
}

int snd_pcm_area_silence(const snd_pcm_channel_area_t *dst_channel, snd_pcm_uframes_t dst_offset,
			 unsigned int samples, snd_pcm_format_t format)
{
    int width = snd_pcm_format_width( format ) / 8;

    memset( dst_channel->addr + dst_offset * width, 0, samples * width );

    return EOK;
}

int snd_pcm_areas_silence(const snd_pcm_channel_area_t *dst_channels, snd_pcm_uframes_t dst_offset,
			  unsigned int channels, snd_pcm_uframes_t frames, snd_pcm_format_t format)
{
    int width = snd_pcm_format_width( format ) / 8;

    memset( dst_channels->addr + dst_offset * width, 0, frames * width );

    return EOK;
}

int snd_pcm_area_size(snd_pcm_channel_area_t *area, int size, int channels, snd_pcm_format_t format)
{
    void *newaddr = realloc( area->addr, size * (snd_pcm_format_width( format ) >> 3) * channels );
    if( newaddr == NULL ) {
        return -ENOMEM;
    }

    area->addr = newaddr;
    area->first = 0;
    area->step = snd_pcm_format_width( format );

    return EOK;
}

const char *snd_pcm_stream_name(const snd_pcm_stream_t stream)
{
    switch(stream)
    {
        case SND_PCM_STREAM_PLAYBACK:
            return "playback";
        case SND_PCM_STREAM_CAPTURE:
            return "capture";
        default:
            return "unknown direction";
    }
}

snd_pcm_format_t snd_pcm_format_value(const char* name)
{
    if( !strcmp(name, "SND_PCM_FORMAT_S8") ) {
        return SND_PCM_FORMAT_S8;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U8") ) {
        return SND_PCM_FORMAT_U8;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S16_LE") ) {
        return SND_PCM_FORMAT_S16_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S16_BE") ) {
        return SND_PCM_FORMAT_S16_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U16_LE") ) {
        return SND_PCM_FORMAT_U16_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U16_BE") ) {
        return SND_PCM_FORMAT_U16_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S24_LE") ) {
        return SND_PCM_FORMAT_S24_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S24_BE") ) {
        return SND_PCM_FORMAT_S24_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U24_LE") ) {
        return SND_PCM_FORMAT_U24_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U24_BE") ) {
        return SND_PCM_FORMAT_U24_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S32_LE") ) {
        return SND_PCM_FORMAT_S32_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S32_BE") ) {
        return SND_PCM_FORMAT_S32_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U32_LE") ) {
        return SND_PCM_FORMAT_U32_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U32_BE") ) {
        return SND_PCM_FORMAT_U32_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_FLOAT_LE") ) {
        return SND_PCM_FORMAT_FLOAT_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_FLOAT_BE") ) {
        return SND_PCM_FORMAT_FLOAT_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_FLOAT64_LE") ) {
        return SND_PCM_FORMAT_FLOAT64_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_FLOAT64_BE") ) {
        return SND_PCM_FORMAT_FLOAT64_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_IEC958_SUBFRAME_LE") ) {
        return SND_PCM_FORMAT_IEC958_SUBFRAME_LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_IEC958_SUBFRAME_BE") ) {
        return SND_PCM_FORMAT_IEC958_SUBFRAME_BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_MU_LAW") ) {
        return SND_PCM_FORMAT_MU_LAW;
    } else if( !strcmp(name, "SND_PCM_FORMAT_A_LAW") ) {
        return SND_PCM_FORMAT_A_LAW;
    } else if( !strcmp(name, "SND_PCM_FORMAT_IMA_ADPCM") ) {
        return SND_PCM_FORMAT_IMA_ADPCM;
    } else if( !strcmp(name, "SND_PCM_FORMAT_MPEG") ) {
        return SND_PCM_FORMAT_MPEG;
    } else if( !strcmp(name, "SND_PCM_FORMAT_GSM") ) {
        return SND_PCM_FORMAT_GSM;
    } else if( !strcmp(name, "SND_PCM_FORMAT_SPECIAL") ) {
        return SND_PCM_FORMAT_SPECIAL;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S24_3LE") ) {
        return SND_PCM_FORMAT_S24_3LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S24_3BE") ) {
        return SND_PCM_FORMAT_S24_3BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U24_3LE") ) {
        return SND_PCM_FORMAT_U24_3LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U24_3BE") ) {
        return SND_PCM_FORMAT_U24_3BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S20_3LE") ) {
        return SND_PCM_FORMAT_S20_3LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S20_3BE") ) {
        return SND_PCM_FORMAT_S20_3BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U20_3LE") ) {
        return SND_PCM_FORMAT_U20_3LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U20_3BE") ) {
        return SND_PCM_FORMAT_U20_3BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S18_3LE") ) {
        return SND_PCM_FORMAT_S18_3LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S18_3BE") ) {
        return SND_PCM_FORMAT_S18_3BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U18_3LE") ) {
        return SND_PCM_FORMAT_U18_3LE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U18_3BE") ) {
        return SND_PCM_FORMAT_U18_3BE;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S16") ) {
        return SND_PCM_FORMAT_S16;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U16") ) {
        return SND_PCM_FORMAT_U16;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S24") ) {
        return SND_PCM_FORMAT_S24;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U24") ) {
        return SND_PCM_FORMAT_U24;
    } else if( !strcmp(name, "SND_PCM_FORMAT_S32") ) {
        return SND_PCM_FORMAT_S32;
    } else if( !strcmp(name, "SND_PCM_FORMAT_U32") ) {
        return SND_PCM_FORMAT_U32;
    } else if( !strcmp(name, "SND_PCM_FORMAT_FLOAT") ) {
        return SND_PCM_FORMAT_FLOAT;
    } else if( !strcmp(name, "SND_PCM_FORMAT_FLOAT64") ) {
        return SND_PCM_FORMAT_FLOAT64;
    } else if( !strcmp(name, "SND_PCM_FORMAT_IEC958_SUBFRAME") ) {
        return SND_PCM_FORMAT_IEC958_SUBFRAME;
    } else {
        return SND_PCM_FORMAT_UNKNOWN;
    }
}

int snd_pcm_info(snd_pcm_t *pcm, snd_pcm_info_t *info)
{
    return EOK;
}

const char *snd_pcm_format_description(const snd_pcm_format_t format)
{
    switch(format)
    {
        case SND_PCM_FORMAT_S8:
            return "Signed 8 bit";
        case SND_PCM_FORMAT_U8:
            return "Unsigned 8 bit";
        case SND_PCM_FORMAT_S16_LE:
            return "Signed 16 bit Little Endian";
        case SND_PCM_FORMAT_S16_BE:
            return "Signed 16 bit Big Endian";
        case SND_PCM_FORMAT_U16_LE:
            return "Unsigned 16 bit Little Endian";
        case SND_PCM_FORMAT_U16_BE:
            return "Unsigned 16 bit Big Endian";
        case SND_PCM_FORMAT_S24_LE:
            return "Signed 24 bit Little Endian using low three bytes in 32-bit word";
        case SND_PCM_FORMAT_S24_BE:
            return "Signed 24 bit Big Endian using low three bytes in 32-bit word";
        case SND_PCM_FORMAT_U24_LE:
            return "Unsigned 24 bit Little Endian using low three bytes in 32-bit word";
        case SND_PCM_FORMAT_U24_BE:
            return "Unsigned 24 bit Big Endian using low three bytes in 32-bit word";
        case SND_PCM_FORMAT_S32_LE:
            return "Signed 32 bit Little Endian";
        case SND_PCM_FORMAT_S32_BE:
            return "Signed 32 bit Big Endian";
        case SND_PCM_FORMAT_U32_LE:
            return "Unsigned 32 bit Little Endian";
        case SND_PCM_FORMAT_U32_BE:
            return "Unsigned 32 bit Big Endian";
        case SND_PCM_FORMAT_FLOAT_LE:
            return "Float 32 bit Little Endian, Range -1.0 to 1.0";
        case SND_PCM_FORMAT_FLOAT_BE:
            return "Float 32 bit Big Endian, Range -1.0 to 1.0";
        case SND_PCM_FORMAT_FLOAT64_LE:
            return "Float 64 bit Little Endian, Range -1.0 to 1.0";
        case SND_PCM_FORMAT_FLOAT64_BE:
            return "Float 64 bit Big Endian, Range -1.0 to 1.0";
        case SND_PCM_FORMAT_IEC958_SUBFRAME_LE:
            return "IEC-958 Little Endian";
        case SND_PCM_FORMAT_IEC958_SUBFRAME_BE:
            return "IEC-958 Big Endian";
        case SND_PCM_FORMAT_MU_LAW:
            return "Mu-Law";
        case SND_PCM_FORMAT_A_LAW:
            return "A-Law";
        case SND_PCM_FORMAT_IMA_ADPCM:
            return "Ima-ADPCM";
        case SND_PCM_FORMAT_MPEG:
            return "MPEG";
        case SND_PCM_FORMAT_GSM:
            return "GSM";
        case SND_PCM_FORMAT_SPECIAL:
            return "Special";
        case SND_PCM_FORMAT_S24_3LE:
            return "Signed 24bit Little Endian in 3bytes format";
        case SND_PCM_FORMAT_S24_3BE:
            return "Signed 24bit Big Endian in 3bytes format";
        case SND_PCM_FORMAT_U24_3LE:
            return "Unsigned 24bit Little Endian in 3bytes format";
        case SND_PCM_FORMAT_U24_3BE:
            return "Unsigned 24bit Big Endian in 3bytes format";
        case SND_PCM_FORMAT_S20_3LE:
            return "Signed 20bit Little Endian in 3bytes format";
        case SND_PCM_FORMAT_S20_3BE:
            return "Signed 20bit Big Endian in 3bytes format";
        case SND_PCM_FORMAT_U20_3LE:
            return "Unsigned 20bit Little Endian in 3bytes format";
        case SND_PCM_FORMAT_U20_3BE:
            return "Unsigned 20bit Big Endian in 3bytes format";
        case SND_PCM_FORMAT_S18_3LE:
            return "Signed 18bit Little Endian in 3bytes format";
        case SND_PCM_FORMAT_S18_3BE:
            return "Signed 18bit Big Endian in 3bytes format";
        case SND_PCM_FORMAT_U18_3LE:
            return "Unsigned 18bit Little Endian in 3bytes format";
        case SND_PCM_FORMAT_U18_3BE:
            return "Unsigned 18bit Big Endian in 3bytes format";
        default:
            return "Invalid";
    }
}

const char *snd_pcm_state_name(const snd_pcm_state_t state)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_avail_delay(snd_pcm_t *pcm, snd_pcm_sframes_t *availp, snd_pcm_sframes_t *delayp)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_reset(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->reset(pcm);
}

ssize_t snd_pcm_frames_to_bytes(snd_pcm_t *pcm, snd_pcm_sframes_t frames)
{
    const snd_pcm_hw_params_t *rate = find_params_setting(&pcm->hw_params, PARAM_RATE_SET);
    const snd_pcm_hw_params_t *format = find_params_setting(&pcm->hw_params, PARAM_FORMAT_SET);
    if( !rate || !format ) {
        return 0;
    }
    return frames * rate->voices * snd_pcm_format_width( format->format ) / _BITS_BYTE;
}

int snd_pcm_hw_params_set_rate(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(val), SLOG2_FA_UNSIGNED(dir), SLOG2_FA_END);
    return snd_pcm_hw_params_set_rate_near(pcm, params, &val, &dir);
}

int snd_pcm_drop(snd_pcm_t *pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s", SLOG2_FA_STRING(__func__), SLOG2_FA_END);
    return pcm->callbacks->drop(pcm);
}

snd_pcm_sframes_t snd_pcm_bytes_to_frames(snd_pcm_t *pcm, ssize_t bytes)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(bytes), SLOG2_FA_END);
    return bytes / pcm->hw_params.voices / snd_pcm_format_width( pcm->hw_params.format ) * _BITS_BYTE;
}

int snd_lib_error_set_handler(snd_lib_error_handler_t handler)
{
    snd_lib_error = handler;

    return EOK;
}

int snd_pcm_delay(snd_pcm_t *pcm, snd_pcm_sframes_t *delayp)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_hw_params_set_periods(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(val), SLOG2_FA_END);
    const snd_pcm_hw_params_t *period = find_params_setting(params, PARAM_PERIOD_SET);
    const snd_pcm_hw_params_t *buffer = find_params_setting(params, PARAM_BUFFER_SET);
    const snd_pcm_hw_params_t *count = find_params_setting(params, PARAM_PERIOD_COUNT_SET);

    if( (period == NULL || buffer == NULL) && count == NULL ) {
        // Can be anything, so accept the value
        params->periods_min = params->periods_max = val;
        params->params_set |= (1 << PARAM_PERIOD_COUNT_SET);
    } else {
        if( period != NULL && buffer != NULL ) {
            // If the period count limits aren't set yet, set them now
            if( count == NULL ) {
                count = params;
                params->params_set |= (1 << PARAM_PERIOD_COUNT_SET);
                params->periods_min = transform_buffer_size(params, buffer, buffer->buffer_bytes_min)
                                    / transform_buffer_size(params, period, period->period_bytes_max);
                params->periods_max = transform_buffer_size(params, buffer, buffer->buffer_bytes_max)
                                    / transform_buffer_size(params, period, period->period_bytes_min);
            } else {
                // Update the period count limits
                params->periods_min = MAX(count->periods_min, transform_buffer_size(params, buffer, buffer->buffer_bytes_min)
                                                            / transform_buffer_size(params, period, period->period_bytes_max));
                params->periods_max = MIN(count->periods_max, transform_buffer_size(params, buffer, buffer->buffer_bytes_max)
                                                            / transform_buffer_size(params, period, period->period_bytes_min));
            }
        } else {
            // Just take the count limits
            params->periods_min = count->periods_min;
            params->periods_max = count->periods_max;
        }

        if( val < count->periods_min && dir == 1 ) {
            params->periods_max = params->periods_min;
        } else if ( val > count->periods_max && dir == -1 ) {
            params->periods_min = params->periods_max;
        } else if ( val >= count->periods_min && val <= count->periods_max ) {
            params->periods_min = params->periods_max = val;
        } else {
            return -EINVAL;
        }
    }
    // If one of periods and buffer size is set, and the other isn't, it can be set now
    if( period == NULL && buffer != NULL ) {
        params->period_bytes_min = params->period_bytes_max = buffer->buffer_bytes_min / val;
        params->params_set |= (1 << PARAM_PERIOD_SET);
    }
    if( period != NULL && buffer == NULL ) {
        params->buffer_bytes_min = params->buffer_bytes_max = period->period_bytes_min * val;
        params->params_set |= (1 << PARAM_BUFFER_SET);
    }


    params->params_set |= (1 << PARAM_PERIOD_COUNT_SET);

    return EOK;
}

int snd_pcm_hw_params_set_periods_integer(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_hw_params_set_periods_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_sw_params_get_boundary(const snd_pcm_sw_params_t *params, snd_pcm_uframes_t *val)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_sw_params_set_silence_threshold(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_sw_params_set_silence_size(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_sw_params_set_tstamp_mode(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_tstamp_t val)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_hw_params_get_periods_max(const snd_pcm_hw_params_t *params, unsigned int *val, int *dir)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_hw_params_test_period_size(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val, int dir)
{
    UNIMPLEMENTED;
    return 0;
}

int snd_pcm_link(snd_pcm_t *pcm1, snd_pcm_t *pcm2)
{
    UNIMPLEMENTED;
    return 0;
}


int snd_pcm_unlink(snd_pcm_t *pcm)
{
    UNIMPLEMENTED;
    return 0;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/main.c $ $Rev: 781169 $")
#endif
