#include <dlfcn.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>
#include "pcm_internal.h"

int plugin_count = 0;
void **dlhandles = NULL;
struct snd_dlsym_link *snd_dlsym_start = NULL;

#define SND_PCM_PLUGIN_ENTRY_NAME(name) "_snd_pcm_" name "_open"
#define SND_PCM_RATE_PLUGIN_ENTRY_NAME(name) "_snd_pcm_rate_" name "_open"

// Workaround due to PR111938. Switch the codepath once that is resolved
char *MY_strndup(const char *s, size_t n)
{
    char *ret;
    int len;
    for( len = 0; len < n; len ++ ) {
        if( s[len] == 0 ) break;
    }

    ret = malloc(len+1);
    if( ret == NULL ) return NULL;
    memcpy(ret, s, len);
    ret[len] = 0;

    return ret;
}
#define strndup MY_strndup

struct alias_name {
    const char *name;
    const char *descriptor;
    int direction;
} alias_names [] = {
#ifdef __QNX__
    { "default", "name=plug,slave={name=rate,slave={name=hw}}", (1<<SND_PCM_STREAM_PLAYBACK)|(1<<SND_PCM_STREAM_CAPTURE) }
#else
    { "default", "name=plug,slave={name=rate,slave={name=file}}", (1<<SND_PCM_STREAM_PLAYBACK) },
    { "default", "name=plug,slave={name=rate,slave={name=sine_generator}}", (1<<SND_PCM_STREAM_CAPTURE) }
#endif
};

// TODO Where to load plugins from is yet to be determined. But we definitely
// need to load at least the local app
static int load_plugins()
{
    void **new_handles;
    plugin_count = 0;

    plugin_count ++;

    new_handles = realloc( dlhandles, sizeof(void *) * plugin_count );
    if( new_handles == NULL ) {
        return -ENOMEM;
    }

    dlhandles = new_handles;

    new_handles[ plugin_count - 1 ] = dlopen(NULL, RTLD_LAZY | RTLD_LOCAL);
    if( new_handles[ plugin_count - 1 ] == NULL ) {
        plugin_count --;
    }

    return EOK;
}

int snd_pcm_load_plugin_config( snd_pcm_t **pcmp, snd_config_t *root, snd_config_t *conf, snd_pcm_stream_t stream, int mode )
{
    int i;
    int ret;
    char *entry_buf;
    snd_config_iterator_t iter, next;
    const char *slavename = NULL;
    char *name;

    int (*plugin)(snd_pcm_t **pcmp, const char *name, snd_config_t *root, snd_config_t *conf, snd_pcm_stream_t stream, int mode);

    if( snd_config_get_type( conf ) != SND_CONFIG_TYPE_COMPOUND ) {
        return -EINVAL;
    }

    snd_config_for_each(iter, next, conf) {
        snd_config_t *n = snd_config_iterator_entry(iter);
        const char *id;
        if( snd_config_get_id(n, &id) < 0) {
            continue;
        }

        if( strcmp(id, "name") == 0 ) {
            ret = snd_config_get_string(n, &slavename);
            if( ret < 0 ) {
                return ret;
            }
        }
    }

    if( slavename == NULL ) {
        return -EINVAL;
    }

    name = strdup( slavename );
    if( name == NULL ) {
        return -ENOMEM;
    }

    if( plugin_count == 0 ) {
        load_plugins();
    }

    entry_buf = malloc(strlen(slavename)+sizeof(SND_PCM_PLUGIN_ENTRY_NAME("")));
    if( entry_buf == NULL ) {
        free( name );
        return -ENOMEM;
    }
    snprintf(entry_buf, strlen(slavename)+sizeof(SND_PCM_PLUGIN_ENTRY_NAME("")), SND_PCM_PLUGIN_ENTRY_NAME("%s"), slavename);

    ret = -EINVAL;

    for( i = 0; i < plugin_count; i ++ ) {
        plugin = dlsym(dlhandles[i], entry_buf);

        if( plugin ) {
            ret = plugin(pcmp, name, root, conf, stream, mode);
            if( ret == EOK ) {
                (*pcmp)->name = name;
            }
            break;
        }
    }
    if( ret != EOK ) {
        // No plugin found
        free( name );
    }

    free( entry_buf );
    return ret;
}

static int snd_pcm_parse_conf_str(snd_config_t *config, const char *confstr, const char **endstr)
{
    int err;
    const char *param_name, *param_val, *param_end;
    char *tempname, *tempname2;
    param_name = confstr;

    while(param_name != NULL) {
        snd_config_t *child;
        param_val = strchr(param_name, '=');
        if( param_val == NULL ) break;
        param_val ++;
        if( *param_val == '{' ) {
            tempname = strndup(param_name, param_val - param_name - 1);

            if( snd_config_make(&child, tempname, SND_CONFIG_TYPE_COMPOUND) != EOK ) {
                free( tempname );
                return -ENOMEM;
            }
            free( tempname );

            if( (err = snd_pcm_parse_conf_str(child, param_val+1, &param_name)) ) {
                snd_config_delete( child );
                return err;
            }
            param_end = strpbrk(param_name, ",}");
            if( param_end == NULL || *param_end == '}' ) {
                if( endstr ) {
                    *endstr = param_end + 1;
                }
                param_name = NULL;
                param_end = NULL;
            } else {
                param_name = param_end + 1;
            }
        } else {
            param_end = strpbrk(param_val, ",}");
            tempname = strndup(param_name, param_val - param_name - 1);
            if( param_end ) {
                tempname2 = strndup(param_val, param_end - param_val);
            } else {
                tempname2 = strdup(param_val);
            }
            if( snd_config_imake_string(&child, tempname, tempname2) != EOK ) {
                free( tempname );
                free( tempname2 );
                return -ENOMEM;
            }
            free( tempname );
            free( tempname2 );
            if( param_end == NULL || *param_end == '}' ) {
                if( endstr ) {
                    *endstr = param_end + 1;
                }
                param_name = NULL;
                param_end = NULL;
            } else {
                param_name = param_end + 1;
            }
        }
        snd_config_add( config, child );
    }

    return EOK;
}

int snd_pcm_load_plugin( snd_pcm_t **pcmp, const char *name, snd_pcm_stream_t stream, int mode )
{
    int ret;
    snd_config_t *config;

    if( snd_config_make(&config, "slave", SND_CONFIG_TYPE_COMPOUND) != EOK ) {
        return -ENOMEM;
    }

    if( name != NULL ) {
        // First, check if there are any special characters. If not, then it's an alias rather than
        // a full config
        if( strchr(name, '=') ) {
            snd_pcm_parse_conf_str(config, name, NULL);
        } else {
            int i;
            const char *alias_name = NULL;

            for( i = 0; i < sizeof(alias_names)/sizeof(struct alias_name); i ++ ) {
                if( !strcmp( alias_names[ i ].name, name ) && ((1 << stream) & alias_names[ i ].direction) ) {
                    alias_name = alias_names[ i ].descriptor;
                    break;
                }
            }
            if( alias_name ) {
                snd_pcm_parse_conf_str(config, alias_name, NULL);
            } else {
                snd_config_delete( config );
                return -EINVAL;
            }
        }
    }

    if( plugin_count == 0 ) {
        load_plugins();
    }

    ret = snd_pcm_load_plugin_config( pcmp, config, config, stream, mode );

    snd_config_delete( config );

    return ret;
}

int (*snd_pcm_get_rate_plugin( const char *name ))(unsigned int, void **, snd_pcm_rate_ops_t *)
{
    int i;
    int namelen;
    char *namebuf;
    char *buf;

    int (*plugin)(unsigned int version, void **objp, snd_pcm_rate_ops_t *ops) = NULL;
    // Figure out which device to open
    char *param_start = strchr(name, ':');

    if( param_start ) {
        namelen = param_start - name;
        namebuf = strndup(name, namelen);
    } else {
        namelen = strlen(name);
        namebuf = strdup(name);
    }

    if( namebuf == NULL ) {
        return NULL;
    }

    buf = malloc(namelen+sizeof(SND_PCM_RATE_PLUGIN_ENTRY_NAME("")));

    if( plugin_count == 0 ) {
        load_plugins();
    }

    if( buf ) {
        snprintf(buf, namelen+sizeof(SND_PCM_RATE_PLUGIN_ENTRY_NAME("")), SND_PCM_RATE_PLUGIN_ENTRY_NAME("%s"), namebuf);
    }
    for( i = 0; i < plugin_count; i ++ ) {
        if( buf ) {
            plugin = dlsym(dlhandles[i], buf);

            if( plugin ) {
                break;
            }
        }

        if( buf ) {
            plugin = dlsym(dlhandles[i], buf);
            break;
        }
    }

    free(namebuf);
    free(buf);
    return plugin;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/plugins.c $ $Rev: 759464 $")
#endif
